/*
* cxtest - cxtest version 0.1
* Copyright (C)  2005 Seo, Won Ho. All rights reserved.
*
* This file is a part of the cxtest (https://sourceforge.net/projects/cxtest).
* The use and distribution terms for this software are covered by the
* Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
* which can be found in the file epl-v10.html at the root of this distribution.
* By using this software in any fashion, you are agreeing to be bound by
* the terms of this license. You must not remove this notice, or
* any other, from this software.
*/

/*===========================================================================

FILE: TestApp.c
===========================================================================*/


/*===============================================================================
INCLUDES AND VARIABLE DEFINITIONS
=============================================================================== */
//#define WIN_DEBUG
#ifdef WIN_DEBUG
#include <wtypes.h>		// only needed for DebugBreak()
#include <winbase.h>	// only needed for DebugBreak()
#endif /* WIN_DEBUG */

#include "testapp.h"
#if defined(BREW_STATIC_APP)
#include "OEMFeatures.h"
#endif

#define CONFIG_FILE_NAME "testapp.cfg"

#define LOG_FILE_NAME "testapp_log.log"
#define SUMMARY_FILE_NAME "result.txt"

#define LF_CHAR 10 // Line Feed
#define FILE_BUFFER_SIZE 256
#define LINE_BUFFER_SIZE 100

/*-------------------------------------------------------------------
Function Prototypes
-------------------------------------------------------------------*/
static boolean  TestApp_HandleEvent(IApplet * pi, AEEEvent eCode, uint16 wParam, uint32 dwParam);
static boolean  TestApp_InitData (IApplet * pMe);

static boolean  TestApp_Init (TestApp * pMe);
static boolean  TestApp_initMainControls(TestApp* pApp);

static void     TestApp_FreeAppData (TestApp * pMe);
static void     TestApp_releaseObj(TestApp* pApp, void ** ppObj);
static void     TestApp_freeMainControls(TestApp* pApp);

static void     TestApp_displayMainMenu(TestApp *pMe);
static void     TestApp_displayTestMenu(TestApp * pMe);
static void     TestApp_displayProgress( TestApp * pMe, char * bench, char * test, int perct );
static void     TestApp_displayOutput (IApplet * pMe, int nline, char *pszStr, AEEFont font);
static void     TestApp_displaySummaryScreen(TestApp* pMe);
static void     TestApp_displayHelpScreen(TestApp * pApp);

       void     TestApp_reset(TestApp* pMe);
       
       void     TestApp_notifyManualTimerCB(void * pApp);
       void     TestApp_notifyAutoTimerCB(void * pApp);

static boolean  TestApp_initializeLogging(TestApp* pMe);
static void     TestApp_closeLogging(TestApp* pMe);
static int      TestApp_getLine(char *s, char* t);
static boolean  TestApp_loadConfigFile(TestApp* pMe);
static boolean  TestApp_writeConfig(TestApp* pMe, const char * testInfo, uint32 score);
static void     TestApp_getFileError(IFileMgr *p);

static void     TestApp_prompt(TestApp * pApp, 
                            const AECHAR* szTitle, uint16 wIDTitle,  
                            const AECHAR* szText, uint16 wIDText,
                            const char* pFileName,                         
                            uint16* wButtons, uint16 wDefID , 
                            uint32 dwProps) ;


#if defined(BREW_STATIC_APP)
int         TestAppMod_CreateInstance(AEECLSID ClsId,IShell * pIShell,IModule * po,void ** ppObj);
static int  TestApp_Load(IShell *ps, void * pHelpers, IModule ** pMod);

// Constant Data...
static const AEEAppInfo    gaiTestApp = { AEECLSID_TESTCXTEST, 
                                          TESTCXTEST_RES_FILE,
                                          IDS_APP_NAME,
                                          0,
                                          0, 
                                          0,
                                          0,
                                          AFLAG_GAME};
                      

PFNMODENTRY TestApp_GetModInfo(IShell * ps, AEECLSID ** ppClasses, AEEAppInfo ** pApps, uint16 * pnApps,uint16 * pwMinPriv)
{
  *pApps = (AEEAppInfo *)&gaiTestApp;
  *pnApps = 1;
  return((PFNMODENTRY)GUniMon_Load);
}

static int TestApp_Load(IShell *ps, void * pHelpers, IModule ** pMod)
{
  return(AEEStaticMod_New((int16)(sizeof(AEEMod)),ps,pHelpers,pMod, TestAppMod_CreateInstance,NULL));
}

#endif  //BREW_STATIC_APP

/*===============================================================================
FUNCTION DEFINITIONS
=============================================================================== */

/*===========================================================================

FUNCTION: AEEClsCreateInstance

DESCRIPTION
   This function is invoked while the app is being loaded. All Modules must provide this
   function. Ensure to retain the same name and parameters for this function.
   In here, the module must verify the ClassID and then invoke the AEEApplet_New() function
   that has been provided in AEEAppGen.c.

   After invoking AEEApplet_New(), this function can do app specific initialization. In this
   example, a generic structure is provided so that app developers need not change app specific
   initialization section every time except for a call to IDisplay_InitAppData().
   This is done as follows: InitAppData() is called to initialize AppletData
   instance. It is app developers responsibility to fill-in app data initialization
   code of InitAppData(). App developer is also responsible to release memory
   allocated for data contained in AppletData -- this can be done in
   IDisplay_FreeAppData().

PROTOTYPE:
   int AEEClsCreateInstance(AEECLSID ClsId,IShell * pIShell,IModule * po,void ** ppObj)

PARAMETERS:
   clsID: [in]: Specifies the ClassID of the applet which is being loaded

   pIShell: [in]: Contains pointer to the IShell object.

   pIModule: pin]: Contains pointer to the IModule object to the current module to which
   this app belongs

   ppObj: [out]: On return, *ppObj must point to a valid IApplet structure. Allocation
   of memory for this structure and initializing the base data members is done by AEEApplet_New().

DEPENDENCIES
  none

RETURN VALUE
  AEE_SUCCESS: If the app needs to be loaded and if AEEApplet_New() invocation was
     successful
  EFAILED: If the app does not need to be loaded or if errors occurred in
     AEEApplet_New(). If this function returns FALSE, the app will not be loaded.

SIDE EFFECTS
  none
===========================================================================*/
#if defined(BREW_STATIC_APP)
int TestAppMod_CreateInstance(AEECLSID ClsId,IShell * pIShell,IModule * po,void ** ppObj)
{
  *ppObj = NULL;

  if(ClsId == AEECLSID_TESTCXTEST){
    if(AEEApplet_New(sizeof(TestApp), ClsId, pIShell,po,(IApplet**)ppObj,
                     (AEEHANDLER)TestApp_HandleEvent,(PFNFREEAPPDATA)TestApp_FreeAppData)
       == TRUE)
    {
      if (TestApp_InitData ((IApplet *) * ppObj) == TRUE)
        return (AEE_SUCCESS);
    }
  }
  return (EFAILED);
}
#else
int AEEClsCreateInstance(AEECLSID ClsId,IShell * pIShell,IModule * po,void ** ppObj)
{
  *ppObj = NULL;

  if(ClsId == AEECLSID_TESTCXTEST){
    if(AEEApplet_New(sizeof(TestApp), ClsId, pIShell,po,(IApplet**)ppObj,
                     (AEEHANDLER)TestApp_HandleEvent,(PFNFREEAPPDATA)TestApp_FreeAppData)
       == TRUE)
    {
      if (TestApp_InitData ((IApplet *) * ppObj) == TRUE)
        return (AEE_SUCCESS);
    }
  }
  return (EFAILED);
}
#endif


/*===========================================================================

FUNCTION TestApp_HandleEvent

DESCRIPTION
   This is the EventHandler for this app. All events to this app are handled in this
   function. All APPs must supply an Event Handler.

PROTOTYPE:
   boolean TestApp_HandleEvent(IApplet * pi, AEEEvent eCode, uint16 wParam, uint32 dwParam)

PARAMETERS:
   pi: Pointer to the AEEApplet structure. This structure contains information specific
   to this applet. It was initialized during the AEEClsCreateInstance() function.

   ecode: Specifies the Event sent to this applet

   wParam, dwParam: Event specific data.

DEPENDENCIES
  none

RETURN VALUE
  TRUE: If the app has processed the event
  FALSE: If the app did not process the event

SIDE EFFECTS
  none
===========================================================================*/
static boolean TestApp_HandleEvent(IApplet * pi, AEEEvent eCode, uint16 wParam, uint32 dwParam)
{
  TestApp *pMe = (TestApp *) pi;
  AEERect   qrc;   //working screen

  // Get the device info and set the screen rect so that it can be used
  // by all tests here.
  qrc.x    = 0;
  qrc.y    = 0;
  qrc.dx   = pMe->m_di.cxScreen;
  qrc.dy   = pMe->m_di.cyScreen;


  // Let each of the controls have a crack at the event.  If one of them is able to handle
  // the event then return TRUE indicating that the event has been handled.

    if (pMe->m_pIStatic && ISTATIC_HandleEvent(pMe->m_pIStatic, eCode, wParam, dwParam))
     return TRUE;

    if (pMe->m_pIMainMenu && IMENUCTL_HandleEvent (pMe->m_pIMainMenu, eCode, wParam, dwParam))
      {
        return TRUE;
      }
      if (pMe->m_pITestMenu && IMENUCTL_HandleEvent (pMe->m_pITestMenu, eCode, wParam, dwParam))
      {
        return TRUE;
      }
    if (pMe->m_pSK && IMENUCTL_HandleEvent(pMe->m_pSK, eCode, wParam, dwParam))
      {
        return TRUE;
      }

  switch (eCode)
  {
  case EVT_APP_START:
    pMe->activeTestSuite = 0;
    if (TestApp_Init(pMe) == FALSE) {	
		return FALSE;
	}

    ITestSuite_addTest(pMe->m_pITestSuite, TestWasRun_initUnitTestSuite(pMe));

    pMe->m_pActiveTestNode = ITestSuite_getHeadPosition(pMe->m_pITestSuite);
    
    if (pMe->m_bAutoMode) 
    {
      IDISPLAY_ClearScreen(pMe->a.m_pIDisplay);
      pMe->EndOfTestCB = TestApp_notifyAutoTimerCB;
      ISHELL_SetTimer(pMe->a.m_pIShell, pMe->m_nTimerDuration, TestApp_notifyAutoTimerCB,(void *)pMe);
    }
    else // manual mode
    {  
      TestApp_displayMainMenu (pMe);
    }

            
    return (TRUE);

  case EVT_APP_STOP:
    ISHELL_CancelTimer(pMe->a.m_pIShell, TestApp_notifyManualTimerCB, (void *)pMe);
    IDISPLAY_EraseRect (pMe->a.m_pIDisplay, &qrc);  //Clear the display
    return (TRUE);

  case EVT_APP_NO_SLEEP:
    return TRUE;

  case EVT_KEY:
    switch (wParam)
    {
      
    case AVK_CLR:
      if (IMENUCTL_IsActive(pMe->m_pIMainMenu)  || pMe->m_bAutoMode)
      {
        // Main menu already active. Stop the applet
        return FALSE;
      }

      else if (IMENUCTL_IsActive(pMe->m_pITestMenu))
      {
        // back to Main menu
        //IMENUCTL_SetActive(pMe->m_pITestMenu, FALSE);
        //IMENUCTL_SetActive(pMe->m_pIMainMenu, TRUE);
        TestApp_displayMainMenu(pMe);
      }
      else if (!pMe->m_bAutoMode && pMe->m_nExecutionNum == 0)
      {
        //TestApp_reset(pMe);
        TestApp_displayMainMenu(pMe);
        ISHELL_CancelTimer(pMe->a.m_pIShell, TestApp_notifyManualTimerCB, (void *)pMe);
        pMe->EndOfTestCB = NULL;
      }

      return TRUE;

      
    default:
        break;
    }

  case EVT_COMMAND:
    // Control will come here after a API benchmark test is selected from the
    // main menu. There is a delay of CALLBACK_DURATION between successive
    // benchmark tests. A benchmark test is invoked from a timer callback
    // function.

    //if test is from very beginning--main menu
    if(IMENUCTL_IsActive(pMe->m_pIMainMenu))
    {
      if(wParam == USAGE_BENCHMARK_API)
      {
        //IMENUCTL_SetActive(pMe->m_pIMainMenu, FALSE);
        //TestApp_buildTestMenu(pMe);
        TestApp_displayTestMenu(pMe);
        return TRUE;
      }
      else if (wParam == USAGE_BENCHMARK_SETTING)
      {
        IMENUCTL_SetActive(pMe->m_pIMainMenu, FALSE);
        TestApp_displayHelpScreen(pMe);
        return TRUE;
      }
      else
      {
        return TRUE;
      }
    }

    else if(IMENUCTL_IsActive(pMe->m_pITestMenu))
    {
        // Initialize to the first iteration
      pMe->m_nRepeat = 0;

        // select test suite
      if( wParam == ITestSuite_getChildTestCount(pMe->m_pITestSuite))
      {
         pMe->m_IsTestAll = TRUE;
         pMe->m_pActiveTestNode = ITestSuite_getHeadPosition(pMe->m_pITestSuite);
      }
      else
      {
          //  ASSERT(pMe->m_IsTestAll == FALSE);
          pMe->m_pActiveTestNode = ITestSuite_findIndex(pMe->m_pITestSuite, wParam);  
      }

      // start the callback timer
      pMe->EndOfTestCB = TestApp_notifyManualTimerCB;

      ISHELL_SetTimer(pMe->a.m_pIShell, CALLBACK_DURATION,
        TestApp_notifyManualTimerCB, (void*) pMe);

        // Display the main TestApp menu
      IMENUCTL_SetActive(pMe->m_pITestMenu, FALSE);
      IDISPLAY_EraseRect(pMe->a.m_pIDisplay, &qrc);
      TestApp_displayProgress( pMe, NULL, NULL, 0 );
    }
    else if (IMENUCTL_IsActive(pMe->m_pSK))
    {
        if (wParam == IDC_OK)
        {
            //TestApp_displayMainMenu(pMe);
            //TestApp_reset(pMe);
            TestApp_displayMainMenu(pMe);
            return TRUE;   
        }
    }
    return TRUE;
  }
  return (FALSE);
}

/*===============================================================================
                     APP DATA ALLOC-INIT/FREE FUNCTION DEFINITIONS
=============================================================================== */

/*===========================================================================

FUNCTION: TestApp_InitData

DESCRIPTION:
    This function initializes app specific data, allocates memory for app
    data. App developer needs to fill-in the initialization code.

PROTOTYPE:
    boolean TestApp_InitData(IApplet* pi);

PARAMETERS:
    pi [in]: Pointer to the IApplet structure. This structure contains
    information specific to this applet. It was initialized during the
    AEEClsCreateInstance() function.

DEPENDENCIES:
    Assumes pMe is not NULL

RETURN VALUE
    TRUE: If the app has app data is allocated and initialized successfully
    FALSE: Either app data could not be allocated or initialzied

SIDE EFFECTS:
    none
===========================================================================*/
static boolean
TestApp_InitData (IApplet * pi)
{
  int charHeight;
  TestApp *pMe = (TestApp *) pi;

  // init configuration file variables
  pMe->m_pConfigFile = NULL;
  pMe->m_nExecutionNum = NUM_EXECUTION; // run once
  pMe->m_nTimerDuration = 200; // 200 ms
  pMe->m_bOnline = FALSE;
  pMe->m_bLogging = TRUE;
  pMe->m_bAutoMode = TRUE; //FALSE;

  // Initialize the MenuCtl pointer to NULL
  pMe->m_pIMainMenu = NULL;

  ISHELL_GetDeviceInfo (pMe->a.m_pIShell, &pMe->m_di);

  charHeight = IDISPLAY_GetFontMetrics(pMe->a.m_pIDisplay, AEE_FONT_NORMAL, NULL, NULL);

  //if it is toucan display 176*220, with 40 phone infor bar
  if(pMe->m_di.cxScreen ==176 && pMe->m_di.cyScreen >= 180)
  {
    pMe->dispRect.x = 10;
    pMe->dispRect.y = charHeight*3 + 5;
    pMe->dispRect.dx = pMe->m_di.cxScreen-20;
    pMe->dispRect.dy = pMe->m_di.cyScreen - charHeight*5 - 10;
  }
  //else display size are either 100*100 or very samll one
  else{
    //if screen size is bigger than 100*(2*charHeight + 100) 
    //[one charHeight for progress bar, one for display name]
    //set display screen to 100 * 100
    //else set to device screen size
    if(pMe->m_di.cxScreen > 100)
      pMe->dispRect.x = (pMe->m_di.cxScreen - 100)/2;
    else
      pMe->dispRect.x = 0;
	   
    if (pMe->m_di.cyScreen > (2*charHeight + 100))
    {
      pMe->dispRect.y = pMe->m_di.cyScreen - charHeight*2 - 100;
      pMe->dispRect.dy = 100;
      pMe->dispRect.dx = 100;
    }
    else
    {
      pMe->dispRect.y  = charHeight*2;
      pMe->dispRect.dy = pMe->m_di.cyScreen - pMe->dispRect.y;
      pMe->dispRect.dx = pMe->m_di.cxScreen - pMe->dispRect.x;
    }
  }

    //
    //
    CX_TESTSUITE_NEW(pMe->m_pITestSuite, " TestAll");

    CX_TESTRESULT_NEW(pMe->m_pITestResult, NULL);


    SETAEERECT( &pMe->m_rScreenRect, 0, 0 /* charHeight * 2 */, pMe->m_di.cxScreen, pMe->m_di.cyScreen - charHeight * 2 );
    
  return TRUE;
}


/*===========================================================================
FUNCTION: TestApp_Init

DESCRIPTION:
    This function allocates the classes for all the services required by
   TestApp.

PROTOTYPE:
    void TestApp_Init(TestApp * pMe);

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance

RETURN VALUE:
    None
===========================================================================*/
boolean TestApp_Init( TestApp * pMe )
{
    char *pErrorMessage = NULL;
    
#ifdef WIN_DEBUG
    DebugBreak();
#endif
    
    pMe->m_pIFileMgr = NULL;
    pMe->m_pIHeap = NULL;
    pMe->m_pIMainMenu = NULL;
    pMe->m_pITestMenu = NULL;
    
    if(ISHELL_CreateInstance (pMe->a.m_pIShell, AEECLSID_FILEMGR, 
        (void **) &pMe->m_pIFileMgr) != SUCCESS )
    {
        pErrorMessage = "IFileMgr instantiation failed";
    }
    
    else if (!TestApp_loadConfigFile(pMe))
    {
        pErrorMessage = "Error loading config file "CONFIG_FILE_NAME;
    }
    
    else if (ISHELL_CreateInstance(pMe->a.m_pIShell, AEECLSID_HEAP,
        (void **)&pMe->m_pIHeap) != SUCCESS)
    {
        pErrorMessage = "IHeap instantiation failed";
    }
    else if (!TestApp_initMainControls(pMe))
    {
        pErrorMessage = "IMenuCtl, etc. instantiation failed";
    }
    
    if (pErrorMessage)
    {
        TestApp_displayTextLine(pMe, pErrorMessage );
        
        TestApp_FreeAppData( pMe );
        
        return FALSE;
    }

    if (pMe->m_bLogging)
    {
        TestApp_initializeLogging(pMe);
    }
    
    
    // Set Run State
    pMe->eRunState = RUN_STATE_RUNNING;
    return TRUE;
}

/*===========================================================================

FUNCTION: TestApp_createMainControls

DESCRIPTION:
	Creates controls used by the applet

PARAMETERS:
	pApp [in] - Pointer to the CExpenseTrackerApp structure. This structure contains 
    information specific to this applet. 

DEPENDENCIES:
   Assumes pApp is not NULL

RETURN VALUE:
   TRUE - Controls created successfully
   FALSE - Control creation failed

SIDE EFFECTS:
   None
===========================================================================*/
static boolean  TestApp_initMainControls(TestApp* pApp)
{
   // Create the main application controls.
   if((ISHELL_CreateInstance(pApp->a.m_pIShell, AEECLSID_MENUCTL, (void**)(&pApp->m_pIMainMenu)) != SUCCESS) ||
      (ISHELL_CreateInstance(pApp->a.m_pIShell, AEECLSID_MENUCTL, (void**)(&pApp->m_pITestMenu)) != SUCCESS) ||
      (ISHELL_CreateInstance(pApp->a.m_pIShell, AEECLSID_SOFTKEYCTL, (void**)(&pApp->m_pSK)) != SUCCESS) ||
      (ISHELL_CreateInstance(pApp->a.m_pIShell, AEECLSID_STATIC, (void**)(&pApp->m_pIStatic)) != SUCCESS))
   {
      return FALSE;
   }

  return TRUE;   
}

/*===========================================================================
FUNCTION TestApp_FreeAppData

DESCRIPTION
    This function frees any data initialized in the Init function.

PROTOTYPE:
    void TestApp_FreeAppData(IApplet* pi);

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance

RETURN VALUE:
    None
===========================================================================*/
void TestApp_FreeAppData (TestApp * pMe)
{
  // Set App Run State to Shutdown to prevent other tasks from starting up new frames.
  pMe->eRunState = RUN_STATE_SHUTDOWN;

    //
    //    
    if (pMe->m_pITestResult) {
        CX_TESTRESULT_DELETE(pMe->m_pITestResult);
        pMe->m_pITestResult = NULL;
    }
    
    if (pMe->m_pITestSuite) {
        CX_TESTSUITE_DELETE(pMe->m_pITestSuite);
        pMe->m_pITestSuite = NULL;
    }
    
    TestApp_freeMainControls(pMe);

  if (pMe->m_pIHeap)
  {
    IHEAP_Release(pMe->m_pIHeap);
    pMe->m_pIHeap = NULL;
  }

  if (pMe->m_bLogging)
  {
    TestApp_closeLogging(pMe);
  }

  if (pMe->m_pConfigFile)
  {
    IFILE_Release(pMe->m_pConfigFile);
    pMe->m_pConfigFile = NULL;
  }

  if(pMe->m_pIFileMgr)
  {
    IFILEMGR_Release(pMe->m_pIFileMgr);
    pMe->m_pIFileMgr = NULL;
  }


}

/*===========================================================================

FUNCTION: TestApp_ReleaseObj

DESCRIPTION:
  This is a convenience function which releases BREW controls.  It also
  checks that the pointer is not NULL before attempting to release the
  control.

PARAMETERS:
	ReleaseObj [in/out] - Pointer to the data pointer of the control.

DEPENDENCIES:
   None

RETURN VALUE:
   None

SIDE EFFECTS:
   None
===========================================================================*/
static void TestApp_releaseObj(TestApp* pApp, void ** ppObj)
{
   if (pApp && ppObj && *ppObj ) 
   {
      (void)IBASE_Release(((IBase *)*ppObj));
      *ppObj = NULL;
   }
}

/*===========================================================================

FUNCTION: ET_ReleaseMainControls

DESCRIPTION:
  Frees each of the controls used by the application.

PARAMETERS:
	pApp [in] - Applet Structure

DEPENDENCIES:
   Assumes pApp is not NULL

RETURN VALUE:
   None

SIDE EFFECTS:
   None
===========================================================================*/
static void TestApp_freeMainControls(TestApp* pApp)
{   
   // Release each of the controls
   TestApp_releaseObj(pApp, (void**)&pApp->m_pIMainMenu);
   TestApp_releaseObj(pApp, (void**)&pApp->m_pITestMenu);
   TestApp_releaseObj(pApp, (void**)&pApp->m_pIStatic);
   TestApp_releaseObj(pApp, (void**)&pApp->m_pSK);

}
/*===========================================================================
                     APP HELPER FUNCTION DEFINITIONS
 ============================================================================*/
/*===========================================================================
FUNCTION TestApp_displayMainMenu

DESCRIPTION
    This function builds the TestApp main menu.
   it contains "test API" and it also can launch other application to measure
   the performance
PROTOTYPE:
    void TestApp_displayMainMenu(TestApp* pi);

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance

RETURN VALUE:
    None
===========================================================================*/
static void TestApp_displayMainMenu (TestApp * pMe)
{
  AEERect qrc;
  AECHAR szBuf[MAX_STRLEN];

  TestApp_reset(pMe);
    
  // Set display rectangle
  SETAEERECT (&qrc, 0, 0, pMe->m_di.cxScreen, pMe->m_di.cyScreen);
  IDISPLAY_EraseRect (pMe->a.m_pIDisplay, &qrc);  //Clear the display

  STR_TO_WSTR (" cxTestApp ", szBuf, sizeof(szBuf));
  IMENUCTL_SetTitle (pMe->m_pIMainMenu, NULL, 0, szBuf);

  // Set rectangle for the menu
  IMENUCTL_SetRect (pMe->m_pIMainMenu, &qrc);

  STR_TO_WSTR (" Test API", szBuf, sizeof(szBuf));
  IMENUCTL_AddItem (pMe->m_pIMainMenu, 0, 0, USAGE_BENCHMARK_API, szBuf, 0);

  STR_TO_WSTR (" Settings", szBuf, sizeof(szBuf));
  IMENUCTL_AddItem (pMe->m_pIMainMenu, 0, 0, USAGE_BENCHMARK_SETTING, szBuf, 0);
 
  IMENUCTL_SetActive (pMe->m_pIMainMenu, TRUE);

  pMe->m_eAppState = APP_STATE_MAINMENU;
}
/*===========================================================================
FUNCTION TestApp_buildTestMenu

DESCRIPTION
    This function builds the API test menu. it contains "test all" and test each api
PROTOTYPE:
    void TestApp_buildTestMenu(IApplet* pi);

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance

RETURN VALUE:
    None
===========================================================================*/

static void TestApp_displayTestMenu(TestApp * pMe)
{
    AECHAR wszBuf[MAX_STRLEN];
    uint16 nCount;
    uint16 nIndex;
    ITestSuite* pts;
    ITestNode* ptn;
    ITest* pCurTest;

    TestApp_reset(pMe);
    
    pts = pMe->m_pITestSuite;
	nCount = ITestSuite_getChildTestCount(pts);
	
	STR_TO_WSTR (" Test All", wszBuf, sizeof(wszBuf));
	IMENUCTL_AddItem (pMe->m_pITestMenu, 0, 0, nCount, wszBuf, 0);

	nIndex = 0;
	ptn = ITestSuite_getHeadPosition(pts);  
	while (ptn != NULL)
	{ 
		pCurTest = ITestSuite_getNext(pts, &ptn);
		
		STR_TO_WSTR(ITest_name(pCurTest), wszBuf, sizeof(wszBuf));
		IMENUCTL_AddItem(pMe->m_pITestMenu, 0,0, nIndex, wszBuf, 0);
		++nIndex;    
	}       
	
	IMENUCTL_SetActive (pMe->m_pITestMenu, TRUE);

    pMe->m_eAppState = APP_STATE_TESTMENU;
}

/*===========================================================================

FUNCTION: TestApp_displayOutput

DESCRIPTION
    This function displays an output string at a given line number on the
    screen. If the nline parameter is a negative value (-1) the string
    is displayed in the middle of the screen. If the "nline" value is larger
    than or equal to zero the "nline" value is multiplied by 15 and the
    resulting value in pixels is set to the y-coordinate of the start of
    the string display on the screen. If the string does not fit on one line
    the string wraps around to the next line (spaced rougly 10-15 pixels apart).
    By default 5 is used as the starting the x-coordinate of a displayed
    string.

    How many characters that fit on one line is calculated for each line
    that is wrapped around to the next line.

    Note: depending on the phone screen size and the fonts used for characters
          the output might differ on different handsets (skins). Where some
          handsets will have a smaller screen and large default fonts which will
          cause partial overlapping of lines. This function does not try to address
          these issues (this is meant as a simple display function).

PROTOTYPE:
   void TestApp_displayOutput(IApplet * pi, int nline, char *pszStr, AEEFont font)

PARAMETERS:
   pi:     [in]: Contains a pointer to the IApplet class.
   nline:  [in]: Contains the line number to start displaying the text.
        The line numbers are by default spaced 15 pixels apart along the y-axis.
   pszStr: [in]: The character string to be displayed on the screen.

DEPENDENCIES
   None

RETURN VALUE
   None

SIDE EFFECTS
   None

===========================================================================*/
void
TestApp_displayOutput (IApplet * pi, int nline, char *pszStr, AEEFont font)
{
  AEEDeviceInfo di;           // Device Info
  AECHAR szBuf[100] = { 0 };  // a buffer that supports 200 char string
  AECHAR *psz = NULL;
  AEERect rc;
  int pixelWidth;
  int pnFits = 0, dy;
  int totalCh = 0;
  int charHeight = 0;         // Stores the char height in pixels for given font
  int pnAscent = 0;           // Stores the ascent in number of pixels
  int pnDescent = 0;          // Stores the descent in number of pixels
  AEEApplet *pMe = (AEEApplet *) pi;
  if (pMe == NULL)
    return;

  // Get device information
  ISHELL_GetDeviceInfo (pMe->m_pIShell, &di);

  // Get the font metrics info
  charHeight = IDISPLAY_GetFontMetrics (pMe->m_pIDisplay, font, &pnAscent, &pnDescent);

  // Convert to wide string (unicode)
  STR_TO_WSTR ((char *) pszStr, szBuf, sizeof (szBuf));

  // If nlines is zero then print this string starting around the middle of
  // the screen. Or else multiply nlines by 10 to decide the y coordinate of
  // the start of the string.
  if (nline < 0)
  {
    dy = di.cyScreen * 2 / 5;
  }
  else
  {
    dy = nline * charHeight;
  }

  // psz keeps track of the point from which to write from the string buffer
  // in case the string does not fit one line and needs to wrap around in the
  // next line.
  psz = szBuf;

  // Need to calculate the lotal string length to decide if any wrapping
  // around is needed.
  totalCh = STRLEN ((char *) pszStr);

  // Keep displaying text string on multiple lines if the string can't be displayed
  // on one single line. Lines are spaced 15 pixels apart.
  while ((totalCh > 0) && (*psz != NULL))
  {
    // Get information on how many characters will fit in a line.
    // Give the pointer to the buffer to be displayed, and the number of
    // pixels along the x axis you want to display the string in (max number)
    // pnFits will have the max number of chars that will fit in the maxWidth
    // number of pixels (given string can't fit in one line), or the number of
    // chars in the string (if it does fit in one line). pnWidth gives the
    // number of pixels that will be used to display pnFits number of chars.
    pixelWidth = IDISPLAY_MeasureTextEx (pMe->m_pIDisplay, font, (AECHAR *) psz,
                                         -1, di.cxScreen - 5,
                                         &pnFits);  // Number of chars that will fit a line

    // If pnFits is zero there is something wrong in the input to above function.
    // Normally this scenario should not occur. But, have the check anyway.
    if (pnFits == 0)
      return;

    SETAEERECT(&rc, 5, dy, di.cxScreen, charHeight);
    IDISPLAY_EraseRect(pMe->m_pIDisplay, &rc);
    IDISPLAY_DrawText (pMe->m_pIDisplay, font, psz, pnFits,
                       5 /*start dx */ ,
                       dy, 0 /* use default rectangle coordinates */ ,
                       IDF_ALIGN_CENTER);
    psz += pnFits;          // move pointer to the next segment to be displayed
    totalCh -= pnFits;      // reduce the total number of characters to still display
    dy += charHeight;       // Place next line charHeight pixels below the
    // previous line.
    IDISPLAY_Update (pMe->m_pIDisplay);
    if (totalCh < pnFits)
      pnFits = totalCh;   // if total number is less than pnFits, adjust pnFits
  }
  return;
}

/*===========================================================================
FUNCTION: TestApp_displayTextLine

DESCRIPTION
    This function displays a lines of text on the screen, taking into account
    the number of text lines printed so far and the number of lines available
    for output.

    If all available lines are exhausted, this function returns without doing
    anything.

    If the last available line of the screen is reached, this function prints
    "..." to indicate that some lines may not have been printed due to lack
    of space.

    Otherwise, this function prints the line on the screen by calling DisplayOutput,
    and updates the count of lines printed based on how many lines DisplayOutput
    used to print the text.


PROTOTYPE:
    void TestApp_displayTextLine(TestApp * pMe,  char *pszStr)

PARAMETERS:
    pMe:   [in]: Contains a pointer to this app's data structure.
    pszStr:    [in]: The character string to be displayed on the screen.

DEPENDENCIES
    Assumes the pMe->m_cLineNum gets initialized to the starting line for text display
    on the screen, and that pMe->m_cMaxLine contains the last available line for
    output on the screen.

RETURN VALUE
    None.

SIDE EFFECTS
    None

===========================================================================*/
void TestApp_displayTextLine (TestApp * pMe, char *pszStr)
{
  int charHeight;

  charHeight = IDISPLAY_GetFontMetrics (pMe->a.m_pIDisplay, AEE_FONT_NORMAL, NULL, NULL);
  pMe->m_cLineNum++;
  if ((pMe->m_di.cyScreen / charHeight) < pMe->m_cLineNum)
    pMe->m_cLineNum = 0;

  TestApp_displayOutput ((IApplet *) pMe, pMe->m_cLineNum, pszStr, AEE_FONT_NORMAL);
  return;
}

/*===========================================================================
FUNCTION: TestApp_displayProgress

DESCRIPTION:
    This function displays the progress of the test case by displaying progress
   bar. This function is provided to the user to get the feeling that
   benchmark is in progress.

PROTOTYPE:
    void TestApp_displayProgress(TestApp * pMe);

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance

RETURN VALUE:
    None
===========================================================================*/
void TestApp_displayProgress( TestApp * pMe, char * bench, char * test, int perct )
{
  AEERect rc;
   
  // If the phone screen width is less than 100 pixel, then set the scroll
  // bar width to 100 pixels and adjust the percentage of progress accordingly.
  // If phone screen width is more than 100 pixels then set the scroll bar
  // width to 100 pixels.
  rc.x = 5;
  rc.dx = pMe->m_di.cxScreen - 10;
  perct = (perct*(pMe->m_di.cxScreen-10))/100;

  // Display the progress bar in the middle of the phone screen.
  //rc.y  = pMe->di.cyScreen/2;
  rc.dy = IDISPLAY_GetFontMetrics(pMe->a.m_pIDisplay, AEE_FONT_BOLD, NULL, NULL);
  rc.y = pMe->dispRect.y - 2 * rc.dy;
  rc.dy = rc.dy/2;
  IDISPLAY_DrawFrame(pMe->a.m_pIDisplay, &rc, AEE_FT_INDENT, CLR_SYS_SCROLLBAR);
  rc.dx = perct;

  // Fill the rectangle to display the progress of the benchmark tests.
  IDISPLAY_FillRect(pMe->a.m_pIDisplay, &rc, CLR_SYS_SCROLLBAR_FILL);
  IDISPLAY_Update(pMe->a.m_pIDisplay);
}

void TestApp_displayMessage(TestApp* pMe, char *message, boolean top)
{
  AECHAR dest[64];
  int16 nY;
  int charHeight;

  charHeight = IDISPLAY_GetFontMetrics(pMe->a.m_pIDisplay, AEE_FONT_BOLD, NULL, NULL);
  if (top) {
    nY = 2;
  } else {
    nY = pMe->m_di.cyScreen/charHeight - 1;
  }

  STR_TO_WSTR(message, dest, 64);
  TestApp_displayOutput ((IApplet *) pMe,  nY, message, AEE_FONT_NORMAL);
}

void TestApp_displayAutoTest(TestApp *pMe, const char* str)
{
  AECHAR strMessage[MAX_STRLEN];

  STR_TO_WSTR (str, strMessage, sizeof(strMessage));
  IDISPLAY_DrawText(pMe->a.m_pIDisplay,    // Display instance
                    AEE_FONT_BOLD,       // Use normal font
                    strMessage,              // Text - Normally comes from resource
                    -1,                  // -1 = Use full string length
                    0,                   // Ignored - IDF_ALIGN_CENTER
                    0,                   // Ignored - IDF_ALIGN_MIDDLE
                    NULL,                // No clipping
                    IDF_ALIGN_CENTER | IDF_ALIGN_MIDDLE);
  IDISPLAY_UpdateEx(pMe->a.m_pIDisplay, FALSE);
}

static void TestApp_displaySummaryScreen(TestApp* pMe)
{
    uint16 wButtons[2];
    
    // Display the prompt screen.
    
    MEMSET(wButtons, 0, sizeof(wButtons));
    wButtons[0] = IDC_OK;

   TestApp_prompt(pMe, 
        NULL, IDC_HELP,
       NULL/* szText */, 0 /* wIDText*/, LOG_FILE_NAME, 
       wButtons, wButtons[0], ST_ASCII|ST_NOSCROLL);
    
    pMe->m_eAppState = APP_STATE_SUMMARY;
}

/*===========================================================================

FUNCTION: TestApp_displayHelpScreen

DESCRIPTION:
	This function displays the application's help screen.

PARAMETERS:
	pApp [in] - Pointer to the CExpenseTrackerApp structure. This structure contains 
    information specific to this applet. 

DEPENDENCIES:
   None

RETURN VALUE:
   None

SIDE EFFECTS:
  Causes the phone display to be updated.
  Sets the application state to APP_STATE_HELP.
===========================================================================*/
static void TestApp_displayHelpScreen(TestApp * pApp)
{
   uint16 wButtons[2];

   // Display the prompt screen.

   MEMSET(wButtons, 0, sizeof(wButtons));
   wButtons[0] = IDC_OK;
   
   TestApp_prompt(pApp, 
        NULL, IDC_HELP,
       NULL/* szText */, 0 /* wIDText*/, "help.txt", 
       wButtons, wButtons[0], ST_ASCII|ST_NOSCROLL);

   pApp->m_eAppState = APP_STATE_HELP;

   
}

/*===========================================================================
FUNCTION: TestApp_reset

DESCRIPTION:
   Go back to API menu and reset the tests

PROTOTYPE:
    void TestApp_reset(TestApp * pMe);

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance

RETURN VALUE:
    None
===========================================================================*/
void TestApp_reset(TestApp* pMe)
{
    ITestNode* pNode;
    ITestSuite* pITestSuite;

    // reset active child test index for each test suite.
    pNode = ITestSuite_getHeadPosition(pMe->m_pITestSuite);
    while (pNode != NULL)
    {
        pITestSuite = (ITestSuite*)ITestSuite_getNext(pMe->m_pITestSuite, &pNode);

        ITestSuite_setActiveChildTestIndex(pITestSuite, 0);                    
    }
    
    //reset testall falg
    pMe->m_IsTestAll = FALSE;
    pMe->m_pActiveTestNode = ITestSuite_getHeadPosition(pMe->m_pITestSuite);
  
    // Reset Menu Controls
    // This function instructs the menu control to reset (free or delete)
    // its contents and to immediately leave active/focus mode.
    IMENUCTL_Reset(pMe->m_pIMainMenu);
    IMENUCTL_Reset(pMe->m_pITestMenu);
    IMENUCTL_Reset(pMe->m_pSK);

    // disable other controls
    IMENUCTL_SetActive(pMe->m_pIMainMenu, FALSE);
    IMENUCTL_SetActive(pMe->m_pITestMenu, FALSE);

    // release static object
    TestApp_releaseObj(pMe, (void**)&pMe->m_pIStatic);

    // clear screen
    // IDISPLAY_ClearScreen(pMe->a.m_pIDisplay);
}

/*===========================================================================
FUNCTION: TestApp_notifyManualTimerCB

DESCRIPTION:
    This function is a callback function associated with the timer. Benchmark
    tests will be invoked from this function.

PROTOTYPE:
    void TestApp_notifyManualTimerCB(TestApp * pMe);

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance

RETURN VALUE:
    None
===========================================================================*/
void TestApp_notifyManualTimerCB (void * pApp)
{
    ITestSuite* pats; // Active Test Suite
  TestApp * pMe = (TestApp *)pApp;
  int    iPerct;
  //int i;
  //TestSuite * bt;
  int theActiveTestCase;
  int theTotalTestCases;
  int charHeight;
  AEERect qrc;

  //IDISPLAY_ClearScreen(pMe->a.m_pIDisplay);
  charHeight = IDISPLAY_GetFontMetrics(pMe->a.m_pIDisplay, AEE_FONT_BOLD, NULL, NULL);
  SETAEERECT(&qrc, 0, 0, pMe->m_di.cxScreen, charHeight);
  IDISPLAY_EraseRect(pMe->a.m_pIDisplay, &qrc);

  //bt = &pMe->bench[pMe->activeTestSuite];
  //theActiveTestCase = bt->activeTestCase;
  //theTotalTestCases  = bt->totalTestCases;

  pats = (ITestSuite*) ITestSuite_getAt(pMe->m_pITestSuite, pMe->m_pActiveTestNode);
  theActiveTestCase = ITestSuite_getActiveChildTestIndex(pats);
  theTotalTestCases = ITestSuite_getChildTestCount(pats);
  
  if (theActiveTestCase >= theTotalTestCases)
  {
    if (pMe->m_IsTestAll) // We need to run all the tests
    {
      // Make next active test
      //pMe->activeTestSuite++;
      //pMe->m_pActiveTestNode = 
      ITestSuite_getNext(pMe->m_pITestSuite, &pMe->m_pActiveTestNode);

      if( pMe->m_pActiveTestNode == NULL)
      {
        if (pMe->m_nRepeat+1 < pMe->m_nExecutionNum )
        {
            ITestNode* pNode;
            ITestSuite* pITestSuite;

            // reset active child test index for each test suite.
            pNode = ITestSuite_getHeadPosition(pMe->m_pITestSuite);
            while (pNode != NULL)
            {
                pITestSuite = (ITestSuite*)ITestSuite_getNext(pMe->m_pITestSuite, &pNode);

                ITestSuite_setActiveChildTestIndex(pITestSuite, 0);                    
            }

            pMe->m_pActiveTestNode = ITestSuite_getHeadPosition(pMe->m_pITestSuite);
            pMe->m_nRepeat++;

            IDISPLAY_EraseRect(pMe->a.m_pIDisplay, &pMe->dispRect);
            ISHELL_SetTimer(pMe->a.m_pIShell, CALLBACK_DURATION, TestApp_notifyManualTimerCB,(void *)pMe);
        }
        
        // END OF TEST, DO NOT SET TIMER
        else
        {
          pMe->m_nRepeat = 0;
          //GUniMonGetBenchmarkResult(pMe);
          //TestApp_reset(pMe);
          //IMENUCTL_SetActive(pMe->m_pITestMenu, TRUE);

          //  IDISPLAY_ClearScreen(pMe->a.m_pIDisplay);
          //TestApp_displayTextLine(pMe, "Test completed");
          TestApp_displaySummaryScreen(pMe);
        }
        
        return; 
      } // end  of if

      // more test suites to run
      else
      {
        IDISPLAY_EraseRect(pMe->a.m_pIDisplay, &pMe->dispRect);
        TestApp_endOfTest( pMe );
        
        return;
      }
      
    } // END OF if (pMe->m_IsTestAll)

    // individual test suite selected
    if (pMe->m_nRepeat+1 < pMe->m_nExecutionNum )
    {
        
        ITestSuite_setActiveChildTestIndex(pats, 0);
      //bt->activeTestCase = 0;
        IDISPLAY_EraseRect(pMe->a.m_pIDisplay, &pMe->dispRect);
        TestApp_displayProgress( pMe, NULL, NULL, 0 );
        ISHELL_SetTimer(pMe->a.m_pIShell, CALLBACK_DURATION, TestApp_notifyManualTimerCB,(void *)pMe);
        pMe->m_nRepeat++;
        
      return;
    }

    // Reset the active test
    // end of individual test, do not reset timer 
    //bt->activeTestCase = 0;
    ITestSuite_setActiveChildTestIndex(pats, 0);
    //TestApp_reset(pMe);
    TestApp_displaySummaryScreen(pMe);
    return;
  }
  else // still more tests within a suite
  {
    ITestCase* pActiveTestCase;
    
    // clear screen
    IDISPLAY_ClearScreen(pMe->a.m_pIDisplay);

    pActiveTestCase = (ITestCase*)ITestSuite_getChildTestAt(pats, theActiveTestCase); 
    // display test suite name on top
      TestApp_displayOutput((IApplet *)pMe, 0, 
        ITestSuite_name(pats), AEE_FONT_NORMAL);

    // display test case name on bottom
      TestApp_displayOutput((IApplet *)pMe, pMe->m_di.cyScreen/charHeight - 1, 
        ITestCase_name(pActiveTestCase), AEE_FONT_NORMAL);

      if (pMe->m_IsTestAll)
        // Calculate how much percentage of the total execution time is complete
        iPerct = ((theActiveTestCase+1)*100)/theTotalTestCases;
      else
        // Calculate how much percentage of the total execution time is complete
        iPerct = ((theTotalTestCases*pMe->m_nRepeat + (theActiveTestCase+1))*100)/(theTotalTestCases*pMe->m_nExecutionNum );

      TestApp_displayProgress(pMe,NULL,NULL, iPerct);

      ITestCase_run(pActiveTestCase, pMe->m_pITestResult);

      //
      // next test case 
      {
        theActiveTestCase++;
        ITestSuite_setActiveChildTestIndex(pats, theActiveTestCase);
      }
  }

  TestApp_endOfTest(pMe);
  /* If it's not an extended test then immediately queue up the next test */
  //if (!bt->ExtendedTest)
  //{
  //  TestApp_endOfTest( pMe );
  // }
}
/*===========================================================================
FUNCTION: TestApp_notifyAutoTimerCB

DESCRIPTION:
    This function is a callback function associated with the timer. Benchmark
    tests will be invoked from this function.

PROTOTYPE:
    void TestApp_notifyManualTimerCB(TestApp * pMe);

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance

RETURN VALUE:
    None
===========================================================================*/
void TestApp_notifyAutoTimerCB(void * pApp)
{
    ITestSuite* pats; // Active Test Suite
  TestApp * pMe = (TestApp *)pApp;
  int    iPerct;
  //int i;
  //TestSuite * bt;
  int theActiveTestCase;
  int theTotalTestCases;
  int charHeight;
  AEERect qrc;

  //IDISPLAY_ClearScreen(pMe->a.m_pIDisplay);
  charHeight = IDISPLAY_GetFontMetrics(pMe->a.m_pIDisplay, AEE_FONT_BOLD, NULL, NULL);
  SETAEERECT(&qrc, 0, 0, pMe->m_di.cxScreen, charHeight);
  IDISPLAY_EraseRect(pMe->a.m_pIDisplay, &qrc);

  //bt = &pMe->bench[pMe->activeTestSuite];
  //theActiveTestCase = bt->activeTestCase;
  //theTotalTestCases  = bt->totalTestCases;

  pats = (ITestSuite*) ITestSuite_getAt(pMe->m_pITestSuite, pMe->m_pActiveTestNode);
  theActiveTestCase = ITestSuite_getActiveChildTestIndex(pats);
  theTotalTestCases = ITestSuite_getChildTestCount(pats);
  
  if (theActiveTestCase >= theTotalTestCases)
  {
    if (pMe->m_IsTestAll) // We need to run all the tests
    {
      // Make next active test
      //pMe->activeTestSuite++;
      //pMe->m_pActiveTestNode = 
      ITestSuite_getNext(pMe->m_pITestSuite, &pMe->m_pActiveTestNode);

      if( pMe->m_pActiveTestNode == NULL)
      {
        if (pMe->m_nRepeat+1 < pMe->m_nExecutionNum )
        {
            ITestNode* pNode;
            ITestSuite* pITestSuite;

            // reset active child test index for each test suite.
            pNode = ITestSuite_getHeadPosition(pMe->m_pITestSuite);
            while (pNode != NULL)
            {
                pITestSuite = (ITestSuite*)ITestSuite_getNext(pMe->m_pITestSuite, &pNode);

                ITestSuite_setActiveChildTestIndex(pITestSuite, 0);                    
            }

            pMe->m_pActiveTestNode = ITestSuite_getHeadPosition(pMe->m_pITestSuite);
            pMe->m_nRepeat++;

            IDISPLAY_EraseRect(pMe->a.m_pIDisplay, &pMe->dispRect);
            ISHELL_SetTimer(pMe->a.m_pIShell, CALLBACK_DURATION, TestApp_notifyManualTimerCB,(void *)pMe);
        }
        
        // END OF TEST, DO NOT SET TIMER
        else
        {
          pMe->m_nRepeat = 0;
          //GUniMonGetBenchmarkResult(pMe);
          //TestApp_reset(pMe);
          //IMENUCTL_SetActive(pMe->m_pITestMenu, TRUE);

          //  IDISPLAY_ClearScreen(pMe->a.m_pIDisplay);
          //TestApp_displayTextLine(pMe, "Test completed");
          TestApp_displaySummaryScreen(pMe);
        }
        
        return; 
      } // end  of if

      // more test suites to run
      else
      {
        IDISPLAY_EraseRect(pMe->a.m_pIDisplay, &pMe->dispRect);
        TestApp_endOfTest( pMe );
        
        return;
      }
      
    } // END OF if (pMe->m_IsTestAll)

    // individual test suite selected
    if (pMe->m_nRepeat+1 < pMe->m_nExecutionNum )
    {
        
        ITestSuite_setActiveChildTestIndex(pats, 0);
      //bt->activeTestCase = 0;
        IDISPLAY_EraseRect(pMe->a.m_pIDisplay, &pMe->dispRect);
        TestApp_displayProgress( pMe, NULL, NULL, 0 );
        ISHELL_SetTimer(pMe->a.m_pIShell, CALLBACK_DURATION, TestApp_notifyManualTimerCB,(void *)pMe);
        pMe->m_nRepeat++;
        
      return;
    }

    // Reset the active test
    // end of individual test, do not reset timer 
    //bt->activeTestCase = 0;
    ITestSuite_setActiveChildTestIndex(pats, 0);
    //TestApp_reset(pMe);
    TestApp_displaySummaryScreen(pMe);
    return;
  }
  else // still more tests within a suite
  {
    ITestCase* pActiveTestCase;
    
    // clear screen
    IDISPLAY_ClearScreen(pMe->a.m_pIDisplay);

    pActiveTestCase = (ITestCase*)ITestSuite_getChildTestAt(pats, theActiveTestCase); 
    // display test suite name on top
      TestApp_displayOutput((IApplet *)pMe, 0, 
        ITestSuite_name(pats), AEE_FONT_NORMAL);

    // display test case name on bottom
      TestApp_displayOutput((IApplet *)pMe, pMe->m_di.cyScreen/charHeight - 1, 
        ITestCase_name(pActiveTestCase), AEE_FONT_NORMAL);

      if (pMe->m_IsTestAll)
        // Calculate how much percentage of the total execution time is complete
        iPerct = ((theActiveTestCase+1)*100)/theTotalTestCases;
      else
        // Calculate how much percentage of the total execution time is complete
        iPerct = ((theTotalTestCases*pMe->m_nRepeat + (theActiveTestCase+1))*100)/(theTotalTestCases*pMe->m_nExecutionNum );

      TestApp_displayProgress(pMe,NULL,NULL, iPerct);

      ITestCase_run(pActiveTestCase, pMe->m_pITestResult);

      //
      // next test case 
      {
        theActiveTestCase++;
        ITestSuite_setActiveChildTestIndex(pats, theActiveTestCase);
      }
  }

  TestApp_endOfTest(pMe);
  /* If it's not an extended test then immediately queue up the next test */
  //if (!bt->ExtendedTest)
  //{
  //  TestApp_endOfTest( pMe );
  // }    
}

void TestApp_endOfTest( TestApp *pMe )
{
  if (pMe->EndOfTestCB)
  {
    ISHELL_SetTimer(pMe->a.m_pIShell, CALLBACK_DURATION,
                    pMe->EndOfTestCB,(void *)pMe);
  }
}

void TestApp_writeLog(TestApp* pMe, const char * testInfo, uint32 score)
{
  char lineBuffer[LINE_BUFFER_SIZE];
  int length;

  if (pMe->m_bLogging && pMe->m_pLogFile)
  {    
    length = SPRINTF( lineBuffer, "%s,%d\n", testInfo, score );
    IFILE_Write(pMe->m_pLogFile, lineBuffer, length);
  }
}

void TestApp_writeLogString(TestApp* pMe, const char* testInfo)
{
  char lineBuffer[LINE_BUFFER_SIZE];
  int length;
  
  if (pMe->m_bLogging && pMe->m_pLogFile)
  {    
    length = SPRINTF( lineBuffer, "%s\n", testInfo );
    IFILE_Write(pMe->m_pLogFile, lineBuffer, length);
  }
}

/*===================================================================

FUNCTION Log TestApp_writeLogFile

DESCRIPTION
  This funcion generates, updates the log file.
DEPENDENCIES
  none

RETURN VALUE
  none

SIDE EFFECTS
  none
====================================================================*/
void TestApp_writeLogFile(TestApp* pMe, const char * testInfo, const char * testInfo2, int retval)
{
  char fileBuffer[100];
  char *resultText[5] = {
        "SUCCESS", 
        "FAIL", 
        "UNTESTED", 
        "INSPECT",
        "UNSUPPORTED"};
        
  int length;

  if (pMe->m_pLogFile)
  {
    if (retval < 0 || retval > 4)
    {
      retval = 4;                         /* unsupported */
    }

    length = SPRINTF( fileBuffer, "%s%s,%s\n", testInfo, testInfo2,
                      resultText[retval] );

    IFILE_Write(pMe->m_pLogFile, fileBuffer, length);
  }
}

static boolean TestApp_initializeLogging(TestApp* pMe)
{
  if(IFILEMGR_Test(pMe->m_pIFileMgr, LOG_FILE_NAME)==SUCCESS)
  {
    if(IFILEMGR_Remove(pMe->m_pIFileMgr, LOG_FILE_NAME)!=SUCCESS)
      return FALSE;
  }

  pMe->m_pLogFile = IFILEMGR_OpenFile(pMe->m_pIFileMgr, LOG_FILE_NAME,
                                      _OFM_CREATE);

  if (pMe->m_pLogFile)
  {
//TestApp_writeLogString(pMe, "Test starts at timestamp", GET_UPTIMEMS());

//TestApp_writeLogString(pMe, "\t\t --------------TestApp Unit Tests Log File ---------"); 
//TestApp_writeLogString(pMe, "\n");

    TestApp_writeLog(pMe, "Test starts at timestamp", GET_UPTIMEMS());

    TestApp_writeLogString(pMe, "   --TestApp Log File --");
    TestApp_writeLogString(pMe, "\n");    
//    TestApp_writeLogString(pMe, "Manufacture: XXX");
//    TestApp_writeLogString(pMe, "Model #: Model ");
//    TestApp_writeLogString(pMe, "Chipset: MSM6xxx ");
//    TestApp_writeLogString(pMe, "CPU Clock: 32MHz");
//    TestApp_writeLogString(pMe, "SRAM I/F, Qnty & Speed: 16-bit, 4Mbit, 70nsec");
//    TestApp_writeLogString(pMe, "Flash I/F, Qnty, & Speed: 16-bit, 32Mbit, 90nsec");
//    TestApp_writeLogString(pMe, "LCD I/F & format: 16-bit, 565");   
    TestApp_writeLog(pMe, "Screen width", pMe->m_di.cxScreen);
    TestApp_writeLog(pMe, "Screen height", pMe->m_di.cyScreen);
    TestApp_writeLogString(pMe, "\n");
    return TRUE;
  }
  return FALSE;
}

static void TestApp_closeLogging(TestApp* pMe)
{
  if (pMe->m_pLogFile)
  {
    TestApp_writeLogString(pMe, "   --END OF LOG--\n");
    TestApp_writeLog(pMe, "Test ends at timestamp", GET_UPTIMEMS());
    
    IFILE_Release(pMe->m_pLogFile);
    pMe->m_pLogFile = NULL;
  }
}

/*===========================================================================
FUNCTION: MobiMonWriteConfig

DESCRIPTION:
    This function write to the configuration file

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance
    char *strBuf - string
    uint32 nValue - value

RETURN VALUE:
    TRUE    - if successful
    FALSE   - if failed
===========================================================================*/
static boolean TestApp_writeConfig(TestApp* pMe, const char *strBuf, uint32 nValue)
{
  char fileBuffer[100];

  SPRINTF( fileBuffer, "%s,%d\n", strBuf, nValue );

  if (IFILE_Write(pMe->m_pConfigFile, fileBuffer, STRLEN(fileBuffer)) == 0)
    return FALSE;

  return TRUE;
}

/*===========================================================================
FUNCTION: TestApp_getLine

DESCRIPTION:
    This function reads data from buffer t line by line, into array s. 
    It replaces the terminating newline with '\0'

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance
    char *s - buffer to read into
    char *t - buffer to read from

RETURN VALUE:
    i    - position of the first character in the next line
===========================================================================*/
static int TestApp_getLine(char *s, char* t)
{
  int i = 0;
  while (t[i] != LF_CHAR) // while not end of line, i.e LF=10
  {
    s[i] = t[i];
    i++;
  }
  s[i++] = '\0'; // null-terminated string
  return i;
}

/*===========================================================================
FUNCTION: TestApp_loadConfigFile

DESCRIPTION:
    This function loads values from the configuration file
    If no config file exist, a new one will be created with default values

PARAMETERS:
    TestApp * pMe - Pointer to the applet instance


RETURN VALUE:
    TRUE    - if successful
    FALSE   - if failed
===========================================================================*/
static boolean TestApp_loadConfigFile(TestApp* pMe)
{
  char* ConfigFields[] =
    {"ExecutionNum",
     "TimerDuration",
     "Online",
     "Logging", 
     "AutoMode",
    };   
#define NUM_CONFIG_FIELDS (sizeof(ConfigFields) / sizeof(ConfigFields[0]))
  int nValue = -1;
  char fileBuffer[FILE_BUFFER_SIZE];
  char lineBuffer[LINE_BUFFER_SIZE];
  uint32 nPos = 0;
  uint32 nInd = 0;
  int i = 0;
  char* pDest = NULL;
  FileInfo fileInfo;
  MEMSET(fileBuffer, 0, sizeof( fileBuffer));
  MEMSET(lineBuffer, 0, sizeof(lineBuffer));

  if (pMe->m_pIFileMgr == NULL)
  {
    if(ISHELL_CreateInstance (pMe->a.m_pIShell, AEECLSID_FILEMGR, 
                              (void **) &pMe->m_pIFileMgr) != SUCCESS )
      return FALSE;
  }

  if (IFILEMGR_Test(pMe->m_pIFileMgr, CONFIG_FILE_NAME) == SUCCESS) 
  {
    pMe->m_pConfigFile = IFILEMGR_OpenFile(pMe->m_pIFileMgr, CONFIG_FILE_NAME, _OFM_READWRITE);
    if (pMe->m_pConfigFile == NULL)
    {
      TestApp_getFileError(pMe->m_pIFileMgr);
      return FALSE;
    }
    // Parse data from config file
    if (IFILE_GetInfo(pMe->m_pConfigFile, &fileInfo) != SUCCESS) 
      return FALSE;

    if (IFILE_Read(pMe->m_pConfigFile, fileBuffer, fileInfo.dwSize) == 0) // Error: failed to read
      return FALSE;
      
    while (nInd < fileInfo.dwSize)
    {
      nPos = TestApp_getLine(lineBuffer, fileBuffer + nInd);
      nInd += nPos; // next index
         
      // search for matching field name
      for (i=0; i< NUM_CONFIG_FIELDS; i++)
      {
        if (STRSTR(lineBuffer, ConfigFields[i]))
        {
          break;
        }
      }
      // Error: there is no match for field name
      if (i >= NUM_CONFIG_FIELDS)
        return FALSE;

      pDest = STRCHR(lineBuffer, ',');
      if (pDest == NULL) // Error: no comma found
        return FALSE;

      nValue = ATOI(pDest+1);
      if (nValue <0) // Error: invalid value 
        return FALSE;

      switch(i)
      {
      case 0:
        pMe->m_nExecutionNum = nValue;
        break;
      case 1:
        pMe->m_nTimerDuration = nValue;
        break;
      case 2:
        pMe->m_bOnline = (boolean)nValue;
        break;
      case 3:
        pMe->m_bLogging  = (boolean)nValue;
        break;
      case 4:
        pMe->m_bAutoMode = (boolean)nValue;
        break;

      default:
        break;
      }
    } // END OF WHILE
  } // END OF IF
 
/*
  else // otherwise create a config file with defaut values
  {
    pMe->m_pConfigFile = IFILEMGR_OpenFile(pMe->m_pIFileMgr, CONFIG_FILE_NAME, _OFM_CREATE);//_OFM_CREATE);

    if (pMe->m_pConfigFile == NULL)
    {
      //mobimon_getFileError(pMe->m_pIFileMgr);
      return FALSE;
    }

    if (  !TestApp_writeConfig(pMe, "ExecutionNum", pMe->m_nExecutionNum) || 
          !TestApp_writeConfig(pMe, "TimerDuration", pMe->m_nTimerDuration) ||
          !TestApp_writeConfig(pMe, "Online", (int)pMe->m_bOnline) ||
          !TestApp_writeConfig(pMe, "Logging", (int)pMe->m_bLogging) ||
          !TestApp_writeConfig(pMe, "AutoMode", (int)pMe->m_bAutoMode) 
          )

      return FALSE;
  } // END OF ELSE
*/

  return TRUE;
}

static void  TestApp_getFileError(IFileMgr *p)
{
	int i =IFILEMGR_GetLastError(p);
	switch (i)
	{
	case (EFILEEXISTS):
	//	TestApp_displayTextLine(pMe, "EFS failed because EFILEEXISTS");
		break;
	case (EFILENOEXISTS):
	//	TestApp_displayTextLine(pMe, "EFS failed because EFILENOEXISTS");
		break;
	case (EDIRNOTEMPTY):
	//	TestApp_displayTextLine(pMe, "EFS failed because EDIRNOTEMPTY");
		break;
	case (EBADFILENAME):
	//	TestApp_displayTextLine(pMe, "EFS failed because EBADFILENAME");
		break;
	case (EBADSEEKPOS):
	//	TestApp_displayTextLine(pMe, "EFS failed because EBADSEEKPOS");
		break;
	case (EFILEEOF):
	//	TestApp_displayTextLine(pMe, "EFS failed because EFILEEOF");
		break;
	case (EFSFULL):
	//	TestApp_displayTextLine(pMe, "EFS failed because EFSFULL");
		break;
	case (EFILEOPEN):
	//	TestApp_displayTextLine(pMe, "EFS failed because EFILEOPEN");
		break;
	case (EBADPARM):
	//	TestApp_displayTextLine(pMe, "EFS failed because EBADPARM");
		break;
	}
	return;
}


/*===========================================================================

FUNCTION: TestApp_prompt

DESCRIPTION:
	This function displays a prompt on the screen consisting of a static
  control to display text and on or more buttons at the bottom of the 
  screen.

PARAMETERS:
	pApp [in] - Pointer to the CExpenseTrackerApp structure. This structure contains 
    information specific to this applet. 

	pApp [in] -	The CExpenseTrackerApp

  szTitle [in] - Title string.

  wIDTitle [in] - Title string resource ID.

  szText [in] - Text string.

  wIDText [in] - Text string resource ID.

  pFileName [in] - If valid, text is streamed from a file.

  wButtons [in] - Array of button resource IDs.

  wIDDef [in] - Resource ID of default button.

  dwProps [in] - Static control properties.

DEPENDENCIES:
   None

RETURN VALUE:
   None

SIDE EFFECTS:
  Causes the phone display to be updated.
===========================================================================*/
static void     TestApp_prompt(TestApp * pApp, 
                            const AECHAR* szTitle, uint16 wIDTitle,  
                            const AECHAR* szText, uint16 wIDText,
                            const char* pFileName,                         
                            uint16* wButtons, uint16 wDefID , 
                            uint32 dwProps) 
{
	AEERect rc;
	AECHAR * szTitleBuf = NULL;
	AECHAR * szTextBuf = NULL;
  IFile * pIFile = NULL;
  IFileMgr* pIFileMgr = NULL;

  TestApp_reset(pApp);

   // Clear Screen
   IDISPLAY_ClearScreen(pApp->a.m_pIDisplay);


   // Create Static control for prompt
	if (ISHELL_CreateInstance(pApp->a.m_pIShell, AEECLSID_STATIC, (void**)&pApp->m_pIStatic) != AEE_SUCCESS)
      return;

  if (dwProps)
      ISTATIC_SetProperties(pApp->m_pIStatic, dwProps);

	if (!szTitle)
		szTitleBuf = (AECHAR *)MALLOC(MAX_RES_STRING_BUF_SIZE);
	else
		szTitleBuf = (AECHAR *)szTitle;

   if (pFileName) 
   {
      // Create an instance of IFileMgr used to open the file.
      if ( ISHELL_CreateInstance(pApp->a.m_pIShell, AEECLSID_FILEMGR, (void**)(&pIFileMgr)) != SUCCESS )
        return;

      // Open the file.

      pIFile = IFILEMGR_OpenFile(pIFileMgr, pFileName, _OFM_READ);

      // Delete the file manager as we no longer need it.

      IFILEMGR_Release( pIFileMgr );

      // If the file could not open then return.

      if( !pIFile )
         return;
   }
	else if (!szText)
		szTextBuf = (AECHAR *)MALLOC(MAX_RES_STRING_BUF_SIZE);
	else
		szTextBuf = (AECHAR *)szText;

    // setup softkey 
   if (wButtons)
   {
      AEERect rcm;

      while (*wButtons)
	   {
		   IMENUCTL_AddItem(pApp->m_pSK, TESTCXTEST_RES_FILE, *wButtons, *wButtons, NULL, 0);
            wButtons++;
	   }

      IMENUCTL_SetSel(pApp->m_pSK, wDefID);
	   IMENUCTL_SetActive(pApp->m_pSK, TRUE);

      IMENUCTL_GetRect(pApp->m_pSK, &rcm);

      SETAEERECT(&rc, 0, pApp->m_rScreenRect.y, pApp->m_rScreenRect.dx, rcm.y - pApp->m_rScreenRect.y);
   }
   else
      SETAEERECT(&rc, 0, pApp->m_rScreenRect.y, pApp->m_rScreenRect.dx, pApp->m_rScreenRect.dy);
	
   ISTATIC_SetRect(pApp->m_pIStatic, &rc);

   if (!szTitle)			
       ISHELL_LoadResString(pApp->a.m_pIShell, TESTCXTEST_RES_FILE, wIDTitle, szTitleBuf, MAX_RES_STRING_BUF_SIZE);

   if (pIFile)
   {
	 ISTATIC_SetText(pApp->m_pIStatic, szTitleBuf, NULL, AEE_FONT_BOLD, AEE_FONT_NORMAL);
     ISTATIC_SetTextEx(pApp->m_pIStatic, NULL, (IAStream*)pIFile, FALSE);
     TestApp_releaseObj(pApp, (void**)&pIFile);
   }
   else if (!szText)
   {
         // load resource string 
		ISHELL_LoadResString(pApp->a.m_pIShell, TESTCXTEST_RES_FILE, wIDText, szTextBuf, MAX_RES_STRING_BUF_SIZE);
	    ISTATIC_SetText(pApp->m_pIStatic, szTitleBuf, szTextBuf, AEE_FONT_BOLD, AEE_FONT_NORMAL);
		FREEIF(szTextBuf);
   }
   else
   {
	   ISTATIC_SetText(pApp->m_pIStatic, szTitleBuf, szTextBuf, AEE_FONT_BOLD, AEE_FONT_NORMAL);
   }

	ISTATIC_SetActive(pApp->m_pIStatic, TRUE);
	ISTATIC_Redraw(pApp->m_pIStatic);

	if (!szTitle)
	FREEIF(szTitleBuf);

 	IDISPLAY_Update(pApp->a.m_pIDisplay);
}

