package org.argosdic;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.document.Document;
import org.apache.lucene.search.Hits;
import org.argosdic.action.ActionMap;
import org.argosdic.action.BaseAction;
import org.argosdic.action.program.SpeakAction;
import org.argosdic.dictionary.Dictionary;
import org.argosdic.dictionary.DictionaryManager;
import org.argosdic.dictionary.DictionaryServer;
import org.argosdic.preference.ArgosDicPreferenceStore;
import org.argosdic.resource.ResourceManager;
import org.argosdic.speech.SpeechManager;
import org.argosdic.util.ExceptionHandler;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.ToolBarManager;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.JFacePreferences;
import org.eclipse.jface.window.ApplicationWindow;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * ArgosDic.java
 * @author Xavier Cho
 * @version $Revision: 1.9 $ $Date: 2003/09/13 09:11:56 $
 */
public class ArgosDic extends ApplicationWindow {
    private static Log log = LogFactory.getLog(ArgosDic.class);

    private DictionaryServer server;
    private Browser browser;
    private List list;
    private Button searchButton;
    private Text text;
    private ActionMap actionMap;

    /**
     * Defualt constructor of ArgosDic class
     * @param display
     */
    public ArgosDic() {
        super(null);
    }

    /**
     * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
     */
    protected void configureShell(Shell shell) {
        if (log.isDebugEnabled()) {
            log.debug("Configuring shell..."); //$NON-NLS-1$
        }

        this.actionMap = createActionMap();

        ResourceManager resources = ResourceManager.getInstance();

        if (log.isDebugEnabled()) {
            log.debug("Initializing preference store..."); //$NON-NLS-1$
        }

        try {
            IPreferenceStore preferences = new ArgosDicPreferenceStore();
            JFacePreferences.setPreferenceStore(preferences);
        } catch (IOException e) {
            String msg = resources.getString("error.message.preferences"); //$NON-NLS-1$
            ExceptionHandler.handleException(msg, e);
        }

        Thread thread = new Thread(new Runnable() {
            public void run() {
                if (log.isDebugEnabled()) {
                    log.debug("Initializing speech manager..."); //$NON-NLS-1$
                }
                SpeechManager manager = SpeechManager.getInstance();
            }
        });

        thread.start();

        setBlockOnOpen(true);
        addMenuBar();
        addToolBar(SWT.FLAT);
        addStatusLine();

        super.configureShell(shell);

        Window.setExceptionHandler(new ExceptionHandler());

        shell.setText(resources.getString("application.title")); //$NON-NLS-1$
        shell.setImage(resources.getImage("application.icon").createImage()); //$NON-NLS-1$

        Point size = getInitialSize();
        Point location = getInitialLocation(size);

        shell.setBounds(location.x, location.y, size.x, size.y);

        try {
            this.server = new DictionaryServer();
            this.server.start();
        } catch (IOException e) {
            String msg = resources.getString("error.message.server"); //$NON-NLS-1$
            ExceptionHandler.handleException(msg, e);

            System.exit(-1);
        }
    }

    protected ActionMap createActionMap() {
        if (log.isDebugEnabled()) {
            log.debug("Initializing action map..."); //$NON-NLS-1$
        }

        ActionMap map = new ActionMap();

        ResourceManager resources = ResourceManager.getInstance();

        Collection actions = resources.getActionNames();
        Iterator it = actions.iterator();

        while (it.hasNext()) {
            String name = (String) it.next();

            try {
                Constructor constructor =
                    Class.forName(name).getConstructors()[0];

                BaseAction action =
                    (BaseAction) constructor.newInstance(new Object[] { this });

                map.put(action);
            } catch (Exception e) {
                log.warn("Unable to instantiate action class " + name, e); //$NON-NLS-1$
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Loaded " + actions.size() + " action classes."); //$NON-NLS-1$ //$NON-NLS-2$
        }

        return map;
    }

    private void searchWord(String word) {
        try {
            Hits hits = server.search(word);

            if (hits != null) {
                int count = hits.length();
                String[] items = new String[count];

                for (int i = 0; i < count; i++) {
                    Document document = hits.doc(i);
                    String entry = document.getField("word").stringValue(); //$NON-NLS-1$

                    items[i] = entry;
                }

                Arrays.sort(items);
                list.setItems(items);
            }

            if(hits == null || hits.length() == 0) {
                browser.setUrl(server.getUrl() + "/index.html"); //$NON-NLS-1$
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void showDefinition(String word) {
        StringBuffer buffer = new StringBuffer();
        buffer.append(server.getUrl());
        buffer.append("/"); //$NON-NLS-1$
        buffer.append(word);
        buffer.append(".dic"); //$NON-NLS-1$

        this.browser.setUrl(buffer.toString());
    }

    /**
     * @see org.eclipse.jface.window.ApplicationWindow#createMenuManager()
     */
    protected MenuManager createMenuManager() {
        return new ArgosDicMenuManager(this);
    }

    /**
     * @see org.eclipse.jface.window.ApplicationWindow#createToolBarManager(int style)
     */
    protected ToolBarManager createToolBarManager(int style) {
        return new ArgosDicToolBarManager(this);
    }

    /**
     * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
     */
    protected Control createContents(Composite parent) {
        Composite composite = new Composite(parent, SWT.NONE);

        ResourceManager resources = ResourceManager.getInstance();

        FormLayout formLayout = new FormLayout();
        formLayout.marginWidth = 3;
        formLayout.marginHeight = 3;

        composite.setLayout(formLayout);

        Label label = new Label(composite, SWT.NONE);

        FormData formData1 = new FormData();
        formData1.left = new FormAttachment(0, 5);
        formData1.top = new FormAttachment(0, 5);
        formData1.right = new FormAttachment(0, 80);

        label.setText(resources.getString("label.word")); //$NON-NLS-1$
        label.setLayoutData(formData1);

        this.text = new Text(composite, SWT.SINGLE | SWT.BORDER);

        FormData formData2 = new FormData();
        formData2.left = new FormAttachment(label, 0, SWT.RIGHT);
        formData2.top = new FormAttachment(0, 0);
        formData2.right = new FormAttachment(100, -100);

        text.setLayoutData(formData2);
        text.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                handleTextModified();
            }
        });

        this.searchButton = new Button(composite, SWT.PUSH);
        searchButton.setEnabled(false);
        searchButton.setText(resources.getString("label.search")); //$NON-NLS-1$
        searchButton.setToolTipText(resources.getString("label.search.tooltip")); //$NON-NLS-1$

        searchButton.addListener(SWT.Selection, new Listener() {
            public void handleEvent(Event e) {
                String word = text.getText();

                showDefinition(word);
                searchWord(word);
            }
        });

        getShell().setDefaultButton(searchButton);

        FormData formData3 = new FormData();
        formData3.left = new FormAttachment(text, 0, SWT.RIGHT);
        formData3.top = new FormAttachment(text, -2, SWT.TOP);
        formData3.right = new FormAttachment(100, 0);

        searchButton.setLayoutData(formData3);

        SashForm sash = new SashForm(composite, SWT.HORIZONTAL);

        FormData formData4 = new FormData();
        formData4.left = new FormAttachment(0, 0);
        formData4.top = new FormAttachment(label, 10, SWT.BOTTOM);
        formData4.right = new FormAttachment(100, 0);
        formData4.bottom = new FormAttachment(100, 0);

        sash.setLayoutData(formData4);

        this.list = new List(sash, SWT.BORDER | SWT.SINGLE | SWT.V_SCROLL);

        list.addListener(SWT.Selection, new Listener() {
            public void handleEvent(Event e) {
                if (list.getSelectionCount() > 0) {
                    String word = list.getSelection()[0];

                    showDefinition(word);
                    text.setText(word);
                }
            }
        });

        this.browser = new Browser(sash, SWT.WRAP);
        browser.setUrl(server.getUrl() + "/index.html"); //$NON-NLS-1$

        sash.setWeights(new int[] { 20, 80 });

        return composite;
    }

    private void handleTextModified() {
        String keyword = text.getText();
        boolean enabled = (keyword.trim().length() > 0);

        searchButton.setEnabled(enabled);
        actionMap.get(SpeakAction.class.getName()).setEnabled(enabled);
    }

    /**
     * @see org.eclipse.jface.window.Window#getInitialLocation(org.eclipse.swt.graphics.Point)
     */
    protected Point getInitialLocation(Point initialSize) {
        IPreferenceStore preferences = JFacePreferences.getPreferenceStore();

        int x = preferences.getInt("location.x"); //$NON-NLS-1$
        int y = preferences.getInt("location.y"); //$NON-NLS-1$

        return new Point(x, y);
    }

    /**
     * @see org.eclipse.jface.window.Window#getInitialSize()
     */
    protected Point getInitialSize() {
        IPreferenceStore preferences = JFacePreferences.getPreferenceStore();

        int x = preferences.getInt("size.width"); //$NON-NLS-1$
        int y = preferences.getInt("size.height"); //$NON-NLS-1$

        return new Point(x, y);
    }

    public ActionMap getActionMap() {
        return actionMap;
    }

    public String getSearchWord() {
        return text.getText();
    }

    /**
     * @see org.eclipse.jface.window.Window#handleShellCloseEvent()
     */
    protected void handleShellCloseEvent() {
        Shell shell = getShell();

        Point location = shell.getLocation();
        Point size = shell.getSize();

        super.handleShellCloseEvent();

        if (log.isInfoEnabled()) {
            log.info("Saving user preferences..."); //$NON-NLS-1$
        }

        ArgosDicPreferenceStore preferences =
            (ArgosDicPreferenceStore) JFacePreferences.getPreferenceStore();

        preferences.setValue("location.x", location.x); //$NON-NLS-1$
        preferences.setValue("location.y", location.y); //$NON-NLS-1$
        preferences.setValue("size.width", size.x); //$NON-NLS-1$
        preferences.setValue("size.height", size.y); //$NON-NLS-1$

        DictionaryManager manager = DictionaryManager.getInstance();
        Dictionary dictionary = manager.getSelectedDictionary();

        if (dictionary != null) {
            preferences.setValue("default.dictionary", dictionary.getId()); //$NON-NLS-1$
        }

        try {
            preferences.save();
        } catch (IOException e) {
            if (log.isWarnEnabled()) {
                String msg = "Failed to save user preferences data."; //$NON-NLS-1$
                log.warn(msg, e);
            }
        }
    }

    public static void main(String[] args) {
        try {
            ApplicationWindow application = new ArgosDic();
            application.open();
        } catch (Throwable t) {
            if (log.isFatalEnabled()) {
                String msg = "Unable to initialize application."; //$NON-NLS-1$
                log.fatal(msg, t);
            }
        } finally {
            Display.getCurrent().dispose();
            System.exit(0);
        }
    }
}
