package net.kldp.beat.action;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.kldp.beat.annotation.Result;
import net.kldp.beat.exception.InterceptorException;
import net.kldp.beat.interceptor.SystemInterceptor;
import net.kldp.beat.interceptor.UserInterceptor;
import net.kldp.beat.web.interceptor.InterceptorFactory;

public class ActionService {
	private ActionContext context;

	public ActionService(ServletContext context, HttpServletRequest request, HttpServletResponse response)
			throws ServletException {
		this.context = new ActionContext(request, response, context);
	}

	public void doService(Object action, ActionMapper mapper) throws ServletException {
		// 인터셉터 분류
		InterceptorStack stack = new InterceptorStack(action);

		// 시스템 인터셉터 처리
		executeSystemInterceptor(action, stack.getSystemInterceptors());

		// 액션으로 파라미터 주입
		PropertyCopy.mapToBean(context.getParameterMap(), action);

		// Validation 처리
		if (context.hasValidationError()) {
			returnToInput(action, stack.getResult("input"));
			return;
		}
		// 사용자 인터셉터 초기화
		initInterceptors(stack.getBeforeInterceptors());
		initInterceptors(stack.getAfterInterceptors());

		// 전처리 인터셉터 처리
		if (!executeInterceptors(action, stack.getBeforeInterceptors())) {
			executeInterceptors(action, stack.getAfterInterceptors());
			return;
		}
		// 전처리 메서드 실행
		invokeMethod(action, stack.getBeforeMethod());

		// execute 메서드 실행
		String execute = executeAction(action);

		Result result;
		if (stack.hasResults()) {
			result = stack.getResult(execute);
		} else {
			result = mapper.getDefaultResult();
		}

		// 전처리 Result 메서드 실행
		invokeMethod(action, stack.getBeforeResultMethod());

		// Request로 빈 주입
		PropertyCopy.beanToMap(action, context.getRequestMap());

		// View로 디스패치
		dispatchView(result, action);

		// 후처리 메서드 실행
		invokeMethod(action, stack.getAfterMethod());

		// 후처리 인터셉터 실행
		executeInterceptors(action, stack.getAfterInterceptors());
	}

	private void initInterceptors(Map<UserInterceptor, Annotation> interceptors) throws InterceptorException {
		AnnotationFinder finder;
		for (UserInterceptor interceptor : interceptors.keySet()) {
			finder = new AnnotationFinder(interceptor.getClass());
			for (Annotation annotation : finder.getAnnotations()) {
				InterceptorFactory.intecept(interceptor, context, annotation);
			}
		}
	}

	private void dispatchView(Result result, Object action) throws ServletException {
		try {
			context.getDispatcher().dispatch(result, action);
		} catch (IOException e) {
			throw new ServletException(e);
		}
	}

	private void returnToInput(Object action, Result input) throws ServletException {
		if (input == null) {
			throw new ServletException("can not found Result annotation for input.");
		}
		PropertyCopy.beanToMap(action, context.getRequestMap());
		try {
			context.getDispatcher().dispatch(input, action);
		} catch (IOException e) {
			throw new ServletException(e);
		}
	}

	private void invokeMethod(Object action, Method method) throws ServletException {
		if (method != null) {
			try {
				method.invoke(action, (Object) null);
			} catch (Exception e) {
				throw new ServletException("Can not execute Method");
			}
		}
	}

	private void executeSystemInterceptor(Object action, Map<SystemInterceptor, Annotation> systemInterceptors)
			throws InterceptorException {
		InterceptorFactory.intecept(action, systemInterceptors, context);
	}

	private boolean executeInterceptors(Object action, Map<UserInterceptor, Annotation> interceptors)
			throws ServletException {
		boolean result = true;
		for (UserInterceptor interceptor : interceptors.keySet()) {
			// Interceptor가 true일경우에만 Action과 View 처리
			if (!interceptor.intercept(action, interceptors.get(interceptor)))
				result = false;
		}
		return result;
	}

	private String executeAction(Object action) throws ServletException {
		try {
			Method method = action.getClass().getMethod("execute");
			return (String) method.invoke(action);
		} catch (Exception e) {
			throw new ServletException(e);
		}
	}
}