package net.kldp.beat.action;

import javax.servlet.ServletException;

import net.kldp.beat.annotation.Result;
import net.kldp.beat.exception.fatal.ActionExecuteException;
import net.kldp.beat.exception.fatal.ResultException;
import net.kldp.beat.message.ResourceManager;
import net.kldp.beat.servlet.ActionMapper;
import net.kldp.beat.system.aware.ValidationAware;
import net.kldp.beat.util.PropertyCopy;

public class ActionService {
	private ActionContext context;
	private ActionMapper mapper;
	private Object action;

	public ActionService(ActionContext context, Object action,
			ActionMapper mapper) {
		this.context = context;
		this.action = action;
		this.mapper = mapper;
	}

	public void doService() throws ServletException {
		boolean interceptResult = true;
		ActionInvoker actionInvoker = new ActionInvoker(action);

		// 리소스 초기화
		ResourceManager resourceManager = new ResourceManager(mapper.getNamespace());

		// 인터셉터 분류
		InterceptorStack stack = new InterceptorStack(action.getClass());

		InterceptorInvoker interceptorInvoker = new InterceptorInvoker(context,	stack, action);

		// 시스템 인터셉터 처리
		interceptorInvoker.executeSystemInterceptors();

		// 액션으로 파라미터 주입
		parameterToAction();

		// 사용자 인터셉터 초기화
		interceptorInvoker.initUserInterceptors();

		// 전처리 인터셉터 처리
		interceptResult = interceptorInvoker.executeBeforeInterceptors();

		// 전처리 메서드 실행
		actionInvoker.executeBefore();

		// Validation 처리
		if (!doValidation()) {
			returnToInput(stack.getResult("input"));
			// 후처리 메서드 실행
			actionInvoker.executeAfter();
			interceptorInvoker.executeAfterInterceptors();
			return;
		}

		if (!interceptResult) {
			// 후처리 메서드 실행
			actionInvoker.executeAfter();
			interceptorInvoker.executeAfterInterceptors();
			return;
		}

		// execute 메서드 실행
		Result result;
		try {
			String execute = actionInvoker.execute();
			if (stack.hasResults()) {
				result = stack.getResult(execute);
			} else {
				result = mapper.getDefaultResult();
			}
		} catch (ActionExecuteException e) {
			try {
				result = stack.getResult("exception");
				context.getRequestMap().put("exception", e);
			} catch (ResultException ex) {
				throw e;
			}
		}

		// 전처리 Result 메서드 실행
		actionInvoker.executeBeforeResult();

		// Request로 빈 주입
		actionToParameter();

		// 리소스 번역
		resourceManager.translateResource();
		try {
			// 리소스 해제.
			resourceManager.finalize();
		} catch (Throwable e) {
			e.printStackTrace();
		}

		// View로 디스패치
		dispatchView(result);

		// 후처리 메서드 실행
		actionInvoker.executeAfter();

		// 후처리 인터셉터 실행
		interceptorInvoker.executeAfterInterceptors();

		// 소멸자 처리
		interceptorInvoker.destroyInterceptors();
	}

	private void actionToParameter() {
		PropertyCopy.beanToMap(action, context.getRequestMap());
	}

	private void parameterToAction() {
		PropertyCopy.mapToBean(context.getParameterMap(), action);
	}

	private boolean doValidation() {
		if (action instanceof ValidationAware) {
			ValidationAware aware = (ValidationAware) action;
			aware.validate();
			if (aware.getErrorsMap().size() > 0) {
				return false;
			}
		}
		return true;
	}

	private void dispatchView(Result result) throws ServletException {
		context.getDispatcher().dispatch(result, action);
	}

	private void returnToInput(Result input) throws ServletException {
		if (input == null) {
			throw new ServletException(
					"can not found Result annotation for input.");
		}
		PropertyCopy.beanToMap(action, context.getRequestMap());
		context.getDispatcher().dispatch(input, action);
	}
}