package net.kldp.beat.configuration;

import java.io.IOException;
import java.net.URL;
import java.util.Properties;

import net.kldp.beat.exception.fatal.ConfigException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.ApplicationContext;

/**
 * Beat framework의 환경 설정을 로드합니다.
 * 
 */
public class Config {
	private static final String SYSTEM_FILE = "/net/kldp/beat/configuration/beat.xml";
	private static final String USER_FILE = "/beat.xml";
	private static final String VERSION = "0.5";
	private static Log logger = LogFactory.getLog(Config.class);
	private static Class<?> cl;
	private static ConfigParser parser;

	/**
	 * Beat Framework 버전번호를 리턴합니다.
	 */
	public static String version() {
		return VERSION;
	}

	/**
	 * 파일 업로드 버퍼 사이즈를 리턴합니다.
	 * @return
	 */
	public static int bufferSize() {
		return parser.bufferSize;
	}

	/**
	 * 개발 모드인지 리턴합니다.
	 * @return
	 */
	public static boolean devMode() {
		return parser.devMode;
	}

	/**
	 * 최대 업로드 사이즈를 리턴합니다.
	 * @return
	 */
	public static int maxSize() {
		return parser.maxSize;
	}

	/**
	 * 파일 업로드 디렉터리를 리턴합니다.
	 * @return
	 */
	public static String saveDir() {
		if (parser.saveDir.equals("")) {
			parser.saveDir = getDefaultTempDir();
		}
		return parser.saveDir;
	}

	public static ActionMapping defaultMapping() {
		return parser.defaultMapping;
	}

	public static ActionMapping getMapping(String namespace) {
		for (ActionMapping mapping : parser.mappings) {
			if (mapping.getMappingLocation().equals(namespace))
				return mapping;
		}
		return parser.defaultMapping;
	}

	public static String viewType() {
		return parser.viewType;
	}

	public static String viewSuffix() {
		if (parser.viewType.equals("jsp")) {
			return "jsp";
		} else if (parser.viewType.equals("velocity")) {
			return "vm";
		} else if (parser.viewType.equals("freemaker")) {
			return "fm";
		}
		return "jsp";
	}

	/**
	 * Beat Framework 환경 설정 파일을 로드합니다.
	 * @param filename
	 * @return
	 * @throws ConfigException
	 */
	private static boolean loadConfiguration(String filename)
			throws ConfigException {
		URL url = cl.getResource(filename);
		if (url != null) {
			try {
				parser = new ConfigParser(url.getPath());
				System.out.println(parser);
				return true;
			} catch (Exception e) {
				logger.fatal(e);
				throw new ConfigException(
						"can not parsing beat configuration file:" + filename);
			}
		}
		return false;
	}

	public static void load(Class<?> clazz) throws ConfigException {
		cl = clazz;
		if (!loadConfiguration(USER_FILE))
			if (!loadConfiguration(SYSTEM_FILE))
				throw new ConfigException("can not load beat configuration file");
		parser.parse();
	}

	/**
	 * 프로퍼티 파일을 로드합니다.
	 * 
	 * @param filename
	 * @return
	 * @throws IOException
	 */
	public static Properties loadProperties(String filename) throws IOException {
		Properties properties = new Properties();
		properties.load(cl.getResourceAsStream(filename));
		return properties;
	}

	/**
	 * 기본적으로 임시 파일이 저장될 디렉토리를 리턴합니다.
	 * 이 설정은 환경 설정파일에 저장될 디렉토리가 설정되어 있지 않은 경우에 사용됩니다. 
	 * @return
	 */
	private static String getDefaultTempDir() {
		return System.getProperty("java.io.tmpdir");
	}

	/**
	 * 스프링 컨텍스트 객체를 리턴합니다.
	 * @return
	 */
	public static ApplicationContext getSpringContext() {
		return parser.springContext;
	}

	/**
	 * 글로벌 예외 페이지를 리턴합니다.
	 * @return
	 */
	public static String globalException() {
		return parser.globalException;
	}
}