package net.kldp.beat.system.interceptor;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.lang.annotation.Annotation;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import net.kldp.beat.action.ActionContext;
import net.kldp.beat.configuration.Config;
import net.kldp.beat.exception.fatal.InterceptorException;
import net.kldp.beat.interceptor.SystemInterceptor;
import net.kldp.beat.system.annotation.MultipartRequest;
import net.kldp.beat.system.aware.ValidationAware;
import net.kldp.beat.upload.FormFileImpl;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Multipart인터셉터입니다. Multipart인터셉터가 작동하기 위해서는 사용자 요청이 Multipart/Form-data방식이어야
 * 하며, 액션 클래스에
 * 
 * @Multipart 어노테이션을 선언할 필요가 있습니다. 이 두가지 조건을 만족시키지 못할경우 MultipartInterceptor는
 *            예외를 발생시킵니다.
 * 
 */
final class MultipartInterceptor implements SystemInterceptor {
	private static Log logger = LogFactory.getLog(MultipartInterceptor.class);

	@Override
	public void intercept(Object action, ActionContext context, Annotation annotation)
			throws InterceptorException {
		System.out.println("MULTIPART STARTING....");
		if (!isMultipart(context.getServletRequest())) {
			throw new InterceptorException("Multipart/Form-data요청이 아닙니다.");
		} else if (!(annotation instanceof MultipartRequest)) {
			throw new InterceptorException("Multipart어노테이션이 선언되지 않았습니다.");
		}
		MultipartRequest multipart = (MultipartRequest) annotation;
		try {
			Map<String, Object> parameterMap = multipartParser(context.getServletRequest(), multipart);
			System.out.println(parameterMap.size());
			context.setParameterMap(parameterMap);
		} catch (FileUploadException e) {
			if (action instanceof ValidationAware) {
				addError(action, multipart, e);
			} else {
				throw new InterceptorException(e);
			}
		}
	}

	/**
	 * Request를 해석한 뒤에 파라미터를 맵으로 리턴합니다. 이 파라미터는 ActionContext에 주입되고,
	 * ParameterMap을 대체합니다.
	 * 
	 * @param request
	 * @param multipart
	 * @return parameterMap
	 * @throws FileUploadException
	 * @throws InterceptorException
	 */
	private Map<String, Object> multipartParser(HttpServletRequest request, MultipartRequest multipart)
			throws FileUploadException, InterceptorException {
		Map<String, Object> parameters = new HashMap<String, Object>();
		ServletFileUpload upload = getServletFileUpload();
		// 최대로 저장할 크기 설정
		if (multipart.size() > 0 && multipart.size() < Config.maxSize()) {
			upload.setSizeMax(multipart.size());
		} else if (Config.maxSize() > 0) {
			upload.setSizeMax(Config.maxSize());
		}
		for (Object obj : upload.parseRequest(request)) {
			FileItem item = (FileItem) obj;
			String fieldName = item.getFieldName();
			Object fieldValue;
			try {
				if (item.getSize() > 0) {
					if (item.isFormField()) {
						fieldValue = item.getString("utf-8");
					} else {
						fieldValue = new FormFileImpl(saveFile(item));
					}
					if (parameters.containsKey(fieldName)) {
						Object[] newValues = addValue(parameters.get(fieldName), fieldValue);
						parameters.put(fieldName, newValues);
					} else {
						parameters.put(fieldName, fieldValue);
					}
				}
			} catch (UnsupportedEncodingException e) {
				logger.error(e);
			} catch (Exception e) {
				logger.error(e);
				throw new FileUploadException();
			}
		}
		return parameters;
	}

	/**
	 * 객체를 더한 배열을 리턴합니다. oldValue객체가 배열이라면 배열의 마지막에 추가하고, 그렇지 않으면 두 객체를 담은 배열을
	 * 리턴합니다.
	 * 
	 * @param oldValue
	 * @param value
	 * @return
	 */
	private Object[] addValue(Object oldValue, Object value) {
		Object[] newValues;
		if (oldValue.getClass().isArray()) {
			Object[] oldValues = (Object[]) oldValue;
			newValues = new Object[oldValues.length + 1];
			System.arraycopy(oldValues, 0, newValues, 0, oldValues.length);
			newValues[oldValues.length] = value;
		} else {
			newValues = new Object[2];
			newValues[0] = oldValue;
			newValues[1] = value;
		}
		return newValues;
	}

	/**
	 * 임시 디렉토리에 파일을 저장합니다. 업로드 중복 방지를 위해 파일 확장자 뒤에 현재 시각이 덧붙여 집니다.
	 * 
	 * @param item
	 * @return savedFile
	 * @throws Exception
	 */
	private File saveFile(FileItem item) throws Exception {
		System.out.println("size" + item.getSize());
		String filename = Config.saveDir() + "/" + FilenameUtils.getName(item.getName()) + "."
				+ System.currentTimeMillis();
		File file = new File(filename);
		item.write(file);
		item.delete();
		return file;
	}

	/**
	 * Apache commons upload의 객체를 생성합니다.
	 * 
	 * @return
	 */
	private ServletFileUpload getServletFileUpload() {
		DiskFileItemFactory factory = new DiskFileItemFactory();
		// 메모리 버퍼 설정
		factory.setSizeThreshold(1024 * 1024);
		// 임시 저장 디렉토리 설정
		factory.setRepository(new File(Config.saveDir()));
		return new ServletFileUpload(factory);
	}

	/**
	 * 업로드 허용 크기를 초과한 경우 에러 메세지를 삽입합니다. 만약 액션 클래스가 ValidationAware를 구현하지 않은 경우,
	 * 통상적인 업로드 예외로 처리됩니다.
	 * 
	 * @param action
	 * @param multipart
	 * @param e
	 * @throws InterceptorException
	 */
	private void addError(Object action, MultipartRequest multipart, FileUploadException e)
			throws InterceptorException {
		ValidationAware aware = (ValidationAware) action;
		String key = multipart.errorKey();
		String message = multipart.errorMessage();
		if (!key.equals("") && aware.getErrorsMap() != null) {
			aware.getErrorsMap().put(key, message);
		} else {
			throw new InterceptorException(e);
		}
	}

	/**
	 * 현재 요청이 Multipart/Form-data인지 파악합니다.
	 * 
	 * @param servletRequest
	 * @return
	 */
	private boolean isMultipart(HttpServletRequest servletRequest) {
		return servletRequest.getContentType().trim().contains("multipart/form-data");
	}
}