/**
 * Class Name	: TIFFImage
 * Description	: TIFF handling class.
 * Date 				: 2004/02/24
 * Author			: Moon-Ho, Lee (conv2@nvision.gsnu.ac.kr)
 * History			: 2004/02/24 first created.
 */

package com.conv2.imageGS.IMGFileIO;

import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;

import com.conv2.imageGS.Exception.ImageGSException;
import com.conv2.imageGS.GeometryProc.Mirror;
import com.conv2.imageGS.IMGFileIO.JAI.FileSeekableStream;
import com.conv2.imageGS.IMGFileIO.JAI.ImageCodec;
import com.conv2.imageGS.IMGFileIO.JAI.ImageDecoder;
import com.conv2.imageGS.IMGFileIO.JAI.ImageEncoder;
import com.conv2.imageGS.IMGFileIO.JAI.SeekableStream;
import com.conv2.imageGS.IMGFileIO.JAI.TIFFEncodeParam;
import com.conv2.imageGS.Util.InitIMGBuf;

/**
 * TIFF handling Ŭ with com.conv2.imagegs.JAI package. <p>
 * 
 * Sun翡 ϴ JAI 1.1.1 ҽ  / TIFF  ó <p>
 * 
 *  JAI 1.2 SunOS, Linux, windows ÷ǰ  JDK, JRE ġؾ ϴ <br>
 *  ϰ  Ŭ  Դϴ. <p>
 * 
 * JAI 1.1.1 ҽ com.conv2.imageGS.JAI ְ package  ϸ ˴ϴ.<br>
 *   Ŭ   ϴ. <p>
 * 
 * JAI 1.1.1 ٿε : http://java.sun.com/developer/releases/jai/ <p>
 * 
 *  ҽ test/TestTIFFImage.java Ͽ ֽʽÿ. <p>
 * 
 * [Ư¡] <p>
 * 
 * BufferedImage   32 bit ó  䱸ϴ JAI 1.1.1 ҽ  <br>
 * κ ּ ó.
 * 
 * [History] <p>
 * 
 * 1)  TIFF height width Ųٷ Ǿ ִ 찡 ־, ̸ ϱ  . <br>
 * 2) (̳ʸ)   ุ  . <p>
 * 3)     . <p>
 * 4) TIFF ε  ó ϴ ޼ҵ ߰.<p>
 *
 * @author Moon-Ho, Lee (conv2@nvision.gsnu.ac.kr)
 * @version 1.0.0
 */
public class TIFFImage
{
	private String fname = null;
	private int height = 0;
	private int width = 0;
	private int numPages = 0;
	private RenderedImage[] images = null;
	private int imageType = -1; 

	private short[][] grayChannel = null;
	private short[][] redChannel = null;
	private short[][] greenChannel = null;
	private short[][] blueChannel = null;
	private BufferedImage bi = null;

	public final int BINARY = 0;	
	public final int GRAY = 1;
	public final int COLOR = 2;

	
	private final int ROTATE_N = 0;
	private final int ROTATE_Y = 1;
	
	private int rotateType = ROTATE_N;
	private int compressMethod = 0;
	private boolean fastSplit = false; 
	
	/**
	 *  
	 * 
	 * @param fname ϸ 
	 * @throws ImageGSException
	 */
	public TIFFImage(String fname) throws ImageGSException
	{
		this.fname = fname;
	}
	
	/**
	 * Է  Multi-Tiff  1 ̸̻,  Tiff  ׻ 1̴.
	 * 
	 * @return Է    
	 */
	public int getNumPages()
	{
		return numPages;
	}
	
	/**
	 *  ش  ̸ ȯѴ.
	 * 
	 * @return 
	 */
	public int getHeight()
	{
		return height;	
	}

	/**
	 *  ش  ̸ Ѵ.
	 */
	public void setHeight( int height )
	{
		this.height = height;
	}
	
	/**
	 *  ش  ʺ ȯѴ.
	 * 
	 * @return ʺ
	 */	
	public int getWidth()
	{
		return width;
	}

	/**
	 *  ش  ʺ Ѵ.
	 */
	public void setWidth( int width )
	{
		this.width = width;
	}
	
	/** 
	 * TIFF оδ.
	 */
	public void load() throws ImageGSException
	{
		SeekableStream in = null;
		
		try
		{
			File file = new File( fname );
			in = new FileSeekableStream( file );
			
			ImageDecoder decoder = ImageCodec.createImageDecoder( "tiff", file, null );
			numPages = decoder.getNumPages();
			
			images = new RenderedImage[ numPages ];
			for( int i=0; i<decoder.getNumPages(); i++)
			{
				images[i] = decoder.decodeAsRenderedImage( i );
			}

			height = images[0].getHeight();
			width = images[0].getWidth();
		}
		catch( IOException e )
		{
			throw new ImageGSException(this.getClass().getName() + " >> "
																+ e.getMessage() );	
		}			
		finally
		{
			if(in != null) try { in.close(); } catch(IOException se) {}
		
		}
	}
	
	/**
	 * RenderedImage 迭 ȯѴ.
	 * 
	 * @return RenderedImage[] 
	 */
	public RenderedImage[] getRenderImages()
	{
		return images;
	}

	/**
	 * ش RenderedImage BufferedImage ȯѴ.
	 * 
	 * @return BufferedImage
	 */	
	public BufferedImage getBufferedImage() throws ImageGSException
	{
		return bi;
	}

	/**
	 * ش RenderedImage red channel ȯѴ.  (, ÷ Ÿ 츸)
	 * 
	 * @return short[][] type red channel
	 */		
	public short[][] getRedChannel() throws ImageGSException
	{
		if( imageType == GRAY || imageType == BINARY ) 
		{
			throw new ImageGSException(this.getClass().getName() + " >> "
					+ " gray image!");				
		}
		
		if( fastSplit )
		{
			BufferedImage2Channel bc = new BufferedImage2Channel();
			bc.color( bi );
			redChannel = bc.getRedChannel();
		}
		
		return redChannel;
	}

	/**
	 * ش RenderedImage green channel ȯѴ.  (, ÷ Ÿ 츸)
	 * 
	 * @return short[][] type green channel
	 */	
	public short[][] getGreenChannel() throws ImageGSException
	{
		if( imageType == GRAY || imageType == BINARY ) 
		{
			throw new ImageGSException(this.getClass().getName() + " >> "
					+ " gray image!");				
		}

		if( fastSplit )
		{
			BufferedImage2Channel bc = new BufferedImage2Channel();
			bc.color( bi );
			greenChannel = bc.getGreenChannel();
		}
		
		return greenChannel;
	}

	/**
	 * ش RenderedImage blue channel ȯѴ.  (, ÷ Ÿ 츸)
	 * 
	 * @return short[][] type blue channel
	 */	
	public short[][] getBlueChannel() throws ImageGSException
	{
		if( imageType == GRAY || imageType == BINARY ) 
		{
			throw new ImageGSException(this.getClass().getName() + " >> "
					+ " gray image!");				
		}

		if( fastSplit )
		{
			BufferedImage2Channel bc = new BufferedImage2Channel();
			bc.color( bi );
			blueChannel = bc.getBlueChannel();
		}
		
		return blueChannel;
	}

	/**
	 * ش RenderedImage gray/binary channel ȯѴ.
	 * , binary  gray ȯѴ.
	 * 
	 * @return short[][] type gray channel
	 */		
	public short[][] getGrayChannel() throws ImageGSException
	{
		if( imageType == COLOR ) 
		{
			throw new ImageGSException(this.getClass().getName() + " >> "
					+ " color image!");				
		}

		if( fastSplit )
		{
			grayChannel = getGrayChannel( bi, imageType );
		}
		
		return (short[][])grayChannel;
	}

	private short[][] getGrayChannel( BufferedImage bi, int imageType ) throws ImageGSException
	{
		short [][] image = null;
		try 
		{	
			// ̿ ʺ Ѵ.
			int height = bi.getHeight();
			int width = bi.getWidth();
			int rotateType = 0;
			
			// height width ݴ   ִ ȮѴ.
			int tmpHeight = width;
			int tmpWidth = height;
			
			if( tmpHeight == width && tmpWidth == height )
			{
				rotateType = ROTATE_Y;
			}
			else 
			{
				rotateType = ROTATE_N;
			}
			
			// rotate̸ ٷ ش.
			if( rotateType == ROTATE_Y ) 
			{
				// swap
				int tmp = height ;
				height = width;
				width = tmp;
			}
			
			image = InitIMGBuf.ShortIMGBuf( height, width );
			Raster raster = bi.getData();
			int[] data = new int[3];
			
			for(int i=0; i<height; i++)
			{
				for(int j=0; j<width; j++)
				{
					data = raster.getPixel( i, j, (int[]) null );
					if( imageType == BINARY )
					{
						image[i][j] = (short)( 255 * data[0] );
					}
					else if( imageType == GRAY )
					{
						image[i][j] = (short)data[0];
					}
				}
			}	

			// rotate̸ ٷ ش.
			if( rotateType == ROTATE_Y ) 
			{
				image = rotate( image, height, width );
			}
		}
		catch (Exception e)
		{
			e.printStackTrace();
			throw new ImageGSException( this.getClass().getName() + " >> " + e.getMessage() ); 
		}	

		return image;
	}
	
	/**
	 * ش RenderedImage  ľϿ ϵ/÷  иѴ. <br>
	 * 
	 * ,  TIFF Ѵ.
	 */		
	public void splitImageData() throws ImageGSException
	{
		if( numPages != 1 )
		{
			throw new ImageGSException(this.getClass().getName() + " >> "
					+ " not support multi-TIFF");				
		}
		else if( numPages == 1 )
		{	
			splitImageData( images[0] );
		}
	
		return;
	}
	
	/**
	 * ̿ ʺ ݴ Ǿ ִ ȮѴ.
	 * 
	 * @return  / 
	 */
	public boolean getRotateImage() 
	{
		return rotateType == ROTATE_Y ? true : false; 
	}

	/**
	 * ش RenderedImage  ľϿ /ϵ/÷  иѴ. - fast <br>
	 * 
	 * ,  TIFF Ѵ.
	 */		
	public BufferedImage fastSplitImageData( RenderedImage image ) throws ImageGSException
	{
		try
		{
			int [] rgba = null;
			
			fastSplit = true;
			Raster raster = image.getData();
			rgba = raster.getPixel( 0, 0, (int[]) null);
			
			if(  rgba.length > 3 )
			{	
				throw new ImageGSException(this.getClass().getName() + " >> "
						+ " not support 32-bit in TIFF");	
			}
			
			//  Ǵ ϵ ̴.
			if( rgba.length == 1 )
			{	
				searchBinaryGrayImage( raster, rgba );
				
				if( imageType == BINARY )
				{	
					bi = new BufferedImage( getWidth(), getHeight(), BufferedImage.TYPE_BYTE_BINARY );
				}
				else if( imageType == GRAY )
				{
					bi = new BufferedImage( getWidth(), getHeight(), BufferedImage.TYPE_BYTE_GRAY);				
				}
			}
			// ÷ ̴.
			else if( rgba.length == 3 )
			{	
				bi = new BufferedImage( width, height, BufferedImage.TYPE_INT_RGB );
			}
			
			bi.setData( raster );
		}
		catch (Exception e)
		{
			throw new ImageGSException( this.getClass().getName() + " >> " + e.getMessage() ); 
		}	
		
		return bi;
	}
	
	/**
	 * ش RenderedImage  ľϿ /ϵ/÷  иѴ. - slow <br>
	 * 
	 * ,  TIFF Ѵ.
	 */		
	public void splitImageData( RenderedImage image ) throws ImageGSException
	{
		try
		{
			Raster raster = image.getData();
			
			int [] rgba = null;
			double alpha = 0.0;
			
			// height width ݴ   ִ ȮѴ.
			int tmpHeight = width;
			int tmpWidth = height;
			
			if( tmpHeight == width && tmpWidth == height )
			{
				rotateType = ROTATE_Y;
			}
			else 
			{
				rotateType = ROTATE_N;
			}
		
			// rotate̸ ٷ ش.
			if( rotateType == ROTATE_Y ) 
			{
				// swap
				int tmp = height ;
				height = width;
				width = tmp;
			}
			
			rgba = raster.getPixel( 0, 0, (int[]) null);
			
			if(  rgba.length > 3 )
			{	
				throw new ImageGSException(this.getClass().getName() + " >> "
						+ " not support 32-bit in TIFF");	
			}
			
			//  Ǵ ϵ ̴.
			if( rgba.length == 1 )
			{	
					grayChannel = InitIMGBuf.ShortIMGBuf( height, width );
			}
			// ÷ ̴.
			else if( rgba.length == 3 )
			{	
				redChannel = InitIMGBuf.ShortIMGBuf( height, width );
				greenChannel = InitIMGBuf.ShortIMGBuf( height, width );
				blueChannel = InitIMGBuf.ShortIMGBuf( height, width );
			}
			
			int[] histogram = new int[256];
	
			for(int i=0; i<height; i++)
			{
				for(int j=0; j<width; j++)
				{
					rgba = raster.getPixel( i, j, (int[]) null );

					//  Ǵ ϵ ̴.
					if( rgba.length == 1 )
					{
						imageType = GRAY; 
						grayChannel[i][j] = (short)rgba[0];
						histogram[ grayChannel[i][j] ]++;
					}
					// ÷ ̴.
					else if( rgba.length == 3 )
					{
						imageType = COLOR;
						redChannel[i][j] = (short)(rgba[0] );
						greenChannel[i][j] = (short)(rgba[1] );
						blueChannel[i][j] = (short)(rgba[2] );
					}
				}
	
			}
			
			//  ϵ  ľ
			if( imageType == GRAY )
			{	
				int pixels = 0;
				for(int i=2; i<256; i++)
				{
					if( histogram[i] != 0 ) pixels++;
				}
				
				if( pixels == 0 ) imageType = BINARY;
			}
			
			// binary to gray level
			if( imageType == BINARY )
			{
				// ̳ʸ   츸  ȴ. ׷Ƿ ٷ ش.
				//     ΰ? histogram     .
				//  : ϸ  ̸ _c, ̸ _r  óѴ.
				//   binary to gray method ϸ ȴ.
				grayChannel = inverse( grayChannel );
			}
			
			// rotate̸ ٷ ش.
			if( rotateType == ROTATE_Y ) 
			{
				if( imageType == GRAY ||  imageType == BINARY )
				{
					grayChannel = rotate( grayChannel, height, width );
					
				}
				else if( imageType == COLOR )
				{	
					redChannel = rotate( redChannel, height, width );
					greenChannel = rotate( greenChannel, height, width );
					blueChannel = rotate( blueChannel, height, width );
				}
			}
	
			Channel2BufferedImage cb = new Channel2BufferedImage();		
			
			// /ϵ 
			if( imageType == GRAY ||  imageType == BINARY )
			{		
				cb.gray( grayChannel );
			}
			// ÷  
			else if( imageType == COLOR )
			{	
				cb.color( redChannel, greenChannel, blueChannel );
			}
			
			bi = cb.getBufferedImage();
		}
		catch (Exception e)
		{
			throw new ImageGSException( this.getClass().getName() + " >> " + e.getMessage() ); 
		}	
		
		return;
	}

	private short[][] inverse( short[][] binaryImage ) throws ImageGSException
	{
		int i, j;
		int height = binaryImage.length;
		int width = binaryImage[height-1].length;
		
		short[][] inverseBinaryImage = InitIMGBuf.ShortIMGBuf( height, width );

		for(i=0; i<height; i++)
		{
			for(j=0; j<width; j++)
			{
				inverseBinaryImage[i][j] = binaryImage[i][j] == 0 ? (short)255 : (short)0;	
			}	
		}
		
		return inverseBinaryImage;		
	}
	
	private short[][] rotate( short[][] image, int height, int width ) throws ImageGSException
	{
		//  ´.
		Mirror flip = new Mirror( image );
		flip.FlipOP();
		short[][] tmpImage1 = flip.getShortResultIMG();

		short[][] tmpImage2 = InitIMGBuf.ShortIMGBuf( width, height );
		for(int i=0; i<height; i++)
		{
			for(int j=0; j<width; j++)
			{
				tmpImage2[j][i] = tmpImage1[i][j];	
			}
		}
		
		// ¿ ´.
		flip = new Mirror( tmpImage2 );
		flip.ReflectOP();
		
		return flip.getShortResultIMG();		
	}
	
	/**
	 * Է  Ÿ ȯѴ.
	 * 
	 * @return 0 - gray, 1 - color
	 */
	public int getImageType()
	{
		return imageType;
	}

	/**
	 * Muiti-TIFF Ѵ.
	 * 
	 * @param fname  ϸ
	 * @param image BufferedImage type 迭
	 */	
	public void saveMultiBufferedImage( String fname, BufferedImage[] image ) throws ImageGSException
	{
		OutputStream out = null;
		
		try
		{
			out = new FileOutputStream( fname );
			TIFFEncodeParam encodeParam = new TIFFEncodeParam();
			encodeParam.setCompression( compressMethod );
			ImageEncoder encoder = ImageCodec.createImageEncoder("TIFF", out, encodeParam );

			ArrayList arrayList = new ArrayList();
			for( int i=1; i<image.length; i++ )
			{
				arrayList.add( image[i] );
			}
			encodeParam.setExtraImages( arrayList.iterator() );
			encoder.encode( image[0] );
			out.close();
		}
		catch( IOException e )
		{
			throw new ImageGSException(this.getClass().getName() + " >> "
					+ e.getMessage() );	
		}			
		finally
		{
			if(out != null) try { out.close(); } catch(IOException se) {}
			
		}		
	}

	/**
	 * Single-TIFF Ѵ.
	 * 
	 * @param fname  ϸ
	 * @param image BufferedImage type
	 */	
	public void saveBufferedImage( String fname, BufferedImage image ) throws ImageGSException
	{
		OutputStream out = null;
				
		try
		{
			out = new FileOutputStream( fname );
			TIFFEncodeParam encodeParam = new TIFFEncodeParam();
			encodeParam.setCompression( compressMethod );
			ImageEncoder encoder = ImageCodec.createImageEncoder("TIFF", out, encodeParam );
			encoder.encode( image );
			out.close();
		}
		catch( IOException e )
		{
			throw new ImageGSException(this.getClass().getName() + " >> "
					+ e.getMessage() );	
		}			
		finally
		{
			if(out != null) try { out.close(); } catch(IOException se) {}
			
		}
		
	}

	/**
	 * Muiti-TIFF Ѵ.
	 * 
	 * @param fname  ϸ
	 * @param image RenderedImage type 迭
	 */		
	public void saveMultiRenderedImage( String fname, RenderedImage[] image ) throws ImageGSException
	{
		OutputStream out = null;
		
		try
		{
			out = new FileOutputStream( fname );
			TIFFEncodeParam encodeParam = new TIFFEncodeParam();
			encodeParam.setCompression( compressMethod );
			ImageEncoder encoder = ImageCodec.createImageEncoder("TIFF", out, encodeParam );

			ArrayList arrayList = new ArrayList();
			for( int i=1; i<image.length; i++ )
			{
				arrayList.add( image[i] );
			}
			encodeParam.setExtraImages( arrayList.iterator() );
			encoder.encode( image[0] );
			out.close();
		}
		catch( IOException e )
		{
			throw new ImageGSException(this.getClass().getName() + " >> "
					+ e.getMessage() );	
		}			
		finally
		{
			if(out != null) try { out.close(); } catch(IOException se) {}
			
		}		
	}

	/**
	 * Single-TIFF Ѵ.
	 * 
	 * @param fname  ϸ
	 * @param image BufferedImage type
	 */	
	public void saveRenderedImage( String fname, RenderedImage image ) throws ImageGSException
	{
		OutputStream out = null;
		
		try
		{
			out = new FileOutputStream( fname );
			TIFFEncodeParam encodeParam = new TIFFEncodeParam();
			encodeParam.setCompression( compressMethod );
			ImageEncoder encoder = ImageCodec.createImageEncoder("TIFF", out, encodeParam );
			encoder.encode( image );
			out.close();
		}
		catch( IOException e )
		{
			throw new ImageGSException(this.getClass().getName() + " >> "
					+ e.getMessage() );	
		}			
		finally
		{
			if(out != null) try { out.close(); } catch(IOException se) {}
			
		}
	}
	
	/**
	 * TIFF  JPEG_TTN2  Ѵ.
	 */
	public void setCompressTIFF()
	{
		setCompressTIFF( TIFFEncodeParam.COMPRESSION_JPEG_TTN2 );
	}

	/**
	 * TIFF  Ѵ.
	 * @param method - 0 : No compression
	 *                            1 : Byte-oriented run-length encoding "PackBits" compression
	 *                            2 : Modified Huffman Compression (CCITT Run Length Encoding (RLE))
	 *                            3 : CCITT T.6 bilevel compression (Group 3 facsimile compression)
	 *                            4 : CCITT T.6 bilevel compression (Group 4 facsimile compression)
	 *                            5 : JPEG-in-TIFF compression
	 *                            6 : DEFLATE lossless compression (also known as "Zip-in-TIFF")
	 */	
	public void setCompressTIFF( int method )
	{
		switch( method )
		{
			case 0 : 
						compressMethod = TIFFEncodeParam.COMPRESSION_NONE;
						break;

			case 1 : 
						compressMethod = TIFFEncodeParam.COMPRESSION_PACKBITS;
						break;

			case 2 : 
						compressMethod = TIFFEncodeParam.COMPRESSION_GROUP3_1D;
						break;
			
			case 3 : 
						compressMethod = TIFFEncodeParam.COMPRESSION_GROUP3_2D;
				        break;
				        
	        case 4 : 
	        			compressMethod = TIFFEncodeParam.COMPRESSION_GROUP4;
	        			break;	
	        
	        case 5 : 
	        			compressMethod = TIFFEncodeParam.COMPRESSION_JPEG_TTN2;
	        			break;
	        
	        case 6 : 
	        			compressMethod = TIFFEncodeParam.COMPRESSION_DEFLATE;
	        			break;
	        
	        default :
	        			compressMethod = TIFFEncodeParam.COMPRESSION_NONE;
	        			break;
				        
		}
	}
	
	/**
	 * TIFF    Ұ Ѵ.
	 */
	public String printCompressTIFFMethod()
	{
		StringBuffer sb = new StringBuffer();
		
		sb.append("0 : No compression");
		sb.append("1 : Byte-oriented run-length encoding \"PackBits\" compression");
		sb.append("2 : Modified Huffman Compression (CCITT Run Length Encoding (RLE))");
		sb.append("3 : CCITT T.6 bilevel compression (Group 3 facsimile compression)");
		sb.append("4 : CCITT T.6 bilevel compression (Group 4 facsimile compression)");
		sb.append("5 : JPEG-in-TIFF compression");
		sb.append("6 : DEFLATE lossless compression (also known as \"Zip-in-TIFF\")");
		
		return sb.toString();
	}

	private void searchBinaryGrayImage( Raster raster, int rgba[] )
	{
		// height width ݴ   ִ ȮѴ.
		int tmpHeight = width;
		int tmpWidth = height;
		
		if( tmpHeight == width && tmpWidth == height )
		{
			rotateType = ROTATE_Y;
		}
		else 
		{
			rotateType = ROTATE_N;
		}
		
		// rotate̸ ٷ ش.
		if( rotateType == ROTATE_Y ) 
		{
			// swap
			int tmp = height ;
			height = width;
			width = tmp;
		}
		
		grayChannel = InitIMGBuf.ShortIMGBuf( height, width );
		imageType = GRAY;
		int[] histogram = new int[256];

		for(int i=0; i<height; i++)
		{
			for(int j=0; j<width; j++)
			{

				rgba = raster.getPixel( i, j, (int[]) null );
				grayChannel[i][j] = (short)rgba[0];
				histogram[ grayChannel[i][j] ]++;
			}
		}

		int pixels = 0;
		for(int i=2; i<256; i++)
		{
			if( histogram[i] != 0 ) pixels++;
		}
		
		if( pixels == 0 ) imageType = BINARY;
		
		// rotate̸ ٷ ش.
		if( rotateType == ROTATE_Y ) 
		{
			// swap
			int tmp = width ;
			width = height;
			height = tmp;
		}
	}
	
}
