//===================================================================
//
// uart_gets.c (@haekim)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================

#include "uart.h"

#ifdef UART_M
#include "platform.h"
#include "arch.h"
#include "critical_section.h"
#include "intr.h"

static volatile UINT8 char_cnt;
static volatile INT8 *rx_str;

#ifdef KERNEL_M
static UINT8 running = FALSE;
#endif

static UINT8 max_str_size;
static UINT8 put_port;
extern volatile UINT8 uart_rx_char;
extern void (*rx0_callback)(UINT8);
extern void (*rx1_callback)(UINT8);
extern void (*sched_callback)(void);
#ifdef KERNEL_M
extern void nos_thread_sleep(UINT16 ticks);
#endif

static void nos_uart_rx_bottom_half(UINT8 rx_char)
{
	switch (rx_char)
	{
		case _CR: // carriage return (ENTER Key)
			rx_str[char_cnt] = '\0';
			nos_uart_putc(put_port, '\n');
			nos_uart_putc(put_port, _CR);
			NOS_DISABLE_UART_RX_INTR(put_port);
			break;
		case _BS: // back space
			if ( char_cnt > 0)
			{
				nos_uart_putc(put_port, _BS);
				nos_uart_putc(put_port, ' ');
				nos_uart_putc(put_port, _BS);
				char_cnt--;
			}
			break;
		default:
			if ( char_cnt < max_str_size-1 )	// The last allocated memory is reserved for termination character '\0'.
			{
				rx_str[char_cnt++] = rx_char;
				nos_uart_putc(put_port, rx_char);				
				break;
			}
			else
			{
				rx_str[char_cnt-1] = rx_char;
				nos_uart_putc(put_port, _BS);
				nos_uart_putc(put_port,rx_char);
				break;
			}
	}
}


void nos_uart_gets(UINT8 port_num, INT8 *str, UINT8 str_size)
{
	// saving
	void (*temp_callback)(UINT8);
	volatile BOOL temp_uart_rx_intr;

	if (str_size < 2)
	{
		nos_uart_puts(STDOUT, "\n\rERROR : The string size must be bigger than '2'. One slot  is for termination character.\n\r");
		return;
	}
	

#ifdef KERNEL_M
	// prohibit re-entrance
	while (running)
	{
		nos_thread_sleep(1);		//sleep 1 tick if other thread is using this function now.
	}
#endif

	
	NOS_ENTER_CRITICAL_SECTION();
#ifdef KERNEL_M
	running = TRUE;
#endif
	char_cnt = 0;
	uart_rx_char = '\0';
	rx_str = str;
	max_str_size = str_size;
	put_port = port_num;	
	if (port_num == 0)
	{
		temp_callback = rx0_callback;
		rx0_callback = nos_uart_rx_bottom_half;
		if (USART0_RX_vect_IS_SET())
		{
			temp_uart_rx_intr = TRUE;
		}
		else
		{
			temp_uart_rx_intr = FALSE;
			NOS_ENABLE_UART_RX_INTR(0);			
		}
	}
	else
	{
		temp_callback = rx1_callback;
		rx1_callback = nos_uart_rx_bottom_half;	
		if (USART1_RX_vect_IS_SET())
		{
			temp_uart_rx_intr = TRUE;
		}
		else
		{
			temp_uart_rx_intr = FALSE;
			NOS_ENABLE_UART_RX_INTR(1);			
		}		
	}
	NOS_EXIT_CRITICAL_SECTION();

	
	// receives string
	do {
#ifdef KERNEL_M
		// if "nos_start_sched()" has been called.
		if ( sched_callback )
		{
			nos_thread_sleep(1);
		}
#endif		
	} while ( uart_rx_char != _CR );


	NOS_ENTER_CRITICAL_SECTION();
	// restoring
	if ( port_num == 0 )
	{
		rx0_callback = temp_callback;
		if (temp_uart_rx_intr)
		{
			NOS_ENABLE_UART_RX_INTR(0);
		}
	}
	else
	{
		rx1_callback = temp_callback;
		if (temp_uart_rx_intr)
		{
			NOS_ENABLE_UART_RX_INTR(1);
		}		
	}
#ifdef KERNEL_M
	running = FALSE;
#endif
	NOS_EXIT_CRITICAL_SECTION();
}

#endif // UART_M 
