//===================================================================
//
// sched.c (@jun361, @sheart, @haekim)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================

#include "sched.h"

#ifdef KERNEL_M
#include <string.h>
#include "hal_sched.h"	
#include "heap.h"
#include "arch.h"
#include "critical_section.h"
#include "intr.h"
#include "pwr_mgr.h"

#include "hal_thread.h"
#include "taskq.h"
#include "user_timer.h"
#include "thread.h"

/****sched variables***/
UINT8 	_rtid; 			// running thread id (to indicate which thread is currently running)
UINT8 	_ready_priority_bit;	// priority bit setting for threads, which is set for the priority of the thread when the thread becomes in ready state.
#ifndef THREAD_EXT_M
UINT8   _created_tid_bit;       // used for avoiding duplicated thread id, which is set when a thread is created
UINT8   _suspended_tid_bit;
UINT8   _sleep_tid_bit;         // indicates whether a thread is in sleeping state or not
#else
UINT16  _created_tid_bit;       // used for avoiding duplicated thread id, which is set when a thread is created
UINT16   _suspended_tid_bit;
UINT16  _sleep_tid_bit;         // indicates whether a thread is in sleeping state or not
#endif
BOOL 	_sched_enable_bit;

UINT16 	_init_ticks_left; // ticks left until first wake-up (initially setting)
UINT16 	_ticks_left; // ticks left currently

THREADQ _rdy_q[PRIORITY_LEVEL_COUNT]; 
// array of ready queues for each thread priority
// this queue enables to keep the order among threads with same priorities
// The concept looks like this :
//
//      prio[7] prio[6] prio[5] prio[4] prio[3] prio[2] prio[1] prio[0]
// 	  |	  |       |       |       |       |        |      |
//       ---     ---     ---     ---     ---     ---      ---    ---
//       |-|     |-|     |-|     |-|     |-|     |-|      |-|    |-|
//       |-|     |-|     |-|     |-|     |-|     |-|      |-|    |-|
//       |-|     |-|     |-|     |-|     |-|     |-|      |-|    |-|
//       |-|     |-|     |-|     |-|     |-|     |-|      |-|    |-|
//       ---     ---     ---     ---     ---     ---      ---    ---
//     _rdy_q[7] ...     ...  _rdy_q[4]  ...  _rdy_q[2]  ...   _rdy_q[0]
//

// global and temporal variables for Nano OS kernel
UINT8  _Gi;
UINT16 _Gj;
UINT8  _Gk;

#ifdef SWAPSTK_M
UINT16	   _common_stack_data_size;
STACK_PTR  _common_stack_index;
STACK_PTR  _thread_stack_index;
#endif

_TCB *tcb[MAX_NUM_TOTAL_THREAD]; // tcb pointer array
//

extern volatile UINT8 nested_intr_cnt;
extern volatile INTR_STATUS intr_status;
extern void (*sched_callback)(void);   // this varable indicates the scheduler is working or not.
extern USER_TIMER* timer[MAX_NUM_TOTAL_TIMER];


// __attribute__ ((naked)) keyword does not push any local variables in the function to the stack automatically by the avr-gcc compiler.
// This function consists of three steps. 
// 1) Save current thread state
// 2) Scheduling
// 3) Load new thread state

#ifdef atmega128
void nos_context_switch_core(void)
#endif
#ifdef msp430x1611
__attribute__ ((naked)) void nos_context_switch_core(void)
#endif
{
	//===================Section for thread saving  ===================//
	// Save current thread state
	NOS_THREAD_SAVE_STATE(tcb[_rtid]);

#ifdef SWAPSTK_M
	_common_stack_data_size = RAMEND - (UINT16)(tcb[_rtid]->sptr - 1);// real data + 1 vacant stack (bottom or top, depends on Architecture)
	if (tcb[_rtid]->stack_size != _common_stack_data_size) 
	{
		// free previously allocated swap-stack region
		nos_free(tcb[_rtid]->stack_start_addr);

		// reallocate swap-stack region to store current stack
		tcb[_rtid]->stack_start_addr = nos_malloc(_common_stack_data_size);
		tcb[_rtid]->stack_size = _common_stack_data_size;
	}
	// copy stack data of common stack region to swap-stack region of current stack
	memcpy(tcb[_rtid]->stack_start_addr, tcb[_rtid]->sptr, _common_stack_data_size);
#endif
	// If the thread is in running state, make it to be in ready state
	if (tcb[_rtid]->state == RUNNING_STATE)
	{
		_Gi = tcb[_rtid]->priority;	// see the priority of the current thread
		_BIT_SET(_ready_priority_bit, _Gi);
	       	THREADQ_ENQ(_rdy_q[_Gi], _rtid);   
		tcb[_rtid]->state = READY_STATE;
	}

	//===================Section for thread restoring  ===================//
 	//Scheduling : Preemption RR scheduling

	// find a thread with highest priority
	for (_Gi=PRIORITY_LEVEL_COUNT-1; _Gi>0; _Gi--)
	{
		if (_ready_priority_bit & (1 << _Gi)) break; // found 
	}

	// get the thread id of the thread with highest priority
	THREADQ_FETCH(_rdy_q[_Gi], &_rtid);
	if (THREADQ_IS_EMPTY(_rdy_q[_Gi])) 
		_BIT_CLR(_ready_priority_bit, _Gi); // because the thread queue for this priority is empty, the corresponding bit in _ready_priority_bit is cleared. 

	// the selected thread will run soon after restoring context
	tcb[_rtid]->state = RUNNING_STATE;
#ifdef SWAPSTK_M
	// Must not use memcpy. The return PC for memcpy will be overritten by memcpy itself.	//memcpy((STACK_PTR)(RAMEND - tcb[_rtid]->stack_size) + 1, tcb[_rtid]->stack_start_addr, tcb[_rtid]->stack_size);
	_common_stack_index = (STACK_PTR)(RAMEND - tcb[_rtid]->stack_size ) + 1;
	_thread_stack_index = tcb[_rtid]->stack_start_addr;
	for (int i=0; i<tcb[_rtid]->stack_size; ++i)
	{
		*_common_stack_index = *_thread_stack_index;
		++_thread_stack_index;
		++_common_stack_index;
	}
#endif

	//Load new thread state
	NOS_THREAD_LOAD_STATE(tcb[_rtid]);	
	NOS_RETURN();	// __asm__ volatile ("ret\n\t")
}



// This function is invoked at every SCHED_TIMER_MS to switch context among threads.
// It wakes up sleeping threads whose sleeping time is expired
static void nos_sched_handler(void)
{
	// 1) Handling sleeping threads waiting for time-expiration 

	if (_sleep_tid_bit)
	{
		if (--_ticks_left == 0) // expired
		{
			_ticks_left = 65535; // need to find a minimum for efficiency

			// for all threads
			for (_Gi=1; _Gi<MAX_NUM_TOTAL_THREAD; _Gi++)
			{
				if (NOS_THREAD_IS_SLEEP(_Gi)) // inspect each sleeping thread one by one
				{
					tcb[_Gi]->sleep_tick -= _init_ticks_left;

					if (tcb[_Gi]->sleep_tick == 0) // expired; so wake up the thread
					{
						_BIT_CLR(_sleep_tid_bit, _Gi);
						_Gk = tcb[_Gi]->priority;
						tcb[_Gi]->state = READY_STATE;
						THREADQ_ENQ(_rdy_q[_Gk], _Gi);
						_BIT_SET(_ready_priority_bit, _Gk);
					}

					if (tcb[_Gi]->sleep_tick != 0 && _ticks_left > tcb[_Gi]->sleep_tick) // find a minumum
						_ticks_left = tcb[_Gi]->sleep_tick;
				}
			}
			_init_ticks_left = _ticks_left;
		}
	}

	if (_activated_timer_bit)
	{
		if (--_timer_ticks_left == 0) // expired
		{
			_timer_ticks_left = 65535; // need to find a minimum for efficiency

			// for all threads
			for (_Gi=0; _Gi<MAX_NUM_TOTAL_TIMER; _Gi++)
			{
				if (NOS_TIMER_IS_ACTIVATED(_Gi)) // inspect each activated timer one by one
				{
					timer[_Gi]->timer_tick -= _init_timer_ticks_left;

					if (timer[_Gi]->timer_tick == 0) // expired; so prepare to call the timer function
					{
						_BIT_SET(_expired_timer_bit, _Gi);

						if (timer[_Gi]->opt == TIMER_PERIODIC) // replenish if periodic timer
							timer[_Gi]->timer_tick = timer[_Gi]->init_tick;
					}

					if (timer[_Gi]->timer_tick != 0 && _timer_ticks_left > timer[_Gi]->timer_tick) // find a minumum
						_timer_ticks_left = timer[_Gi]->timer_tick;
				}
			}
			_init_timer_ticks_left = _timer_ticks_left;

			if (_expired_timer_bit)
			{
				// Do something -> make a thread ready for execution
				// Below is the same as nos_thread_priority_change(0, PRIORITY_ULTRA);
			        // if an idle thread is in ready state
        			if (tcb[0]->state == READY_STATE)
        			{
                			_Gi = tcb[0]->priority;
                			THREADQ_DEQ(_rdy_q[_Gi], 0);
                			if (THREADQ_IS_EMPTY(_rdy_q[_Gi]))
                        			_BIT_CLR(_ready_priority_bit, _Gi);

                			THREADQ_ENQ(_rdy_q[PRIORITY_ULTRA], 0);
                			tcb[0]->priority = PRIORITY_ULTRA;
                			_BIT_SET(_ready_priority_bit, PRIORITY_ULTRA);
        			}
        			else // the thread in other states
				{
                			tcb[0]->priority = PRIORITY_ULTRA; // just change its priority
				}
			}
		}
	}

	// 2) Check if context switching is necessary 
	//
	// _ready_priority_bit=0x00 : there are no threads in the ready queue (This happens when the program starts).
	// In this case, no context switching is required because no threads currently exists to be switched.

	if (_ready_priority_bit != 0x00 && _sched_enable_bit) // if there are any threads in the ready queue except an idle thread 
	{
		nos_context_switch_core();
	}
}

void nos_ctx_sw()
{
	//if (nested_intr_cnt || intr_status.cnt )
	if (!NOS_IS_CTX_SW_ALLOWABLE())
	{
		// Set the next scheduling timer flag
		SET_KERNEL_TIMER_FLAG();
		return;
	}

	NOS_DISABLE_GLOBAL_INTERRUPT();

	nos_context_switch_core();

	NOS_ENABLE_GLOBAL_INTERRUPT();
}


void nos_sched_callback(void (*func)(void))
{
        // Set scheduler interrupt callback function
        sched_callback = func;
}

void nos_sched_init()
{
	UINT8 i;

	// When the program starts, only main thread is running.
	// The main thread plays a role as an idle thread in a multi-threaded programming context.
	// The id and priority of the main thread are set to 0.
	// This implies that the main thread is executed only when the other threads are not running.
	//
	// A corresponding bit (for the id of each thread) is set or unset (BIT-OPERATION) for a variable related to a thread.
	// e.g. if a thread of tid=2 is in asleep, _sleep_tid_bit = 0b00000100 = 0x02

	_rtid 		    = 0;  // main thread (tid=0) is initially running
	_ready_priority_bit = 0;  // initially empty (no threads are in ready state)
	_created_tid_bit    = 1;  // the main thread (tid=0) is created ((0-th bit is set) 0b00000001 = 0x01)
	_sleep_tid_bit      = 0;  // no threads in sleep state
	_suspended_tid_bit  = 0;  // no threads in suspended state

	// The next two variables are used only when there are any sleeping threads.
	// for sleeping threads
	_init_ticks_left = 65535;
	_ticks_left = 0;

	// for timers
	_created_timer_bit 	= 0;
	_activated_timer_bit 	= 0;
	_expired_timer_bit 	= 0;

	_init_timer_ticks_left 	= 65535;
	_timer_ticks_left 	= 0;

	// Initialize ready queue
	// There is a corresponding ready queue for each priority.
	for (i=0; i<PRIORITY_LEVEL_COUNT; i++)
	{
		THREADQ_INIT(_rdy_q[i]); // initialize the ready queue for the priority i
	}

	// tcb setting for the main thread (= the idle thread)
	// the main thread is initially running
	// the priority of the main thread is 0 (lowest priority)
	tcb[0] = nos_malloc(sizeof(struct _tcb));
#ifdef SWAPSTK_M
	tcb[0]->stack_start_addr = nos_malloc(DEFAULT_STACK_SIZE);
	tcb[0]->stack_size 	= DEFAULT_STACK_SIZE;
#else
	tcb[0]->stack_start_addr = (STACK_PTR) (RAMEND - SYSTEM_STACK_SIZE + 1);
	tcb[0]->stack_size 	= SYSTEM_STACK_SIZE;
#endif
	tcb[0]->id			= 0;
	tcb[0]->state 		= RUNNING_STATE;
	tcb[0]->priority 	= PRIORITY_LOWEST;

	// Setting of periodic scheduler interrupt
	nos_sched_hal_init();

	// Set scheduler callback function
	nos_sched_callback( nos_sched_handler );
}

// The nos_sched_start() is called by the main thread (= idle thread).
// This function is exeucted by the idle thread infinitely.
// That this funtion (by the idle thread) is about to run implies that no threads are not in ready state.
// Since ihe idle thread (executing this function) does no work but uselessly occupies the CPU,
// it should frequently go into power saving mode to save CPU power.

void nos_sched_start()
{
	// initial settings for power management
	NOS_SET_SLEEP_MODE(NOS_IDLE);	// set sleep mode that MCU can be awaken by sched_timer_interrupt or uart_rx_interrupt
	nos_sched_timer_start();

#ifdef ENABLE_SCHEDULING
	NOS_ENABLE_SCHED();
#else
	NOS_DISABLE_SCHED();
#endif
	nos_ctx_sw(); // context switch for the other threads to occupy the CPU directly
	while(1)
	{
		if (tcb[0]->priority == PRIORITY_ULTRA)
		{
			// Process Task Queue here
			if (!NOS_TASKQ_IS_EMPTY())
			{
				nos_taskq_exe();
			}

			// Process Timer Functions here
			if (_expired_timer_bit)
			{
				nos_timer_exe();
				continue;
			}
			nos_thread_priority_change(0, PRIORITY_LOWEST);
			nos_ctx_sw();
		}

		// go into power saving mode. 
		// this thread will be automatically awaken by hardware.
		else if (tcb[0]->priority == PRIORITY_LOWEST)	// run as idle thread
		{
			NOS_SLEEP_ENABLE();
			NOS_SLEEP_MCU(); 
			NOS_SLEEP_DISABLE();
		}
	}
}
#endif
