//===================================================================
//
// thread.h (@jun361, @sheart)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================
#ifndef THREAD_H
#define THREAD_H
#include "kconf.h"
#ifdef KERNEL_M

#include "common.h"

#include "hal_thread.h"
#include "hal_sched.h"
#ifndef THREAD_EXT_M
#include "threadq.h"
#else
#include "threadq_ext.h"
#endif


// the maximum number of threads supported
#ifndef THREAD_EXT_M
#define MAX_NUM_USER_THREAD     (5)  // the number of user threads
#else
#define MAX_NUM_USER_THREAD     (15)  // the number of user threads
#endif

#define MAX_NUM_TOTAL_THREAD	(MAX_NUM_USER_THREAD+1)  // the number of user threads + idle thread

// thread states
#define READY_STATE		(1)
#define RUNNING_STATE		(2)
#define WAITING_STATE		(3)
#define SLEEPING_STATE		(4)
#define EXIT_STATE		(5)

// define errors
#define THREAD_NO_ERROR				(0)
#ifdef THREAD_MANUAL_NUMBERING_M
#define THREAD_ID_DUPLICATE_ERROR		(1)
#define THREAD_PRIORITY_OUT_OF_RANGE_ERROR	(2)
#else
#define THREAD_ID_EXHAUST_ERROR			(-1)
#define THREAD_PRIORITY_OUT_OF_RANGE_ERROR	(-2)
#endif
// priority errors
#define THREAD_PRIORITY_CHANGE_ERROR		(3)
// resume errors
#define THREAD_RESUME_ERROR			(4)
#define THREAD_WAKEUP_ERROR			(5)
#define THREAD_SLEEP_ERROR			(6)
#define THREAD_KILL_ERROR			(7)
#define THREAD_EXIT_ERROR			(8)

// Variables and Functions
// thread control block


typedef struct _tcb
{
	void 		(*func)(void *);  // function pointer
	void		*args_data;	  // function arguments
	UINT8 		id;		  // thread id (tid)
	UINT8 		state;		  // thread state
	UINT16  	sleep_tick;	  // the number of ticks to sleep; sleepdetected whenever timer event occurs
	UINT8   	priority; 	  // thread priority
	THREADQ		waitq;		  // a set of threads for this thread to exit
	STACK_PTR	sptr;   	  // thread stack pointer
	UINT16		stack_size;	  // thread stack size
	STACK_PTR  	stack_start_addr; // to detect stack overflow, but (main) idle thread has no limit
}_TCB;


//extern variables
extern UINT8 	_rtid; 			// running thread id (to indicate which thread is currently running)

#ifndef THREAD_EXT_M
extern UINT8   _created_tid_bit;       // used for avoiding duplicated thread id, which is set when a thread is created
extern UINT8   _suspended_tid_bit;
extern UINT8   _sleep_tid_bit;         // indicates whether a thread is in sleeping state or not
#else
extern UINT16  _created_tid_bit;       // used for avoiding duplicated thread id, which is set when a thread is created
extern UINT16   _suspended_tid_bit;
extern UINT16  _sleep_tid_bit;         // indicates whether a thread is in sleeping state or not
#endif

extern UINT16 	_init_ticks_left; // ticks left until first wake-up (initially setting)
extern UINT16 	_ticks_left; // ticks left currently

extern UINT8  _Gi;
extern UINT16 _Gj;
extern UINT8  _Gk;

extern _TCB *tcb[MAX_NUM_TOTAL_THREAD]; // tcb pointer array


// BIT OPERATION VARIABLES; for a thread with tid k, k-th bit of the variable is set or clear.
// The next defitinitions were not currently be used (will be considered later).
#define NOS_THREAD_IS_READY(tid)		(_ready_priority_bit & (1 << tid))
#define NOS_THREAD_IS_CREATED(tid)		(_created_tid_bit & (1 << tid))
#define NOS_THREAD_IS_SLEEP(tid)		(_sleep_tid_bit & (1 << tid))
#define NOS_THREAD_IS_SUSPENDED(tid)		(_suspended_tid_bit & (1 << tid))
#define NOS_THREAD_IS_NOT_READY(tid)		(~_ready_priority_bit & (1 << tid))
#define NOS_THREAD_IS_NOT_CREATED(tid)		(~_created_tid_bit & (1 << tid))
#define NOS_THREAD_IS_NOT_SLEEP(tid)		(~_sleep_tid_bit & (1 << tid))
#define NOS_THREAD_IS_NOT_SUSPENDED(tid)	(~_suspended_tid_bit & (1 << tid))


#ifdef THREAD_MANUAL_NUMBERING_M
UINT8 nos_thread_create(UINT8 id, void (*func)(void *args), void *args_data, UINT16 additional_stack_size, UINT8 priority);
#else
INT8 nos_thread_create(void (*func)(void *args), void *args_data, UINT16 stack_size, UINT8 priority);
#endif
UINT8 nos_thread_exit(void);
UINT8 nos_thread_join(UINT8 tid);
UINT8 nos_thread_suspend(UINT8 tid);
UINT8 nos_thread_resume(UINT8 tid);
UINT8 nos_thread_priority_change(UINT8 tid, UINT8 new_priority);
UINT8 nos_thread_sleep(UINT16 ticks);
UINT8 nos_thread_kill(UINT8 tid);
UINT8 nos_thread_wakeup(UINT8 tid);

// returns thread information
#define NOS_GET_THREAD_ID()			(tcb[_rtid]->id)
#define NOS_GET_THREAD_STATE(tid)		(tcb[tid]->state)
#define NOS_GET_THREAD_PRIORITY(tid)		(tcb[tid]->priority)
#define NOS_GET_THREAD_STACK_POINTER(tid)	(tcb[tid]->sptr)
#define NOS_GET_THREAD_STACK_UNUSED_SPACE(tid)	\
	((UINT8*)(tcb[tid]->sptr - tcb[tid]->stack_start_addr))
UINT8 nos_get_num_user_threads(void);

// thread sleep(..)
#define nos_thread_sleep_ms(ms)		nos_thread_sleep((UINT16)(ms-1+SCHED_TIMER_MS)/SCHED_TIMER_MS)
#define nos_thread_sleep_sec(sec)	nos_thread_sleep((UINT16)(1000*sec-1+SCHED_TIMER_MS)/SCHED_TIMER_MS)
#endif // KERNEL_M
#endif // ~THREAD_H
