//===================================================================
//
// thread_sleep.c (@jun361, @sheart)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================

#include "thread.h"

#ifdef KERNEL_M
#include "arch.h"
#include "critical_section.h"
#include "sched.h"
// The thread_sleep(UINT16 ms) function suspends the thread for 'ms' milli-seconds.
// In the mean time, other threads may be running.
// But note that accurate sleeping time is not guaranteed for two reasons. 
//
// First, because it is calculated as unit of scheduling time interval,
// the sleeping time values may be rounded.
// For example, if SCHED_TIMER_MS is 250ms, this function sleeps the thread only in the unit of 250ms (250ms, 500ms, 750ms, etc).
//
// Second, after this thread is awaken, it is not executed immediately. 
// It is inserted into the ready queue and waits for being selected by the scheduler.
//
// Thus, the sleeping time may be more longer than expected.
// Don't use this function when the exact time requirement is needed.

UINT8 nos_thread_sleep(UINT16 ticks)
{
	UINT8 i;
	UINT16 ticks_diff;

	if (ticks == 0) 
		return THREAD_SLEEP_ERROR;

NOS_ENTER_CRITICAL_SECTION();

	ticks_diff = _init_ticks_left - _ticks_left; // time elapsed in ticks_diff
	_ticks_left = ticks;

	if (_sleep_tid_bit) // if there is a sleeping thread, then, ticks_init_left and _ticks_left will be greater than 0
	{
		// for all sleeping threads, find a minimum _init_ticks_left after substracting ticks_diff
		for (i=1; i<MAX_NUM_TOTAL_THREAD; i++)
		{
			if (NOS_THREAD_IS_SLEEP(i)) 
			{
				tcb[i]->sleep_tick -= ticks_diff;
				if (_ticks_left > tcb[i]->sleep_tick)
					_ticks_left = tcb[i]->sleep_tick;
			}
		}
	}

	_init_ticks_left = _ticks_left;

	tcb[_rtid]->sleep_tick = ticks;  // sleeping time of this thread in ticks
	_BIT_SET(_sleep_tid_bit, _rtid);   // indicates that this thread goes into sleep
	tcb[_rtid]->state = SLEEPING_STATE; // this thread is being in sleep state

NOS_EXIT_CRITICAL_SECTION();
	// context switching so that other thread can occupy the CPU while this thread is sleeping.
	nos_ctx_sw();
	return THREAD_NO_ERROR;
}
#endif
