//===================================================================
//
// user_timer.c (@sheart)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================

#include "user_timer.h"

#ifdef KERNEL_M
#include "sched.h"
#include "heap.h"
#include "critical_section.h"

USER_TIMER* timer[MAX_NUM_TOTAL_TIMER];

UINT8 nos_timer_create(void (*func)(void), UINT16 ticks, UINT8 opt)
{
	UINT8 tmid;

	NOS_ENTER_CRITICAL_SECTION();
	if (ticks == 0 || (opt!=TIMER_ONE_SHOT && opt!=TIMER_PERIODIC))
	{	
		NOS_EXIT_CRITICAL_SECTION();
		return TIMER_CREATE_ERROR;
	}
	if (_created_timer_bit == 0xff) // full in slots
	{
		NOS_EXIT_CRITICAL_SECTION();
		return TIMER_CREATE_ERROR;
	}
	for (tmid=0; tmid<MAX_NUM_TOTAL_TIMER; ++tmid)
	{
		if (~_created_timer_bit & (1 << tmid))
		{
			break; // found an empty slot
		}
	}
	if (tmid >= MAX_NUM_TOTAL_TIMER)
	{
		NOS_EXIT_CRITICAL_SECTION();
		return TIMER_CREATE_ERROR;
	}

	_BIT_SET(_created_timer_bit, tmid);   // indicates that this timer is ready

	timer[tmid] = nos_malloc(sizeof(USER_TIMER));

	timer[tmid]->tmid		= tmid; // timer id
	timer[tmid]->init_tick  	= ticks;  // initial sleeping time of this thread in ticks
	timer[tmid]->timer_tick 	= ticks;  // sleeping time of this timer in ticks
	timer[tmid]->func		= func;
	timer[tmid]->opt		= opt;

	// Any created timer will be activated automatically
	if (nos_timer_activate(tmid) == TIMER_ACTIVATE_ERROR)
	{
		NOS_EXIT_CRITICAL_SECTION();
		return TIMER_CREATE_ERROR;
	}

	NOS_EXIT_CRITICAL_SECTION();
	return tmid;
}

UINT8 nos_timer_destroy(UINT8 tmid)
{
NOS_ENTER_CRITICAL_SECTION();
	if (NOS_TIMER_IS_NOT_CREATED(tmid))
	{
		NOS_EXIT_CRITICAL_SECTION();
		return TIMER_DESTROY_ERROR;
	}

	nos_timer_deactivate(tmid);
	_BIT_CLR(_created_timer_bit, tmid);   // indicates that this timer is deleted
	nos_free(timer[tmid]);
NOS_EXIT_CRITICAL_SECTION();
	return TIMER_NO_ERROR;
}

UINT8 nos_timer_activate(UINT8 tmid)
{
	UINT8 i;
	UINT16 ticks_diff;

NOS_ENTER_CRITICAL_SECTION();
        if (NOS_TIMER_IS_NOT_CREATED(tmid))
        {
                NOS_EXIT_CRITICAL_SECTION();
                return TIMER_ACTIVATE_ERROR;
        }

	ticks_diff = _init_timer_ticks_left - _timer_ticks_left; // time elapsed in ticks_diff
	_timer_ticks_left = timer[tmid]->init_tick;

	if (_activated_timer_bit) // if there is an activated timer, then, _init_timer_ticks_left and _timer_ticks_left will be greater than 0
	{
		// for all sleeping threads, find a minimum _init_ticks_left after substracting ticks_diff
		for (i=0; i<MAX_NUM_TOTAL_TIMER; i++)
		{
			if (NOS_TIMER_IS_ACTIVATED(i)) 
			{
				timer[i]->timer_tick -= ticks_diff;
				if (_timer_ticks_left > timer[i]->timer_tick)
					_timer_ticks_left = timer[i]->timer_tick;
			}
		}
	}

	_init_timer_ticks_left = _timer_ticks_left;
	_BIT_SET(_activated_timer_bit, tmid);   // indicates that this timer is activated

NOS_EXIT_CRITICAL_SECTION();
        return TIMER_NO_ERROR;
}

UINT8 nos_timer_deactivate(UINT8 tmid)
{
	UINT8 i;
    UINT16 ticks_diff;

NOS_ENTER_CRITICAL_SECTION();
        if (NOS_TIMER_IS_NOT_ACTIVATED(tmid))
        {
                NOS_EXIT_CRITICAL_SECTION();
                return TIMER_DEACTIVATE_ERROR;
        }
        else // activated 
        {
                // deactivate this timer
                _BIT_CLR(_activated_timer_bit, tmid);
                timer[tmid]->timer_tick = 0;

                if (_activated_timer_bit)
                {
                        ticks_diff = _init_timer_ticks_left - _timer_ticks_left;
                        _timer_ticks_left = 65535;

                        // for all activated timers, find a minimum _init_ticks_left after substracting ticks_diff
                        for (i=0; i<MAX_NUM_TOTAL_TIMER; i++)
                        {
                                if (NOS_TIMER_IS_ACTIVATED(i))
                                {
                                        timer[i]->timer_tick -= ticks_diff;
                                        if (_timer_ticks_left > timer[i]->timer_tick)
                                        	_timer_ticks_left = timer[i]->timer_tick;
                                }
                        }
                        _init_timer_ticks_left = _timer_ticks_left;
                }

NOS_EXIT_CRITICAL_SECTION();
                return TIMER_NO_ERROR;
        }
}

void nos_timer_exe()
{
	UINT8 i;
	NOS_ENTER_CRITICAL_SECTION();
	for (i=0; i<MAX_NUM_TOTAL_TIMER; ++i)
	{
		if (NOS_TIMER_IS_EXPIRED(i)) // inspect each activated timer one by one
		{
			// execute the timer function
			NOS_EXIT_CRITICAL_SECTION();
			(timer[i]->func)();
			NOS_ENTER_CRITICAL_SECTION();

			_BIT_CLR(_expired_timer_bit, i);
			if (timer[i]->opt == TIMER_ONE_SHOT) // destroy one shot-timer
			{
				nos_timer_destroy(i);
			}
		}
	}
	NOS_EXIT_CRITICAL_SECTION();
}

#endif
