//===================================================================
//
// sensor_us.c (@sheart, @haekim, @jun361)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================
#include "sensor_us.h"
#ifdef SENSOR_US_M
#include <avr/interrupt.h>
#include "critical_section.h"
#include "arch.h"

static void (*callback_func)(void);
static volatile BOOL us_trigger;
static volatile UINT16 round_trip_counter;

void nos_us_init()
{
	callback_func = NULL;
	_BIT_SET(DDRA, US_SENSOR_POWER_SW); 
	_BIT_SET(DDRF, US_SENSOR_TX_SW);
	NOS_US_POWER_ON();
	NOS_US_TX_OFF();
	
	// US RX interrupt setting
	EICRB |= (1 << ISC71) | (0 << ISC70); // 00 : low level, 01 : any change, 10: falling edge, 11:rising edge
	DISABLE_US_INTR();
	
	// Timer3 on for US
	TCCR3B |= (0 << CS32) | (1 << CS31) | (1 << CS30); // 1/64 clk = 800000/64 = 125000 Hz
}

void nos_us_callback(void (*func)(void))
{
	callback_func = func;
}

void nos_us_trigger()
{
	UINT8 waiting_us;
	us_trigger=FALSE;
	ENABLE_US_INTR();
	NOS_US_TX_ON();
	TCNT3 = 0;
	// Maximum delay 65535tick / (125tick/msec) = 524.28ms
	for ( waiting_us =0; waiting_us < 250; waiting_us++ )
	{
		nos_delay_ms(3);
		if ( us_trigger )
		{
			return;
		}
	}
	DISABLE_US_INTR();
	NOS_US_TX_OFF();
}


ISR(INT7_vect)
{
	if (us_trigger)
	{
		return;
	}
	NOS_ENTER_ISR();

	round_trip_counter = TCNT3;		// read timer
	NOS_US_TX_OFF();	// turn off US TX
	DISABLE_US_INTR();
	us_trigger = TRUE;
	if (callback_func)
	{
		callback_func();
	}

	NOS_EXIT_ISR();
}

UINT16 nos_us_get_data()
{
	// Assumption : the speed of ultra-sonic = 340 m/sec (Actually,  331.5 + 0.61 * Celsius)
	// Round trip distance (cm) = (TCNT3 / 125000(Hz) ) * 34000(cm/s)  = TCNT3 * 0.272
	// Maximum 90m (theoretically)
	// returns distance (cm), range : 2cm ~ 8912cm
	return ( (round_trip_counter-11) * 0.136);
}

#endif
