/*  -
   Copyright (C) 2006 Weongyo Jeong (weongyo@gmail.com)

This file is part of ROVM.

ROVM is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

ROVM is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.  */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include "types.h"
#include "mpool.h"
#include "tktree.h"
#include "clstree.h"
#include "sha1.h"
#include "listen.h"
#include "rovm.h"

#include "thread.h"
#include "thread_mutex.h"
#include "thread_cond.h"

#include "mpm_worker_fdqueue.h"
#include "mpm_worker_pod.h"
#include "mpm_worker.h"

#include "utils.h"
#include "connection.h"
#include "request.h"

#include "ticket.h"

/**
   struct ticket ü  TK   ϱ   ʿ伺
   ִ ׸ ؼ   ֵ Ѵ.

   @param tk     TK 
   @return       Ϸ 0  ׷   -1  ȯѴ.
 */
int
ticket_pre_config (tk)
     struct ticket *tk;
{
  /*
    struct ticket ü   RET  ʱȭѴ.  ̷ ʱȭ 
     ׻ request `OPCODE'  óϰ  , ׿  return  û
    client   ֱ  return   ʿ䰡  ̴.
  */
  STACK_TYPE (TICKET_RET (tk)) = STACK_TYPE_VOID;

  return 0;
}

/**
   Ŭ method  Ǳ , ѹ ȣǴ Լν, struct ticket ü
    Ҹ  Լ ȣ⿡ °  ִ Ȱ մϴ.   Ǿ
   δ Ʒ   ֽϴ.

     - Local 

   @param tk    Ticket ü
   @param m     ȣ Ǵ ޽ 
   @param argc  ޽ argument count.
   @param argv  ޽ argument value.
   @return          ߻Ͽ  -1 , ׷ ʴٸ
                0  ȯѴ.
 */
int
ticket_beforecall (tk, m, argc, argv)
     struct ticket *tk;
     RvMethod *m;
     u2 argc, *argv;
{
  TICKET_LOCAL (tk) = TICKET_SP (tk) - ((int) (argc - 1));
  TICKET_MAXLOCAL (tk) = TICKET_SP (tk) + 1;

  return 0;
}

/**
   ü ticket  Ѵ.

   @return Ӱ  ticket  ȯѴ.
 */
struct ticket *
create_ticket (r)
     struct rovm *r;
{
  rc_status_t rv;
  struct ticket *t;

  t = (struct ticket *) rc_calloc (1, sizeof (struct ticket));
  if (!t)
    return NULL;

  TICKET_TIME (t) = rc_time_now ();
  TICKET_MAX_STACK_SLOT (t) = CONF_DEFAULT_STACK_SLOT (ROVM_CONF (r));
  TICKET_STACK (t) = (rovm_stack_t *) rc_calloc (1, sizeof (rovm_stack_t) * TICKET_MAX_STACK_SLOT (t));
  if (!TICKET_STACK (t))
    return NULL;

  TICKET_SP (t) = &TICKET_STACK_IDX (t, -1);
  TICKET_LOCAL (t) = TICKET_SP (t);
  TICKET_MAXLOCAL (t) = TICKET_SP (t);
  TICKET_MAX_SP (t) = &TICKET_STACK_IDX (t, TICKET_MAX_STACK_SLOT (t));
  TICKET_MIN_SP (t) = &TICKET_STACK_IDX (t, -1);

  mp_create (&TICKET_POOL (t), PROCREC_PCONF (ROVM_PROCESS (r)));
  mp_tag (TICKET_POOL (t), "ticket");

  rv = rc_thread_mutex_create (&TICKET_MUTEX (t), RC_THREAD_MUTEX_DEFAULT, TICKET_POOL (t));
  if (rv != RC_SUCCESS)
    return NULL;

  if (ticket_pre_config (t))
    return NULL;

  return t;
}

/**
   ü ticket  ϴ  ڿ free Ѵ.

   @param tk    ıϰ ϴ tk ü 
 */
void
destroy_ticket (tk)
     struct ticket *tk;
{
  if (!tk)
    return;

  rc_free (TICKET_STACK (tk));
  rc_free (tk);
}

/**
   Ticket ID  Ticket   Ѵ.

   @param r     Request ü
   @param tid    Ticket ID.  Ticket ID   Ticket  Բ
                ŵȴ.
   @note        destroy_ticket () Լ ܼ `s'   ̹Ƿ 
                 ǰ ʿϴ.
 */
int
destroy_tickets (r, tid)
     request_rec *r;
     char *tid;
{
  struct rovm *rovm = CONN_TS (REQUEST_CONN (r))->arg;

  tktree_remove (ROVM_TKTREE (rovm), tid);

  return 0;
}

/**
   TICKETID  Ӱ  , װ ȯѴ.

   @param r     Request ü
   @return      Ӱ  ticketid  ȯѴ.
 */
char *
create_ticketid (r)
     request_rec *r;
{
  char *tbuf;

  tbuf = (char *) rc_malloc (TICKETID_SIZE);
  if (!tbuf)
    return NULL;

  if (gen_unique_ticketid (tbuf))
    {
      rc_free (tbuf);
      return NULL;
    }

  return tbuf;
}

/**
   Unique Ticket ID  Ͽ Ѵ.  ̸ ϱ  ϴ δ
   PID   ð ̿ϸ, SHA1  ̿Ͽ unique key  Ѵ.

   @param r	ä ü.
   @param tbuf	߱޵ ticket ۰   ϴ .    ũ
                ݵ TICKET_SIZE ũ⿩ Ѵ.   
 */
int
gen_unique_ticketid (tbuf)
     char *tbuf;
{
  pid_t pid;
  unsigned short counter;
#define MAX_TICKETSTR	256
  char tmpbuf[MAX_TICKETSTR];

  /*  Unique Ticket   ؼ PID  COUNTER  ϰ ִ.
         ϱ ؼ Ʒ  ׸ ߰   ̴.

       - Channel  open  ð.  (Request   ð)
       - Client IP ּҰ  */
  pid = getpid ();
  counter = (unsigned short) (rc_time_usec (rc_time_now ()) / 10);

  /* PID  COUNTER   ڿ ̰ MAX_TICKETSTR ũ⸦  ʴ´ٰ 
      Ѵ.  */
  snprintf (tmpbuf, MAX_TICKETSTR, "%d%d", pid, counter);

  rovm_sha1 ((const char *) tmpbuf, strlen (tmpbuf), tbuf);

  return 0;
}

/**
   ڰ û Ticket ID  ̿Ͽ ׿  ticket ü ´.

   @param r     Request ü
   @param tbuf  Ticket ID  ִ 
   @return      TKTREE  ã ticket ü.    ʴ´ٸ 
                NULL  ȯѴ.
 */
struct ticket *
lookup_ticket (r, tbuf)
     request_rec *r;
     char *tbuf;
{
  struct rovm *rovm = CONN_TS (REQUEST_CONN (r))->arg;

  return (struct ticket *) tktree_lookup (ROVM_TKTREE (rovm), tbuf);
}

/**
    TBUF  ü ticket tree ü Ѵ.  ̷ ϵ TBUF  ߿ 
   ̿ شϴ request   , ش  ϰ ȴ.

   ,   Session   Ȱ Ѵٰ    ̴.

   @param r	ä ü.
   @param tbuf	߱޵ ticket ۰  ִ .   ticket  
                gen_unique_ticket () Լ   Ǿ.

   @see gen_unique_ticketid
 */
int
register_ticketid (r, tbuf)
     request_rec *r;
     char *tbuf;
{
  int ret = 0;
  struct rovm *rovm = CONN_TS (REQUEST_CONN (r))->arg;

  if (!tktree_lookup (ROVM_TKTREE (rovm), tbuf))
    {
      struct ticket *tk = create_ticket (rovm);

      if (!tk)
        {
          request_add_error (r, ERRLOG_ERR, ERRLOG_MARK, "Failed to create ticket.");
          return -1;
        }

      /* Insert ̹Ƿ ̿  serialize ( )  ʿϴ. */
      rc_thread_mutex_lock (ROVM_MUTEX (rovm));

      if (tktree_insert (ROVM_TKTREE (rovm), tbuf, (void *) tk))
        {
          rc_thread_mutex_unlock (ROVM_MUTEX (rovm));
          return -1;
        }

      rc_thread_mutex_unlock (ROVM_MUTEX (rovm));
    }

  return ret;
}
