#!/usr/bin/env python
# -*- coding: utf-8 -*-

""" rovmclient.py - ROVM Client
    Copyright (C) 2006 Weongyo Jeong (weongyo@gmail.com)

This file is part of ROVM.

ROVM is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

ROVM is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.
"""

import os, sys
import socket
import struct
import string

# ---->3--------------------------------------------------->3----

# 도움말의 경우, 한글로 작성이 되어 있기 때문에 여러분의 환경에서 제대로
# 된 출력물을 얻기 위해서는 인코딩을 제대로 맞추어야 합니다.  기본적으로
# UTF-8 로 되어 있으며, EUC-KR 환경을 쓰시는 분은 `euc-kr' 로 변경
# 하시기 바랍니다.
DEFAULT_ENCODING = "utf-8"

# ---->3--------------------------------------------------->3----

# 현재 ROVM Client 의 버전 정보입니다.
VERSION = "0.0.1a"

if os.name == 'nt':
    DEFAULT_ENCODING = "mbcs"

COMMAND_REQ             = "\x01"
COMMAND_REQEND          = "\x02"
COMMAND_OPCODE          = "\x03"
COMMAND_TICKET          = "\x04"
COMMAND_OK              = "\x05"
COMMAND_ERROR           = "\x06"
COMMAND_RETURN          = "\x07"
COMMAND_GETSTACK        = "\x08"

OPCODE_FPUSH            = "\x0e" # dec : 14
OPCODE_DPUSH            = "\x0f" # dec : 15
OPCODE_CPUSH            = "\x10" # dec : 16
OPCODE_HPUSH            = "\x11" # dec : 17
OPCODE_IPUSH            = "\x12" # dec : 18
OPCODE_SPUSH            = "\x13" # dec : 19
OPCODE_DUP              = "\x59" # dec : 89
OPCODE_IADD             = "\x60" # dec : 96
OPCODE_CALL             = "\xb6" # dec : 182
OPCODE_NEW              = "\xbb" # dec : 187
OPCODE_NEWARRAY         = "\xbc" # dec : 188

"""
하나의 티켓에 관련된 여러 정보들을 저장하고 처리하는 클래스입니다.
"""
class Ticket:
    def __init__ (self):
        self.host = None
        self.port = None
        self.key = None
        self.opcode = []

    def set_host (self, host, port):
        """
        Host 를 설정한다.
        """
        self.host = host
        self.port = port
        
    def set_ticket (self, key):
        """
        Ticket 를 설정한다.
        """
        self.key = key

    def add_opcode (self, opcode, arg1=None, arg2=None, arg3=None, arg4=None, arg5=None, arg6=None):
        """
        Opcode 를 추가한다.
        """
        total = ""
        total += opcode
        
        if arg1: total += arg1
        if arg2: total += arg2
        if arg3: total += arg3
        if arg4: total += arg4
        if arg5: total += arg5
        if arg6: total += arg6

        self.opcode.append (total)

    def printopcode (self):
        """
        현재 쌓여있는 opcode 들을 화면에 출력합니다.
        """
        z = 0

        if len (self.opcode) == 0:
            print "Nothing in opcode table."
            return

        for o in self.opcode:
            print "#%d (%02d)" % (z, len (o)), repr (o)
            z += 1

    def reset_opcode (self):
        """
        현재 쌓여있는 opcode 들을 reset 합니다.
        """
        self.opcode = []

# Request 요청을 처리하고 나면, proc_res () 메쏘드에서 이 변수가 설정된다.
TICKET = Ticket ()
OPCODE_MODE = 0
PROGRAM_ABORTED = False

#
# `일반 모드' 에서 사용될 수 있는 명령어에 대한 자세한 설명입니다.
#

NormalCommands = { "help" : u"""
HELP - 도움말 출력

지금보고 계시는 도움말을 출력하는 명령어입니다.
""",
                   "req" : u"""
REQ - 티켓을 끊습니다.

ROVM 에서는 어떤 명령어를 원격에서 실행시키기 위해서는 해당 서버의 허락을
받아야 합니다.  그러한 허락을 받을 수 있는 방법이 바로 서버에서 티켓을
끊는 것입니다.  이 명령어를 통해서 서버로 부터 Unique 티켓을 획득할 수
있으며 다음 명령어 부터는 이 티켓을 사용해서 명령어를 실행할 수 있습니다.
즉, 실행하고자 하는 모든 명령어에는 이 티켓이 함께 동봉됩니다.  이를 통해
서버에서는 해당 티켓의 유효성을 테스트하게 됩니다.

형식 : req <URL 주소>
예제 :
        >>> req e://192.168.59.128:8889
""",
                   "reqend" : u"""
REQEND - 티켓을 버립니다.

req 명령어를 통해서 구입한 티켓을 더 이상 사용할 필요가 없어서 이를
버립니다.  이 명령어를 실행할 경우, ROVM 서버 상에서의 티켓과 관련된
모든 resource 들을 해제하게 되며, 더 이상 해당 티켓을 사용할 수 없게
됩니다.

형식 : reqend
예제 :
        >>> req e://192.168.59.128:8889
        >>> reqend
""",
                   "opcode" : u"""
OPCODE - Opcode 모드로 진입합니다.

ROVM Client 는 현재 두가지 모드를 가지고 있습니다.  첫번째는 `>>>'
프롬프트로 시작하는 `일반 모드'와 `...' 프롬프트로 시작하느 `Opcode 모드'가
그것입니다.  일반 모드에서는 티켓을 끊거나 서버의 스택을 확인하거나 등등의
여러 실행, 디버깅, 기타 일들을 위한 명령어를 제공합니다.  하지만 `Opcode 모드'
에서는 실제 ROVM 에서 실행되는 opcode 들을 중심으로 구성되어 있습니다.
Opcode 모드를 빠져나오기 위해서는 `...' 프롬프트 상태에서 단순히 <ENTER>
키만 눌러주면 자동으로 빠져나오게 됩니다.  사용자가 입력한 opcode 들이
자동으로 해석되어 저장되어 있다가 `일반 모드' 의 send 명령어가 실행되었을 때
Server 측에 보내게 됩니다.  즉, opcode 가 완성된 형태일 필요는 전혀 없습니다.
디버깅을 위해서 하나의 opcode 만 넣고 send 명령어를 실행시켜도 됩니다.
`일반 모드'의 stack 명령어를 통해서 *항상* 티켓에 할당된 스택 상태를 확인할 수
있습니다.

형식 : opcode
예제 :
        >>> opcode
        ... <현재 opcode 모드 상태에 진입해 있습니다.>
""",
                   "printopcode" : u"""
PRINTOPCODE - 사용자가 현재까지 입력한 opcode 목록을 보여줍니다.

사용자가 `Opcode 모드' 에서 입력한 opcode 가 실제로 어떻게 변환되었는지를
확인할 수 있으며, 자신이 입력한 opcode 가 어떤 것들을 입력하였는지를 보여주는
명령어입니다.

형식 : printopcode
예제 :
        >>> opcode
        ... cpush 1
        ... cpush 2
        ...
        >>> printopcode
        <입력한 내용이 출력됨>
""",
                   "quit" : u"""
QUIT - ROVM Client 를 종료합니다.

프로그램을 종료합니다.
""",
                   "send" : u"""
SEND - Opcode 를 보냅니다.

사용자가 `Opcode 모드'에서 입력해 둔 명령어들을 실제 서버측에 보냅니다.
이 명령어가 실행되기 전에 반드시 서버로부터 티켓을 발급받아야 합니다.
이 명령어가 실행된 후, 사용자가 입력한 opcode 들은 지워지게 되며, 다시
`Opcode 모드'로 명령어를 입력하고, send 명령어를 실행했을 때는, 이제 막
`Opcode 모드'에서 입력했던 opcode 들만 전송되게 됩니다.

형식 : send
예제 :
        >>> req e://192.168.59.128:8889
        >>> opcode
        ... <실행할 opcode 를 입력합니다.>
        ... <ENTER>
        >>> send
""",
                   "stack" : u"""
STACK - 스택 상태를 출력합니다.

현재 티켓에 설정되어 있는 스택 상태를 원격 서버로 부터 받아와 화면에
출력합니다.

형식 : stack
예제 :
        >>> req e://192.168.59.128:8889
        >>> opcode
        ... <실행할 opcode 를 입력합니다.>
        ... <ENTER>
        >>> send
        >>> stack
        <스택 내용이 출력된 화면>
"""
                   }

#
# 아래는 Opcode 명령어에 대한 설명입니다.
#

OpcodeCommands = { "call" : u"""
CALL - 클래스의 메쏘드를 호출합니다.

Class 내부에 선언되어 있는 메쏘드를 호출하는데, 사용됩니다.  이 opcode 가
사용되기 전에 반드시 스택상에 objectref 가 존재해야 하며, 메쏘드 type 에
맞는 argument 또한 먼저 push 되어 있어야 합니다.

번호 : 182 (0xb6)
형식 : call <name> <type>
스택 : (..., objectref, [arg1, [arg2 ...]]) -> (...)
예제 :
        ... call __init__ (S)V
        ... call abc (SII)I
""",
                   "cpush" : u"""
CPUSH - Char (C 언어에서 char 타입) 를 PUSH 합니다.

Char 을 PUSH 합니다.  Char 은 1 바이트의 type 으로써 C 언어의
`char' 와 완벽하게 같습니다.  숫자의 범위 -128 ~ 127 까지 입니다.
더하기 등등의 연산시 이 값은 integer 로써 계산이 되게 됩니다.

번호 : 16 (0x10)
형식 : cpush <1 byte number>
스택 : (...) -> (..., <byte number>)
예제 :
        ... cpush -128
        ... cpush 127
""",
                   "iadd" : u"""
IADD - Integer 를 더합니다.

스택에서 2 개의 integer entry 를 pop 한 후에, 더하고 그에 대한 결과를 다시
push 합니다.

번호 : 90 (0x60)
형식 : iadd
스택 : (..., <value1>, <value2>) -> (..., <result>)
예제 :
        ... cpush 1
        ... cpush 2
        ... iadd
""",
                   "ipush" : u"""
IPUSH - Integer 번호를 스택에 Push 합니다.

Integer 타입은 4 바이트의 정수이다.  이 명령어를 원격 서버에서 실행되면
스택에 4 바이트 integer 타입의 정수를 push 하게 된다.  참고로 Integer 의
범위는 -2147483648 ~ 2147483647 이다.

번호 : 18 (0x12)
형식 : ipush  <integer number>
스택 : (...) -> (..., <integer number>)
예제 :
        ... ipush -2147483648
        ... ipush 2147483647
""",
                   "new" : u"""
NEW - 지정한 Class 에 대한 Object Reference 를 PUSH 합니다.

지정한 Class 에 대한 Object Reference 를 구해서 Stack 에 PUSH 하게
됩니다.  만약 해당 Class 를 찾지 못하였거나, 성공적으로 Load 하지
못하였거나, 해당 Object Reference 를 생성하지 못하였다면, `ERROR' 메세지를
반환하게 됩니다.  

번호 : 187 (0xbb)
형식 : new      e://<hostname>[:<port>]<path>
스택 : (...) -> (..., <objectref>)
예제:
        ... new e://192.168.58.129:4390/testsuite/insert
        ... new e://envlang.kldp.net/testsuite/insert
""",
                   "newarray" : u"""
NEWARRAY - 지정한 Type 에 대한 Array Ref 를 생성합니다.

n 개의 배열을 만들기 위해서는 이 명령어가 실행되기
전에 해당 정보를 스택에 PUSH 해 놓아야 합니다.  해당 정보를 바탕으로
배열을 생성하여 그에 맞는 ArrayRef 를 반환하게 됩니다.

번호 : 188 (0xbc)
형식 : newarray <type number>
스택 : (..., <count>) -> (..., <arrayref>)
예제 :
        ... ipush 256
        ... newarray 3
""",
                   "spush" : u"""
SPUSH - 문자열을 stack 상에 PUSH 합니다. 

PUSH 된 문자열은 자동으로 StringRef (ObjectRef 의 다른 이름으로써,
문자열과 관련된 점만 강조하기 위해서 지은 이름입니다.) 로 변환되게
되어 저장되게 됩니다.  가변길이를 가지는 opcode 입니다.

번호 : 19 (0x13)
형식 : spush "<string>"
스택 : (...) -> (..., <StringRef>)
예제 :
        ... spush "Hello! ROVM Server"
        ... spush "I love you."
"""
                   }

def open_socket (host, port):
    """
    Open a TCP connection to a given host and port
    """
    if not port:
        port = DEF_PORT
    elif type(port) == type(''):
        port = string.atoi (port)

    print "Opening Socket %s:%s" % (host, port)

    s = socket.socket (socket.AF_INET, socket.SOCK_STREAM)
    s.connect ((host, port))

    return s

def close_socket (s):
    """
    Close Socket.
    """
    print "Closing Socket."
    s.close ()

def greeting ():
    """
    프로그램을 처음 실행시키면, 나오는 인사말입니다.
    """
    global VERSION
    
    print "ROVM Client", VERSION
    print 'Type "help" for more information.'

def normal_usage (cl):
    '''
    This method prints messages how this client can be used.
    '''
    global NormalCommands, DEFAULT_ENCODING
    
    msg = """
Normal mode commands
-----------------------------------------------------------------
 help         Show this message.
 opcode       Entering OPCODE mode.
 printopcode  Print saved opcodes.
 quit         Exit rovmclient.
 req <URL>    Get Ticket.
 reqend       Remove Ticket.
 send         Send opcodes.
 stack        Print remote stack status.
-----------------------------------------------------------------
If you want to see a detailed explanation, Type `help <command>'.
"""
    p = cl.split (' ')
    if len (p) > 1:
        if NormalCommands.has_key (p[1]):
            print NormalCommands[p[1]].encode (DEFAULT_ENCODING)
        else:
            print "No such command."
    else:
        print msg

def opcode_mode_usage (cl):
    """
    Opcode 모드일 경우, 사용할 수 있는 명령어를 보여줍니다.
    """
    global OpcodeCommands, DEFAULT_ENCODING
    
    msg = """
Opcode mode commands
-----------------------------------------------------------------
 call <name> <type>
              Call the method.
 cpush <Char Number>
              Push `char type' number.
 ipush <Int Number>
              Push `int type' number.
 spush "<string>"
              Push a string into stack.
 iadd         Add int.
 new <URL>
              Allocate a ObjectRef of class.
 newarray <type number>
              Allocate a new ArrayRef of Type.
 <ENTER>      Escape opcode mode
-----------------------------------------------------------------
If you want to see a detailed explanation, Type `help <command>'.
"""
    p = cl.split (' ')
    if len (p) > 1:
        if OpcodeCommands.has_key (p[1]):
            print OpcodeCommands[p[1]].encode (DEFAULT_ENCODING)
        else:
            print "No such opcode command."
    else:
        print msg

def print_errlevel (level):
    """
    """
    errors = [ "EMERG",
               "ALERT",
               "CRIT",
               "ERROR",
               "WARNING",
               "NOTICE",
               "INFO",
               "DEBUG" ]
    return errors[level]

def proc_res (s):
    """
    응답을 기다리다가 응답이 올 경우, 그것을 적절히 처리하여 준다.
    """
    global TICKET
    
    rc = s.recv (1)

    if rc == COMMAND_TICKET:
        key = s.recv (20)
        print "Ticket", repr (key)
        TICKET.set_ticket (key)
    elif rc == COMMAND_ERROR:
        import StringIO
        from socket import ntohs
        
        kkk = s.recv (2)
        # struct.unpack 에서 ! 는 네트워크를 가르킨다.
        nop = struct.unpack ('!H', kkk)
        errmsgcount = nop[0]
        if errmsgcount == 0:
            print "(empty)"
            
        for i in range (errmsgcount):
            b = s.recv (1)
            if b == None or len (b) == 0:
                break
            (errlevel, ) = struct.unpack ('B', b)
            (errlen, ) = struct.unpack ('!H', s.recv (2))
            errmsg = s.recv (errlen)
            print "[%s]" % print_errlevel (errlevel), errmsg
    elif rc == COMMAND_RETURN:
        rettype = s.recv (1)
        if rettype == '\x00':
            print "Return VOID"
        elif rettype == '\x01':
            retarray = []
            (arraylen, ) = struct.unpack ('!I', s.recv (4))
            for i in range (arraylen):
                (rettype, ) = struct.unpack ('B', s.recv (1))
                if rettype == 0x9:
                    (strlen, ) = struct.unpack ('!I', s.recv (4))
                    strmsg = s.recv (strlen)

                    retarray.append ((rettype, strmsg))
                else:
                    print "[TODO] Need array upgrade!!!", repr (rettype)
            print "Return Array", retarray
        elif rettype == '\x04':
            (ival, ) = struct.unpack ('!i', s.recv (4))
            print "Return INT =", ival
        else:
            print "[TODO] Need ROVMCLIENT upgrade!!!"
    elif rc == COMMAND_OK:
        print "<- OK"
    else:
        print "proc_res: TODO, rc ->", rc, repr (rc)

    close_socket (s)

def send_rcmd (s, cmd):
    """
    소켓 S 에 CMD 에 해당하는 명령어를 보냅니다.
    """
    global TICKET
    
    if cmd == COMMAND_REQ:
        s.send (cmd)
        proc_res (s)
    elif cmd == COMMAND_REQEND:
        s.send (cmd)
        s.send (TICKET.key)
        proc_res (s)
    elif cmd == COMMAND_OPCODE:
        """
        현재 쌓여있는 opcode 를 원격 서버에 보낸다.
        """
        opcodelen = 0

        if len (TICKET.opcode) == 0:
            print "Nothing in opcode table."
            return

        for o in TICKET.opcode:
            opcodelen += len (o)

        msg = COMMAND_OPCODE \
              + TICKET.key \
              + struct.pack ("!I", opcodelen)
        
        for o in TICKET.opcode:
            msg += o

        s.send (msg)
        proc_res (s)
        TICKET.reset_opcode ()
    elif cmd == COMMAND_GETSTACK:
        s.send (cmd)
        s.send (TICKET.key)
        proc_res (s)
    else:
        raise ValueError, "ERROR.REQ: Invalid command."

def rcmd_req (c):
    """
    REQ 명령어를 처리한다.
    """
    global TICKET
    
    c = c.lstrip ()

    if c[0:4] != "e://":
        print "ERROR.REQ: Invalid prefix. It must be started `e://'"
        return

    host, port = c[4:].split (':')

    try:
        s = open_socket (host, port)
        TICKET.set_host (host, port)
        send_rcmd (s, COMMAND_REQ)
    except socket.error, (errno, strerror):
        print "Socket error(%s): %s" % (errno, strerror)

def opcode_fpush (c):
    """
    OPCODE `fpush' 를 처리하여 TICKET 클래스에 추가한다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"
    
    c = float (c[5:].rstrip ())

    TICKET.add_opcode (OPCODE_FPUSH, struct.pack ("<f", c))

def opcode_dpush (c):
    """
    OPCODE `dpush' 를 처리하여 TICKET 클래스에 추가한다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"
    
    c = float (c[5:].rstrip ())

    TICKET.add_opcode (OPCODE_DPUSH, struct.pack ("<d", c))

def opcode_cpush (c):
    """
    OPCODE `cpush' 를 처리하여 TICKET 클래스에 추가한다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"
    
    c = int (c[5:].rstrip ())

    if (c < -128 or c > 127):
        print "CPUSH: Invalid number."
        return

    TICKET.add_opcode (OPCODE_CPUSH, struct.pack ("<b", c))

def opcode_hpush (c):
    """
    OPCODE `hpush' 를 처리하여 TICKET 클래스에 추가한다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"
    
    c = int (c[5:].rstrip ())

    if (c < -32768 or c > 32767):
        print "HPUSH: Invalid number."
        return

    TICKET.add_opcode (OPCODE_HPUSH, struct.pack ("<h", c))

def opcode_ipush (c):
    """
    OPCODE `ipush' 를 처리하여 TICKET 클래스에 추가한다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"
    
    c = int (c[5:].rstrip ())

    if (c < -2147483648 or c > 2147483647):
        print "IPUSH: Invalid number."
        return

    TICKET.add_opcode (OPCODE_IPUSH, struct.pack ("<i", c))

def opcode_spush (c):
    """
    OPCODE `spush' 를 처리하여 TICKET 클래스에 추가한다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"
    
    c = c[5:].lstrip ()

    if c[0] == '"':
        c = c[1:]
    if c[-1] == '"':
        c = c[0:-1]

    TICKET.add_opcode (OPCODE_SPUSH,
                       struct.pack ("<I", len (c)),
                       c)

def opcode_iadd (c):
    """
    OPCODE `iadd' 를 처리하여 TICKET 클래스에 추가합니다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"
    
    TICKET.add_opcode (OPCODE_IADD)

def opcode_dup (c):
    """
    OPCODE `dup' 를 처리하여 TICKET 클래스에 추가합니다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"
    
    TICKET.add_opcode (OPCODE_DUP)

def strip_eaddr (addr):
    """
    e://<hostname>[:<port>]<path> 형식을 (<hostname>, <port>, <path>) 로 변환하여
    반환합니다.  만약 중간에 오류가 발생하였을 경우, None 을 반환합니다.

    [결론] 지저분합니다.
    """
    if addr[0:4] != "e://":
        return None
    
    addr = addr[4:]
    idx = addr.find ("/")
    if idx == -1:
        return None
    
    hostname = addr[0:idx]
    hostname = hostname.split (":")
    port = 0
    if len (hostname) > 1:
        port = int (hostname[1])

    hostname = hostname[0]
    path = addr[idx:]

    return (hostname, port, path)

def opcode_new (c):
    """
    OPCODE `new' 를 처리하여 TICKET 클래스에 추가한다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"

    addr = c[3:].rstrip ().lstrip ()
    (hostname, port, path) = strip_eaddr (addr)
    
    TICKET.add_opcode (OPCODE_NEW,
                       struct.pack ("<B", 0), # IPv4
                       struct.pack ("<H", len (hostname)),
                       hostname,
                       struct.pack ("<H", port),
                       struct.pack ("<H", len (path)),
                       path)

def opcode_newarray (c):
    """
    OPCODE `newarray' 를 처리하여 TICKET 클래스에 추가한다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"
    
    TICKET.add_opcode (OPCODE_NEWARRAY)

def opcode_call (c):
    """
    OPCODE `call' 을 처리하여 TICKET 클래스에 추가합니다.
    """
    global TICKET

    if TICKET == None:
        raise ValueError, "TICKET == None"

    nametype = c[4:].rstrip ().lstrip ()
    (mname, mtype) = nametype.split (' ')

    TICKET.add_opcode (OPCODE_CALL,
                       struct.pack ("<B", len (mname)),
                       mname,
                       struct.pack ("<H", len (mtype)),
                       mtype)

def proc_opcode_cmd (c):
    """
    Opcode 모드에 들어왔을 경우, 처리할 행동 패턴을 정의한다.
    """
    global OPCODE_MODE
    
    # 만약 c 값이 공백이나 아무것도 입력이 되지 않았을 경우, opcode 의 끝으로
    # 인식을 한다.
    if len (c) <= 0:
        OPCODE_MODE = 0
        return

    cl = c.lower ()

    if cl[0:4] == "help":
        opcode_mode_usage (cl)
    elif cl == "dup":
        opcode_dup (c)
    elif cl[0:5] == "cpush":
        opcode_cpush (c)
    elif cl[0:5] == "hpush":
        opcode_hpush (c)
    elif cl[0:5] == "ipush":
        opcode_ipush (c)
    elif cl[0:5] == "fpush":
        opcode_fpush (c)
    elif cl[0:5] == "dpush":
        opcode_dpush (c)
    elif cl[0:5] == "spush":
        opcode_spush (c)
    elif cl == "iadd":
        opcode_iadd (c)
    elif cl == "newarray":
        opcode_newarray (c)
    elif cl[0:3] == "new":
        opcode_new (c)
    elif cl[0:4] == "call":
        opcode_call (c)
    else:
        print "Can't process your opcode command"
    
def proc_cmd (c):
    """
    This method parses `C' command.  And processes command's meaning.
    """
    global OPCODE_MODE, TICKET, PROGRAM_ABORTED

    if len (c) <= 0:
        return
    
    cl = c.lower ()

    if cl[0:4] == "help":
        normal_usage (cl)
    elif cl == "opcode":
        OPCODE_MODE = 1
        if TICKET:
            print "Using ticket", repr (TICKET.key)
        else:
            print "Testing status, not connected"
    elif cl == "printopcode":
        TICKET.printopcode ()
    elif cl[0:3] == "req" and (cl[3:4] == " " or cl[3:4] == "\t"):
        rcmd_req (c[3:])
    elif cl == "reqend":
        if TICKET.host == None and TICKET.port == None:
            print "TICKET's host and port isn't filled.  Run `req' command first."
            return
        s = open_socket (TICKET.host, TICKET.port)
        send_rcmd (s, COMMAND_REQEND)
    elif cl == "send":
        if TICKET.host == None and TICKET.port == None:
            print "TICKET's host and port isn't filled.  Run `req' command first."
            return
        s = open_socket (TICKET.host, TICKET.port)
        send_rcmd (s, COMMAND_OPCODE)
    elif cl == "stack":
        if TICKET.host == None and TICKET.port == None:
            print "TICKET's host and port isn't filled.  Run `req' command first."
            return
        s = open_socket (TICKET.host, TICKET.port)
        send_rcmd (s, COMMAND_GETSTACK)
    elif cl == "quit":
        PROGRAM_ABORTED = True
    else:
        print "Can't process your command"

def main_loop ():
    """
    This method loops forever. This takes some command by stdin and pass over commands to
    proc_cmd () method.
    """
    while 1:
        if OPCODE_MODE == 1:
            prompt = "... "
        else:
            prompt = ">>> "
        sys.stdout.write (prompt)

        c = sys.stdin.readline ()
        c = c.rstrip ()

        if OPCODE_MODE == 1:
            proc_opcode_cmd (c)
        else:
            proc_cmd (c)

        # 프로그램 종료 flag 가 설정되었다면 종료한다.
        if PROGRAM_ABORTED:
            break

def main ():
    greeting ()    
    main_loop ()
        
if __name__ == "__main__":
    main ()
