#include "types.h"
#include "rovm.h"

enum opcode
  {
#define DEF_OPCODE(ENUM, NAME, NUMBER, OPFUNC, TYPE)   OPCODE_##ENUM ,
#include "opcode.def"
#undef DEF_OPCODE
    MAX_OPCODE
  };

static void
_prealloc_ (rovm_t *r, size_t s)
{
  unsigned char *p;

  r->max_oplen = r->max_oplen + DEFAULT_OPCODE_LEN + s;
  p = (unsigned char *) apr_pcalloc (r->p, r->max_oplen);
  memcpy (p, r->op, r->oplen);

  r->op = p;
}

#define ROVM_OPFUNCS(NAME, TYPE)                \
  static inline void                            \
  NAME (rovm_t *r, TYPE val)                    \
  {                                             \
    TYPE *p = (TYPE *) (r->op + r->oplen);      \
                                                \
    r->oplen += sizeof (TYPE);                  \
    if (r->oplen >= r->max_oplen)               \
      _prealloc_ (r, 0);                        \
                                                \
    p[0] = val;                                 \
  }

ROVM_OPFUNCS (_opcode_, char);
ROVM_OPFUNCS (_char_, char);
ROVM_OPFUNCS (_uchar_, unsigned char);
ROVM_OPFUNCS (_short_, short);
ROVM_OPFUNCS (_ushort_, unsigned short);
ROVM_OPFUNCS (_int_, int);
ROVM_OPFUNCS (_uint_, unsigned int);
ROVM_OPFUNCS (_float_, float);
ROVM_OPFUNCS (_double_, double);

static void
_buf_ (rovm_t *r, const char *buf, size_t len)
{
  char *p = (char *) (r->op + r->oplen);

  r->oplen += len;
  if (r->oplen >= r->max_oplen)
    _prealloc_ (r, len);

  memcpy (p, buf, len);
}

/* 0  opcode.  */
static void
opcode_nop (rovm_t *r)
{
  _opcode_ (r, OPCODE_NOP);
}

/* 13 */
static void
opcode_bpush (rovm_t *r, unsigned char value)
{
  _opcode_ (r, OPCODE_BPUSH);
  _uchar_ (r, value);
}

/* 16 */
static void
opcode_cpush (rovm_t *r, char value)
{
  _opcode_ (r, OPCODE_CPUSH);
  _char_ (r, value);
}

/* 17 */
static void
opcode_hpush (rovm_t *r, short value)
{
  _opcode_ (r, OPCODE_HPUSH);
  _short_ (r, value);
}

/* 18 */
static void 
opcode_ipush (rovm_t *r, int value)
{
  _opcode_ (r, OPCODE_IPUSH);
  _int_ (r, value);
}

/* 14 */
static void 
opcode_fpush (rovm_t *r, float value)
{
  _opcode_ (r, OPCODE_FPUSH);
  _float_ (r, value);
}

/* 15 */
static void 
opcode_dpush (rovm_t *r, double value)
{
  _opcode_ (r, OPCODE_DPUSH);
  _double_ (r, value);
}

/* 19 */
static void 
opcode_spush (rovm_t *r, const char *buf, size_t len)
{
  _opcode_ (r, OPCODE_SPUSH);
  _uint_ (r, len);
  _buf_ (r, buf, len);
}

/* 21 */
static void 
opcode_iload (rovm_t *r, unsigned char value)
{
  _opcode_ (r, OPCODE_SPUSH);
  _uchar_ (r, value);
}

/* 87 */
static void 
opcode_pop (rovm_t *r)
{
  _opcode_ (r, OPCODE_POP);
}

/* 89 */
static void 
opcode_dup (rovm_t *r)
{
  _opcode_ (r, OPCODE_DUP);
}

/* 95 */
static void 
opcode_swap (rovm_t *r)
{
  _opcode_ (r, OPCODE_SWAP);
}

/* 96 */
static void 
opcode_iadd (rovm_t *r)
{
  _opcode_ (r, OPCODE_IADD);
}

/* 98 */
static void 
opcode_fadd (rovm_t *r)
{
  _opcode_ (r, OPCODE_FADD);
}

/* 99 */
static void 
opcode_dadd (rovm_t *r)
{
  _opcode_ (r, OPCODE_DADD);
}

/* 100 */
static void 
opcode_isub (rovm_t *r)
{
  _opcode_ (r, OPCODE_ISUB);
}

/* 102 */
static void 
opcode_fsub (rovm_t *r)
{
  _opcode_ (r, OPCODE_FSUB);
}

/* 103 */
static void 
opcode_dsub (rovm_t *r)
{
  _opcode_ (r, OPCODE_DSUB);
}

/* 104 */
static void 
opcode_imul (rovm_t *r)
{
  _opcode_ (r, OPCODE_IMUL);
}

/* 106 */
static void 
opcode_fmul (rovm_t *r)
{
  _opcode_ (r, OPCODE_FMUL);
}

/* 107 */
static void 
opcode_dmul (rovm_t *r)
{
  _opcode_ (r, OPCODE_DMUL);
}

/* 108 */
static void 
opcode_idiv (rovm_t *r)
{
  _opcode_ (r, OPCODE_IDIV);
}

/* 110 */
static void 
opcode_fdiv (rovm_t *r)
{
  _opcode_ (r, OPCODE_FDIV);
}

/* 111 */
static void 
opcode_ddiv (rovm_t *r)
{
  _opcode_ (r, OPCODE_DDIV);
}

/* 112 */
static void 
opcode_irem (rovm_t *r)
{
  _opcode_ (r, OPCODE_IREM);
}

/* 114 */
static void 
opcode_frem (rovm_t *r)
{
  _opcode_ (r, OPCODE_FREM);
}

/* 115 */
static void 
opcode_drem (rovm_t *r) 
{
  _opcode_ (r, OPCODE_DREM); 
}

/* 172 */
static void 
opcode_ireturn (rovm_t *r) 
{
  _opcode_ (r, OPCODE_IRETURN); 
}

/* 174 */
static void 
opcode_freturn (rovm_t *r) 
{
  _opcode_ (r, OPCODE_FRETURN); 
}

/* 175 */
static void 
opcode_dreturn (rovm_t *r) 
{
  _opcode_ (r, OPCODE_DRETURN); 
}

/* 176 */
static void 
opcode_areturn (rovm_t *r) 
{ 
  _opcode_ (r, OPCODE_ARETURN); 
}

/* 177 */
static void 
opcode_return (rovm_t *r) 
{ 
  _opcode_ (r, OPCODE_RETURN); 
}

/* 182 */
static void 
opcode_call (rovm_t *r, const char *mname, const char *mtype)
{
  _opcode_ (r, OPCODE_CALL);
  _uchar_ (r, strlen (mname));
  _buf_ (r, mname, strlen (mname));
  _ushort_ (r, strlen (mtype));
  _buf_ (r, mtype, strlen (mtype));
}

/* 187 */
static void
opcode_new (rovm_t *r, int type, const char *host, unsigned short port, const char *path)
{
  _opcode_ (r, OPCODE_NEW);
  _uchar_ (r, (unsigned char) type);
  _ushort_ (r, strlen (host));
  _buf_ (r, host, strlen (host));
  _ushort_ (r, port);
  _ushort_ (r, strlen (path));
  _buf_ (r, path, strlen (path));
}

/* 188 */
static void 
opcode_newarray (rovm_t *r)
{
  _opcode_ (r, OPCODE_NEWARRAY);
}

static void
opcode_reserved (rovm_t *r)
{
  fprintf (stderr, "error!\n");
}

struct rovmops rovm_opcodes =
  {
#define DEF_OPCODE(ENUM, NAME, NUMBER, OPFUNC, TYPE)   OPFUNC ,
#include "opcode.def"
#undef DEF_OPCODE
  };
