#ifndef _ROVMI_ROVM_H
#define _ROVMI_ROVM_H

#include "types.h"

#ifdef HAVE_OPENSSL
/** OpenSSL headers */
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/crypto.h>
#include <openssl/evp.h>
#include <openssl/rand.h>
#include <openssl/x509v3.h>
#include <openssl/bio.h>
#endif

#define ROVMLOG_EMERG           0       /* system is unusable */
#define ROVMLOG_ALERT           1       /* action must be taken immediately */
#define ROVMLOG_CRIT            2       /* critical conditions */
#define ROVMLOG_ERR             3       /* error conditions */
#define ROVMLOG_WARNING         4       /* warning conditions */
#define ROVMLOG_NOTICE          5       /* normal but significant condition */
#define ROVMLOG_INFO            6       /* informational */
#define ROVMLOG_DEBUG           7       /* debug-level messages */

#define ROVMLOG_MARK		__FILE__,__LINE__

typedef struct rovmret rovmret_t;
typedef struct rovm rovm_t;
typedef struct rovmres rovmres_t;

struct rovmerror
{
  char *msg;
  int level;

  struct rovmerror *next;
};

#define TICKET_SIZE     20

struct rovmstr
{
  char *ptr;
  size_t len;
};

#define RETTYPE_VOID            0x00
#define RETTYPE_ARRAYREF        0x01
#define RETTYPE_BOOLEAN         0x02
#define RETTYPE_CHAR            0x03
#define RETTYPE_SHORT           0x04
#define RETTYPE_INT             0x05
#define RETTYPE_FLOAT           0x06
#define RETTYPE_DOUBLE          0x07
#define RETTYPE_OBJREF          0x08
#define RETTYPE_STRINGREF       0x09

struct rovmarray
{
  char type;

  union
  {
    int _int;
    struct rovmarray *_array;
    struct rovmstr *_string;
  } v;  
};

struct rovmret
{
  char type;
  int is_array;
  size_t arraylen;

  union
  {
    int _int;
    struct rovmarray *_array;
    struct rovmstr *_string;
  } v;
};

struct rovmops
{
#define DEF_OPCODE(ENUM, NAME, NUMBER, OPFUNC, TYPE)   void (*ENUM) TYPE;
#include "opcode.def"
#undef DEF_OPCODE
};

#define DEFAULT_OPCODE_LEN      1024

struct rovm
{
  const char *host;
  apr_port_t port;
  unsigned char ticket[TICKET_SIZE];

  apr_pool_t *p;
  apr_socket_t *sock;

  int errcount;
  struct rovmerror *errmsg;

  unsigned char *op;
  size_t oplen;
  size_t max_oplen;

  struct rovmops ops;

  rovmret_t *(*send) (rovm_t *r);
  void (*printerror) (rovm_t *r);
  void (*printstack) (rovm_t *r);
  void (*printreturn) (rovm_t *r, rovmret_t *ret);

  /* For OpenSSL.  */
  BIO *bio_err;
  SSL_CTX *ctx;
  SSL *ssl;
  SSL_SESSION *sess;
  char *keyfile;
  char *pass;
  int need_shutdown;
};

struct rovmres
{
  char type;
};

extern rovm_t * rovm_req (const char *host, apr_port_t port, const char *keyfile, const char *password);
extern void rovm_reqend (rovm_t *r);
extern void rovmi_log (rovm_t *r, int level, const char *filename, int line, const char *fmt,  ...);
extern int ri_ssl_socket_send (rovm_t *r, char *buf, int *size);
extern int ri_ssl_socket_recv (rovm_t *r, char *buf, int *size);
extern int ri_ssl_connect (rovm_t *r);
extern int ri_ssl_initialize_ctx (rovm_t *r, const char *keyfile, const char *password);
extern int ri_ssl_destory (rovm_t *r);

#endif
