/* 
 * FFS File System Driver for Windows
 *
 * ffsdrv.c
 *
 * 2004.5.6 ~
 *
 * Lee Jae-Hong, http://www.pyrasis.com
 *
 * See License.txt
 *
 */

#include "ntifs.h"
#include "ffsdrv.h"

/* Globals */
PFFS_GLOBAL FFSGlobal = NULL;


LARGE_INTEGER
FFSSysTime(
	IN ULONG i_time)
{
	LARGE_INTEGER SysTime;
	SysTime.QuadPart = (LONGLONG)(i_time) * 10000000;
	SysTime.QuadPart += FFSGlobal->TimeZone.QuadPart;
	return SysTime;
}


NTSTATUS
FFSQueryVolumeInformation(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT          DeviceObject;
	NTSTATUS                Status = STATUS_UNSUCCESSFUL;
	PFFS_VCB                Vcb;
	PIRP                    Irp;
	PIO_STACK_LOCATION      IoStackLocation;
	FS_INFORMATION_CLASS    FsInformationClass;
	ULONG                   Length;
	PVOID                   Buffer;
	BOOLEAN                 VcbResourceAcquired = FALSE;

	__try
	{
		ASSERT(IrpContext != NULL);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;
	
		/*
		 * This request is not allowed on the main device object
		 */
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_INVALID_DEVICE_REQUEST;
			__leave;
		}
		
		Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
		Vcb->ffs_block = 1024;
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		if (!ExAcquireResourceSharedLite(
			&Vcb->MainResource,
			IrpContext->IsSynchronous))
		{
			Status = STATUS_PENDING;
			__leave;
		}
		
		VcbResourceAcquired = TRUE;
		
		Irp = IrpContext->Irp;
		
		IoStackLocation = IoGetCurrentIrpStackLocation(Irp);
		
		FsInformationClass =
			IoStackLocation->Parameters.QueryVolume.FsInformationClass;
		
		Length = IoStackLocation->Parameters.QueryVolume.Length;
		
		Buffer = Irp->AssociatedIrp.SystemBuffer;
		
		RtlZeroMemory(Buffer, Length);
		
		switch (FsInformationClass)
		{
		case FileFsVolumeInformation:
			{
				PFILE_FS_VOLUME_INFORMATION FsVolInfo;
				ULONG                       VolumeLabelLength;
				ULONG                       RequiredLength;
				
				if (Length < sizeof(FILE_FS_VOLUME_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FsVolInfo = (PFILE_FS_VOLUME_INFORMATION) Buffer;
				
				FsVolInfo->VolumeCreationTime.QuadPart = 0;
				
				FsVolInfo->VolumeSerialNumber = 0;

				VolumeLabelLength = 3;
				
				FsVolInfo->VolumeLabelLength = VolumeLabelLength * 2;
				
				/* I don't know what this means */
				FsVolInfo->SupportsObjects = FALSE;
				
				RequiredLength = sizeof(FILE_FS_VOLUME_INFORMATION)
					+ VolumeLabelLength * 2 - sizeof(WCHAR);
				
				if (Length < RequiredLength)
				{
					Irp->IoStatus.Information =
						sizeof(FILE_FS_VOLUME_INFORMATION);
					Status = STATUS_BUFFER_OVERFLOW;
					__leave;
				}
				
				FFSCharToWchar(
					FsVolInfo->VolumeLabel,
					"FFS",
					VolumeLabelLength);
				
				Irp->IoStatus.Information = RequiredLength;
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		case FileFsSizeInformation:
			{
				PFILE_FS_SIZE_INFORMATION FsSizeInfo;
				
				if (Length < sizeof(FILE_FS_SIZE_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FsSizeInfo = (PFILE_FS_SIZE_INFORMATION) Buffer;
				
				{
					FsSizeInfo->TotalAllocationUnits.QuadPart =
						Vcb->ffs_super_block->fs_ncg;
					
					FsSizeInfo->AvailableAllocationUnits.QuadPart = 0;
				}
				
				FsSizeInfo->SectorsPerAllocationUnit =
					Vcb->ffs_block / Vcb->DiskGeometry.BytesPerSector;
				
				FsSizeInfo->BytesPerSector =
					Vcb->DiskGeometry.BytesPerSector;
				
				Irp->IoStatus.Information = sizeof(FILE_FS_SIZE_INFORMATION);
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		case FileFsDeviceInformation:
			{
				PFILE_FS_DEVICE_INFORMATION FsDevInfo;
				
				if (Length < sizeof(FILE_FS_DEVICE_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FsDevInfo = (PFILE_FS_DEVICE_INFORMATION) Buffer;
				
				FsDevInfo->DeviceType =
					Vcb->TargetDeviceObject->DeviceType;
				
				FsDevInfo->Characteristics =
					Vcb->TargetDeviceObject->Characteristics;
				
#ifndef FFS_RO
				if (Vcb->ReadOnly)
#endif
				{
					FsDevInfo->Characteristics |=
						FILE_READ_ONLY_DEVICE;
				}
				
				Irp->IoStatus.Information = sizeof(FILE_FS_DEVICE_INFORMATION);
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		case FileFsAttributeInformation:
			{
				PFILE_FS_ATTRIBUTE_INFORMATION  FsAttrInfo;
				ULONG                           RequiredLength;
				
				if (Length < sizeof(FILE_FS_ATTRIBUTE_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FsAttrInfo =
					(PFILE_FS_ATTRIBUTE_INFORMATION) Buffer;
				
				FsAttrInfo->FileSystemAttributes =
					FILE_CASE_SENSITIVE_SEARCH | FILE_CASE_PRESERVED_NAMES;
				
				FsAttrInfo->MaximumComponentNameLength = MAXNAMLEN;
				
				FsAttrInfo->FileSystemNameLength = sizeof(DRIVER_NAME) * 2;
				
				RequiredLength = sizeof(FILE_FS_ATTRIBUTE_INFORMATION) +
					sizeof(DRIVER_NAME) * 2 - sizeof(WCHAR);
				
				if (Length < RequiredLength)
				{
					Irp->IoStatus.Information =
						sizeof(FILE_FS_ATTRIBUTE_INFORMATION);
					Status = STATUS_BUFFER_OVERFLOW;
					__leave;
				}
				
				FFSCharToWchar(
					FsAttrInfo->FileSystemName,
					DRIVER_NAME,
					sizeof(DRIVER_NAME));
				
				Irp->IoStatus.Information = RequiredLength;
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		default:
			Status = STATUS_INVALID_INFO_CLASS;
        }
    }
    __finally
    {
	    if (VcbResourceAcquired)
	    {
		    ExReleaseResourceForThreadLite(
			    &Vcb->MainResource,
			    ExGetCurrentResourceThread());
	    }
	    
	    if (!IrpContext->ExceptionInProgress)
	    {
		    if (Status == STATUS_PENDING)
		    {
			    FFSQueueRequest(IrpContext);
		    }
		    else
		    {
			    IrpContext->Irp->IoStatus.Status = Status;
			    
			    FFSCompleteRequest(
				    IrpContext->Irp,
				    (CCHAR)
				    (NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
			    
			    FFSFreeIrpContext(IrpContext);
		    }
	    }
    }
    
    return Status;
}


VOID
FFSCharToWchar(
	IN OUT PWCHAR   Destination,
	IN PCHAR        Source,
	IN ULONG        Length)
{
	ULONG Index;
	
	ASSERT(Destination != NULL);
	ASSERT(Source != NULL);
	
	for (Index = 0; Index < Length; Index++)
	{
		Destination[Index] = (WCHAR) Source[Index];
	}
}


NTSTATUS
FFSWcharToChar(
	IN OUT PCHAR    Destination,
	IN PWCHAR       Source,
	IN ULONG        Length)
{
	ULONG       Index;
	NTSTATUS    Status;
	
	ASSERT(Destination != NULL);
	ASSERT(Source != NULL);
	
	for (Index = 0, Status = STATUS_SUCCESS; Index < Length; Index++)
	{
		Destination[Index] = (CHAR) Source[Index];
		
		/*
		 * Check that the wide character fits in a normal character
		 * but continue with the conversion anyway
		 */
		if ( ((WCHAR) Destination[Index]) != Source[Index] )
		{
			Status = STATUS_OBJECT_NAME_INVALID;
		}
	}
	
	return Status;
}


VOID
FFSFreeFcb(
	IN PFFS_FCB Fcb)
{
	ASSERT(Fcb != NULL);
	
	ASSERT((Fcb->Identifier.Type == FCB) &&
		(Fcb->Identifier.Size == sizeof(FFS_FCB)));

	FsRtlUninitializeFileLock(&Fcb->FileLockAnchor);

	ExDeleteResourceLite(&Fcb->MainResource);
	
	ExDeleteResourceLite(&Fcb->PagingIoResource);
	
	RemoveEntryList(&Fcb->Next);
	
	ExFreePool(Fcb->FileName.Buffer);
	
#if DBG
	ExFreePool(Fcb->DbgFileName);
#endif
	
	ExFreePool(Fcb->dinode);
	
	ExFreePool(Fcb);
}


PFFS_CCB
FFSAllocateCcb(VOID)
{
	PFFS_CCB Ccb;
	
	Ccb = (PFFS_CCB)ExAllocatePool(NonPagedPool, sizeof(FFS_CCB));
	
	if (!Ccb)
	{
		return NULL;
	}
	
	Ccb->Identifier.Type = CCB;
	Ccb->Identifier.Size = sizeof(FFS_CCB);
	
	Ccb->CurrentByteOffset = 0;
	
	Ccb->DirectorySearchPattern.Length = 0;
	Ccb->DirectorySearchPattern.MaximumLength = 0;
	Ccb->DirectorySearchPattern.Buffer = 0;
	
	return Ccb;
}


PFFS_FCB
FFSAllocateFcb(
	IN PFFS_VCB             Vcb,
	IN PUNICODE_STRING      FileName,
	IN ULONG                IndexNumber,
	IN ULONG                inode,
	IN ULONG                dir_inode,
	IN struct ufs1_dinode*  dinode)
{
	PFFS_FCB Fcb;
	
	Fcb = (PFFS_FCB)ExAllocatePool(NonPagedPool, sizeof(FFS_FCB));
	
	if (!Fcb)
	{
		return NULL;
	}
	
	Fcb->Identifier.Type = FCB;
	Fcb->Identifier.Size = sizeof(FFS_FCB);
	
#ifndef FFS_RO
	
	RtlZeroMemory(&Fcb->ShareAccess, sizeof(SHARE_ACCESS));
	
#endif
	
	FsRtlInitializeFileLock (
		&Fcb->FileLockAnchor,
		NULL,
		NULL);
	
	Fcb->OpenHandleCount = 0;
	Fcb->ReferenceCount = 0;
	
#ifndef FFS_RO
	
	Fcb->IsPageFile = FALSE;
	
	Fcb->DeletePending = FALSE;
	
#endif
	
	Fcb->FileName.Length = 0;
	
	Fcb->FileName.MaximumLength = FileName->MaximumLength;
	
	Fcb->FileName.Buffer = (PWSTR) ExAllocatePool(
		NonPagedPool,
		Fcb->FileName.MaximumLength);
	
	if (!Fcb->FileName.Buffer)
	{
		ExFreePool(Fcb);
		return NULL;
	}
	
	RtlCopyUnicodeString(
		&Fcb->FileName,
		FileName);
	
#if DBG
	
	Fcb->DbgFileName = (PUCHAR) ExAllocatePool(
		NonPagedPool,
		Fcb->FileName.MaximumLength / sizeof(WCHAR) + 1);
	
	if (!Fcb->DbgFileName)
	{
		ExFreePool(Fcb->FileName.Buffer);
		ExFreePool(Fcb);
		return NULL;
	}
	
	FFSWcharToChar(
		Fcb->DbgFileName,
		Fcb->FileName.Buffer,
		Fcb->FileName.Length / sizeof(WCHAR));
	
	Fcb->DbgFileName[Fcb->FileName.Length / sizeof(WCHAR)] = 0;

#endif // DBG
	
	Fcb->FileAttributes = FILE_ATTRIBUTE_NORMAL;

	if ((dinode->di_mode & IFMT) == IFDIR)
	{
		Fcb->FileAttributes |= FILE_ATTRIBUTE_DIRECTORY;
	}
	
	Fcb->IndexNumber.QuadPart = IndexNumber;
	
	Fcb->dinode = dinode;
	Fcb->inode = inode;
	Fcb->dir_inode = dir_inode;
	
	RtlZeroMemory(&Fcb->CommonFCBHeader, sizeof(FSRTL_COMMON_FCB_HEADER));
	
	//Fcb->CommonFCBHeader.NodeTypeCode = 
	Fcb->CommonFCBHeader.NodeByteSize = sizeof(FSRTL_COMMON_FCB_HEADER);
	Fcb->CommonFCBHeader.IsFastIoPossible = FastIoIsNotPossible;
	Fcb->CommonFCBHeader.Resource = &(Fcb->MainResource);
	Fcb->CommonFCBHeader.PagingIoResource = &(Fcb->PagingIoResource);
	Fcb->CommonFCBHeader.AllocationSize.QuadPart = dinode->di_size;
	Fcb->CommonFCBHeader.FileSize.QuadPart = dinode->di_size;
	Fcb->CommonFCBHeader.ValidDataLength.QuadPart = dinode->di_size;
	
	Fcb->SectionObject.DataSectionObject = NULL;
	Fcb->SectionObject.SharedCacheMap = NULL;
	Fcb->SectionObject.ImageSectionObject = NULL;
	
	ExInitializeResourceLite(&(Fcb->MainResource));
	ExInitializeResourceLite(&(Fcb->PagingIoResource));
	
	InsertTailList(&Vcb->FcbList, &Fcb->Next);
	
	return Fcb;
}


VOID
FFSFreeCcb(
	IN PFFS_CCB Ccb)
{
	ASSERT(Ccb != NULL);
	
	ASSERT((Ccb->Identifier.Type == CCB) &&
		(Ccb->Identifier.Size == sizeof(FFS_CCB)));
	
	if (Ccb->DirectorySearchPattern.Buffer != NULL)
	{
		ExFreePool(Ccb->DirectorySearchPattern.Buffer);
	}
	
	ExFreePool(Ccb);
}


VOID
FFSFreeVcb(
	IN PFFS_VCB Vcb)
{
	ASSERT(Vcb != NULL);
	
	ASSERT((Vcb->Identifier.Type == VCB) &&
		(Vcb->Identifier.Size == sizeof(FFS_VCB)));
	
	FFSClearVpbFlag(Vcb->Vpb, VPB_MOUNTED);
	
	ExAcquireResourceExclusiveLite(
		&FFSGlobal->Resource,
		TRUE);
	
	RemoveEntryList(&Vcb->Next);
	
	ExReleaseResourceForThreadLite(
		&FFSGlobal->Resource,
		ExGetCurrentResourceThread());
	
	ExDeleteResourceLite(&Vcb->MainResource);
	
	ExDeleteResourceLite(&Vcb->PagingIoResource);
	
	if (Vcb->ffs_super_block)
		ExFreePool(Vcb->ffs_super_block);
	
	IoDeleteDevice(Vcb->DeviceObject);
}

VOID
FFSSetVpbFlag(
	IN PVPB     Vpb,
	IN USHORT   Flag)
{
	KIRQL OldIrql;
	
	IoAcquireVpbSpinLock(&OldIrql);
	
	Vpb->Flags |= Flag;
	
	IoReleaseVpbSpinLock(OldIrql);
}


VOID
FFSClearVpbFlag(
	IN PVPB     Vpb,
	IN USHORT   Flag)
{
	KIRQL OldIrql;
	
	IoAcquireVpbSpinLock(&OldIrql);
	
	Vpb->Flags &= ~Flag;
	
	IoReleaseVpbSpinLock(OldIrql);
}


ULONG FFSGetBlock(
	IN PFFS_VCB vcb,
	ULONG       dwContent,
	ULONG       Index,
	int         layer)
{
	ULONG *pData = NULL;
	ULONG lba = 0;
	ULONG i = 0, j = 0, temp = 1;
	ULONG dwBlk = 0;

	vcb->ffs_block = 1024;

	lba = dwContent * vcb->ffs_block / SECTOR_SIZE;

	if (layer == 0)
	{
		dwBlk = dwContent;
	}
	else if (layer <= 3)
	{
		pData = (ULONG *)ExAllocatePool(NonPagedPool,
					vcb->ffs_super_block->fs_bsize);
		if (!pData)
		{
			KdPrint(("FFSGetBlock: no enough memory.\n"));
			return dwBlk;
		}

		FFSReadDisk(vcb->TargetDeviceObject, lba, 0, vcb->ffs_super_block->fs_bsize, (PVOID)pData);

		for (i = 0; i < (ULONG)(layer - 1); i++)
		{
			temp *= (vcb->ffs_block / 4);
		}

#if DBG
		KdPrint(("FFSGetBlock: Index : %d, temp : %d, fs_bsize : %d\n", Index, temp, vcb->ffs_super_block->fs_bsize));
#endif

		i = Index / temp;
		j = Index % temp;

		dwBlk = FFSGetBlock(vcb, pData[i], j, layer - 1);

		ExFreePool(pData);
	}

#if DBG
	KdPrint(("FFSGetBlock(): Index : %d, temp : %d, lba : %x, dwBlk : %04x, Data Block : %X\n", 
		Index, temp, lba, dwBlk, (dwBlk * 0x400)));
#endif

	return dwBlk;
}


ULONG
FFSBlockMap(
	IN PFFS_VCB            vcb,
	IN struct ufs1_dinode* dinode,
	IN ULONG               Index)
{
	ULONG dwSizes[3] = {vcb->ffs_block / 4, 0, 0};
	int   i;

#if DBG
	KdPrint(("FFSBlockMap() start\n"));
#endif

	if (Index >= dinode->di_blocks)
	{
		KdPrint(("FFSBlockMap: error input paramters.\n"));
		return 0;
	}

	if (Index < 12)
	{
#if DBG
		KdPrint(("FFSBlockMap() di_db[%d] : %04x, Data Blcok : %X\n", Index, dinode->di_db[Index], 
			(dinode->di_db[Index] * 0x400)));
#endif
		return dinode->di_db[Index];
	}

	Index -= 12;

	for (i = 0; i < 3; i++)
	{
		if (i > 0)
		{
			dwSizes[i] = (vcb->ffs_block / 4) * dwSizes[i - 1];
		}
		if (Index < dwSizes[i])
		{
			return FFSGetBlock(vcb, dinode->di_ib[i], Index , i + 1); 
		}
		Index -= dwSizes[i];
	}

#if DBG
	KdPrint(("FFSBlockMap() end\n"));
#endif

	return 0;
}


ULONG
FFSReadInode(
	IN PFFS_VCB            vcb,
	IN struct ufs1_dinode* dinode,
	IN ULONG               offset,
	IN PVOID               Buffer,
	IN ULONG               Size,
	IN BOOLEAN             ReadFileData)
{
	ULONG	dwBytes = 0;
	ULONG	nBeg, nEnd;
	ULONG	dwBlk;
	ULONG	lba, off, dwSize;
	ULONG   i;

	/* XXX */
	vcb->ffs_block = 1024;

	if (ReadFileData == FALSE)
	{
		nBeg = offset / vcb->ffs_block;
		nEnd = (Size + offset + vcb->ffs_block - 1) / vcb->ffs_block;
	}
	else
	{
		nBeg = offset / vcb->ffs_super_block->fs_bsize;
		nEnd = (Size + offset + vcb->ffs_super_block->fs_bsize - 1) / vcb->ffs_super_block->fs_bsize;
	}


#if DBG
	KdPrint(("FFSReadInode() nBeg : %d, nEnd : %d, Size: %d, offset : %d\n", nBeg, nEnd, Size, offset));
#endif

	for (i = nBeg; i < nEnd; i++)
	{
		dwBlk = FFSBlockMap(vcb, dinode, i);
		lba = dwBlk * vcb->ffs_block / SECTOR_SIZE;
		off = 0;
		if (ReadFileData == FALSE)
			dwSize = vcb->ffs_block;
		else
			dwSize = vcb->ffs_super_block->fs_bsize;

		if (i == nBeg)
		{
			lba += (offset % vcb->ffs_block) / SECTOR_SIZE; 
			off = offset % SECTOR_SIZE;
			if (ReadFileData == FALSE)
				dwSize = vcb->ffs_block - (offset % vcb->ffs_block);
			else
				dwSize = vcb->ffs_super_block->fs_bsize - (offset % vcb->ffs_super_block->fs_bsize);
		}
		else if (i == nEnd)
		{
			/* 
			 *   Ÿ б  ũ  
			 *  ũ - ݱ  Ÿ ũ
			 */
			dwSize = Size - dwBytes;
		}
		else if (i == (nEnd - 1))
		{
			/*
			 * 4096  Length Ƿ 8192   
			 * ޸ 翡  ߻Ѵ. 
			 * Length / 8192 x.5( 4096)   dwSize 4096 Ѵ.
			 * nEnd - 1  Index  dwSize 4096 ϱ ؼ̴.
			 *  ƴ ٸ ƹ 4096 ָ  Index   4096 
			 * Ƿ ߸  ´.
			 */
			if ((Size % 8192) == 4096)
				dwSize = vcb->ffs_super_block->fs_bsize / 2;
		}
		FFSReadDisk(vcb->TargetDeviceObject,
			lba, off, dwSize, &((PUCHAR)Buffer)[dwBytes]);
		dwBytes += dwSize;

#if DBG
		KdPrint(("FFSReadInode() lba : %x, offset: %X dwSize : %d, i : %d, off: %d, dwBytes : %d\n", 
			lba, lba * 0x200, dwSize, i, off, dwBytes));
#endif

	}

	return dwBytes;
}


BOOLEAN
FFSGetInodeLba(
	IN PFFS_VCB vcb,
	IN ULONG    inode,
	OUT PULONG  lba,
	OUT PULONG  offset)
{
	ULONG loc;

#if DBG
	KdPrint(("FFSGetInodeLba() start\n"));
#endif

#if 0
	if (inode < 1 || inode > vcb->ffs_super_block->fs_ipg)
	{
		KdPrint(("FFSGetInodeLba: Inode value %xh is invalid.\n",inode));
		*lba = 0;
		*offset = 0;
		return FALSE;
	}
#endif

	loc = (ULONG)cgimin(vcb->ffs_super_block, ino_to_cg(vcb->ffs_super_block, inode)) 
		* vcb->ffs_super_block->fs_fsize + ((inode % vcb->ffs_super_block->fs_ipg) * 128);

	*lba = loc / SECTOR_SIZE;
	*offset = loc % SECTOR_SIZE;

	KdPrint(("FFSGetInodeLba: inode=%xh lba=%xh offset=%xh\n", inode, *lba, *offset));


#if DBG
	KdPrint(("FFSGetInodeLba() end\n"));
#endif

	return TRUE;
}


BOOLEAN
FFSLoadInode(
	IN PFFS_VCB           vcb,
	IN ULONG              inode,
	IN struct ufs1_dinode *dinode)
{
	ULONG		lba;
	ULONG		offset;
	NTSTATUS	Status;

#if DBG
	KdPrint(("FFSLoadInode() start\n"));
#endif

	if (!FFSGetInodeLba(vcb, inode, &lba, &offset))
	{
		KdPrint(("FFSLoadInode: error get inode(%xh)'s addr.\n", inode));
		return FALSE;
	}

	Status = FFSReadDisk(vcb->TargetDeviceObject,
		lba,
		offset,
		sizeof(FFS_INODE),
		(PVOID)dinode);

	if (!NT_SUCCESS(Status))
	{
		return FALSE;
	}

#if DBG
	KdPrint(("FFSLoadInode() end\n"));
#endif

	return TRUE;
}


NTSTATUS
FFSLookupFileName(
	IN PFFS_VCB         Vcb,
	IN PUNICODE_STRING  FullFileName,
	IN OUT PULONG       Offset,
	IN OUT PULONG       Inode,
	IN OUT PULONG       DirInode,
	IN OUT PFFS_INODE   dinode)
{
	ULONG           RootInode = 2;
	UNICODE_STRING  FileName;
	NTSTATUS        Status = STATUS_OBJECT_NAME_NOT_FOUND;
	FFS_DIR_ENTRY   ffs_dir;
	USHORT          i = 0;
	BOOLEAN	        bRun = TRUE;
	BOOLEAN	        bCurrent = FALSE;
	FFS_INODE       in;
	ULONG           off = 0;

	*Offset = 0;
	*Inode = 0;

#if DBG
	KdPrint(("FFSLookupFileName() start\n"));
#endif

	if (*DirInode != 0 && FullFileName->Buffer[0] != L'\\')
	{
		bCurrent = TRUE;
		RootInode = *DirInode;
	}
	else
	{
		RootInode = 2;
		*DirInode = 2;
	}
	
	RtlZeroMemory(&ffs_dir, sizeof(FFS_DIR_ENTRY));

	if (FullFileName->Length == 0)
	{
		return Status;
	}
	
	if (FullFileName->Length == 2 && FullFileName->Buffer[0] == L'\\')
	{
		if (!FFSLoadInode(Vcb, RootInode, dinode))
		{
			return Status;
		}
		*Inode = 2;
		*DirInode = 2;
		return STATUS_SUCCESS;
	}

	while (bRun && i < FullFileName->Length / 2)
	{
		ULONG Length;

		if (bCurrent)
		{
			bCurrent = FALSE;
		}
		else
		{
			if(FullFileName->Buffer[i++] != L'\\')
				break;
		}

		Length = i;

		while(i < FullFileName->Length / 2 && (FullFileName->Buffer[i] != L'\\'))
			i++;
		if (i - Length >0)
		{
			if (NT_SUCCESS(Status))
				RootInode = ffs_dir.d_ino;

			FileName = *FullFileName;
			FileName.Buffer += Length;
			FileName.Length = (USHORT)((i - Length) * 2);

			if (!FFSLoadInode(Vcb, RootInode, &in))
			{
				return STATUS_OBJECT_NAME_NOT_FOUND;
			}

			if (!((in.di_mode & IFMT) == IFDIR))
				break;

			Status = FFSScanDir(
				Vcb,
				RootInode,
				&FileName,
				&off,
				&in,
				&ffs_dir);

			if (!NT_SUCCESS(Status))
				bRun = FALSE;

		}
	}

	if (NT_SUCCESS(Status))
	{
		if (Inode)
			*Inode = ffs_dir.d_ino;
		if (DirInode)
			*DirInode = RootInode;
		
		if (Offset)
			*Offset = 0;
		if (dinode)
			FFSLoadInode(Vcb, ffs_dir.d_ino, dinode);
	}

#if DBG
	KdPrint(("FFSLookupFileName() end\n"));
#endif

	return Status;
}


NTSTATUS
FFSScanDir(
	IN PFFS_VCB            Vcb,
	IN ULONG               inode,
	IN PUNICODE_STRING     FileName,
	IN OUT PULONG          Index,
	IN PFFS_INODE          dinode,
	IN OUT PFFS_DIR_ENTRY  ffs_dir)
{
	NTSTATUS                Status = STATUS_UNSUCCESSFUL;
	USHORT                  InodeFileNameLength;
	UNICODE_STRING          InodeFileName;

	PVOID           DirectoryContent = NULL;
	struct direct*  pDir = NULL;
	ULONG           dwBytes;
	BOOLEAN         bFound = FALSE;
	

	InodeFileName.Buffer = NULL;
	
	__try
	{

		DirectoryContent = ExAllocatePool(NonPagedPool,
			(ULONG)dinode->di_size * 2);
		
		if (!DirectoryContent)
		{
			Status = STATUS_INSUFFICIENT_RESOURCES;
			__leave;
		}

		FFSReadInode(Vcb, dinode, 0, DirectoryContent, (ULONG)dinode->di_size, FALSE);
		dwBytes = 0;

		pDir = (struct direct *) ((PUCHAR)DirectoryContent + dwBytes);

		while (!bFound && dwBytes < (ULONG)dinode->di_size && pDir->d_ino)
		{
			InodeFileNameLength = pDir->d_namlen & 0xff;
			
			InodeFileName.Length = InodeFileName.MaximumLength =
				InodeFileNameLength * 2;
			
			InodeFileName.Buffer = ExAllocatePool(
				NonPagedPool,
				InodeFileNameLength * 2 + 2);

			RtlZeroMemory(InodeFileName.Buffer, InodeFileNameLength * 2 + 2);
			
			FFSCharToWchar(
				InodeFileName.Buffer,
				pDir->d_name,
				InodeFileNameLength);

			KdPrint(("FFSScanDir: file:%S to comp.\n", InodeFileName.Buffer));

			if (!RtlCompareUnicodeString(
				FileName,
				&InodeFileName,
				TRUE))
			{
				bFound = TRUE;
				*Index = dwBytes;
				RtlCopyMemory(ffs_dir, pDir, pDir->d_reclen > sizeof(FFS_DIR_ENTRY)
					? sizeof(FFS_DIR_ENTRY) : pDir->d_reclen);
				Status = STATUS_SUCCESS;
				KdPrint(("FFSScanDir: Found: inode=%xh\n", pDir->d_ino));
			}
			
			if (InodeFileName.Buffer != NULL)
			{
				ExFreePool(InodeFileName.Buffer);
				InodeFileName.Buffer = NULL;
			}
			
			dwBytes +=pDir->d_reclen;
			pDir = (struct direct *) ((PUCHAR)DirectoryContent + dwBytes);

		}

		if (!bFound)
			Status = STATUS_NO_SUCH_FILE;
	}

	__finally
	{
		if (InodeFileName.Buffer != NULL)
		{
			ExFreePool(InodeFileName.Buffer);
		}

		if (DirectoryContent)
			ExFreePool(DirectoryContent);
	}
	
	return Status;
}


NTSTATUS
FFSOpenFile(
	PFFS_VCB Vcb, PIRP Irp)
{
	NTSTATUS            Status = STATUS_OBJECT_NAME_NOT_FOUND;
	PIO_STACK_LOCATION  io_stack;
	PFFS_FCB            Fcb = NULL;
	PFFS_FCB            pParentFcb = NULL;
	PFFS_CCB            Ccb = NULL;
	ULONG               found_index = 0;
	struct ufs1_dinode* dinode;
	PLIST_ENTRY         Link;
	PFFS_FCB            TmpFcb;
	BOOLEAN             VcbResourceAcquired = FALSE;
	BOOLEAN             bFound = FALSE;
	BOOLEAN	            bDir = FALSE;
	ULONG               inode, dir_inode;
	UNICODE_STRING      FileName;

	FileName.Buffer = NULL;
	io_stack = IoGetCurrentIrpStackLocation(Irp);

	__try
	{
		ExAcquireResourceExclusiveLite(
			&Vcb->MainResource,
			TRUE);
		
		VcbResourceAcquired = TRUE;
		
		dinode = ExAllocatePool(
			NonPagedPool,
			sizeof(struct ufs1_dinode));
		if (!dinode)
			__leave;

		FileName.MaximumLength = 0x400;

		if (io_stack->FileObject->RelatedFileObject)
		{
			pParentFcb = (PFFS_FCB)(io_stack->FileObject->RelatedFileObject->FsContext);
		}

		FileName = io_stack->FileObject->FileName;


		if (pParentFcb)
		{
			dir_inode = pParentFcb->inode;
		}
		else
		{
			dir_inode = 2;
		}

		KdPrint(("FFSOpenFile: %S Opt: %xh.\n",
			FileName.Buffer,
			io_stack->Parameters.Create.Options));

		Status = FFSLookupFileName(
			Vcb,
			&FileName,
			&found_index,
			&inode,
			&dir_inode,
			dinode);

		if (!NT_SUCCESS(Status))
		{
			__leave;
		}

		Link = Vcb->FcbList.Flink;
		
		while (!bFound && Link != &Vcb->FcbList)
		{
			TmpFcb = CONTAINING_RECORD(Link, FFS_FCB, Next);
			
			if (TmpFcb && TmpFcb->Identifier.Type == FCB)
			{
#if DBG
				KdPrint(("FFSOpenFile: [%s,%xh]\n", TmpFcb->DbgFileName, TmpFcb->inode));
#endif				
				if (TmpFcb->inode == inode)
				{
					KdPrint(("FFSOpenFile: Found FCB for %xh.\n", inode));
					Fcb = TmpFcb;
					bFound = TRUE;
				}
			}
			Link = Link->Flink;
		}

		
		if (!Fcb)
		{
			Fcb = FFSAllocateFcb(
				Vcb, &FileName, found_index, inode,
				dir_inode, dinode);
		}
		
		if (Fcb)
		{
			Ccb = FFSAllocateCcb();
			
			Fcb->OpenHandleCount++;
			Vcb->OpenFileHandleCount++;
			Fcb->ReferenceCount++;
			Vcb->ReferenceCount++;
#if DBG
			KdPrint(("FFSOpenFile: %s refercount=%xh\n", Fcb->DbgFileName, Fcb->ReferenceCount));
#endif			
			io_stack->FileObject->FsContext = (void*) Fcb;
			io_stack->FileObject->FsContext2 = (void*) Ccb;
			io_stack->FileObject->PrivateCacheMap = NULL;
			io_stack->FileObject->SectionObjectPointer = &(Fcb->SectionObject);
			io_stack->FileObject->Vpb = Vcb->Vpb;
			
			Irp->IoStatus.Information = FILE_OPENED;
			Status = STATUS_SUCCESS;
#if DBG			
			KdPrint(("FFSOpenFile: %s OpenCount: %u ReferCount: %u\n",
				Fcb->DbgFileName, Fcb->OpenHandleCount,	Fcb->ReferenceCount));
#endif
		}
		else
		{
			Irp->IoStatus.Information = 0;
			Status = STATUS_OBJECT_NAME_NOT_FOUND;
			KdPrint(("FFSOpenFile: Create denied (CCB allocate error.)\n"));
		}
	
	}

	__finally
	{
		if (VcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Vcb->MainResource,
				ExGetCurrentResourceThread());
		}

		if (bFound)
		{
			if (dinode)
				ExFreePool(dinode);
		}
		else
		{
			if (!Fcb && dinode)
				ExFreePool(dinode);
		}
/*
		if (FileName.Buffer)
			ExFreePool(FileName.Buffer);
*/
	}
	
	return Status;
}


NTSTATUS
FFSCreate(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT      DeviceObject;
	PIRP                Irp;
	PIO_STACK_LOCATION  io_stack;
	PFFS_VCB            Vcb = 0;
	NTSTATUS            Status = STATUS_OBJECT_NAME_NOT_FOUND;

#if DBG
	KdPrint(("FFSCreate() start\n"));
#endif

	DeviceObject = IrpContext->DeviceObject;

	Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
	
	Irp = IrpContext->Irp;
	
	io_stack = IoGetCurrentIrpStackLocation(Irp);
	
	if (DeviceObject == FFSGlobal->DeviceObject)
	{
		KdPrint(("FFSCreate: Create on main device object.\n"));
		
		Irp->IoStatus.Information = FILE_OPENED;
		Status = STATUS_SUCCESS;
		IrpContext->Irp->IoStatus.Status = Status;
		
		FFSCompleteRequest(IrpContext->Irp, IO_NO_INCREMENT);
		
		FFSFreeIrpContext(IrpContext);
		
		return Status;
		
	}
	else if (io_stack->FileObject->FileName.Length == 0)
	{
		io_stack->FileObject->FsContext = Vcb;
		
		ExAcquireResourceExclusiveLite(
			&Vcb->MainResource,
			TRUE);
		
		Vcb->ReferenceCount++;
		
		ExReleaseResourceForThreadLite(
			&Vcb->MainResource,
			ExGetCurrentResourceThread());
		
		Irp->IoStatus.Information = FILE_OPENED;
		Status = STATUS_SUCCESS;
//		KdPrint(("FFSCreate: Create on volume succeded.\n"));
		IrpContext->Irp->IoStatus.Status = Status;
		
		FFSCompleteRequest(IrpContext->Irp, IO_NO_INCREMENT);
		
		FFSFreeIrpContext(IrpContext);
		
		return Status;
	}
	
//	KdPrint(("FFSCreate: FileName: (%S)\n", io_stack->FileObject->FileName.Buffer));

	__try
	{
		Status = FFSOpenFile(Vcb, Irp);
	}

	
	__finally
	{

		if (!IrpContext->ExceptionInProgress)
		{
			IrpContext->Irp->IoStatus.Status = Status;
			
			FFSCompleteRequest(
				IrpContext->Irp,
				(CCHAR)
				(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
			
			FFSFreeIrpContext(IrpContext);
			
			if (Vcb &&
				Vcb->Flags & VCB_DISMOUNT_PENDING &&
				!Vcb->ReferenceCount)
			{
				FFSFreeVcb(Vcb);
			}
		}
	}

#if DBG
	KdPrint(("FFSCreate() end\n"));	
#endif

	return Status;
}


NTSTATUS 
FFSDiskIoControl(
	IN PDEVICE_OBJECT   pDeviceObject,
	IN ULONG            IoctlCode,
	IN PVOID            InputBuffer,
	IN ULONG            InputBufferSize,
	IN OUT PVOID        OutputBuffer,
	IN OUT PULONG       OutputBufferSize)
{
	ULONG           OutBufferSize = 0;
	KEVENT          Event;
	PIRP            Irp;
	IO_STATUS_BLOCK IoStatus;
	NTSTATUS        Status;
	
	PAGED_CODE();
	
	ASSERT(pDeviceObject != NULL);
	
	if (OutputBufferSize)
	{
		OutBufferSize = *OutputBufferSize;
	}
	
	KeInitializeEvent(&Event, NotificationEvent, FALSE);
	
	Irp = IoBuildDeviceIoControlRequest(
		IoctlCode,
		pDeviceObject,
		InputBuffer,
		InputBufferSize,
		OutputBuffer,
		OutBufferSize,
		FALSE,
		&Event,
		&IoStatus);
	
	if (Irp == NULL)
	{
		KdPrint(("FFSDiskIoControl: Building IRQ error!\n"));
		return STATUS_INSUFFICIENT_RESOURCES;
	}
	
	Status = IoCallDriver(pDeviceObject, Irp);
	
	if (Status == STATUS_PENDING)
	{
		KeWaitForSingleObject(&Event, Executive, KernelMode, FALSE, NULL);
		Status = IoStatus.Status;
	}
	
	if (OutputBufferSize)
	{
		*OutputBufferSize = IoStatus.Information;
	}
	
	return Status;
}


BOOLEAN
FFSAcquireForLazyWrite(
	IN PVOID    Context,
	IN BOOLEAN  Wait)
{
	/*
	 * On a readonly filesystem this function still has to exist but it
	 * doesn't need to do anything.
	 */
	
	PFFS_FCB    Fcb;
#ifndef FFS_RO
	BOOLEAN     Status;
#endif
	
	PAGED_CODE();
	
	Fcb = (PFFS_FCB) Context;
	
	ASSERT(Fcb != NULL);
	
	ASSERT((Fcb->Identifier.Type == FCB) &&
		(Fcb->Identifier.Size == sizeof(FFS_FCB)));
	
#ifdef FFS_RO
	return TRUE;
#else
	if (ExAcquireResourceExclusiveLite(
		&Fcb->MainResource,Wait))
	{
		Status = TRUE;
	}
	else 
	{
		Status = FALSE;
	}
	
	return Status;
#endif
}


VOID
FFSReleaseFromLazyWrite(
	IN PVOID Context)
{
	/*
	 * On a readonly filesystem this function still has to exist but it
	 * doesn't need to do anything.
	 */
	PFFS_FCB Fcb;
	
	PAGED_CODE();
	
	Fcb = (PFFS_FCB) Context;
	
	ASSERT(Fcb != NULL);
	
	ASSERT((Fcb->Identifier.Type == FCB) &&
		(Fcb->Identifier.Size == sizeof(FFS_FCB)));
	
#ifdef FFS_RO
	return;
#else
	
	ExReleaseResourceForThreadLite(
		&Fcb->MainResource,
		ExGetCurrentResourceThread());
#endif
}


BOOLEAN
FFSAcquireForReadAhead(
	IN PVOID    Context,
	IN BOOLEAN  Wait)
{
	PFFS_FCB    Fcb;
	BOOLEAN     Status;
	
	PAGED_CODE();
	
	Fcb = (PFFS_FCB) Context;
	
	ASSERT(Fcb != NULL);
	
	ASSERT((Fcb->Identifier.Type == FCB) &&
		(Fcb->Identifier.Size == sizeof(FFS_FCB)));
	
	if (ExAcquireResourceSharedLite(
		&Fcb->MainResource, Wait))
	{
		Status = TRUE;
	}
	else
	{
		Status = FALSE;
	}
	
	return Status;
}


VOID
FFSReleaseFromReadAhead(
	IN PVOID Context)
{
	PFFS_FCB Fcb;
	
	PAGED_CODE();
	
	Fcb = (PFFS_FCB) Context;
	
	ASSERT(Fcb != NULL);
	
	ASSERT((Fcb->Identifier.Type == FCB) &&
		(Fcb->Identifier.Size == sizeof(FFS_FCB)));
	
	ExReleaseResourceForThreadLite(
		&Fcb->MainResource,
		ExGetCurrentResourceThread());
}


NTSTATUS
FFSClose(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT  DeviceObject;
	NTSTATUS        Status = STATUS_SUCCESS;
	PFFS_VCB        Vcb;
	BOOLEAN         VcbResourceAcquired = FALSE;
	PFILE_OBJECT    FileObject;
	PFFS_FCB        Fcb;
	BOOLEAN         FcbResourceAcquired = FALSE;
	PFFS_CCB        Ccb;
	BOOLEAN         FreeVcb = FALSE;
	
	__try
	{
		ASSERT(IrpContext != NULL);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;
		
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_SUCCESS;
			__leave;
		}
		
		Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
		
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		if (!ExAcquireResourceExclusiveLite(
			&Vcb->MainResource,
			IrpContext->IsSynchronous))
		{
			KdPrint(("FFSClose: PENDING ... Vcb: %xh/%xh\n", Vcb->OpenFileHandleCount, Vcb->ReferenceCount));
			Status = STATUS_PENDING;
			__leave;
		}
		
		VcbResourceAcquired = TRUE;
		
		FileObject = IrpContext->FileObject;
		
		Fcb = (PFFS_FCB) FileObject->FsContext;
		
		ASSERT(Fcb != NULL);
		
		if (Fcb->Identifier.Type == VCB)
		{
			if (Vcb->Flags & VCB_VOLUME_LOCKED)
			{
				Vcb->Flags &= ~VCB_VOLUME_LOCKED;
				
				FFSClearVpbFlag(Vcb->Vpb, VPB_LOCKED);
			}
			
			Vcb->ReferenceCount--;
			
			if (!Vcb->ReferenceCount && Vcb->Flags & VCB_DISMOUNT_PENDING)
			{
				FreeVcb = TRUE;
			}
			
			Status = STATUS_SUCCESS;
			
			__leave;
		}
		
		ASSERT((Fcb->Identifier.Type == FCB) &&
			(Fcb->Identifier.Size == sizeof(FFS_FCB)));
		
#ifndef FFS_RO
		if (!Fcb->IsPageFile)
#endif
		{
			if (!ExAcquireResourceExclusiveLite(
				&Fcb->MainResource,
				IrpContext->IsSynchronous))
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			FcbResourceAcquired = TRUE;
		}
		
		Ccb = (PFFS_CCB) FileObject->FsContext2;
		
		ASSERT(Ccb != NULL);
		
		ASSERT((Ccb->Identifier.Type == CCB) &&
			(Ccb->Identifier.Size == sizeof(FFS_CCB)));
		
		Fcb->ReferenceCount--;
		
		Vcb->ReferenceCount--;
		
		if (!Vcb->ReferenceCount && Vcb->Flags & VCB_DISMOUNT_PENDING)
		{
			FreeVcb = TRUE;
		}
		
		
		FFSFreeCcb(Ccb);
		
		if (!Fcb->ReferenceCount)
		{
			ExReleaseResourceForThreadLite(
				&Fcb->MainResource,
				ExGetCurrentResourceThread());
			
			FcbResourceAcquired = FALSE;
			
			FFSFreeFcb(Fcb);
		}
		
		Status = STATUS_SUCCESS;
    }
    __finally
    {
		if (VcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Vcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (FcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Fcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (!IrpContext->ExceptionInProgress)
		{
			if (Status == STATUS_PENDING)
			{
				
				Status = STATUS_SUCCESS;
				
				if (IrpContext->Irp != NULL)
				{
					IrpContext->Irp->IoStatus.Status = Status;
					
					FFSCompleteRequest(
						IrpContext->Irp,
						(CCHAR)
						(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
					
					IrpContext->Irp = NULL;
				}
				
				FFSQueueCloseRequest(IrpContext);
			}
			else
			{
				if (IrpContext->Irp != NULL)
				{
					IrpContext->Irp->IoStatus.Status = Status;
					
					FFSCompleteRequest(
						IrpContext->Irp,
						(CCHAR)
						(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
				}
				
				FFSFreeIrpContext(IrpContext);
				
				if (FreeVcb)
				{
					FFSFreeVcb(Vcb);
				}
			}
		}
    }
    
    return Status;
}


VOID
FFSQueueCloseRequest(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	ASSERT(IrpContext);
	
	ASSERT((IrpContext->Identifier.Type == ICX) &&
		(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
	
	/* IsSynchronous means we can block (so we don't requeue it) */
	IrpContext->IsSynchronous = TRUE;
	
	ExInitializeWorkItem(
		&IrpContext->WorkQueueItem,
		FFSDeQueueCloseRequest,
		IrpContext);
	
	ExQueueWorkItem(&IrpContext->WorkQueueItem, CriticalWorkQueue);
}


VOID
FFSDeQueueCloseRequest(
	IN PVOID Context)
{
	PFFS_IRP_CONTEXT IrpContext;
	
	IrpContext = (PFFS_IRP_CONTEXT) Context;
	
	ASSERT(IrpContext);
	
	ASSERT((IrpContext->Identifier.Type == ICX) &&
		(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
	
	__try
	{
		__try
		{
			FsRtlEnterFileSystem();
			FFSClose(IrpContext);
		}
		__except (FFSExceptionFilter(IrpContext, GetExceptionCode()))
		{
			FFSExceptionHandler(IrpContext);
		}
	}
	__finally
	{
		FsRtlExitFileSystem();
	}
}


BOOLEAN
FFSFastIoUnlockAll(
	IN PFILE_OBJECT         FileObject,
	IN PEPROCESS            ProcessId,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject)
{
	BOOLEAN     Status = FALSE;
	PFFS_FCB    Fcb;
	
	PAGED_CODE();
	
	__try
	{
		__try
		{
			if (DeviceObject == FFSGlobal->DeviceObject)
			{
				IoStatus->Status = STATUS_INVALID_DEVICE_REQUEST;
				Status = TRUE;
				__leave;
			}
			
			Fcb = (PFFS_FCB) FileObject->FsContext;
			
			ASSERT(Fcb != NULL);
			
			if (Fcb->Identifier.Type == VCB)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
			
			ASSERT((Fcb->Identifier.Type == FCB) &&
				(Fcb->Identifier.Size == sizeof(FFS_FCB)));
			
			if (Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
#if DBG			
			KdPrint((
				DRIVER_NAME ": %-16.16s %-31s %s\n",
				ProcessId->ImageFileName,
				"FASTIO_UNLOCK_ALL",
				Fcb->DbgFileName));
#endif			
			FsRtlEnterFileSystem();
			
			IoStatus->Status = FsRtlFastUnlockAll(
				&Fcb->FileLockAnchor,
				FileObject,
				ProcessId,
				NULL);
			
			IoStatus->Information = 0;
			
			Status =  TRUE;
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
			IoStatus->Status = GetExceptionCode();
			Status = TRUE;
		}
	}
	__finally
	{
		FsRtlExitFileSystem();
	}
	
	if (Status == FALSE)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: FALSE ***\n",
			ProcessId->ImageFileName,
			"FASTIO_UNLOCK_ALL"));
	}
	else if (IoStatus->Status != STATUS_SUCCESS)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: %s (%#x) ***\n",
			ProcessId->ImageFileName,
			"FASTIO_UNLOCK_ALL",
			FFSNtStatusToString(IoStatus->Status),
			IoStatus->Status));
	}
	
	return Status;
}


BOOLEAN
FFSFastIoUnlockAllByKey(
	IN PFILE_OBJECT         FileObject,
	IN PEPROCESS            ProcessId,
	IN ULONG                Key,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject)
{
	BOOLEAN     Status = FALSE;
	PFFS_FCB   Fcb;
	
	PAGED_CODE();
	
	__try
	{
		__try
		{
			if (DeviceObject == FFSGlobal->DeviceObject)
			{
				IoStatus->Status = STATUS_INVALID_DEVICE_REQUEST;
				Status = TRUE;
				__leave;
			}
			
			Fcb = (PFFS_FCB) FileObject->FsContext;
			
			ASSERT(Fcb != NULL);
			
			if (Fcb->Identifier.Type == VCB)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
			
			ASSERT((Fcb->Identifier.Type == FCB) &&
				(Fcb->Identifier.Size == sizeof(FFS_FCB)));
			
			if (Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
#if DBG			
			KdPrint((
				DRIVER_NAME ": %-16.16s %-31s %s\n",
				ProcessId->ImageFileName,
				"FASTIO_UNLOCK_ALL_BY_KEY",
				Fcb->DbgFileName));
#endif			
			KdPrint((
				DRIVER_NAME ": Key: %u\n",
				Key));
			
			FsRtlEnterFileSystem();
			
			IoStatus->Status = FsRtlFastUnlockAllByKey(
				&Fcb->FileLockAnchor,
				FileObject,
				ProcessId,
				Key,
				NULL);  
			
			IoStatus->Information = 0;
			
			Status =  TRUE;
		}

		__except (EXCEPTION_EXECUTE_HANDLER)
		{
			IoStatus->Status = GetExceptionCode();
			Status = TRUE;
		}
	}

	__finally
	{
		FsRtlExitFileSystem();
	}
	
	if (Status == FALSE)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: FALSE ***\n",
			ProcessId->ImageFileName,
			"FASTIO_UNLOCK_ALL_BY_KEY"));
	}
	else if (IoStatus->Status != STATUS_SUCCESS)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: %s (%#x) ***\n",
			ProcessId->ImageFileName,
			"FASTIO_UNLOCK_ALL_BY_KEY",
			FFSNtStatusToString(IoStatus->Status),
			IoStatus->Status));
	}
	
	return Status;
}


BOOLEAN
FFSFastIoUnlockSingle(
	IN PFILE_OBJECT         FileObject,
	IN PLARGE_INTEGER       FileOffset,
	IN PLARGE_INTEGER       Length,
	IN PEPROCESS            ProcessId,
	IN ULONG                Key,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject)
{
	BOOLEAN     Status = FALSE;
	PFFS_FCB    Fcb;
	
	PAGED_CODE();
	
	__try
	{
		__try
		{
			if (DeviceObject == FFSGlobal->DeviceObject)
			{
				IoStatus->Status = STATUS_INVALID_DEVICE_REQUEST;
				Status = TRUE;
				__leave;
			}
			
			Fcb = (PFFS_FCB) FileObject->FsContext;
			
			ASSERT(Fcb != NULL);
			
			if (Fcb->Identifier.Type == VCB)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
			
			ASSERT((Fcb->Identifier.Type == FCB) &&
				(Fcb->Identifier.Size == sizeof(FFS_FCB)));
			
			if (Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
#if DBG			
			KdPrint((
				DRIVER_NAME ": %-16.16s %-31s %s\n",
				ProcessId->ImageFileName,
				"FASTIO_UNLOCK_SINGLE",
				Fcb->DbgFileName));
#endif			
			KdPrint((
				DRIVER_NAME ": Offset: %I64u Length: %I64u Key: %u\n",
				FileOffset->QuadPart,
				Length->QuadPart,
				Key));
			
			FsRtlEnterFileSystem();
			
			IoStatus->Status = FsRtlFastUnlockSingle(
				&Fcb->FileLockAnchor,
				FileObject,
				FileOffset,
				Length,
				ProcessId,
				Key,
				NULL,
				NULL);                      
			
			IoStatus->Information = 0;
			
			Status =  TRUE;
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
			IoStatus->Status = GetExceptionCode();
			Status = TRUE;
		}
	}
	__finally
	{
		FsRtlExitFileSystem();
	}
	
	if (Status == FALSE)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: FALSE ***\n",
			ProcessId->ImageFileName,
			"FASTIO_UNLOCK_SINGLE"));
	}
	else if (IoStatus->Status != STATUS_SUCCESS)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: %s (%#x) ***\n",
			ProcessId->ImageFileName,
			"FASTIO_UNLOCK_SINGLE",
			FFSNtStatusToString(IoStatus->Status),
			IoStatus->Status));
	}
	
	return Status;
}


BOOLEAN
FFSFastIoLock(
	IN PFILE_OBJECT         FileObject,
	IN PLARGE_INTEGER       FileOffset,
	IN PLARGE_INTEGER       Length,
	IN PEPROCESS            ProcessId,
	IN ULONG                Key,
	IN BOOLEAN              FailImmediately,
	IN BOOLEAN              ExclusiveLock,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject)
{
	BOOLEAN     Status = FALSE;
	PFFS_FCB    Fcb;
	
	PAGED_CODE();
	
	__try
	{
		__try
		{
			if (DeviceObject == FFSGlobal->DeviceObject)
			{
				IoStatus->Status = STATUS_INVALID_DEVICE_REQUEST;
				Status = TRUE;
				__leave;
			}
			
			Fcb = (PFFS_FCB) FileObject->FsContext;
			
			ASSERT(Fcb != NULL);
			
			if (Fcb->Identifier.Type == VCB)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
			
			ASSERT((Fcb->Identifier.Type == FCB) &&
				(Fcb->Identifier.Size == sizeof(FFS_FCB)));
			
			if (Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
#if DBG			
			KdPrint((
				DRIVER_NAME ": %-16.16s %-31s %s\n",
				ProcessId->ImageFileName,
				"FASTIO_LOCK",
				Fcb->DbgFileName));
#endif			
			KdPrint((
				DRIVER_NAME ": Offset: %I64u Length: %I64u Key: %u %s%s\n",
				FileOffset->QuadPart,
				Length->QuadPart,
				Key,
				(FailImmediately ? "FailImmediately " : ""),
				(ExclusiveLock ? "ExclusiveLock " : "")	));
			
			if (Fcb->CommonFCBHeader.IsFastIoPossible != FastIoIsQuestionable)
			{
#if DBG
				KdPrint((
					DRIVER_NAME ": %-16.16s %-31s %s\n",
					ProcessId->ImageFileName,
					"FastIoIsQuestionable",
					Fcb->DbgFileName));
#endif				
				Fcb->CommonFCBHeader.IsFastIoPossible = FastIoIsQuestionable;
			}
			
			FsRtlEnterFileSystem();
			
			Status = FsRtlPrivateLock(
				&Fcb->FileLockAnchor,
				FileObject,
				FileOffset,
				Length,
				ProcessId,
				Key,
				FailImmediately,
				ExclusiveLock,
				IoStatus,
				NULL,
				NULL,
				NULL);
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
			IoStatus->Status = GetExceptionCode();
			Status = TRUE;
		}
	}

	__finally
	{
		FsRtlExitFileSystem();
	}
	
	if (Status == FALSE)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: FALSE ***\n",
			ProcessId->ImageFileName,
			"FASTIO_LOCK"));
	}
	else if (IoStatus->Status != STATUS_SUCCESS)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: %s (%#x) ***\n",
			ProcessId->ImageFileName,
			"FASTIO_LOCK",
			FFSNtStatusToString(IoStatus->Status),
			IoStatus->Status));
	}
	
	return Status;
}


BOOLEAN
FFSFastIoQueryStandardInfo(
	IN PFILE_OBJECT                 FileObject,
	IN BOOLEAN                      Wait,
	OUT PFILE_STANDARD_INFORMATION  Buffer,
	OUT PIO_STATUS_BLOCK            IoStatus,
	IN PDEVICE_OBJECT               DeviceObject)
{
	
	BOOLEAN     Status = FALSE;
	PFFS_FCB    Fcb;
	BOOLEAN     FcbMainResourceAcquired = FALSE;
	
	PAGED_CODE();
	
	__try
	{
		__try
		{
			if (DeviceObject == FFSGlobal->DeviceObject)
			{
				IoStatus->Status = STATUS_INVALID_DEVICE_REQUEST;
				Status = TRUE;
				__leave;
			}
			
			Fcb = (PFFS_FCB) FileObject->FsContext;
			
			ASSERT(Fcb != NULL);
			
			if (Fcb->Identifier.Type == VCB)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
			
			ASSERT((Fcb->Identifier.Type == FCB) &&
				(Fcb->Identifier.Size == sizeof(FFS_FCB)));
#if DBG			
			KdPrint((
				DRIVER_NAME ": %-16.16s %-31s %s\n",
				PsGetCurrentProcess()->ImageFileName,
				"FASTIO_QUERY_STANDARD_INFO",
				Fcb->DbgFileName));
#endif			
			FsRtlEnterFileSystem();
			
			if (!ExAcquireResourceSharedLite(
				&Fcb->MainResource,
				Wait))
			{
				Status = FALSE;
				__leave;
			}
			
			FcbMainResourceAcquired = TRUE;
			
			RtlZeroMemory(Buffer, sizeof(FILE_STANDARD_INFORMATION));
			
			/*
			typedef struct _FILE_STANDARD_INFORMATION {
			LARGE_INTEGER   AllocationSize;
			LARGE_INTEGER   EndOfFile;
			ULONG           NumberOfLinks;
			BOOLEAN         DeletePending;
			BOOLEAN         Directory;
			} FILE_STANDARD_INFORMATION, *PFILE_STANDARD_INFORMATION;
			*/
			
			Buffer->AllocationSize.QuadPart =
				Fcb->dinode->di_size;
			Buffer->EndOfFile.QuadPart =
				Fcb->dinode->di_size;			
			Buffer->NumberOfLinks = Fcb->dinode->di_nlink;
			
#ifndef FFS_RO
			Buffer->DeletePending = Fcb->DeletePending;
#else
			Buffer->DeletePending = FALSE;
#endif
			
			if (Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY)
			{
				Buffer->Directory = TRUE;
			}
			else
			{
				Buffer->Directory = FALSE;
			}
			
			IoStatus->Information = sizeof(FILE_STANDARD_INFORMATION);
			
			IoStatus->Status = STATUS_SUCCESS;
			
			Status =  TRUE;
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
			IoStatus->Status = GetExceptionCode();
			Status = TRUE;
		}
	}
	__finally
	{
		if (FcbMainResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Fcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		FsRtlExitFileSystem();
	}
	
	if (Status == FALSE)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: FALSE ***\n",
			PsGetCurrentProcess()->ImageFileName,
			"FASTIO_QUERY_STANDARD_INFO"));
	}
	else if (IoStatus->Status != STATUS_SUCCESS)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: %s (%#x) ***\n",
			PsGetCurrentProcess()->ImageFileName,
			"FASTIO_QUERY_STANDARD_INFO",
			FFSNtStatusToString(IoStatus->Status),
			IoStatus->Status));
	}
	
	return Status;
}


BOOLEAN
FFSFastIoQueryBasicInfo(
	IN PFILE_OBJECT             FileObject,
	IN BOOLEAN                  Wait,
	OUT PFILE_BASIC_INFORMATION Buffer,
	OUT PIO_STATUS_BLOCK        IoStatus,
	IN PDEVICE_OBJECT           DeviceObject)
{
	BOOLEAN     Status = FALSE;
	PFFS_FCB    Fcb;
	BOOLEAN     FcbMainResourceAcquired = FALSE;
	
	PAGED_CODE();
	__try
	{
		__try
		{
			if (DeviceObject == FFSGlobal->DeviceObject)
			{
				IoStatus->Status = STATUS_INVALID_DEVICE_REQUEST;
				Status = TRUE;
				__leave;
			}
			
			Fcb = (PFFS_FCB) FileObject->FsContext;
			
			ASSERT(Fcb != NULL);
			
			if (Fcb->Identifier.Type == VCB)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
			
			ASSERT((Fcb->Identifier.Type == FCB) &&
				(Fcb->Identifier.Size == sizeof(FFS_FCB)));
#if DBG			
			KdPrint((
				DRIVER_NAME ": %-16.16s %-31s %s\n",
				PsGetCurrentProcess()->ImageFileName,
				"FASTIO_QUERY_BASIC_INFO",
				Fcb->DbgFileName));
#endif			
			FsRtlEnterFileSystem();
			
			if (!ExAcquireResourceSharedLite(
				&Fcb->MainResource,
				Wait))
			{
				Status = FALSE;
				__leave;
			}
			
			FcbMainResourceAcquired = TRUE;
			
			RtlZeroMemory(Buffer, sizeof(FILE_BASIC_INFORMATION));
			
			/*
			typedef struct _FILE_BASIC_INFORMATION {
			LARGE_INTEGER   CreationTime;
			LARGE_INTEGER   LastAccessTime;
			LARGE_INTEGER   LastWriteTime;
			LARGE_INTEGER   ChangeTime;
			ULONG           FileAttributes;
			} FILE_BASIC_INFORMATION, *PFILE_BASIC_INFORMATION;
			*/

			Buffer->CreationTime = FFSSysTime(Fcb->dinode->di_ctime + TIMEZONE);
			Buffer->LastAccessTime = FFSSysTime(Fcb->dinode->di_atime + TIMEZONE);
			Buffer->LastWriteTime = FFSSysTime(Fcb->dinode->di_mtime + TIMEZONE);
			Buffer->ChangeTime = FFSSysTime(Fcb->dinode->di_mtime + TIMEZONE);
			
			
			Buffer->FileAttributes = Fcb->FileAttributes;
			
			IoStatus->Information = sizeof(FILE_BASIC_INFORMATION);
			
			IoStatus->Status = STATUS_SUCCESS;
			
			Status =  TRUE;
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
			IoStatus->Status = GetExceptionCode();
			Status = TRUE;
		}
	}

	__finally
	{
		if (FcbMainResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Fcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		FsRtlExitFileSystem();
	}
	
	

	if (Status == FALSE)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: FALSE ***\n",
			PsGetCurrentProcess()->ImageFileName,
			"FASTIO_QUERY_BASIC_INFO"));
	}
	else if (IoStatus->Status != STATUS_SUCCESS)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: %s (%#x) ***\n",
			PsGetCurrentProcess()->ImageFileName,
			"FASTIO_QUERY_BASIC_INFO",
			FFSNtStatusToString(IoStatus->Status),
			IoStatus->Status));
	}
	
	return Status;
}


BOOLEAN
FFSFastIoQueryNetworkOpenInfo(
	IN PFILE_OBJECT                     FileObject,
	IN BOOLEAN                          Wait,
	OUT PFILE_NETWORK_OPEN_INFORMATION  Buffer,
	OUT PIO_STATUS_BLOCK                IoStatus,
	IN PDEVICE_OBJECT                   DeviceObject)
{
	BOOLEAN     Status = FALSE;
	PFFS_FCB    Fcb;
	BOOLEAN     FcbMainResourceAcquired = FALSE;
	
	PAGED_CODE();
	
	__try
	{
		__try
		{
			if (DeviceObject == FFSGlobal->DeviceObject)
			{
				IoStatus->Status = STATUS_INVALID_DEVICE_REQUEST;
				Status = TRUE;
				__leave;
			}
			
			Fcb = (PFFS_FCB) FileObject->FsContext;
			
			ASSERT(Fcb != NULL);
			
			if (Fcb->Identifier.Type == VCB)
			{
				IoStatus->Status = STATUS_INVALID_PARAMETER;
				Status = TRUE;
				__leave;
			}
			
			ASSERT((Fcb->Identifier.Type == FCB) &&
				(Fcb->Identifier.Size == sizeof(FFS_FCB)));
#if DBG			
			KdPrint((
				DRIVER_NAME ": %-16.16s %-31s %s\n",
				PsGetCurrentProcess()->ImageFileName,
				"FASTIO_QUERY_NETWORK_OPEN_INFO",
				Fcb->DbgFileName));
#endif			
			FsRtlEnterFileSystem();
			
			if (!ExAcquireResourceSharedLite(
				&Fcb->MainResource,
				Wait))
			{
				Status = FALSE;
				__leave;
			}
			
			FcbMainResourceAcquired = TRUE;
			
			RtlZeroMemory(Buffer, sizeof(FILE_NETWORK_OPEN_INFORMATION));
			
			/*
			typedef struct _FILE_NETWORK_OPEN_INFORMATION {
			LARGE_INTEGER   CreationTime;
			LARGE_INTEGER   LastAccessTime;
			LARGE_INTEGER   LastWriteTime;
			LARGE_INTEGER   ChangeTime;
			LARGE_INTEGER   AllocationSize;
			LARGE_INTEGER   EndOfFile;
			ULONG           FileAttributes;
			} FILE_NETWORK_OPEN_INFORMATION, *PFILE_NETWORK_OPEN_INFORMATION;
			*/
			
			Buffer->CreationTime = FFSSysTime(Fcb->dinode->di_ctime + TIMEZONE);
			Buffer->LastAccessTime = FFSSysTime(Fcb->dinode->di_atime + TIMEZONE);
			Buffer->LastWriteTime = FFSSysTime(Fcb->dinode->di_mtime + TIMEZONE);
			Buffer->ChangeTime = FFSSysTime(Fcb->dinode->di_mtime + TIMEZONE);
			Buffer->FileAttributes = Fcb->FileAttributes;
			Buffer->AllocationSize.QuadPart = Fcb->dinode->di_size;
			Buffer->EndOfFile.QuadPart = Fcb->dinode->di_size;
			
			Buffer->FileAttributes = Fcb->FileAttributes;
			
			IoStatus->Information = sizeof(FILE_NETWORK_OPEN_INFORMATION);
			
			IoStatus->Status = STATUS_SUCCESS;
			
			Status =  TRUE;
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
			IoStatus->Status = GetExceptionCode();
			Status = TRUE;
		}
	}
	__finally
	{
		if (FcbMainResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Fcb->MainResource,
				ExGetCurrentResourceThread()
				);
		}
		
		FsRtlExitFileSystem();
	}
	
	
	
	if (Status == FALSE)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: FALSE ***\n",
			PsGetCurrentProcess()->ImageFileName,
			"FASTIO_QUERY_NETWORK_OPEN_INFO"));
	}
	else if (IoStatus->Status != STATUS_SUCCESS)
	{
		KdPrint((
			DRIVER_NAME ": %-16.16s %-31s *** Status: %s (%#x) ***\n",
			PsGetCurrentProcess()->ImageFileName,
			"FASTIO_QUERY_NETWORK_OPEN_INFO",
			FFSNtStatusToString(IoStatus->Status),
			IoStatus->Status));
	}
	
	return Status;
}


BOOLEAN
FFSFastIoRead(
	IN PFILE_OBJECT         FileObject,
	IN PLARGE_INTEGER       FileOffset,
	IN ULONG                Length,
	IN BOOLEAN              Wait,
	IN ULONG                LockKey,
	OUT PVOID               Buffer,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject)
{
	BOOLEAN     Status;
	PFFS_FCB    Fcb;
	
	PAGED_CODE();
	
	Fcb = (PFFS_FCB) FileObject->FsContext;
	
	ASSERT(Fcb != NULL);
	
	ASSERT((Fcb->Identifier.Type == FCB) &&
		(Fcb->Identifier.Size == sizeof(FFS_FCB)));
	
	Status = FsRtlCopyRead(
		FileObject, FileOffset,	Length,	Wait,
		LockKey, Buffer, IoStatus, DeviceObject);
	
	return Status;
}


BOOLEAN
FFSFastIoCheckIfPossible(
	IN PFILE_OBJECT         FileObject,
	IN PLARGE_INTEGER       FileOffset,
	IN ULONG                Length,
	IN BOOLEAN              Wait,
	IN ULONG                LockKey,
	IN BOOLEAN              CheckForReadOperation,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject)
{
	BOOLEAN          Status = FALSE;
	PFFS_FCB         Fcb;
	LARGE_INTEGER    lLength;
	
	lLength.QuadPart = Length;
	
	PAGED_CODE();
	
	__try
	{
		__try
		{
			if (DeviceObject == FFSGlobal->DeviceObject)
			{
				Status = FALSE;
				__leave;
			}
			
			Fcb = (PFFS_FCB) FileObject->FsContext;
			
			ASSERT(Fcb != NULL);
			
			if (Fcb->Identifier.Type == VCB)
			{
				Status = FALSE;
				__leave;
			}
			
			ASSERT((Fcb->Identifier.Type == FCB) &&
				(Fcb->Identifier.Size == sizeof(FFS_FCB)));
			
			if (Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY)
			{
				Status = FALSE;
				__leave;
			}
			
			FsRtlEnterFileSystem();
			
			if (CheckForReadOperation)
			{
				Status = FsRtlFastCheckLockForRead(
					&Fcb->FileLockAnchor,
					FileOffset,
					&lLength,
					LockKey,
					FileObject,
					PsGetCurrentProcess());
			}
			else
			{
				Status = FsRtlFastCheckLockForWrite(
					&Fcb->FileLockAnchor,
					FileOffset,
					&lLength,
					LockKey,
					FileObject,
					PsGetCurrentProcess());
			}
#if DBG			
			KdPrint(("FFSFastIOCheckPossible: %-16.16s %-31s %s\n",
				PsGetCurrentProcess()->ImageFileName,
				"FASTIO_CHECK_IF_POSSIBLE",
				Fcb->DbgFileName));
#endif			
			KdPrint(("FFSFastIIOCheckPossible: Offset: %I64u Length: %u Key: %u %s %s\n",
				FileOffset->QuadPart,
				Length,
				LockKey,
				(CheckForReadOperation ? "CheckForReadOperation:" : "CheckForWriteOperation:"),
				(Status ? "Succeeded" : "Failed")));
		}
		__except (EXCEPTION_EXECUTE_HANDLER)
		{
			Status = FALSE;
		}
	}
	__finally
	{
		FsRtlExitFileSystem();
	}
	
	return Status;
}


VOID
FFSFreeIrpContext(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	ASSERT(IrpContext != NULL);
	
	ASSERT((IrpContext->Identifier.Type == ICX) &&
		(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
	
	ExFreePool(IrpContext);
}


PFFS_IRP_CONTEXT
FFSAllocateIrpContext(
	IN PDEVICE_OBJECT DeviceObject,
	IN PIRP Irp)
{
	PIO_STACK_LOCATION  IoStackLocation;
	PFFS_IRP_CONTEXT    IrpContext;
	
	ASSERT(DeviceObject != NULL);
	ASSERT(Irp != NULL);
	
	IoStackLocation = IoGetCurrentIrpStackLocation(Irp);
	
	IrpContext = (PFFS_IRP_CONTEXT)ExAllocatePool(NonPagedPool, sizeof(FFS_IRP_CONTEXT));
	
	if (!IrpContext)
	{
		return NULL;
	}
	
	IrpContext->Identifier.Type = ICX;
	IrpContext->Identifier.Size = sizeof(FFS_IRP_CONTEXT);
	
	IrpContext->Irp = Irp;
	
	IrpContext->MajorFunction = IoStackLocation->MajorFunction;
	IrpContext->MinorFunction = IoStackLocation->MinorFunction;
	
	IrpContext->DeviceObject = DeviceObject;
	
	IrpContext->FileObject = IoStackLocation->FileObject;
	
	if (IrpContext->MajorFunction == IRP_MJ_FILE_SYSTEM_CONTROL ||
		IrpContext->MajorFunction == IRP_MJ_DEVICE_CONTROL ||
		IrpContext->MajorFunction == IRP_MJ_SHUTDOWN)
	{
		IrpContext->IsSynchronous = TRUE;
	}
	else if (IrpContext->MajorFunction == IRP_MJ_CLEANUP ||
		IrpContext->MajorFunction == IRP_MJ_CLOSE)
	{
		IrpContext->IsSynchronous = FALSE;
	}
	else
	{
		IrpContext->IsSynchronous = IoIsOperationSynchronous(Irp);
	}
	
	/*
	 * Temporary workaround for a bug in close that makes it reference a
	 * fileobject when it is no longer valid.
	 */
	if (IrpContext->MajorFunction == IRP_MJ_CLOSE)
	{
		IrpContext->IsSynchronous = TRUE;
	}
	
	IrpContext->IsTopLevel = (IoGetTopLevelIrp() == Irp);
	
	IrpContext->ExceptionInProgress = FALSE;
	
	return IrpContext;
}


NTSTATUS
FFSReadSectors(
	IN PDEVICE_OBJECT pDeviceObject,
	IN ULONG          DiskSector,
	IN ULONG          SectorCount,
	IN PVOID          Buffer)
{
	LARGE_INTEGER   sectorNumber;
	PIRP            irp;
	IO_STATUS_BLOCK ioStatus;
	KEVENT          event;
	NTSTATUS        Status;
	ULONG           sectorSize;

#if 0
	KdPrint(("FFSReadSectors(pDeviceObject %xh, lba %xh, sectors: %xh Buffer %xh)\n",
		pDeviceObject,DiskSector, SectorCount, Buffer));
#endif

	sectorNumber.u.HighPart = 0;
	sectorNumber.u.LowPart = DiskSector * SECTOR_SIZE;
	
	KeInitializeEvent(&event, NotificationEvent, FALSE);
	
	sectorSize = SECTOR_SIZE * SectorCount;
	
	irp = IoBuildSynchronousFsdRequest(IRP_MJ_READ,
		pDeviceObject,
		Buffer,
		sectorSize,
		&sectorNumber,
		&event,
		&ioStatus);
	
	if (irp == NULL) 
	{
		return STATUS_UNSUCCESSFUL;
	}
	
	Status = IoCallDriver(pDeviceObject, irp);
	
	if (Status == STATUS_PENDING) 
	{
		KeWaitForSingleObject(&event,
			Suspended,
			KernelMode,
			FALSE,
			NULL);
		Status = ioStatus.Status;
	}
	
	if (!NT_SUCCESS(Status)) 
	{
		KdPrint(("FFSReadSectors IO failed!!! Error code: %d(%x)\n", Status, Status));
		return Status;
	}
	
	return Status;
}


NTSTATUS
FFSReadDisk(
	IN PDEVICE_OBJECT pDeviceObject,
	IN ULONG          lba,
	IN ULONG          offset,
	IN ULONG          Size,
	IN PVOID          Buffer)
{
	ULONG		Sectors;
	NTSTATUS	Status;
	PUCHAR		Buf;
	ULONG		off = 0;

	if (offset >= SECTOR_SIZE)
		lba += offset / SECTOR_SIZE;
	off = offset % SECTOR_SIZE;
	Sectors = (off + Size + SECTOR_SIZE - 1) / SECTOR_SIZE;

#if 0
	KdPrint(("FFSReadDisk: lba %xh, offset: %xh size: %xh Sectors:%xh.\n",
		lba, offset, Size, Sectors));
#endif
	KdPrint(("FFSReadDisk: Se * SE: %d, Sectors : %x, offset: %X, off: %x, Size: %x\n", 
		Sectors * SECTOR_SIZE, Sectors, offset, off, Size));

	Buf = ExAllocatePool(NonPagedPool, Sectors * SECTOR_SIZE);
	if (!Buf)
	{
		KdPrint(("FFSReadDisk: no enough memory.\n"));
		return STATUS_UNSUCCESSFUL;
	}

	Status = FFSReadSectors(pDeviceObject, lba, Sectors, Buf);

	if (!NT_SUCCESS(Status))
	{
		KdPrint(("FFSReadDisk: Read Block Device error.\n"));
		ExFreePool(Buf);
		return Status;
	}

	RtlCopyMemory(Buffer, &Buf[off], Size);

	ExFreePool((PVOID)Buf);

	return Status;
}


struct fs *
FFSLoadSuper(
	IN PDEVICE_OBJECT pDeviceObject)
{
	PVOID		Buffer;
	NTSTATUS	Status;

#if DBG
	KdPrint(("FFSLoadSuper() start\n"));
#endif

	Buffer = ExAllocatePool(NonPagedPool, 3 * SECTOR_SIZE);
	if (!Buffer)
	{
		KdPrint(("FFSLoadSuper: no enough memory.\n"));
		return NULL;
	}

	/* 
	 * FFSv1 Super Block. LBA 0, Ƽ ó  8192 (2000h) ŭ  ִ.
	 * Super Block ũ⵵ 1376̱  SECTOR_SIZE(512) * 3 = 1536 ŭ о´.
	 * Ext2  (pDeviceObject, 2, 0, SECTOR_SIZE * 2, buffer)
	 * Super Block LBA 2 = Ƽ ó  400h  ִ. offset 0
	 * ũ 1024̱  SECTOR_SIZE(512) * 2 = 1024 ŭ о´.
	 */
	Status = FFSReadDisk(pDeviceObject, 0, 8192, SECTOR_SIZE * 3, Buffer);

	if (!NT_SUCCESS(Status))
	{
		KdPrint(("FFSReadDisk: Read Block Device error.\n"));
		ExFreePool(Buffer);
		return NULL;
	}

#if DBG
	KdPrint(("FFSLoadSuper() end\n"));
#endif

	return (struct fs *)Buffer;
}


NTSTATUS
FFSVerifyVolume(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT          DeviceObject;
	NTSTATUS                Status = STATUS_UNSUCCESSFUL;
	PFFS_SUPER_BLOCK        ffs_sb = NULL;
	PFFS_VCB                Vcb;
	BOOLEAN                 VcbResourceAcquired = FALSE;
	PIRP                    Irp;
	PIO_STACK_LOCATION      IoStackLocation;
	
	__try
	{
		ASSERT(IrpContext != NULL);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;
		//
		// This request is not allowed on the main device object
		//
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_INVALID_DEVICE_REQUEST;
			__leave;
		}
		
		Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
		
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		ExAcquireResourceExclusiveLite(
			&Vcb->MainResource,
			TRUE);
		
		VcbResourceAcquired = TRUE;
		
		if (!(Vcb->TargetDeviceObject->Flags & DO_VERIFY_VOLUME))
		{
			Status = STATUS_SUCCESS;
			__leave;
		}
		
		Irp = IrpContext->Irp;
		
		IoStackLocation = IoGetCurrentIrpStackLocation(Irp);
		
		ffs_sb = FFSLoadSuper(Vcb->TargetDeviceObject);

		if (!ffs_sb)
		{
			__leave;
		}

		if (ffs_sb->fs_magic != FS_UFS1_MAGIC)
		{
			__leave;
		}

		Vcb->TargetDeviceObject->Flags &= ~DO_VERIFY_VOLUME;
			
//		KdPrint(("FFSVerifyVolume: Volume verify succeeded.\n"));

		Status = STATUS_SUCCESS;
			
		__leave;
	}
	__finally
	{
		if (ffs_sb)
			ExFreePool(ffs_sb);

		if (VcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Vcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (!IrpContext->ExceptionInProgress)
		{
			IrpContext->Irp->IoStatus.Status = Status;
			
			FFSCompleteRequest(
				IrpContext->Irp,
				(CCHAR)
				(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
			
			FFSFreeIrpContext(IrpContext);
		}
	}
	
	return Status;
}


NTSTATUS
FFSIsVolumeMounted(
	IN PFFS_IRP_CONTEXT IrpContext)
{
    ASSERT(IrpContext);

    ASSERT((IrpContext->Identifier.Type == ICX) &&
           (IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));

    return FFSVerifyVolume(IrpContext);
}


NTSTATUS
FFSDismountVolume(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT  DeviceObject;
	NTSTATUS        Status = STATUS_UNSUCCESSFUL;
	PFFS_VCB        Vcb;
	BOOLEAN         VcbResourceAcquired = FALSE;
	
	__try
	{
		ASSERT(IrpContext != NULL);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;

		//
		// This request is not allowed on the main device object
		//
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_INVALID_DEVICE_REQUEST;
			__leave;
		}
		
		Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
		
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		ExAcquireResourceExclusiveLite(
			&Vcb->MainResource,
			TRUE);
		
		VcbResourceAcquired = TRUE;
		
		if (!(Vcb->Flags & VCB_VOLUME_LOCKED))
		{
			KdPrint(("FFSDismount: Volume is not locked.\n"));
			
			Status = STATUS_ACCESS_DENIED;
			
			__leave;
		}
		
		Vcb->Flags |= VCB_DISMOUNT_PENDING;
		
		KdPrint(("FFSDismount: Volume dismount pending.\n"));
		
		Status = STATUS_SUCCESS;
	}
	__finally
	{
		if (VcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Vcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (!IrpContext->ExceptionInProgress)
		{
			IrpContext->Irp->IoStatus.Status = Status;
			
			FFSCompleteRequest(
				IrpContext->Irp,
				(CCHAR)
				(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
			
			FFSFreeIrpContext(IrpContext);
		}
	}
	
	return Status;
}


VOID
FFSPurgeVolume(
	IN PFFS_VCB Vcb,
	IN BOOLEAN  FlushBeforePurge)
{
	BOOLEAN         VcbResourceAcquired = FALSE;
	PFFS_FCB        Fcb;
	LIST_ENTRY      FcbList;
	PLIST_ENTRY     ListEntry;
	PFCB_LIST_ENTRY FcbListEntry;
	
	__try
	{
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		ExAcquireResourceSharedLite(
			&Vcb->MainResource,
			TRUE);
		
		VcbResourceAcquired = TRUE;
		
#ifndef FFS_RO
		if (Vcb->ReadOnly)
		{
			Flush = FALSE;
		}
#endif
		
		InitializeListHead(&FcbList);
		
		for (
			ListEntry = Vcb->FcbList.Flink;
			ListEntry != &Vcb->FcbList;
			ListEntry = ListEntry->Flink
			)
		{
			Fcb = CONTAINING_RECORD(ListEntry, FFS_FCB, Next);
			
			ExAcquireResourceExclusiveLite(
				&Fcb->MainResource,
				TRUE);
			
			Fcb->ReferenceCount++;
#if DBG
			KdPrint(("FFSPurgeVolume: %s refercount=%xh\n", Fcb->DbgFileName, Fcb->ReferenceCount));
#endif	
			
			ExReleaseResourceForThreadLite(
				&Fcb->MainResource,
				ExGetCurrentResourceThread());
			
			FcbListEntry = ExAllocatePool(NonPagedPool, sizeof(FCB_LIST_ENTRY));
			
			FcbListEntry->Fcb = Fcb;
			
			InsertTailList(&FcbList, &FcbListEntry->Next);
		}
		
		ExReleaseResourceForThreadLite(
			&Vcb->MainResource,
			ExGetCurrentResourceThread());
		
		VcbResourceAcquired = FALSE;
		
		while (!IsListEmpty(&FcbList))
		{
			ListEntry = RemoveHeadList(&FcbList);
			
			FcbListEntry = CONTAINING_RECORD(ListEntry, FCB_LIST_ENTRY, Next);
			
			Fcb = FcbListEntry->Fcb;
			
			FFSPurgeFile(Fcb, FlushBeforePurge);
			
			if (!Fcb->OpenHandleCount && Fcb->ReferenceCount == 1)
			{
#if DBG
				KdPrint(("FFSFreeFcb %s.\n", Fcb->DbgFileName));
#endif
				FFSFreeFcb(Fcb);
			}
			
			ExFreePool(FcbListEntry);
		}
		
		KdPrint(("FFSPurgeVolume: Volume flushed and purged.\n"));
	}
	__finally
	{
		if (VcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Vcb->MainResource,
				ExGetCurrentResourceThread());
		}
	}
}


VOID
FFSPurgeFile(
	IN PFFS_FCB Fcb,
	IN BOOLEAN  FlushBeforePurge)
{
	ASSERT(Fcb != NULL);
        
	ASSERT((Fcb->Identifier.Type == FCB) &&
		(Fcb->Identifier.Size == sizeof(FFS_FCB)));
	
#ifndef FFS_RO
	if (FlushBeforePurge)
	{
#if DBG
		KdPrint(("FFSPurgeFile: CcFlushCache on %s.\n", Fcb->DbgFileName));
#endif
		
		CcFlushCache(&Fcb->SectionObject, NULL, 0, &IoStatus);
	}
#endif
	
	if (Fcb->SectionObject.ImageSectionObject)
	{
#if DBG
		KdPrint(("FFSPurgeFile: MmFlushImageSection on %s.\n", Fcb->DbgFileName));
#endif
	
		MmFlushImageSection(&Fcb->SectionObject, MmFlushForWrite);
	}
	
	if (Fcb->SectionObject.DataSectionObject)
	{
#if DBG
		KdPrint(("FFSPurgeFile: CcPurgeCacheSection on %s.\n", Fcb->DbgFileName));
#endif		
		CcPurgeCacheSection(&Fcb->SectionObject, NULL, 0, FALSE);
	}
}


NTSTATUS
FFSLockVolume(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT  DeviceObject;
	NTSTATUS        Status = STATUS_UNSUCCESSFUL;
	PFFS_VCB        Vcb;
	BOOLEAN         VcbResourceAcquired = FALSE;
	
	__try
	{
		ASSERT(IrpContext != NULL);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;
		
		//
		// This request is not allowed on the main device object
		//
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_INVALID_DEVICE_REQUEST;
			__leave;
		}
		
		Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
		
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		ExAcquireResourceSharedLite(
			&Vcb->MainResource,
			TRUE);
		
		VcbResourceAcquired = TRUE;
		
		if (Vcb->Flags & VCB_VOLUME_LOCKED)
		{
			KdPrint(("FFSLockVolume: Volume is already locked.\n"));
			
			Status = STATUS_ACCESS_DENIED;
			
			__leave;
		}
		
		if (Vcb->OpenFileHandleCount)
		{
			KdPrint(("FFSLockVolume: Open files exists.\n"));
			
			Status = STATUS_ACCESS_DENIED;
			
			__leave;
		}
		
		ExReleaseResourceForThreadLite(
			&Vcb->MainResource,
			ExGetCurrentResourceThread());
		
		VcbResourceAcquired = FALSE;
		
		FFSPurgeVolume(Vcb, TRUE);
		
		ExAcquireResourceExclusiveLite(
			&Vcb->MainResource,
			TRUE);
		
		VcbResourceAcquired = TRUE;
		
		if (!IsListEmpty(&Vcb->FcbList))
		{
			KdPrint(("FFSLockVolume: Could not purge cached files.\n"));
			
			Status = STATUS_ACCESS_DENIED;
			
			__leave;
		}
		
		Vcb->Flags |= VCB_VOLUME_LOCKED;
		
		FFSSetVpbFlag(Vcb->Vpb, VPB_LOCKED);
		
		KdPrint(("FFSLockVolume: Volume locked.\n"));
		
		Status = STATUS_SUCCESS;
	}
	__finally
	{
		if (VcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Vcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (!IrpContext->ExceptionInProgress)
		{
			IrpContext->Irp->IoStatus.Status = Status;
			
			FFSCompleteRequest(
				IrpContext->Irp,
				(CCHAR)
				(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
			
			FFSFreeIrpContext(IrpContext);
		}
	}
	
	return Status;
}


NTSTATUS
FFSUnlockVolume(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT  DeviceObject;
	NTSTATUS        Status = STATUS_UNSUCCESSFUL;
	PFFS_VCB        Vcb;
	BOOLEAN         VcbResourceAcquired = FALSE;
	
	__try
	{
		ASSERT(IrpContext != NULL);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;
		
		//
		// This request is not allowed on the main device object
		//
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_INVALID_DEVICE_REQUEST;
			__leave;
		}
		
		Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
		
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		ExAcquireResourceExclusiveLite(
			&Vcb->MainResource,
			TRUE);
		
		VcbResourceAcquired = TRUE;
		
		if (!(Vcb->Flags & VCB_VOLUME_LOCKED))
		{
			KdPrint((": FFSUnlockVolume: Volume is not locked .\n"));
			
			Status = STATUS_ACCESS_DENIED;
			
			__leave;
		}
		
		Vcb->Flags &= ~VCB_VOLUME_LOCKED;
		
		FFSClearVpbFlag(Vcb->Vpb, VPB_LOCKED);
		
		KdPrint(("FFSUnlockVolume: Volume unlocked.\n"));
		
		Status = STATUS_SUCCESS;
	}
	__finally
	{
		if (VcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Vcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (!IrpContext->ExceptionInProgress)
		{
			IrpContext->Irp->IoStatus.Status = Status;
			
			FFSCompleteRequest(
				IrpContext->Irp,
				(CCHAR)
				(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
			
			FFSFreeIrpContext(IrpContext);
		}
	}
	
	return Status;
}


NTSTATUS
FFSUserFsRequest(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PIRP                Irp;
	PIO_STACK_LOCATION  IoStackLocation;
	ULONG               FsControlCode;
	NTSTATUS            Status;
	
	ASSERT(IrpContext);
	
	ASSERT((IrpContext->Identifier.Type == ICX) &&
		(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
	
	Irp = IrpContext->Irp;
	
	IoStackLocation = IoGetCurrentIrpStackLocation(Irp);
	
#ifndef _GNU_NTIFS_
	FsControlCode =
		IoStackLocation->Parameters.FileSystemControl.FsControlCode;
#else
	FsControlCode = ((PEXTENDED_IO_STACK_LOCATION)
		IoStackLocation)->Parameters.FileSystemControl.FsControlCode;
#endif
	
	switch (FsControlCode)
	{
	case FSCTL_LOCK_VOLUME:
		Status = FFSLockVolume(IrpContext);
		break;
		
	case FSCTL_UNLOCK_VOLUME:
		Status = FFSUnlockVolume(IrpContext);
		break;
		
	case FSCTL_DISMOUNT_VOLUME:
		Status = FFSDismountVolume(IrpContext);
		break;
		
	case FSCTL_IS_VOLUME_MOUNTED:
		Status = FFSIsVolumeMounted(IrpContext);
		break;
		
	default:
		Status = STATUS_INVALID_DEVICE_REQUEST;
		IrpContext->Irp->IoStatus.Status = Status;
		FFSCompleteRequest(IrpContext->Irp, IO_NO_INCREMENT);
		FFSFreeIrpContext(IrpContext);
	}
	
	return Status;
}


NTSTATUS
FFSMountVolume(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT              MainDeviceObject;
	BOOLEAN                     GlobalDataResourceAcquired = FALSE;
	PIRP                        Irp;
	PIO_STACK_LOCATION          IoStackLocation;
	PDEVICE_OBJECT              TargetDeviceObject;
	NTSTATUS                    Status = STATUS_UNSUCCESSFUL;
	PDEVICE_OBJECT              VolumeDeviceObject = NULL;
	PFFS_VCB                    Vcb;
	BOOLEAN                     VcbResourceInitialized = FALSE;
	struct fs*                  ffs_super_block = NULL;
	USHORT                      VolumeLabelLength;
	ULONG                       IoctlSize;

#if DBG
	KdPrint(("FFSMountVolume() start\n"));
#endif

	__try
	{
		ASSERT(IrpContext != NULL);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		MainDeviceObject = IrpContext->DeviceObject;

		/*
		 * This request is only allowed on the main device object
		 */
		if (MainDeviceObject != FFSGlobal->DeviceObject)
		{
			Status = STATUS_INVALID_DEVICE_REQUEST;
			__leave;
		}
		
		ExAcquireResourceExclusiveLite(
			&(FFSGlobal->Resource),
			TRUE);
		
		GlobalDataResourceAcquired = TRUE;
		
		if (FFSGlobal->Flags & FFS_UNLOAD_PENDING)
		{
			Status = STATUS_UNRECOGNIZED_VOLUME;
			__leave;
		}
	
		Irp = IrpContext->Irp;
		
		IoStackLocation = IoGetCurrentIrpStackLocation(Irp);
		
		TargetDeviceObject =
			IoStackLocation->Parameters.MountVolume.DeviceObject;
		
		ffs_super_block = FFSLoadSuper(TargetDeviceObject);

#if DBG
		KdPrint(("fs_firstfield : %X\nffs magic : %X\nffs fs_sblockloc : %X\n", 
			ffs_super_block->fs_firstfield, 
			ffs_super_block->fs_magic,
			ffs_super_block->fs_sblockloc));
		/*
		 * fs_sblockloc Super Block ִ ġ Ǿ ִ. 
		 * FFSv1    0x2000 (8192)̴.
		 */
#endif

		Status = STATUS_UNSUCCESSFUL;
		if (ffs_super_block)
		{
			if (ffs_super_block->fs_magic == FS_UFS1_MAGIC)
			{
				KdPrint(("FFS: FFSv1 is found.\n"));
				Status = STATUS_SUCCESS;
			}
		}

		if (!NT_SUCCESS(Status))
		{
			__leave;
		}

		Status = IoCreateDevice(
			MainDeviceObject->DriverObject,
			sizeof(FFS_VCB),
			NULL,
			FILE_DEVICE_DISK_FILE_SYSTEM,
			0,
			FALSE,
			&VolumeDeviceObject);
		
		if (!NT_SUCCESS(Status))
		{
			__leave;
		}
		
		VolumeDeviceObject->StackSize = TargetDeviceObject->StackSize;
		
		(IoStackLocation->Parameters.MountVolume.Vpb)->DeviceObject =
			VolumeDeviceObject;
		
		Vcb = (PFFS_VCB) VolumeDeviceObject->DeviceExtension;
		
		RtlZeroMemory(Vcb, sizeof(FFS_VCB));
		
		Vcb->Identifier.Type = VCB;
		Vcb->Identifier.Size = sizeof(FFS_VCB);

#ifndef FFS_RO
		Vcb->ReadOnly = TRUE;
#endif
		
		ExInitializeResourceLite(&Vcb->MainResource);
		ExInitializeResourceLite(&Vcb->PagingIoResource);
		
		VcbResourceInitialized = TRUE;
		
		Vcb->Vpb = IoStackLocation->Parameters.MountVolume.Vpb;
		
		InitializeListHead(&Vcb->FcbList);
		
		Vcb->DeviceObject = VolumeDeviceObject;
		
		Vcb->TargetDeviceObject = TargetDeviceObject;
		
		Vcb->OpenFileHandleCount = 0;
		
		Vcb->ReferenceCount = 0;
		
		Vcb->Flags = 0;
		
		Vcb->ffs_super_block = ffs_super_block;
		
		VolumeLabelLength = 0;
		
		if (VolumeLabelLength > MAXIMUM_VOLUME_LABEL_LENGTH / 2)
		{
			VolumeLabelLength = MAXIMUM_VOLUME_LABEL_LENGTH / 2;
		}
		
		Vcb->Vpb->VolumeLabelLength = VolumeLabelLength * 2;
/*		
		FFSCharToWchar(
			Vcb->Vpb->VolumeLabel,
			Vcb->romfs_super_block->name,
			VolumeLabelLength
			);
*/
		
		Vcb->Vpb->SerialNumber = 'PS';


		IoctlSize = sizeof(DISK_GEOMETRY);
		
		Status = FFSDiskIoControl(
			TargetDeviceObject,
			IOCTL_DISK_GET_DRIVE_GEOMETRY,
			NULL,
			0,
			&Vcb->DiskGeometry,
			&IoctlSize);
		
		if (!NT_SUCCESS(Status))
		{
			__leave;
		}
		
		IoctlSize = sizeof(PARTITION_INFORMATION);
		
		Status = FFSDiskIoControl(
			TargetDeviceObject,
			IOCTL_DISK_GET_PARTITION_INFO,
			NULL,
			0,
			&Vcb->PartitionInformation,
			&IoctlSize);
		
		if (!NT_SUCCESS(Status))
		{
			Vcb->PartitionInformation.StartingOffset.QuadPart = 0;
			
			Vcb->PartitionInformation.PartitionLength.QuadPart =
				Vcb->DiskGeometry.Cylinders.QuadPart *
				Vcb->DiskGeometry.TracksPerCylinder *
				Vcb->DiskGeometry.SectorsPerTrack *
				Vcb->DiskGeometry.BytesPerSector;
			
			Status = STATUS_SUCCESS;
		}

		InsertTailList(&(FFSGlobal->VcbList), &Vcb->Next);
    }

    __finally
    {
	    if (GlobalDataResourceAcquired)
	    {
		    ExReleaseResourceForThreadLite(
			    &FFSGlobal->Resource,
			    ExGetCurrentResourceThread() );
	    }
	    
	    if (!NT_SUCCESS(Status))
	    {
		    if (ffs_super_block)
		    {
			    ExFreePool(ffs_super_block);
		    }
		    
		    if (VcbResourceInitialized)
		    {
			    ExDeleteResourceLite(&Vcb->MainResource);
			    ExDeleteResourceLite(&Vcb->PagingIoResource);
		    }
		    
		    if (VolumeDeviceObject)
		    {
			    IoDeleteDevice(VolumeDeviceObject);
		    }
	    }
	    
	    if (!IrpContext->ExceptionInProgress)
	    {
		    if (NT_SUCCESS(Status))
		    {
			    VolumeDeviceObject->Flags &= ~DO_DEVICE_INITIALIZING;
		    }
		    
		    IrpContext->Irp->IoStatus.Status = Status;
		    
		    FFSCompleteRequest(
			    IrpContext->Irp,
			    (CCHAR)
			    (NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
		    
		    FFSFreeIrpContext(IrpContext);
	    }
    }
    

#if DBG
	KdPrint(("FFSMountVolume() end\n"));
#endif

	return Status;
}


NTSTATUS
FFSFileSystemControl(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	NTSTATUS Status;

	ASSERT(IrpContext);
	
	ASSERT((IrpContext->Identifier.Type == ICX) &&
		(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
	
	switch (IrpContext->MinorFunction)
	{
	case IRP_MN_USER_FS_REQUEST:
		Status = FFSUserFsRequest(IrpContext);
		break;

	case IRP_MN_MOUNT_VOLUME:
		Status = FFSMountVolume(IrpContext);
		break;
#if 0	
	case IRP_MN_VERIFY_VOLUME:
		Status = FFSVerifyVolume(IrpContext);
		break;
#endif	
	default:
		Status = STATUS_INVALID_DEVICE_REQUEST;
		IrpContext->Irp->IoStatus.Status = Status;
		FFSCompleteRequest(IrpContext->Irp, IO_NO_INCREMENT);
		FFSFreeIrpContext(IrpContext);
	}

	return Status;
}


NTSTATUS
FFSReadFileData(
	IN PFFS_VCB             Vcb,
	IN ULONG                Index,
	IN struct ufs1_dinode*  dinode,
	IN PLARGE_INTEGER       Offset,
	IN ULONG                Length,
	IN OUT PVOID            Buffer)
{
	NTSTATUS Status = STATUS_UNSUCCESSFUL;
	
	PAGED_CODE();
	
	ASSERT(Vcb != NULL);
	ASSERT(dinode != NULL);
	ASSERT(Offset != NULL);
	ASSERT(Buffer != NULL);
	
	KdPrint(("FFSReadFileData: Index: %#x Offset: %I64u Length: %u\n",
		Index,
		Offset->QuadPart,
		Length));

	FFSReadInode(Vcb, dinode, (ULONG)Offset->QuadPart, Buffer, Length, TRUE);
	
	return STATUS_SUCCESS;
}


NTSTATUS
FFSRead(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	NTSTATUS Status;
	
	ASSERT(IrpContext);
	
	ASSERT((IrpContext->Identifier.Type == ICX) &&
		(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
	
	if (IrpContext->MinorFunction & IRP_MN_COMPLETE)
	{
		Status = FFSReadComplete(IrpContext);
	}
	else
	{
		Status = FFSReadNormal(IrpContext);
	}
	
	return Status;
}


NTSTATUS
FFSReadNormal(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT      DeviceObject;
	NTSTATUS            Status = STATUS_UNSUCCESSFUL;
	PFFS_VCB            Vcb;
	PFILE_OBJECT        FileObject;
	PFFS_FCB            Fcb;
	PFFS_CCB            Ccb;
	PIRP                Irp;
	PIO_STACK_LOCATION  IoStackLocation;
	ULONG               Length;
	ULONG               ReturnedLength;
	LARGE_INTEGER       ByteOffset;
	BOOLEAN             PagingIo;
	BOOLEAN             Nocache;
	BOOLEAN             SynchronousIo;
	BOOLEAN             VcbResourceAcquired = FALSE;
	BOOLEAN             FcbMainResourceAcquired = FALSE;
	BOOLEAN             FcbPagingIoResourceAcquired = FALSE;
	PUCHAR              Buffer;
	PDEVICE_OBJECT      DeviceToVerify;

	__try
	{
		ASSERT(IrpContext);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;
	
		/*
		 * This request is not allowed on the main device object
		 */
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_INVALID_DEVICE_REQUEST;
			__leave;
		}
		
		Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
		
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		FileObject = IrpContext->FileObject;
		
		Fcb = (PFFS_FCB) FileObject->FsContext;
		
		ASSERT(Fcb);
		
		if (Fcb->Identifier.Type == VCB)
		{
			Irp = IrpContext->Irp;
			
			IoStackLocation = IoGetCurrentIrpStackLocation(Irp);
			
			Length = IoStackLocation->Parameters.Read.Length;
			ByteOffset = IoStackLocation->Parameters.Read.ByteOffset;
			
			PagingIo = (Irp->Flags & IRP_PAGING_IO ? TRUE : FALSE);
			Nocache = (Irp->Flags & IRP_NOCACHE ? TRUE : FALSE);
			SynchronousIo = (FileObject->Flags & FO_SYNCHRONOUS_IO ? TRUE : FALSE);
			
			if (Length == 0)
			{
				Irp->IoStatus.Information = 0;
				Status = STATUS_SUCCESS;
				__leave;
			}
			
			if (!Nocache)
			{
				Status = STATUS_INVALID_PARAMETER;
				__leave;
			}
			
			if (!ExAcquireResourceSharedLite(
				&Vcb->MainResource,
				IrpContext->IsSynchronous))
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			VcbResourceAcquired = TRUE;
			
			if (ByteOffset.QuadPart >=
#ifndef FFS_RO
				Vcb->PartitionInformation.PartitionLength.QuadPart
#else
				(Vcb->ffs_super_block->fs_size - Vcb->ffs_super_block->fs_pendingblocks) * Vcb->ffs_block
#endif
				)
			{
				Irp->IoStatus.Information = 0;
				Status = STATUS_END_OF_FILE;
				__leave;
			}
				
			if ((ByteOffset.QuadPart + Length) >
#ifndef FFS_RO
					Vcb->PartitionInformation.PartitionLength.QuadPart
#else
					(Vcb->ffs_super_block->fs_size - Vcb->ffs_super_block->fs_pendingblocks) * Vcb->ffs_block
#endif
					)
			{
				Length = (ULONG) (
#ifndef FFS_RO
					Vcb->PartitionInformation.PartitionLength.QuadPart -
#else
					(Vcb->ffs_super_block->fs_size - Vcb->ffs_super_block->fs_pendingblocks) * Vcb->ffs_block -
#endif
					ByteOffset.QuadPart);
	
				Length &= ~(SECTOR_SIZE - 1);
			}
			
			if (KeGetCurrentIrql() > PASSIVE_LEVEL)
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			Buffer = FFSGetUserBuffer(Irp);
			
			if (Buffer == NULL)
			{
				Status = STATUS_INVALID_USER_BUFFER;
				__leave;
			}

			Status = FFSReadDisk(
				Vcb->TargetDeviceObject,
				(ULONG)ByteOffset.QuadPart / SECTOR_SIZE,
				(ULONG)ByteOffset.QuadPart % SECTOR_SIZE,
				Length, Buffer);
			
			if (Status == STATUS_VERIFY_REQUIRED)
			{
				Status = IoVerifyVolume(Vcb->TargetDeviceObject, FALSE);
				
				if (NT_SUCCESS(Status))
				{
					Status = FFSReadDisk(
						Vcb->TargetDeviceObject,
						(ULONG)ByteOffset.QuadPart / SECTOR_SIZE,
						(ULONG)ByteOffset.QuadPart % SECTOR_SIZE,
						Length, Buffer);
				}
			}
		
			if (NT_SUCCESS(Status))
			{
				Irp->IoStatus.Information = Length;
			}
			
			__leave;
		}
	
		ASSERT((Fcb->Identifier.Type == FCB) &&
			(Fcb->Identifier.Size == sizeof(FFS_FCB)));
		
		/*
		 * This request is not allowed on directories
		 */
		if (Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY)
		{
			Status = STATUS_INVALID_PARAMETER;
			__leave;
		}
		
		Ccb = (PFFS_CCB) FileObject->FsContext2;
		
		ASSERT(Ccb);
		
		ASSERT((Ccb->Identifier.Type == CCB) &&
			(Ccb->Identifier.Size == sizeof(FFS_CCB)));
		
		Irp = IrpContext->Irp;
		
		IoStackLocation = IoGetCurrentIrpStackLocation(Irp);
		
		Length = IoStackLocation->Parameters.Read.Length;
		ByteOffset = IoStackLocation->Parameters.Read.ByteOffset;
#if DBG
		KdPrint(("FFSReadNormal() Length : %d, ByteOffset %ld\n", Length, ByteOffset));
#endif
		PagingIo = (Irp->Flags & IRP_PAGING_IO ? TRUE : FALSE);
		Nocache = (Irp->Flags & IRP_NOCACHE ? TRUE : FALSE);
		SynchronousIo = (FileObject->Flags & FO_SYNCHRONOUS_IO ? TRUE : FALSE);
		
		if (Length == 0)
		{
			Irp->IoStatus.Information = 0;
			Status = STATUS_SUCCESS;
			__leave;
		}
		
		if (IrpContext->MinorFunction & IRP_MN_DPC)
		{
			IrpContext->MinorFunction &= ~IRP_MN_DPC;
			Status = STATUS_PENDING;
			__leave;
		}
		
		if (!PagingIo)
		{
			if (!ExAcquireResourceSharedLite(
				&Fcb->MainResource,
				IrpContext->IsSynchronous ))
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			FcbMainResourceAcquired = TRUE;
		}
		else
		{
			if (!ExAcquireResourceSharedLite(
				&Fcb->PagingIoResource,
				IrpContext->IsSynchronous ))
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			FcbPagingIoResourceAcquired = TRUE;
		}
		
		if ((ULONG)ByteOffset.QuadPart >= (Fcb->dinode->di_size))
		{
			Irp->IoStatus.Information = 0;
			Status = STATUS_END_OF_FILE;
			__leave;
		}
		
		if (!PagingIo)
		{
			if (!FsRtlCheckLockForReadAccess(
				&Fcb->FileLockAnchor,
				Irp))
			{
				Status = STATUS_FILE_LOCK_CONFLICT;
				__leave;
			}
		}
		
		if (!Nocache)
		{
			if ((ULONG)(ByteOffset.QuadPart + Length) >
				Fcb->dinode->di_size)
			{
				Length =
					(ULONG)Fcb->dinode->di_size - ByteOffset.LowPart;
			}
			
			if (FileObject->PrivateCacheMap == NULL)
			{
				CcInitializeCacheMap(
					FileObject,
					(PCC_FILE_SIZES)(&Fcb->CommonFCBHeader.AllocationSize),
					FALSE,
					&FFSGlobal->CacheManagerCallbacks,
					Fcb);
			}
			
			if (IrpContext->MinorFunction & IRP_MN_MDL)
			{
				CcMdlRead(
					FileObject,
					&ByteOffset,
					Length,
					&Irp->MdlAddress,
					&Irp->IoStatus);
				
				Status = Irp->IoStatus.Status;
			}
			else
			{
				Buffer = FFSGetUserBuffer(Irp);
				
				if (Buffer == NULL)
				{
					Status = STATUS_INVALID_USER_BUFFER;
					__leave;
				}
				
				if (!CcCopyRead(
					FileObject,
					&ByteOffset,
					Length,
					IrpContext->IsSynchronous,
					Buffer,
					&Irp->IoStatus))
				{
					Status = STATUS_PENDING;
					__leave;
				}
				
				Status = Irp->IoStatus.Status;
			}
		}
		else
		{
			ReturnedLength = Length;
			
			/* signed/unsigned mismatch  ߻. (ULONG)߰. */
			if (((ULONG)(ByteOffset.QuadPart + Length)) >
				Fcb->dinode->di_size)
			{
				ReturnedLength =
					(ULONG)Fcb->dinode->di_size - ByteOffset.LowPart;
				
				Length = (ReturnedLength & ~(SECTOR_SIZE - 1)) + SECTOR_SIZE;
			}
			
			if (KeGetCurrentIrql() > PASSIVE_LEVEL)
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			Buffer = FFSGetUserBuffer(Irp);
			
			if (Buffer == NULL)
			{
				Status = STATUS_INVALID_USER_BUFFER;
				__leave;
			}
#if DBG
			KdPrint(("FFSReadNormal() ReturnedLength: %x, Length: %x down\n", ReturnedLength, Length));
#endif
			Status = FFSReadFileData(
				Vcb,
				(ULONG) Fcb->IndexNumber.QuadPart,
				Fcb->dinode,
				&ByteOffset,
				Length,
				Buffer);
			
			if (Status == STATUS_VERIFY_REQUIRED)
			{
				DeviceToVerify = IoGetDeviceToVerify(PsGetCurrentThread());
				
				IoSetDeviceToVerify(PsGetCurrentThread(), NULL);
				
				Status = IoVerifyVolume(DeviceToVerify, FALSE);
				
				if (NT_SUCCESS(Status))
				{
					Status = FFSReadFileData(
						Vcb,
						(ULONG) Fcb->IndexNumber.QuadPart,
						Fcb->dinode,
						&ByteOffset,
						Length,
						Buffer);
				}
			}
			
			if (NT_SUCCESS(Status))
			{
				Irp->IoStatus.Information = ReturnedLength;
			}
		}
	}


	__finally
	{
		if (FcbPagingIoResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Fcb->PagingIoResource,
				ExGetCurrentResourceThread());
		}
		
		if (FcbMainResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Fcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (VcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Vcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (!IrpContext->ExceptionInProgress)
		{
			if (Status == STATUS_PENDING)
			{
				Status = FFSLockUserBuffer(
					IrpContext->Irp,
					Length,
					IoWriteAccess);
				
				if (NT_SUCCESS(Status))
				{
					Status = FFSQueueRequest(IrpContext);
				}
				else
				{
					IrpContext->Irp->IoStatus.Status = Status;
					FFSCompleteRequest(IrpContext->Irp, IO_NO_INCREMENT);
					FFSFreeIrpContext(IrpContext);
				}
			}
			else
			{
				IrpContext->Irp->IoStatus.Status = Status;
				
				if (SynchronousIo && !PagingIo && NT_SUCCESS(Status))
				{
					FileObject->CurrentByteOffset.QuadPart =
						ByteOffset.QuadPart + Irp->IoStatus.Information;
				}
				
				if (!PagingIo && NT_SUCCESS(Status))
				{
					FileObject->Flags &= ~FO_FILE_FAST_IO_READ;
				}
				
				FFSCompleteRequest(
					IrpContext->Irp,
					(CCHAR)
					(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
				
				FFSFreeIrpContext(IrpContext);
			}
		}
	}
	
	return Status;
}


NTSTATUS
FFSReadComplete(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	NTSTATUS        Status = STATUS_UNSUCCESSFUL;
	PFILE_OBJECT    FileObject;
	PIRP            Irp;
	
	__try
	{
		ASSERT(IrpContext);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		FileObject = IrpContext->FileObject;
		
		Irp = IrpContext->Irp;
		
		CcMdlReadComplete(FileObject, Irp->MdlAddress);
		
		Irp->MdlAddress = NULL;
		
		Status = STATUS_SUCCESS;
	}
	__finally
	{
		if (!IrpContext->ExceptionInProgress)
		{
			IrpContext->Irp->IoStatus.Status = Status;
			
			FFSCompleteRequest(
				IrpContext->Irp,
				(CCHAR)
				(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT)
				);
			
			FFSFreeIrpContext(IrpContext);
		}
	}
	
	return Status;
}


NTSTATUS
FFSQueryInformation(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT          DeviceObject;
	NTSTATUS                Status = STATUS_UNSUCCESSFUL;
	PFILE_OBJECT            FileObject;
	PFFS_FCB                Fcb;
	PFFS_CCB                Ccb;
	PIRP                    Irp;
	PIO_STACK_LOCATION      IoStackLocation;
	FILE_INFORMATION_CLASS  FileInformationClass;
	ULONG                   Length;
	PVOID                   Buffer;
	BOOLEAN                 FcbResourceAcquired = FALSE;
	
	__try
	{
		ASSERT(IrpContext != NULL);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;
		
		/*
		 * This request is not allowed on the main device object
		 */
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_INVALID_DEVICE_REQUEST;
			__leave;
		}
		
		FileObject = IrpContext->FileObject;
		
		Fcb = (PFFS_FCB) FileObject->FsContext;
		
		ASSERT(Fcb != NULL);
		
		/*
		 * This request is not allowed on volumes
		 */
		if (Fcb->Identifier.Type == VCB)
		{
			Status = STATUS_INVALID_PARAMETER;
			__leave;
		}
		
		ASSERT((Fcb->Identifier.Type == FCB) &&
			(Fcb->Identifier.Size == sizeof(FFS_FCB)));
		
#ifndef FFS_RO
		if (!Fcb->IsPageFile)
#endif
		{
			if (!ExAcquireResourceSharedLite(
				&Fcb->MainResource,
				IrpContext->IsSynchronous))
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			FcbResourceAcquired = TRUE;
		}
		
		Ccb = (PFFS_CCB) FileObject->FsContext2;
		
		ASSERT(Ccb != NULL);
		
		ASSERT((Ccb->Identifier.Type == CCB) &&
			(Ccb->Identifier.Size == sizeof(FFS_CCB)));
		
		Irp = IrpContext->Irp;
		
		IoStackLocation = IoGetCurrentIrpStackLocation(Irp);
		
		FileInformationClass =
			IoStackLocation->Parameters.QueryFile.FileInformationClass;
		
		Length = IoStackLocation->Parameters.QueryFile.Length;
		
		Buffer = Irp->AssociatedIrp.SystemBuffer;
		
		RtlZeroMemory(Buffer, Length);
		
		switch (FileInformationClass)
		{
		case FileBasicInformation:
			{
				PFILE_BASIC_INFORMATION FileBasicInformation;
				
				if (Length < sizeof(FILE_BASIC_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FileBasicInformation = (PFILE_BASIC_INFORMATION) Buffer;
				
				FileBasicInformation->CreationTime = FFSSysTime(Fcb->dinode->di_ctime + TIMEZONE);
				
				FileBasicInformation->LastAccessTime = FFSSysTime(Fcb->dinode->di_atime + TIMEZONE);
				
				FileBasicInformation->LastWriteTime = FFSSysTime(Fcb->dinode->di_mtime + TIMEZONE);
				
				FileBasicInformation->ChangeTime = FFSSysTime(Fcb->dinode->di_mtime + TIMEZONE);
				
				FileBasicInformation->FileAttributes = Fcb->FileAttributes;
				
				Irp->IoStatus.Information = sizeof(FILE_BASIC_INFORMATION);
				Status = STATUS_SUCCESS;
				__leave;
			}
		case FileAttributeTagInformation:
			{
				PFILE_ATTRIBUTE_TAG_INFORMATION FATI;
				
				if (Length < sizeof(FILE_ATTRIBUTE_TAG_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FATI = (PFILE_ATTRIBUTE_TAG_INFORMATION) Buffer;
				
				FATI->FileAttributes = Fcb->FileAttributes;
				FATI->ReparseTag = 0;
				
				Irp->IoStatus.Information = sizeof(FILE_ATTRIBUTE_TAG_INFORMATION);
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		case FileStandardInformation:
			{
				PFILE_STANDARD_INFORMATION FileStandardInformation;
				
				if (Length < sizeof(FILE_STANDARD_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FileStandardInformation = (PFILE_STANDARD_INFORMATION) Buffer;
				
				FileStandardInformation->AllocationSize.QuadPart =
					Fcb->dinode->di_size;
				
				FileStandardInformation->EndOfFile.QuadPart =
					Fcb->dinode->di_size;
				
				FileStandardInformation->NumberOfLinks = Fcb->dinode->di_nlink;
				
#ifndef FFS_RO
				
				FileStandardInformation->DeletePending = Fcb->DeletePending;
				
#else
				
				FileStandardInformation->DeletePending = FALSE;
				
#endif
				
				if (Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY)
				{
					FileStandardInformation->Directory = TRUE;
				}
				else
				{
					FileStandardInformation->Directory = FALSE;
				}
				
				Irp->IoStatus.Information = sizeof(FILE_STANDARD_INFORMATION);
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		case FileInternalInformation:
			{
				PFILE_INTERNAL_INFORMATION FileInternalInformation;
				
				if (Length < sizeof(FILE_INTERNAL_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FileInternalInformation = (PFILE_INTERNAL_INFORMATION) Buffer;
				
				/* The "inode number" */
				FileInternalInformation->IndexNumber = Fcb->IndexNumber;
				
				Irp->IoStatus.Information = sizeof(FILE_INTERNAL_INFORMATION);
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		case FileEaInformation:
			{
				PFILE_EA_INFORMATION FileEaInformation;
				
				if (Length < sizeof(FILE_EA_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FileEaInformation = (PFILE_EA_INFORMATION) Buffer;
				
				/* Romfs doesn't have any extended attributes */
				FileEaInformation->EaSize = 0;
				
				Irp->IoStatus.Information = sizeof(FILE_EA_INFORMATION);
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		case FileNameInformation:
			{
				PFILE_NAME_INFORMATION FileNameInformation;
				
				if (Length < sizeof(FILE_NAME_INFORMATION) +
					Fcb->FileName.Length - sizeof(WCHAR))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FileNameInformation = (PFILE_NAME_INFORMATION) Buffer;
				
				FileNameInformation->FileNameLength = Fcb->FileName.Length;
				
				RtlCopyMemory(
					FileNameInformation->FileName,
					Fcb->FileName.Buffer,
					Fcb->FileName.Length);
				
				Irp->IoStatus.Information = sizeof(FILE_NAME_INFORMATION) +
					Fcb->FileName.Length - sizeof(WCHAR);
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		case FilePositionInformation:
			{
				PFILE_POSITION_INFORMATION FilePositionInformation;
				
				if (Length < sizeof(FILE_POSITION_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FilePositionInformation = (PFILE_POSITION_INFORMATION) Buffer;
				
				FilePositionInformation->CurrentByteOffset =
					FileObject->CurrentByteOffset;
				
				Irp->IoStatus.Information = sizeof(FILE_POSITION_INFORMATION);
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		case FileAllInformation:
			{
				PFILE_ALL_INFORMATION       FileAllInformation;
				PFILE_BASIC_INFORMATION     FileBasicInformation;
				PFILE_STANDARD_INFORMATION  FileStandardInformation;
				PFILE_INTERNAL_INFORMATION  FileInternalInformation;
				PFILE_EA_INFORMATION        FileEaInformation;
				PFILE_POSITION_INFORMATION  FilePositionInformation;
				PFILE_NAME_INFORMATION      FileNameInformation;
				
				if (Length < sizeof(FILE_ALL_INFORMATION))
				{
					Status = STATUS_INFO_LENGTH_MISMATCH;
					__leave;
				}
				
				FileAllInformation = (PFILE_ALL_INFORMATION) Buffer;
				
				FileBasicInformation =
					&FileAllInformation->BasicInformation;
				
				FileStandardInformation =
					&FileAllInformation->StandardInformation;
				
				FileInternalInformation =
					&FileAllInformation->InternalInformation;
				
				FileEaInformation =
					&FileAllInformation->EaInformation;
				
				FilePositionInformation =
					&FileAllInformation->PositionInformation;
				
				FileNameInformation =
					&FileAllInformation->NameInformation;
				
				FileBasicInformation->CreationTime = FFSSysTime(Fcb->dinode->di_ctime + TIMEZONE);
				
				FileBasicInformation->LastAccessTime = FFSSysTime(Fcb->dinode->di_atime + TIMEZONE);
				
				FileBasicInformation->LastWriteTime = FFSSysTime(Fcb->dinode->di_mtime + TIMEZONE);
				
				FileBasicInformation->ChangeTime = FFSSysTime(Fcb->dinode->di_mtime + TIMEZONE);
				
				FileBasicInformation->FileAttributes = Fcb->FileAttributes;
				
				FileStandardInformation->AllocationSize.QuadPart =
					Fcb->dinode->di_size;
				
				FileStandardInformation->EndOfFile.QuadPart =
					Fcb->dinode->di_size;
				
				FileStandardInformation->NumberOfLinks = Fcb->dinode->di_nlink;
#ifndef FFS_RO
				FileStandardInformation->DeletePending = Fcb->DeletePending;
#else
				FileStandardInformation->DeletePending = FALSE;
#endif
				
				if (Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY)
				{
					FileStandardInformation->Directory = TRUE;
				}
				else
				{
					FileStandardInformation->Directory = FALSE;
				}
				
				/* The "inode number" */
				FileInternalInformation->IndexNumber = Fcb->IndexNumber;
				
				/* Romfs doesn't have any extended attributes */
				FileEaInformation->EaSize = 0;
				
				FilePositionInformation->CurrentByteOffset =
					FileObject->CurrentByteOffset;
				
				if (Length < sizeof(FILE_ALL_INFORMATION) +
					Fcb->FileName.Length - sizeof(WCHAR))
				{
					Irp->IoStatus.Information = sizeof(FILE_ALL_INFORMATION);
					Status = STATUS_BUFFER_OVERFLOW;
					__leave;
				}
				
				FileNameInformation->FileNameLength = Fcb->FileName.Length;
				
				RtlCopyMemory(
					FileNameInformation->FileName,
					Fcb->FileName.Buffer,
					Fcb->FileName.Length);
				
				Irp->IoStatus.Information = sizeof(FILE_ALL_INFORMATION) +
					Fcb->FileName.Length - sizeof(WCHAR);
				Status = STATUS_SUCCESS;
				__leave;
            }
	    
	    /*
	    case FileAlternateNameInformation:
            {
	    // TODO: Handle FileAlternateNameInformation
	    
	      // Here we would like to use RtlGenerate8dot3Name but I don't
	      // know how to use the argument PGENERATE_NAME_CONTEXT
	      }
	      */
	      
        case FileNetworkOpenInformation:
		{
			PFILE_NETWORK_OPEN_INFORMATION FileNetworkOpenInformation;
			
			if (Length < sizeof(FILE_NETWORK_OPEN_INFORMATION))
			{
				Status = STATUS_INFO_LENGTH_MISMATCH;
				__leave;
			}
			
			FileNetworkOpenInformation =
				(PFILE_NETWORK_OPEN_INFORMATION) Buffer;
			
			FileNetworkOpenInformation->CreationTime = FFSSysTime(Fcb->dinode->di_ctime + TIMEZONE);
			
			FileNetworkOpenInformation->LastAccessTime = FFSSysTime(Fcb->dinode->di_atime + TIMEZONE);
			
			FileNetworkOpenInformation->LastWriteTime = FFSSysTime(Fcb->dinode->di_mtime + TIMEZONE);
			
			FileNetworkOpenInformation->ChangeTime = FFSSysTime(Fcb->dinode->di_mtime + TIMEZONE);
			
			FileNetworkOpenInformation->AllocationSize.QuadPart =
				Fcb->dinode->di_size;
			
			FileNetworkOpenInformation->EndOfFile.QuadPart =
				Fcb->dinode->di_size;
			
			FileNetworkOpenInformation->FileAttributes =
				Fcb->FileAttributes;
			
			Irp->IoStatus.Information =
				sizeof(FILE_NETWORK_OPEN_INFORMATION);
			Status = STATUS_SUCCESS;
			__leave;
		}
		
        default:
		Status = STATUS_INVALID_INFO_CLASS;
        }
    }
    __finally
    {
	    if (FcbResourceAcquired)
	    {
		    ExReleaseResourceForThreadLite(
			    &Fcb->MainResource,
			    ExGetCurrentResourceThread());
	    }
	    
	    if (!IrpContext->ExceptionInProgress)
	    {
		    if (Status == STATUS_PENDING)
		    {
			    FFSQueueRequest(IrpContext);
		    }
		    else
		    {
			    IrpContext->Irp->IoStatus.Status = Status;
			    
			    FFSCompleteRequest(
				    IrpContext->Irp,
				    (CCHAR)
				    (NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
			    
			    FFSFreeIrpContext(IrpContext);
		    }
	    }
    }
    
    return Status;
}


NTSTATUS
FFSSetInformation(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT          DeviceObject;
	NTSTATUS                Status = STATUS_UNSUCCESSFUL;
	PFFS_VCB                Vcb;
	PFILE_OBJECT            FileObject;
	PFFS_FCB                Fcb;
	PFFS_CCB                Ccb;
	PIRP                    Irp;
	PIO_STACK_LOCATION      IoStackLocation;
	FILE_INFORMATION_CLASS  FileInformationClass;
	ULONG                   Length;
	PVOID                   Buffer;
#ifndef FFS_RO
	BOOLEAN                 VcbResourceAcquired = FALSE;
#endif
	BOOLEAN                 FcbMainResourceAcquired = FALSE;
#ifndef FFS_RO
	BOOLEAN                 FcbPagingIoResourceAcquired = FALSE;
#endif

	__try
	{
		ASSERT(IrpContext != NULL);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;

		/*
		 * This request is not allowed on the main device object
		 */
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_INVALID_DEVICE_REQUEST;
			__leave;
		}
		
		Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
		
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		FileObject = IrpContext->FileObject;
		
		Fcb = (PFFS_FCB) FileObject->FsContext;
		
		ASSERT(Fcb != NULL);
		
		//
		// This request is not allowed on volumes
		//
		if (Fcb->Identifier.Type == VCB)
		{
			Status = STATUS_INVALID_PARAMETER;
			__leave;
		}
		
		ASSERT((Fcb->Identifier.Type == FCB) &&
			(Fcb->Identifier.Size == sizeof(FFS_FCB)));
		
		Ccb = (PFFS_CCB) FileObject->FsContext2;
		
		ASSERT(Ccb != NULL);
		
		ASSERT((Ccb->Identifier.Type == CCB) &&
			(Ccb->Identifier.Size == sizeof(FFS_CCB)));
		
		Irp = IrpContext->Irp;
		
		IoStackLocation = IoGetCurrentIrpStackLocation(Irp);
		
		FileInformationClass =
			IoStackLocation->Parameters.SetFile.FileInformationClass;
		
		Length = IoStackLocation->Parameters.SetFile.Length;
		
		Buffer = Irp->AssociatedIrp.SystemBuffer;
		
#ifndef FFS_RO
		
		if (FileInformationClass == FileDispositionInformation ||
			FileInformationClass == FileRenameInformation ||
			FileInformationClass == FileLinkInformation)
		{
			if (!ExAcquireResourceExclusiveLite(
				&Vcb->MainResource,
				IrpContext->IsSynchronous))
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			VcbResourceAcquired = TRUE;
		}
		
#endif // !FFS_RO
		
#ifndef FFS_RO
		if (!Fcb->IsPageFile)
#endif
		{
			if (!ExAcquireResourceExclusiveLite(
				&Fcb->MainResource,
				IrpContext->IsSynchronous ))
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			FcbMainResourceAcquired = TRUE;
		}
		
#ifndef FFS_RO
		
		if (FileInformationClass == FileDispositionInformation ||
			FileInformationClass == FileRenameInformation ||
			FileInformationClass == FileLinkInformation ||
			FileInformationClass == FileAllocationInformation ||
			FileInformationClass == FileEndOfFileInformation)
		{
			if (!ExAcquireResourceExclusiveLite(
				&Fcb->PagingIoResource,
				IrpContext->IsSynchronous))
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			FcbPagingIoResourceAcquired = TRUE;
		}
		
#endif // !FFS_RO
		
#ifndef FFS_RO
		if (Vcb->ReadOnly)
#endif
		{
			if (FileInformationClass != FilePositionInformation)
			{
				Status = STATUS_MEDIA_WRITE_PROTECTED;
				__leave;
			}
		}
		
		switch (FileInformationClass)
		{
			/*
			 * This is the only set file information request supported on read
			 * only file systems
			 */
		case FilePositionInformation:
			{
				PFILE_POSITION_INFORMATION FilePositionInformation;
				
				if (Length < sizeof(FILE_POSITION_INFORMATION))
				{
					Status = STATUS_INVALID_PARAMETER;
					__leave;
				}
				
				FilePositionInformation = (PFILE_POSITION_INFORMATION) Buffer;
				
				if ((FileObject->Flags & FO_NO_INTERMEDIATE_BUFFERING) &&
					(FilePositionInformation->CurrentByteOffset.LowPart &
					DeviceObject->AlignmentRequirement) )
				{
					Status = STATUS_INVALID_PARAMETER;
					__leave;
				}
				
				FileObject->CurrentByteOffset =
					FilePositionInformation->CurrentByteOffset;
				
				Status = STATUS_SUCCESS;
				__leave;
			}
			
		default:
			Status = STATUS_INVALID_INFO_CLASS;
		}
    }
    __finally
    {
	    
#ifndef FFS_RO
	    
	    if (FcbPagingIoResourceAcquired)
	    {
		    ExReleaseResourceForThreadLite(
			    &Fcb->PagingIoResource,
			    ExGetCurrentResourceThread());
	    }
	    
#endif // !FFS_RO
	    
	    if (FcbMainResourceAcquired)
	    {
		    ExReleaseResourceForThreadLite(
			    &Fcb->MainResource,
			    ExGetCurrentResourceThread());
	    }
	    
#ifndef FFS_RO
	    
	    if (VcbResourceAcquired)
	    {
		    ExReleaseResourceForThreadLite(
			    &Vcb->MainResource,
			    ExGetCurrentResourceThread());
	    }
	    
#endif // !FFS_RO
	    
	    if (!IrpContext->ExceptionInProgress)
	    {
		    if (Status == STATUS_PENDING)
		    {
			    FFSQueueRequest(IrpContext);
		    }
		    else
		    {
			    IrpContext->Irp->IoStatus.Status = Status;
			    
			    FFSCompleteRequest(
				    IrpContext->Irp,
				    (CCHAR)
				    (NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
			    
			    FFSFreeIrpContext(IrpContext);
		    }
	    }
    }
    
    return Status;
}


NTSTATUS
FFSLockUserBuffer(
	IN PIRP             Irp,
	IN ULONG            Length,
	IN LOCK_OPERATION   Operation)
{
	NTSTATUS Status;
	
	ASSERT(Irp != NULL);
	
	if (Irp->MdlAddress != NULL)
	{
		return STATUS_SUCCESS;
	}
	
	IoAllocateMdl(Irp->UserBuffer, Length, FALSE, FALSE, Irp);
	
	if (Irp->MdlAddress == NULL)
	{
		return STATUS_INSUFFICIENT_RESOURCES;
	}
	
	__try
	{
		MmProbeAndLockPages(Irp->MdlAddress, Irp->RequestorMode, Operation);
		
		Status = STATUS_SUCCESS;
	}
	__except (EXCEPTION_EXECUTE_HANDLER)
	{
		IoFreeMdl(Irp->MdlAddress);
		
		Irp->MdlAddress = NULL;
		
		Status = STATUS_INVALID_USER_BUFFER;
	}
	
	return Status;
}


PVOID
FFSGetUserBuffer(
	IN PIRP Irp)
{
	ASSERT(Irp != NULL);
	
	if (Irp->MdlAddress != NULL)
	{
		return MmGetSystemAddressForMdl(Irp->MdlAddress);
	}
	else
	{
		return Irp->UserBuffer;
	}
}


ULONG
FFSGetInfoLength(
	IN FILE_INFORMATION_CLASS  FileInformationClass)
{
	switch (FileInformationClass)
	{
	case FileDirectoryInformation:
		return sizeof(FILE_DIRECTORY_INFORMATION);
		break;
		
	case FileFullDirectoryInformation:
		return sizeof(FILE_FULL_DIR_INFORMATION);
		break;
		
	case FileBothDirectoryInformation:
		return sizeof(FILE_BOTH_DIR_INFORMATION);
		break;
		
	case FileNamesInformation:
		return sizeof(FILE_NAMES_INFORMATION);
		break;
		
	default:
		break;
	}

	return 0;
}


ULONG
FFSProcessDirEntry(
	IN PFFS_VCB                Vcb,
	IN FILE_INFORMATION_CLASS  FileInformationClass,
	IN ULONG                   in,
	IN PVOID                   Buffer,
	IN ULONG                   UsedLength,
	IN ULONG                   Length,
	IN ULONG                   FileIndex,
	IN PUNICODE_STRING         pName,
	IN BOOLEAN                 Single)
{
	FFS_INODE inode;
	PFILE_DIRECTORY_INFORMATION FDI;
	PFILE_FULL_DIR_INFORMATION FFI;
	PFILE_BOTH_DIR_INFORMATION FBI;
	PFILE_NAMES_INFORMATION	FNI;

	ULONG InfoLength = 0;
	ULONG NameLength = 0;
	ULONG dwBytes = 0;

	NameLength = pName->Length;
//	KdPrint(("FFSProcessDirEntry: NameLength: %xh.\n", NameLength));

	if (!in)
	{
		KdPrint(("FFSProcessDirEntry: direct is empty.\n"));
		return 0;
	}

	InfoLength = FFSGetInfoLength(FileInformationClass);
	if (!InfoLength || InfoLength + NameLength - sizeof(WCHAR)> Length)
	{
		KdPrint(("FFSProcessDirEntry: Size/Length error.\n"));
		return 0;
	}

	if(!FFSLoadInode(Vcb, in, &inode))
	{
		KdPrint(("FFSProcessDirEntry: Loading inode %xh error.\n", in));
		return 0;
	}

	switch(FileInformationClass)
	{
	case FileDirectoryInformation:
		FDI = (PFILE_DIRECTORY_INFORMATION) ((PUCHAR)Buffer + UsedLength);
		//FillInfo (FDI, InfoLength, inode, FileIndex, NameLength, pName, Single)
		if (!Single)
			FDI->NextEntryOffset = InfoLength + NameLength - sizeof(WCHAR);
		else
			FDI->NextEntryOffset = 0;
		FDI->FileIndex = FileIndex;
		FDI->CreationTime = FFSSysTime(inode.di_ctime + TIMEZONE);
		FDI->LastAccessTime = FFSSysTime(inode.di_atime + TIMEZONE);
		FDI->LastWriteTime = FFSSysTime(inode.di_mtime + TIMEZONE);
		FDI->ChangeTime = FFSSysTime(inode.di_mtime + TIMEZONE);
		FDI->EndOfFile.QuadPart = inode.di_size;
		FDI->AllocationSize.QuadPart = inode.di_size;
		FDI->FileAttributes = FILE_ATTRIBUTE_NORMAL;
		if ((inode.di_mode & IFMT) == IFDIR)
			FDI->FileAttributes |= FILE_ATTRIBUTE_DIRECTORY;
		FDI->FileNameLength = NameLength;
		RtlCopyMemory(FDI->FileName, pName->Buffer, NameLength);
		dwBytes = InfoLength + NameLength - sizeof(WCHAR); 
		break;
		
	case FileFullDirectoryInformation:
		FFI = (PFILE_FULL_DIR_INFORMATION) ((PUCHAR)Buffer + UsedLength);
//		FillInfo (FFI, InfoLength, inode, FileIndex, NameLength, pName, Single)
		if (!Single)
			FFI->NextEntryOffset = InfoLength + NameLength - sizeof(WCHAR);
		else
			FFI->NextEntryOffset = 0;
		FFI->FileIndex = FileIndex;
		FFI->CreationTime = FFSSysTime(inode.di_ctime + TIMEZONE);
		FFI->LastAccessTime = FFSSysTime(inode.di_atime + TIMEZONE);
		FFI->LastWriteTime = FFSSysTime(inode.di_mtime + TIMEZONE);
		FFI->ChangeTime = FFSSysTime(inode.di_mtime + TIMEZONE);
		FFI->EndOfFile.QuadPart = inode.di_size;
		FFI->AllocationSize.QuadPart = inode.di_size;
		FFI->FileAttributes = FILE_ATTRIBUTE_NORMAL;
		if ((inode.di_mode & IFMT) == IFDIR)
			FFI->FileAttributes |= FILE_ATTRIBUTE_DIRECTORY;
		FFI->FileNameLength = NameLength;
		RtlCopyMemory(FFI->FileName, pName->Buffer, NameLength);
		dwBytes = InfoLength + NameLength - sizeof(WCHAR); 

		break;
		
	case FileBothDirectoryInformation:
		FBI = (PFILE_BOTH_DIR_INFORMATION) ((PUCHAR)Buffer + UsedLength);
//		FillInfo (FBI, InfoLength, inode, FileIndex, NameLength, pName, Single)
		if (!Single)
			FBI->NextEntryOffset = InfoLength + NameLength - sizeof(WCHAR);
		else
			FBI->NextEntryOffset = 0;
		FBI->CreationTime = FFSSysTime(inode.di_ctime + TIMEZONE);
		FBI->LastAccessTime = FFSSysTime(inode.di_atime + TIMEZONE);
		FBI->LastWriteTime = FFSSysTime(inode.di_mtime + TIMEZONE);
		FBI->ChangeTime = FFSSysTime(inode.di_mtime + TIMEZONE);

/*
		FBI->CreationTime.QuadPart = inode.i_ctime + TIMEZONE;
		FBI->LastAccessTime.QuadPart = inode.i_atime + TIMEZONE;
		FBI->LastWriteTime.QuadPart = inode.i_mtime + TIMEZONE;
		FBI->ChangeTime.QuadPart = inode.i_mtime + TIMEZONE;
*/
		FBI->EndOfFile.QuadPart = inode.di_size;
		FBI->AllocationSize.QuadPart = inode.di_size;
		FBI->FileAttributes = FILE_ATTRIBUTE_NORMAL;
		if ((inode.di_mode & IFMT) == IFDIR)
			FBI->FileAttributes |= FILE_ATTRIBUTE_DIRECTORY;
		FBI->FileNameLength = NameLength;
		RtlCopyMemory(FBI->FileName, pName->Buffer, NameLength);
		dwBytes = InfoLength + NameLength - sizeof(WCHAR); 

		break;
		
	case FileNamesInformation:
		FNI = (PFILE_NAMES_INFORMATION) ((PUCHAR)Buffer + UsedLength);
//		FillInfo (FNI, InfoLength, inode, FileIndex, NameLength, pName, Single)
		if (!Single)
			FNI->NextEntryOffset = InfoLength + NameLength - sizeof(WCHAR);
		else
			FNI->NextEntryOffset = 0;
		FNI->FileNameLength = NameLength;
		RtlCopyMemory(FNI->FileName, pName->Buffer, NameLength);
		dwBytes = InfoLength + NameLength - sizeof(WCHAR); 

		break;
		
	default:
		break;
	}

	return dwBytes;
}


NTSTATUS
FFSQueryDirectory(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT          DeviceObject;
	NTSTATUS                Status = STATUS_UNSUCCESSFUL;
	PFFS_VCB                Vcb;
	PFILE_OBJECT            FileObject;
	PFFS_FCB                Fcb;
	PFFS_CCB                Ccb;
	PIRP                    Irp;
	PIO_STACK_LOCATION      IoStackLocation;
	FILE_INFORMATION_CLASS  FileInformationClass;
	ULONG                   Length;
	PUNICODE_STRING         FileName;
	ULONG                   FileIndex;
	BOOLEAN                 RestartScan;
	BOOLEAN                 ReturnSingleEntry;
	BOOLEAN                 IndexSpecified;
	PUCHAR                  Buffer;
	BOOLEAN                 FirstQuery;
	struct ufs1_dinode*     Inode = NULL;
	BOOLEAN                 FcbResourceAcquired = FALSE;
	ULONG                   UsedLength = 0;
	USHORT                  InodeFileNameLength;
	UNICODE_STRING          InodeFileName;
	PVOID                   DirectoryContent = NULL;
	struct direct*          pDir;
	ULONG                   dwBytes;
	ULONG                   dwTemp = 0;
	ULONG                   dwSize = 0;
	ULONG                   dwReturn = 0;
	BOOLEAN                 bRun = TRUE;

	InodeFileName.Buffer = NULL;
	
	__try
	{
		ASSERT(IrpContext);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;
		
		/*
		 * This request is not allowed on the main device object
		 */
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_INVALID_DEVICE_REQUEST;
			__leave;
		}
		
		Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
		
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		FileObject = IrpContext->FileObject;
		
		Fcb = (PFFS_FCB) FileObject->FsContext;
		
		ASSERT(Fcb);
		
		/*
		 * This request is not allowed on volumes
		 */
		if (Fcb->Identifier.Type == VCB)
		{
			Status = STATUS_INVALID_PARAMETER;
			__leave;
		}
		
		ASSERT((Fcb->Identifier.Type == FCB) &&
			(Fcb->Identifier.Size == sizeof(FFS_FCB)));
		
		if (!ExAcquireResourceSharedLite(
			&Fcb->MainResource,
			IrpContext->IsSynchronous))
		{
			Status = STATUS_PENDING;
			__leave;
		}
		
		FcbResourceAcquired = TRUE;
		
		/*
		 * This request is only allowed on directories
		 */
		if (!(Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY))
		{
			Status = STATUS_INVALID_PARAMETER;
			__leave;
		}
		
		Ccb = (PFFS_CCB) FileObject->FsContext2;
		
		ASSERT(Ccb);
		
		ASSERT((Ccb->Identifier.Type == CCB) &&
			(Ccb->Identifier.Size == sizeof(FFS_CCB)));
		
		Irp = IrpContext->Irp;
		
		IoStackLocation = IoGetCurrentIrpStackLocation(Irp);
		
#ifndef _GNU_NTIFS_
		
		FileInformationClass =
			IoStackLocation->Parameters.QueryDirectory.FileInformationClass;
		
		Length = IoStackLocation->Parameters.QueryDirectory.Length;
		
		FileName = IoStackLocation->Parameters.QueryDirectory.FileName;
		
		FileIndex = IoStackLocation->Parameters.QueryDirectory.FileIndex;
		
#else // _GNU_NTIFS_
		
		FileInformationClass = ((PEXTENDED_IO_STACK_LOCATION)
			IoStackLocation)->Parameters.QueryDirectory.FileInformationClass;
		
		Length = ((PEXTENDED_IO_STACK_LOCATION)
			IoStackLocation)->Parameters.QueryDirectory.Length;
		
		FileName = ((PEXTENDED_IO_STACK_LOCATION)
			IoStackLocation)->Parameters.QueryDirectory.FileName;
		
		FileIndex = ((PEXTENDED_IO_STACK_LOCATION)
			IoStackLocation)->Parameters.QueryDirectory.FileIndex;
		
#endif // _GNU_NTIFS_
		
		RestartScan = IoStackLocation->Flags & SL_RESTART_SCAN;
		ReturnSingleEntry = IoStackLocation->Flags & SL_RETURN_SINGLE_ENTRY;
		IndexSpecified = IoStackLocation->Flags & SL_INDEX_SPECIFIED;
		
		if (!IrpContext->IsSynchronous)
		{
			Status = STATUS_PENDING;
			__leave;
		}
		
		Buffer = FFSGetUserBuffer(Irp);
		
		if (Buffer == NULL)
		{
			Status = STATUS_INVALID_USER_BUFFER;
			__leave;
		}
		
		if (FileName != NULL)
		{
	
			if (Ccb->DirectorySearchPattern.Buffer != NULL)
			{
				FirstQuery = FALSE;
			}
			else
			{
				FirstQuery = TRUE;
				
				Ccb->DirectorySearchPattern.Length =
					Ccb->DirectorySearchPattern.MaximumLength =
					FileName->Length;
				
				Ccb->DirectorySearchPattern.Buffer =
					ExAllocatePool(NonPagedPool, FileName->Length);
				
				if (Ccb->DirectorySearchPattern.Buffer == NULL)
				{
					Status = STATUS_INSUFFICIENT_RESOURCES;
					__leave;
				}
				
				RtlCopyMemory(
					Ccb->DirectorySearchPattern.Buffer,
					FileName->Buffer,
					FileName->Length);
			}
		}
		else if (Ccb->DirectorySearchPattern.Buffer != NULL)
		{
			FirstQuery = FALSE;
			FileName = &Ccb->DirectorySearchPattern;
		}
		else
		{
			FirstQuery = TRUE;
			
			Ccb->DirectorySearchPattern.Length =
				Ccb->DirectorySearchPattern.MaximumLength =
				sizeof(L"*");
			
			Ccb->DirectorySearchPattern.Buffer =
				ExAllocatePool(NonPagedPool, sizeof(L"*"));
			
			if (Ccb->DirectorySearchPattern.Buffer == NULL)
			{
				Status = STATUS_INSUFFICIENT_RESOURCES;
				__leave;
			}
			
			RtlCopyMemory(
				Ccb->DirectorySearchPattern.Buffer,
				L"*", sizeof(L"*"));
		}
		
		if (!IndexSpecified)
		{
			if (RestartScan || FirstQuery)
			{
				FileIndex = (ULONG)Fcb->IndexNumber.QuadPart;
			}
			else
			{
				FileIndex = Ccb->CurrentByteOffset;
			}
		}
		
		Inode = (struct ufs1_dinode*) ExAllocatePool(
			NonPagedPool,
			sizeof(struct ufs1_dinode));
		
		if (Inode == NULL)
		{
			Status = STATUS_INSUFFICIENT_RESOURCES;
			__leave;
		}
		
		RtlZeroMemory(Buffer, Length);

		dwSize = (ULONG)Fcb->dinode->di_size - FileIndex;
		if (dwSize <= 0)
		{
			Status = STATUS_NO_MORE_FILES;
			__leave;
		}
		
		DirectoryContent = ExAllocatePool(NonPagedPool,
			dwSize * 2); // XXX üũ
		if (!DirectoryContent)
		{
			Status = STATUS_INSUFFICIENT_RESOURCES;
			__leave;
		}
		
		dwBytes = 0;
		RtlZeroMemory(DirectoryContent, dwSize * 2); // XXX üũ
		FFSReadInode(Vcb, Fcb->dinode, FileIndex, DirectoryContent, dwSize, FALSE);
		dwTemp = 0;
		
		pDir = (struct direct *) ((PUCHAR)DirectoryContent + dwBytes);

		while (bRun && UsedLength < Length  && dwBytes < dwSize && pDir->d_ino)
		{

			InodeFileNameLength = pDir->d_namlen & 0xff;
			
			InodeFileName.Length = InodeFileName.MaximumLength =
				InodeFileNameLength * 2;
			
			if (InodeFileName.Length <=0 )
				break;

			InodeFileName.Buffer = ExAllocatePool(
				NonPagedPool,
				InodeFileNameLength * 2 + 2);

			RtlZeroMemory(InodeFileName.Buffer, InodeFileNameLength * 2 + 2);
			
			FFSCharToWchar(
				InodeFileName.Buffer,
				pDir->d_name,
				InodeFileNameLength);

//			KdPrint(("FFSQueryDirectory: Found %S to comp.\n", InodeFileName.Buffer));
/*			
			if (FsRtlIsNameInExpression(
				FileName,
				&InodeFileName,
				FALSE,
				NULL ))
*/
			if (FsRtlDoesNameContainWildCards(FileName) ?
				FsRtlIsNameInExpression(
					FileName,
					&InodeFileName,
					TRUE,
					NULL) :
				!RtlCompareUnicodeString(
					FileName,
					&InodeFileName,
					TRUE))
			{
				dwReturn = FFSProcessDirEntry(
					Vcb, FileInformationClass,
					pDir->d_ino, Buffer, UsedLength, Length - UsedLength,
					dwBytes, &InodeFileName,
					ReturnSingleEntry);
				if (dwReturn <= 0)
				{
					bRun = FALSE;
				}
				else
				{
					dwTemp = UsedLength;
					UsedLength += dwReturn;
				}
			}
			
			if (InodeFileName.Buffer != NULL)
			{
				ExFreePool(InodeFileName.Buffer);
				InodeFileName.Buffer = NULL;
			}
			
			if (bRun)
			{
				dwBytes +=pDir->d_reclen;
				Ccb->CurrentByteOffset = FileIndex + dwBytes;
				pDir = (struct direct *) ((PUCHAR)DirectoryContent + dwBytes);
			}

			if (UsedLength && ReturnSingleEntry)
			{
				Status = STATUS_SUCCESS;
				__leave;
			}
		}

		FileIndex += dwBytes;

		((PULONG)((PUCHAR)Buffer + dwTemp)) [0] = 0;

		if (!UsedLength)
		{
			if (FirstQuery)
			{
				Status = STATUS_NO_SUCH_FILE;
			}
			else
			{
				Status = STATUS_NO_MORE_FILES;
			}
		}
		else
		{
			Status = STATUS_SUCCESS;
		}
	}

	__finally
	{
	
		if (FcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Fcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (Inode != NULL)
		{
			ExFreePool(Inode);
		}
		
		if (DirectoryContent != NULL)
		{
			ExFreePool(DirectoryContent);
			DirectoryContent = NULL;
		}
		
		if (InodeFileName.Buffer != NULL)
		{
			ExFreePool(InodeFileName.Buffer);
		}
		
		if (!IrpContext->ExceptionInProgress)
		{
			if (Status == STATUS_PENDING)
			{
				Status = FFSLockUserBuffer(
					IrpContext->Irp,
					Length,
					IoWriteAccess);
				
				if (NT_SUCCESS(Status))
				{
					Status = FFSQueueRequest(IrpContext);
				}
				else
				{
					IrpContext->Irp->IoStatus.Status = Status;
					FFSCompleteRequest(IrpContext->Irp, IO_NO_INCREMENT);
					FFSFreeIrpContext(IrpContext);
				}
			}
			else
			{
				IrpContext->Irp->IoStatus.Information = UsedLength;
				IrpContext->Irp->IoStatus.Status = Status;
				
				FFSCompleteRequest(
					IrpContext->Irp,
					(CCHAR)
					(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
				
				FFSFreeIrpContext(IrpContext);
			}
		}
	}
	
	return Status;
}


NTSTATUS
FFSDirectoryControl(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	NTSTATUS Status;
	
	ASSERT(IrpContext);
	
	ASSERT((IrpContext->Identifier.Type == ICX) &&
		(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
	
	switch (IrpContext->MinorFunction)
	{
	case IRP_MN_QUERY_DIRECTORY:
		Status = FFSQueryDirectory(IrpContext);
		break;
		
	default:
		Status = STATUS_INVALID_DEVICE_REQUEST;
		IrpContext->Irp->IoStatus.Status = Status;
		FFSCompleteRequest(IrpContext->Irp, IO_NO_INCREMENT);
		FFSFreeIrpContext(IrpContext);
	}
	
	return Status;
}


NTSTATUS
FFSQueueRequest(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	ASSERT(IrpContext);
	
	ASSERT((IrpContext->Identifier.Type == ICX) &&
		(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
	
	/* IsSynchronous means we can block (so we don't requeue it) */
	IrpContext->IsSynchronous = TRUE;
	
	IoMarkIrpPending(IrpContext->Irp);
	
	ExInitializeWorkItem(
		&IrpContext->WorkQueueItem,
		FFSDeQueueRequest,
		IrpContext);
	
	ExQueueWorkItem(&IrpContext->WorkQueueItem, CriticalWorkQueue);
	
	return STATUS_PENDING;
}


VOID
FFSDeQueueRequest(
	IN PVOID Context)
{
    PFFS_IRP_CONTEXT IrpContext;

    IrpContext = (PFFS_IRP_CONTEXT) Context;

    ASSERT(IrpContext);

    ASSERT((IrpContext->Identifier.Type == ICX) &&
           (IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));

    __try
    {
        __try
        {
            FsRtlEnterFileSystem();

            if (!IrpContext->IsTopLevel)
            {
                IoSetTopLevelIrp((PIRP) FSRTL_FSP_TOP_LEVEL_IRP);
            }

            FFSDispatchRequest(IrpContext);
        }
        __except (FFSExceptionFilter(IrpContext, GetExceptionCode()))
        {
            FFSExceptionHandler(IrpContext);
        }
    }
    __finally
    {
        IoSetTopLevelIrp(NULL);

        FsRtlExitFileSystem();
    }
}


NTSTATUS
FFSCleanup(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	PDEVICE_OBJECT  DeviceObject;
	NTSTATUS        Status = STATUS_SUCCESS;
	PFFS_VCB        Vcb;
	BOOLEAN         VcbResourceAcquired = FALSE;
	PFILE_OBJECT    FileObject;
	PFFS_FCB        Fcb;
	BOOLEAN         FcbResourceAcquired = FALSE;
	PFFS_CCB        Ccb;
	PIRP            Irp;

	__try
	{
		ASSERT(IrpContext != NULL);
		
		ASSERT((IrpContext->Identifier.Type == ICX) &&
			(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
		
		DeviceObject = IrpContext->DeviceObject;
		
		if (DeviceObject == FFSGlobal->DeviceObject)
		{
			Status = STATUS_SUCCESS;
			__leave;
		}
		
		Vcb = (PFFS_VCB) DeviceObject->DeviceExtension;
		
		ASSERT(Vcb != NULL);
		
		ASSERT((Vcb->Identifier.Type == VCB) &&
			(Vcb->Identifier.Size == sizeof(FFS_VCB)));
		
		FileObject = IrpContext->FileObject;
		
		Fcb = (PFFS_FCB) FileObject->FsContext;
		
		ASSERT(Fcb != NULL);
		
		if (Fcb->Identifier.Type == VCB)
		{
			Status = STATUS_SUCCESS;
			__leave;
		}
		
		ASSERT((Fcb->Identifier.Type == FCB) &&
			(Fcb->Identifier.Size == sizeof(FFS_FCB)));
		
		Ccb = (PFFS_CCB) FileObject->FsContext2;
		
		ASSERT(Ccb != NULL);
		
		ASSERT((Ccb->Identifier.Type == CCB) &&
			(Ccb->Identifier.Size == sizeof(FFS_CCB)));
		
		Irp = IrpContext->Irp;

		if (!ExAcquireResourceExclusiveLite(
			&Vcb->MainResource,
			IrpContext->IsSynchronous))
		{
			Status = STATUS_PENDING;
			__leave;
		}
		
		VcbResourceAcquired = TRUE;
		
#ifndef FFS_RO
		if (!Fcb->IsPageFile)
#endif
		{
			if (!ExAcquireResourceExclusiveLite(
				&Fcb->MainResource,
				IrpContext->IsSynchronous))
			{
				Status = STATUS_PENDING;
				__leave;
			}
			
			FcbResourceAcquired = TRUE;
		}
		
		Fcb->OpenHandleCount--;
		
		Vcb->OpenFileHandleCount--;
		
		CcUninitializeCacheMap(FileObject, NULL, NULL);
		
		if (!(Fcb->FileAttributes & FILE_ATTRIBUTE_DIRECTORY))
		{
			/*
			 * Drop any byte range locks this process may have on the file.
			 */
			FsRtlFastUnlockAll(
				&Fcb->FileLockAnchor,
				FileObject,
				IoGetRequestorProcess(Irp),
				NULL);
			
			/*
			 * If there are no byte range locks owned by other processes on the
			 * file the fast I/O read/write functions doesn't have to check for
			 * locks so we set IsFastIoPossible to FastIoIsPossible again.
			 */
			if (!FsRtlGetNextFileLock(&Fcb->FileLockAnchor, TRUE))
			{
				if (Fcb->CommonFCBHeader.IsFastIoPossible != FastIoIsPossible)
				{
#if DBG
					KdPrint(("%-16.16s %-31s %s\n",
						PsGetCurrentProcess()->ImageFileName,
						"FastIoIsPossible",
						Fcb->DbgFileName));
#endif					
					Fcb->CommonFCBHeader.IsFastIoPossible = FastIoIsPossible;
				}
			}
		}
		
#ifndef FFS_RO
		
		IoRemoveShareAccess(FileObject, &Fcb->ShareAccess);
		
#endif
#if DBG		
		KdPrint(("FFSCleanup: OpenCount: %u ReferCount: %-7u %s\n",
			Fcb->OpenHandleCount,
			Fcb->ReferenceCount,
			Fcb->DbgFileName));
#endif
		Status = STATUS_SUCCESS;
	}
	__finally
	{
		if (IrpContext->FileObject)
		{
			IrpContext->FileObject->Flags |= FO_CLEANUP_COMPLETE;
		}
		
		if (FcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Fcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (VcbResourceAcquired)
		{
			ExReleaseResourceForThreadLite(
				&Vcb->MainResource,
				ExGetCurrentResourceThread());
		}
		
		if (!IrpContext->ExceptionInProgress)
		{
			if (Status == STATUS_PENDING)
			{
				FFSQueueRequest(IrpContext);
			}
			else
			{
				IrpContext->Irp->IoStatus.Status = Status;
				
				FFSCompleteRequest(
					IrpContext->Irp,
					(CCHAR)
					(NT_SUCCESS(Status) ? IO_DISK_INCREMENT : IO_NO_INCREMENT));
				
				FFSFreeIrpContext(IrpContext);
			}
		}
	}
	
	return Status;
}


NTSTATUS
FFSDispatchRequest(
	IN PFFS_IRP_CONTEXT IrpContext)
{
#if 0
	/* KdPrint(("FFSDispatchRequest() start\n")); */
	KdPrint(("FFSDipatchRequest() : %2x\n", IrpContext->MajorFunction));
#endif

	ASSERT(IrpContext);
	
	ASSERT((IrpContext->Identifier.Type == ICX) &&
		(IrpContext->Identifier.Size == sizeof(FFS_IRP_CONTEXT)));
	
	switch (IrpContext->MajorFunction)
	{
	case IRP_MJ_CREATE:
		return FFSCreate(IrpContext);

	case IRP_MJ_CLOSE:
		return FFSClose(IrpContext);

	case IRP_MJ_READ:
		return FFSRead(IrpContext);

	case IRP_MJ_QUERY_INFORMATION:
		return FFSQueryInformation(IrpContext);

	case IRP_MJ_SET_INFORMATION:
		return FFSSetInformation(IrpContext);

	case IRP_MJ_QUERY_VOLUME_INFORMATION:
		return FFSQueryVolumeInformation(IrpContext);

	case IRP_MJ_DIRECTORY_CONTROL:
		return FFSDirectoryControl(IrpContext);

	case IRP_MJ_FILE_SYSTEM_CONTROL:
		return FFSFileSystemControl(IrpContext);
#if 0	
	case IRP_MJ_DEVICE_CONTROL:
		return FFSDeviceControl(IrpContext);

	case IRP_MJ_LOCK_CONTROL:
		return FFSLockControl(IrpContext);
#endif		
	case IRP_MJ_CLEANUP:
		return FFSCleanup(IrpContext);

	default:
		KdPrint(("FFSDispatchRequest: Unexpected major function: %#x\n",
			IrpContext->MajorFunction));
		IrpContext->Irp->IoStatus.Status = STATUS_DRIVER_INTERNAL_ERROR;
		FFSCompleteRequest(IrpContext->Irp, IO_NO_INCREMENT);
		FFSFreeIrpContext(IrpContext);
		return STATUS_DRIVER_INTERNAL_ERROR;
	}

#if 0
	KdPrint(("FFSDispatchRequest() end\n"));
#endif
}


NTSTATUS
FFSExceptionFilter(
	IN PFFS_IRP_CONTEXT IrpContext,
	IN NTSTATUS ExceptionCode)
{
	return STATUS_SUCCESS;
}


NTSTATUS
FFSExceptionHandler(
	IN PFFS_IRP_CONTEXT IrpContext)
{
	return STATUS_SUCCESS;
}


NTSTATUS
FFSBuildRequest(
	IN PDEVICE_OBJECT DeviceObject,
	IN PIRP Irp)
{
	BOOLEAN             AtIrqlPassiveLevel = FALSE;
	BOOLEAN             IsTopLevelIrp = FALSE;
	PFFS_IRP_CONTEXT    IrpContext = NULL;
	NTSTATUS            Status = STATUS_UNSUCCESSFUL;

#if 0
	KdPrint(("FFSBuildRequest() start\n"));
#endif

	__try
	{
		__try
		{
			/* FFSKdPrintCall(DeviceObject, Irp); */
			
			AtIrqlPassiveLevel = (KeGetCurrentIrql() == PASSIVE_LEVEL);
			
			if (AtIrqlPassiveLevel)
			{
				FsRtlEnterFileSystem();
			}
			
			if (!IoGetTopLevelIrp())
			{
				IsTopLevelIrp = TRUE;
				IoSetTopLevelIrp(Irp);
			}
			
			IrpContext = FFSAllocateIrpContext(DeviceObject, Irp);
			
			if (!IrpContext)
			{
				Status = STATUS_INSUFFICIENT_RESOURCES;
				Irp->IoStatus.Status = Status;
				FFSCompleteRequest(Irp, IO_NO_INCREMENT);
			}
			else
			{
				Status = FFSDispatchRequest(IrpContext);
			}
		}
		__except (FFSExceptionFilter(IrpContext, GetExceptionCode()))
		{
			Status = FFSExceptionHandler(IrpContext);
		}
	}
	__finally
	{
		if (IsTopLevelIrp)
		{
			IoSetTopLevelIrp(NULL);
		}
		
		if (AtIrqlPassiveLevel)
		{
			FsRtlExitFileSystem();
		}		
	}

#if 0
	KdPrint(("FFSBuildRequest() end\n"));
#endif

	return Status;
}


NTSTATUS
DriverEntry(
	IN PDRIVER_OBJECT DriverObject,
	IN PUNICODE_STRING RegistryPath)
{
	PDEVICE_OBJECT              DeviceObject;
	PFAST_IO_DISPATCH           FastIoDispatch;
	PCACHE_MANAGER_CALLBACKS    CacheManagerCallbacks;
	PFFS_EXT	                DeviceExt;

	UNICODE_STRING              DeviceName;
#if DBG
	UNICODE_STRING              DosDeviceName;
#endif
	NTSTATUS Status;

#if DBG
	KdPrint(("DriverEntry() start\n"));
	KdPrint(("FFS : Entered the FFS driver!\n"));
#endif

	RtlInitUnicodeString(&DeviceName, DEVICE_NAME);
	
	Status = IoCreateDevice(
			DriverObject,
			sizeof(FFS_EXT),
			&DeviceName,
			FILE_DEVICE_DISK_FILE_SYSTEM,
			0,
			FALSE,
			&DeviceObject);

	if (!NT_SUCCESS(Status))
	{
		KdPrint(("FFS IoCreateDevice error.\n"));
		return Status;
	}
	
	DeviceExt = (PFFS_EXT)DeviceObject->DeviceExtension;
	RtlZeroMemory(DeviceExt, sizeof(FFS_EXT));

	FFSGlobal = &(DeviceExt->FFSGlobal);
#if 0 
	FFSGlobal->TimeZone.QuadPart = LocalTime.QuadPart; 
	KdPrint(("TimeZone: %I64xh\n", FFSGlobal->TimeZone.QuadPart)); 
#endif 
	FFSGlobal->Identifier.Type = FGD; 
	FFSGlobal->Identifier.Size = sizeof(FFS_GLOBAL); 
	FFSGlobal->DeviceObject = DeviceObject; 
	FFSGlobal->DriverObject = DriverObject; 

	DriverObject->MajorFunction[IRP_MJ_CREATE] = FFSBuildRequest;
	DriverObject->MajorFunction[IRP_MJ_CLOSE] = FFSBuildRequest;
	DriverObject->MajorFunction[IRP_MJ_READ] = FFSBuildRequest;
	DriverObject->MajorFunction[IRP_MJ_QUERY_INFORMATION] = FFSBuildRequest;
	DriverObject->MajorFunction[IRP_MJ_SET_INFORMATION] = FFSBuildRequest;
	DriverObject->MajorFunction[IRP_MJ_QUERY_VOLUME_INFORMATION] = FFSBuildRequest;
	DriverObject->MajorFunction[IRP_MJ_DIRECTORY_CONTROL] = FFSBuildRequest;
	DriverObject->MajorFunction[IRP_MJ_FILE_SYSTEM_CONTROL] = FFSBuildRequest;
	DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = FFSBuildRequest;
	DriverObject->MajorFunction[IRP_MJ_LOCK_CONTROL] = FFSBuildRequest;
	DriverObject->MajorFunction[IRP_MJ_CLEANUP] = FFSBuildRequest;
	DriverObject->DriverUnload = NULL;


	/*
	 * Initialize the fast I/O entry points
	 */
	
	FastIoDispatch = &(FFSGlobal->FastIoDispatch);
	
	FastIoDispatch->SizeOfFastIoDispatch = sizeof(FAST_IO_DISPATCH);
	FastIoDispatch->FastIoCheckIfPossible = FFSFastIoCheckIfPossible;
#if DBG
	FastIoDispatch->FastIoRead = FFSFastIoRead;
#else
	FastIoDispatch->FastIoRead = FsRtlCopyRead;
#endif
	FastIoDispatch->FastIoQueryBasicInfo = FFSFastIoQueryBasicInfo;
	FastIoDispatch->FastIoQueryStandardInfo = FFSFastIoQueryStandardInfo;
	FastIoDispatch->FastIoLock = FFSFastIoLock;
	FastIoDispatch->FastIoUnlockSingle = FFSFastIoUnlockSingle;
	FastIoDispatch->FastIoUnlockAll = FFSFastIoUnlockAll;
	FastIoDispatch->FastIoUnlockAllByKey = FFSFastIoUnlockAllByKey;
	FastIoDispatch->FastIoQueryNetworkOpenInfo = FFSFastIoQueryNetworkOpenInfo;

	DriverObject->FastIoDispatch = FastIoDispatch;
	
	/*
	 * Initialize the Cache Manager callbacks
	 */
	
	CacheManagerCallbacks = &(FFSGlobal->CacheManagerCallbacks);
	CacheManagerCallbacks->AcquireForLazyWrite = FFSAcquireForLazyWrite;
	CacheManagerCallbacks->ReleaseFromLazyWrite = FFSReleaseFromLazyWrite;
	CacheManagerCallbacks->AcquireForReadAhead = FFSAcquireForReadAhead;
	CacheManagerCallbacks->ReleaseFromReadAhead = FFSReleaseFromReadAhead;

	/*
	 * Initialize the global data
	 */

	InitializeListHead(&(FFSGlobal->VcbList));
	ExInitializeResourceLite(&(FFSGlobal->Resource));


#if DBG
	RtlInitUnicodeString(&DosDeviceName, DOS_DEVICE_NAME);
	IoCreateSymbolicLink(&DosDeviceName, &DeviceName);
#endif
	IoRegisterFileSystem(DeviceObject);

#if DBG
	KdPrint(("DriverEntry() end\n"));
#endif

	return Status;
}


VOID
FFSUnload(
	IN PDRIVER_OBJECT DriverObject)
{
	UNICODE_STRING DosDeviceName;
	
	KdPrint(("FFS: Unloading!!\n"));

	RtlInitUnicodeString(&DosDeviceName, DOS_DEVICE_NAME);
	
	IoDeleteSymbolicLink(&DosDeviceName);
	
	IoDeleteDevice(DriverObject->DeviceObject);
}
