#ifndef CFRUNLOOP_H
#define CFRUNLOOP_H

#ifndef CFBASE_H
#include <CoreFoundation/CFBase.h>
#endif

#ifndef CFARRAY_H
#include <CoreFoundation/CFArray.h>
#endif

#ifndef CFDATE_H
#include <CoreFoundation/CFDate.h>
#endif

#ifndef CFSTRING_H
#include <CoreFoundation/CFString.h>
#endif


#include <mach/port.h>


#if defined(__cplusplus)
extern "C" {
#endif


/*Default Run Loop Mode

Default run loop mode.

const CFStringRef kCFRunLoopDefaultMode;
Constants
kCFRunLoopDefaultMode

Run loop mode that should be used when a thread is in its default, or idle, state, waiting for an event. This mode is used when the run loop is started with CFRunLoopRun.

Available in Mac OS X v10.0 and later.*/

const CFStringRef kCFRunLoopDefaultMode;

/*Common Mode Flag

A run loop pseudo-mode that manages objects monitored in the “common” modes.

const CFStringRef kCFRunLoopCommonModes;
Constants
kCFRunLoopCommonModes

Objects added to a run loop using this value as the mode are monitored by all run loop modes that have been declared as a member of the set of “common” modes with CFRunLoopAddCommonMode.

Available in Mac OS X v10.0 and later.
Discussion

Run loops never run in this mode. This pseudo-mode is used only as a special set of sources, timers, and observers that is shared by other modes. See “Managing Objects in a Run Loop” for more details.*/
const CFStringRef kCFRunLoopCommonModes;

/*CFRunLoopRef

A reference to a run loop object.

typedef struct __CFRunLoop *CFRunLoopRef;
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct __CFRunLoop *CFRunLoopRef;


/*CFRunLoopSourceRef

A reference to a run loop source object.

typedef struct __CFRunLoopSource *CFRunLoopSourceRef;
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct __CFRunLoopSource *CFRunLoopSourceRef;

/*CFRunLoopEqualCallBack

Callback invoked to test two CFRunLoopSource objects for equality.

typedef Boolean (*CFRunLoopEqualCallBack) (
   const void *info1,
   const void *info2
);

If you name your function MyCallBack, you would declare it like this:

Boolean MyCallBack (
   const void *info1,
   const void *info2
);
Parameters
info1

The info member of the CFRunLoopSourceContext or CFRunLoopSourceContext1 structure that was used when creating the first run loop source to test.
info2

The info member of the CFRunLoopSourceContext or CFRunLoopSourceContext1 structure that was used when creating the second run loop source to test.
Return Value

true if info1 and info2 should be considered equal; otherwise false. 
Discussion

You specify this callback in the CFRunLoopSourceContext or CFRunLoopSourceContext1 structure when creating the run loop source.*/
typedef Boolean (*CFRunLoopEqualCallBack) (
   const void *info1,
   const void *info2
);

/*CFRunLoopHashCallBack

Callback invoked to compute a hash code for the info pointer of a CFRunLoopSource object.

typedef CFHashCode (*CFRunLoopHashCallBack) (
   const void *info
);

If you name your function MyCallBack, you would declare it like this:

CFHashCode MyCallBack (
   const void *info
);
Parameters
info

The info member of the CFRunLoopSourceContext or CFRunLoopSourceContext1 structure that was used when creating the run loop source.
Return Value

A hash code value for info. 
Discussion

If a hash callback is not provided for a source, the info pointer is used.

You specify this callback in the CFRunLoopSourceContext or CFRunLoopSourceContext1 structure when creating the run loop source.*/

typedef CFHashCode (*CFRunLoopHashCallBack) (
   const void *info
);

/*CFRunLoopScheduleCallBack

Callback invoked when a version 0 CFRunLoopSource object is added to a run loop mode.

typedef void (*CFRunLoopScheduleCallBack) (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);

If you name your function MyCallBack, you would declare it like this:

void MyCallBack (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);
Parameters
info

The info member of the CFRunLoopSourceContext structure that was used when creating the run loop source.
rl

The run loop in which the source is being scheduled. 
mode

The run loop mode in which the source is being scheduled. 
Discussion

You specify this callback in the CFRunLoopSourceContext structure when creating the run loop source.*/
typedef void (*CFRunLoopScheduleCallBack) (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);

/*CFRunLoopCancelCallBack

Callback invoked when a version 0 CFRunLoopSource object is removed from a run loop mode.

typedef void (*CFRunLoopCancelCallBack) (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);

If you name your function MyCallBack, you would declare it like this:

void MyCallBack (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);
Parameters
info

The info member of the CFRunLoopSourceContext structure that was used when creating the run loop source.
rl

The run loop from which the run loop source is being removed. 
mode

The run loop mode from which the run loop source is being removed. 
Discussion

You specify this callback in the CFRunLoopSourceContext structure when creating the run loop source.*/
typedef void (*CFRunLoopCancelCallBack) (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);

/*CFRunLoopPerformCallBack

Callback invoked when a message is received on a version 0 CFRunLoopSource object.

typedef void (*CFRunLoopPerformCallBack) (
   void *info
);

If you name your function MyCallBack, you would declare it like this:

void MyCallBack (
   void *info
);
Parameters
info

The info member of the CFRunLoopSourceContext structure that was used when creating the run loop source.
Discussion

You only need to provide this callback if you create your own version 0 run loop source. CFSocket run loop sources already implement this callback to forward the received message to the CFSocket’s own callback function, which you do need to implement.

You specify this callback in the CFRunLoopSourceContext structure when creating the run loop source.*/
typedef void (*CFRunLoopPerformCallBack) (
   void *info
);


/*CFRunLoopSourceContext

A structure that contains program-defined data and callbacks with which you can configure a version 0 CFRunLoopSource’s behavior.

struct CFRunLoopSourceContext {
   CFIndex version;
   void *info;
   CFAllocatorRetainCallBack retain;
   CFAllocatorReleaseCallBack release;
   CFAllocatorCopyDescriptionCallBack copyDescription;
   CFRunLoopEqualCallBack equal;
   CFRunLoopHashCallBack hash;
   CFRunLoopScheduleCallBack schedule;
   CFRunLoopCancelCallBack cancel;
   CFRunLoopPerformCallBack perform;
};
typedef struct CFRunLoopSourceContext CFRunLoopSourceContext;
Fields
version

Version number of the structure. Must be 0. 
info

An arbitrary pointer to program-defined data, which can be associated with the CFRunLoopSource at creation time. This pointer is passed to all the callbacks defined in the context.
retain

A retain callback for your program-defined info pointer. Can be NULL.
release

A release callback for your program-defined info pointer. Can be NULL. 
copyDescription

A copy description callback for your program-defined info pointer. Can be NULL. 
equal

An equality test callback for your program-defined info pointer. Can be NULL.
hash

A hash calculation callback for your program-defined info pointer. Can be NULL.
schedule

A scheduling callback for the run loop source. This callback is called when the source is added to a run loop mode. Can be NULL. 
cancel

A cancel callback for the run loop source. This callback is called when the source is removed from a run loop mode. Can be NULL. 
perform

A perform callback for the run loop source. This callback is called when the source has fired.*/
struct CFRunLoopSourceContext {
   CFIndex version;
   void *info;
   CFAllocatorRetainCallBack retain;
   CFAllocatorReleaseCallBack release;
   CFAllocatorCopyDescriptionCallBack copyDescription;
   CFRunLoopEqualCallBack equal;
   CFRunLoopHashCallBack hash;
   CFRunLoopScheduleCallBack schedule;
   CFRunLoopCancelCallBack cancel;
   CFRunLoopPerformCallBack perform;
};
typedef struct CFRunLoopSourceContext CFRunLoopSourceContext;

/*CFRunLoopSourceCreate

Creates a CFRunLoopSource object.

CFRunLoopSourceRef CFRunLoopSourceCreate (
   CFAllocatorRef allocator,
   CFIndex order,
   CFRunLoopSourceContext *context
);
Parameters
allocator

The allocator to use to allocate memory for the new object. Pass NULL or kCFAllocatorDefault to use the current default allocator.
order

A priority index indicating the order in which run loop sources are processed. When multiple run loop sources are firing in a single pass through the run loop, the sources are processed in increasing order of this parameter. If the run loop is set to process only one source per loop, only the highest priority source, the one with the lowest order value, is processed. This value is ignored for version 1 sources. Pass 0 unless there is a reason to do otherwise.
context

A structure holding contextual information for the run loop source. The function copies the information out of the structure, so the memory pointed to by context does not need to persist beyond the function call.
Return Value

The new CFRunLoopSource object. You are responsible for releasing this object.
Discussion

The run loop source is not automatically added to a run loop. Ownership follows the Create Rule.
Availability
Available in Mac OS X v10.0 and later.*/
CFRunLoopSourceRef CFRunLoopSourceCreate (
   CFAllocatorRef allocator,
   CFIndex order,
   CFRunLoopSourceContext *context
);

/*CFRunLoopAddSource

Adds a CFRunLoopSource object to a run loop mode.

void CFRunLoopAddSource (
   CFRunLoopRef rl,
   CFRunLoopSourceRef source,
   CFStringRef mode
);
Parameters
rl

The run loop to modify. 
source

The run loop source to add. 
mode

The run loop mode to which to add source. Use the constant kCFRunLoopCommonModes to add source to the set of objects monitored by all the common modes. 
Discussion

If source is a version 0 source, this function calls the schedule callback function specified in the context structure for source. See CFRunLoopSourceContext for more details.

A run loop source can be registered in multiple run loops and run loop modes at the same time. When the source is signaled, whichever run loop that happens to detect the signal first will fire the source.

If rl already contains source in mode, this function does nothing.
Availability
Available in Mac OS X v10.0 and later.*/
void CFRunLoopAddSource (
   CFRunLoopRef rl,
   CFRunLoopSourceRef source,
   CFStringRef mode
);

/*CFRunLoopGetCurrent

Returns the CFRunLoop object for the current thread.

CFRunLoopRef CFRunLoopGetCurrent ();
Return Value

Current thread’s run loop. Ownership follows the Get Rule.
Discussion

Each thread has exactly one run loop associated with it.
Availability
Available in Mac OS X v10.0 and later.*/
CFRunLoopRef CFRunLoopGetCurrent ();

/*CFRunLoopRun

Runs the current thread’s CFRunLoop object in its default mode indefinitely.

void CFRunLoopRun ();
Discussion

The current thread’s run loop runs in the default mode (see "Default Run Loop Mode") until the run loop is stopped with CFRunLoopStop or all the sources and timers are removed from the default run loop mode.

Run loops can be run recursively. You can call CFRunLoopRun from within any run loop callout and create nested run loop activations on the current thread’s call stack.
Availability
Available in Mac OS X v10.0 and later.*/
void CFRunLoopRun ();


#ifdef __cplusplus
}
#endif

#endif