#ifndef CFNOTIFICATIONCENTER_H
#define CFNOTIFICATIONCENTER_H

#include <CoreFoundation/CFBase.h>
#include <CoreFoundation/CFDictionary.h>

#if defined(__cplusplus)
extern "C" {
#endif

/*CFNotificationCenterRef

The type of a reference to a CFNotificationCenter.

typedef struct *CFNotificationCenterRef;
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct _CFNotificationCenterRef *CFNotificationCenterRef;

/*CFNotificationCallback

Callback function invoked for each observer of a notification when the notification is posted.

typedef void (*CFNotificationCallback) (
   CFNotificationCenterRef center,
   void *observer,
   CFStringRef name,
   const void *object,
   CFDictionaryRef userInfo
);

If you name your function MyCallBack, you would declare it like this:

void MyCallBack (
   CFNotificationCenterRef center,
   void *observer,
   CFStringRef name,
   const void *object,
   CFDictionaryRef userInfo
);
Parameters
center

The notification center handling the notification. 
observer

An arbitrary value, other than NULL, that identifies the observer.
name

The name of the notification being posted. 
object

An arbitrary value that identifies the object posting the notification. For distributed notifications, object is always a CFString object. This value could be NULL.

If the notification center is a Darwin notification center, this value must be ignored.
userInfo

A dictionary containing additional information regarding the notification. This value could be NULL. 

If the notification center is a Darwin notification center, this value must be ignored.*/

typedef void (*CFNotificationCallback) (
   CFNotificationCenterRef center,
   void *observer,
   CFStringRef name,
   const void *object,
   CFDictionaryRef userInfo
);

/*Notification Delivery Suspension Behavior

Suspension flags that indicate how distributed notifications should be handled when the receiving application is in the background.

enum CFNotificationSuspensionBehavior {
   CFNotificationSuspensionBehaviorDrop = 1,
   CFNotificationSuspensionBehaviorCoalesce = 2,
   CFNotificationSuspensionBehaviorHold = 3,
   CFNotificationSuspensionBehaviorDeliverImmediately = 4
};
typedef enum CFNotificationSuspensionBehavior CFNotificationSuspensionBehavior;
Constants
CFNotificationSuspensionBehaviorDrop

The server will not queue any notifications of the specified name and object while the receiving application is in the background.

Available in Mac OS X v10.0 and later.
CFNotificationSuspensionBehaviorCoalesce

The server will only queue the last notification of the specified name and object; earlier notifications are dropped.

Available in Mac OS X v10.0 and later.
CFNotificationSuspensionBehaviorHold

The server will hold all matching notifications until the queue has been filled (queue size determined by the server) at which point the server may flush queued notifications.

Available in Mac OS X v10.0 and later.
CFNotificationSuspensionBehaviorDeliverImmediately

The server will deliver notifications of the specified name and object whether or not the application is in the background. When a notification with this suspension behavior is matched, it has the effect of first flushing any queued notifications.

Available in Mac OS X v10.0 and later.
Discussion

An application selects the suspension behavior for a given notification when it registers an observer for that notification with CFNotificationCenterAddObserver.*/
enum CFNotificationSuspensionBehavior {
   CFNotificationSuspensionBehaviorDrop = 1,
   CFNotificationSuspensionBehaviorCoalesce = 2,
   CFNotificationSuspensionBehaviorHold = 3,
   CFNotificationSuspensionBehaviorDeliverImmediately = 4
};
typedef enum CFNotificationSuspensionBehavior CFNotificationSuspensionBehavior;

/*CFNotificationCenterAddObserver

Registers an observer to receive notifications.

void CFNotificationCenterAddObserver (
   CFNotificationCenterRef center,
   const void *observer,
   CFNotificationCallback callBack,
   CFStringRef name,
   const void *object,
   CFNotificationSuspensionBehavior suspensionBehavior
);
Parameters
center

The notification center to which to add the observer. 
observer

The observer. In Mac OS X v10.3 and later, this parameter may be NULL.

If the notification center is a Darwin notification center, this value must not be NULL.
callBack

The callback function to call when object posts the notification named name. 
name

The name of the notification to observe. If NULL, callback is called for any notification posted by object. 
object

The object to observe. For distributed notifications, object must be a CFString object. If NULL, callback is called when a notification named name is posted by any object.

If the notification center is a Darwin notification center, this value is ignored.
suspensionBehavior

Flag indicating how notifications should be handled when the application is in the background. See “Notification Delivery Suspension Behavior” for the list of available values.

If the notification center is a Darwin notification center, this value is ignored.
Discussion

The first time an observer is registered with a distributed notification center, the notification center creates a connection with the system-wide notification server and places a listening port into the common modes of the current thread’s run loop. When a notification is delivered, it is processed on this initial thread, even if the observer that is receiving the notification registered for the notification on a different thread.

Because loaded frameworks may potentially spawn threads and add their own observers before your code executes, you cannot know for certain which thread will receive distributed notifications. If you need to control which thread processes a notification, your callback function must be able to forward the notification to the proper thread. You can use a CFMessagePort object or a custom CFRunLoopSource object to send notifications to the correct thread’s run loop.
Availability
Available in Mac OS X v10.0 and later.*/

void CFNotificationCenterAddObserver (
   CFNotificationCenterRef center,
   const void *observer,
   CFNotificationCallback callBack,
   CFStringRef name,
   const void *object,
   CFNotificationSuspensionBehavior suspensionBehavior
);

/*CFNotificationCenterGetDistributedCenter

Returns the application’s distributed notification center.

CFNotificationCenterRef CFNotificationCenterGetDistributedCenter();
Return Value

The application’s distributed notification center. An application has only one distributed notification center, so this function returns the same value each time it is called. 
Discussion

A distributed notification center delivers notifications between applications. A notification object used with a distributed notification center must always be a CFString object and the notification dictionary must contain only property list values.
Availability
Available in Mac OS X v10.0 and later.*/
CFNotificationCenterRef CFNotificationCenterGetDistributedCenter();

/*CFNotificationCenterRemoveObserver

Stops an observer from receiving certain notifications.

void CFNotificationCenterRemoveObserver (
   CFNotificationCenterRef center,
   const void *observer,
   CFStringRef name,
   const void *object
);
Parameters
center

The notification center to modify. 
observer

The observer. This value must not be NULL.
name

The name of the notification to stop observing. If NULL, observer stops receiving callbacks for all notifications posted by object. 
object

The object to stop observing. For distributed notifications, object must be a CFString object. If NULL, observer stops receiving callbacks for all objects posting notifications named name.

If the notification center is a Darwin notification center, this value is ignored.
Discussion

If both name and object are NULL, this function unregisters observer from all the notifications for which it had previously registered with center.
Availability
Available in Mac OS X v10.0 and later.*/
void CFNotificationCenterRemoveObserver (
   CFNotificationCenterRef center,
   const void *observer,
   CFStringRef name,
   const void *object
);

#ifdef __cplusplus
}
#endif

#endif