#ifndef DRAG_H
#define DRAG_H

#ifndef APPLICATIONSERVICES_H
#include <ApplicationServices/ApplicationServices.h>
#endif

#ifndef EVENTS_H
#include <HIToolbox/Events.h>
#endif


#include <HYS/HIToolbox/Drag.h>
#include <AvailabilityMacros.h>

#if PRAGMA_ONCE
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*FlavorType

Defines a flavor type.

typedef OSType FlavorType;
Discussion

The flavor type is a four character type that describes the format of drag item flavor data. The flavor type has the same function as a scrap type; it designates the format of the associated data. Any scrap type or resource type may be used. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef OSType FlavorType;


/*DragItemRef

Defines a reference to a drag item.

typedef UInt32 DragItemRef;
Discussion

The drag item reference is a reference number used to refer to a single item in a drag. Drag item reference numbers are created by the sender application when adding drag item flavor information to a drag. Drag item reference numbers are created by and should only be interpreted by the sender application. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef UInt32 DragItemRef;

/*Flavor Flags

Provide additional information about drag item flavors.

typedef UInt32 FlavorFlags;
enum {
   flavorSenderOnly = (1 << 0),
   flavorSenderTranslated = (1 << 1),
   flavorNotSaved = (1 << 2),
   flavorSystemTranslated = (1 << 8),
   flavorDataPromised = (1 << 9) };
Constants
flavorSenderOnly

Set by the sender if the flavor should only be available to the sender of a drag. If this flag is set when adding the flavor to a drag, no Drag Manager clients other than the sender can receive this flavor.

Available in Mac OS X v10.0 and later.
flavorSenderTranslated

Set by the sender if the flavor data is translated by the sender. This flag is useful to a receiver if the receiver needs to determine if the sender is performing its own translation to generate this data type. Typically, receivers that store dragged data without interpreting each data type do not store translated types. Flavor types marked with this flag are not stored by the Finder in clipping files.

Available in Mac OS X v10.0 and later.
flavorNotSaved

Set by the sender if the flavor data should not be stored by the receiver. This flag is useful for marking flavor data that will become stale after the drag has completed. Receivers that store dragged data should not store flavors that are marked with this flag. Flavor types marked with this flag are not stored by the Finder in clipping files.

Available in Mac OS X v10.0 and later.
flavorSystemTranslated

Set if the flavor data is provided by the Translation Manager. If this flavor is requested, the Drag Manager will obtain any required data types from the sender and then it will use the Translation Manager to provide the data that the receiver requested. Typically, receivers that store dragged data without interpreting each data type do not store translated types. Flavor types marked with this flag are not stored by the Finder in clipping files.

Available in Mac OS X v10.0 and later.
flavorDataPromised

Set by the sender if the flavor data is promised at a later time.

Available in Mac OS X v10.1 and later.
Discussion

These constants are used when calling the AddDragItemFlavor function and can be obtained by calling the GetFlavorFlags function.*/

typedef UInt32 FlavorFlags;
enum {
   flavorSenderOnly = (1 << 0),
   flavorSenderTranslated = (1 << 1),
   flavorNotSaved = (1 << 2),
   flavorSystemTranslated = (1 << 8),
   flavorDataPromised = (1 << 9) };


/*Drag Tracking Messages

Define messages that may be sent to your drag tracking handler.

typedef SInt16 DragTrackingMessage;
enum {
   kDragTrackingEnterHandler = 1,
   kDragTrackingEnterWindow = 2,
   kDragTrackingInWindow = 3,
   kDragTrackingLeaveWindow = 4,
   kDragTrackingLeaveHandler = 5
};
Constants
kDragTrackingEnterHandler

Your tracking handler receives this message when the focus of a drag enters a window that is handled by your tracking handler. If the user moves the drag directly to another window that is handled by the same tracking handler, a second kDragTrackingEnterHandler message is not received. Your tracking handler only receives this message when the drag enters the domain of your function after leaving another.

Available in Mac OS X v10.0 and later.
kDragTrackingEnterWindow

Your tracking handler receives this message when a drag enters any window that is handled by your tracking handler. This message is sent to your tracking handler for each window that the drag may enter. Your tracking handler will always receive this message within a pair of kDragTrackingEnterHandler and kDragTrackingLeaveHandler messages.

Available in Mac OS X v10.0 and later.
kDragTrackingInWindow

Your tracking handler receives this message as the user is dragging within a window handled by your tracking handler. You can use this message to track the dragging process through your window. Your tracking handler will always receive this message within a pair of kDragTrackingEnterWindow and kDragTrackingLeaveWindow messages.Your tracking handler would typically draw the majority of your window highlighting and track objects in your window when you receive this message from the Drag Manager.

Available in Mac OS X v10.0 and later.
kDragTrackingLeaveWindow

Your tracking handler receives this message when a drag leaves any window that is handled by your tracking handler. You are guaranteed to receive this message after receiving a corresponding kDragTrackingEnterWindow message. Your tracking handler will always receive this message within a pair of kDragTrackingEnterHandler and kDragTrackingLeaveHandler messages.

Available in Mac OS X v10.0 and later.
kDragTrackingLeaveHandler

Your tracking handler receives this message when the focus of a drag enters a window that is not handled by your tracking handler. Your tracking handler is guaranteed to receive this message after receiving a corresponding kDragTrackingEnterHandler message.

Available in Mac OS X v10.0 and later.
Discussion

See DragTrackingHandlerProcPtr for more information on drag tracking handlers.*/

typedef SInt16 DragTrackingMessage;
enum {
   kDragTrackingEnterHandler = 1,
   kDragTrackingEnterWindow = 2,
   kDragTrackingInWindow = 3,
   kDragTrackingLeaveWindow = 4,
   kDragTrackingLeaveHandler = 5
};


/*DragRef

Defines a reference to a drag object.

typedef struct OpaqueDragRef * DragRef;
Discussion

Before calling any other Drag Manager function, you must first create a new drag reference by calling the NewDrag function. The drag reference that is returned by the NewDrag function is used in all subsequent calls to the Drag Manager. Use the DisposeDrag function to dispose of a drag reference after you are finished using it.

The meaning of the bits in a drag reference is internal to the Drag Manager. You should not attempt to interpret the value of the drag reference. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct OpaqueDragRef * DragRef;

/*DragTrackingHandlerProcPtr

Defines a pointer to a drag tracking handler.

typedef OSErr (*DragTrackingHandlerProcPtr)
(
   DragTrackingMessage message,
   WindowRef theWindow,
   void * handlerRefCon,
   DragRef theDrag);

If you name your function MyDragTrackingHandler, you would declare it like this:

OSErr MyDragTrackingHandler (
   DragTrackingMessage message,
   WindowRef theWindow,
   void * handlerRefCon,
   DragRef theDrag);
Parameters
message

A tracking message from the Drag Manager indicating the action your tracking handler should take. These messages are described further in “Drag Tracking Messages”. 
theWindow

A reference to the window that the mouse is currently over.
handlerRefCon

A pointer to the reference constant that was provided to the InstallTrackingHandler function when this handler was installed. 
theDrag

The drag reference of the drag. 
Return Value

A result code. See “Drag Manager Result Codes”.
Discussion

When the user drags an item or collection of items through a window, the Drag Manager calls any tracking handlers that have been installed on that window. Your tracking handler can determine the contents of the drag by calling the drag item information functions, such as CountDragItems, CountDragItemFlavors, GetFlavorType and GetFlavorFlags, and highlighting or modifying the objects in your window accordingly.

When the Drag Manager calls your tracking handler, the port will always be set to the window that the mouse is over. 
Special Considerations

For Classic applications, the Drag Manager guarantees that your application’s A5 world and low-memory environment is properly set up for your application’s use. Therefore, you can allocate memory, and use your application’s global variables. You can also rely on low-memory globals being valid.

Although it is possible to call WaitNextEvent or other functions that run the event loop from within your drag tracking handler, it is not recommended as it can cause the drag to timeout and may result in a crash or in corrupt data.

Note that the Process Manager's process switching mechanism is disabled during calls to your handler. If your application is not frontmost when calling these functions, your application will not be able to switch forward. This could result in a situation where a modal dialog appears behind the front process but will not be able to come forward in order to interact with the user.*/

typedef OSErr (*DragTrackingHandlerProcPtr)
(
   DragTrackingMessage message,
   WindowRef theWindow,
   void * handlerRefCon,
   DragRef theDrag);


/*DragTrackingHandlerUPP

Defines a universal procedure pointer (UPP) to a drag tracking handler.

typedef DragTrackingHandlerProcPtr DragTrackingHandlerUPP;
Discussion

For more information, see the description of the DragTrackingHandlerProcPtr callback function.*/
typedef DragTrackingHandlerProcPtr DragTrackingHandlerUPP;

/*DragReceiveHandlerProcPtr

Defines a pointer to a drag receive handler.

typedef OSErr (*DragReceiveHandlerProcPtr)
(
   WindowRef theWindow,
   void * handlerRefCon,
   DragRef theDrag);

If you name your function MyDragReceiveHandler, you would declare it like this:

OSErr MyDragReceiveHandler (
   WindowRef theWindow,
   void * handlerRefCon,
   DragRef theDrag);
Parameters
theWindow

A reference to the window that the drop occurred in.
handlerRefCon

A pointer to the reference constant that was provided to the InstallReceiveHandler function when this handler was installed. 
theDrag

The drag reference of the drag. 
Return Value

A result code. See “Drag Manager Result Codes”.
Discussion

When the user releases a drag in a window, the Drag Manager calls any drag receive handler functions that have been installed on that window. You can get the information about the data that is being dragged, to determine if your window will accept the drop, by using the drag information functions provided by the Drag Manager. After your drag receive handler decides that it can accept the drop, use the GetFlavorData function to get the data from the sender of the drag.

When the Drag Manager calls your drag receive handler, the port is set to the window that the drop occurred in. If you want to provide an optional Apple Event descriptor of the drop location for the sender, use the SetDropLocation function to set the drop location descriptor before calling the sender with the GetFlavorData or GetFlavorDataSize functions.

If you return any result code other than noErr from your drag receive handler, the Drag Manager will "zoomback" the drag region to the source location and return the userCanceledErr result code from the TrackDrag function. If the drag is dropped into a location that cannot accept the drag (such as the window title bar or window scroll bars) or no acceptable data types were available, your drag receive handler should return the dragNotAcceptedErr result code, which will cause the Drag Manager to provide the "zoomback" animation described above. 
Special Considerations

For Classic applications, the Drag Manager guarantees that your application’s A5 world and low-memory environment is properly set up for your application’s use. Therefore, you can allocate memory, and use your application’s global variables. You can also rely on low-memory globals being valid.

Although it is possible to call WaitNextEvent or other functions that run the event loop from within your drag receive handler, it is not recommended as it can cause the drag to timeout and may result in a crash or in corrupt data.

Note that the Process Manager's process switching mechanism is disabled during calls to your handler. If your application is not frontmost when calling these functions, your application will not be able to switch forward. This could result in a situation where a modal dialog appears behind the front process but will not be able to come forward in order to interact with the user.*/

typedef OSErr (*DragReceiveHandlerProcPtr)
(
   WindowRef theWindow,
   void * handlerRefCon,
   DragRef theDrag);



/*DragReceiveHandlerUPP

Defines a universal procedure pointer (UPP) to a drag receive handler.

typedef DragReceiveHandlerProcPtr DragReceiveHandlerUPP;
Discussion

For more information, see the description of the DragReceiveHandlerProcPtr callback function.*/
typedef DragReceiveHandlerProcPtr DragReceiveHandlerUPP;



//function

/*GetDragMouse

Gets the current mouse and pinned mouse locations.

OSErr GetDragMouse (
   DragRef theDrag,
   Point * mouse,
   Point * globalPinnedMouse
);
Parameters
theDrag

A drag reference. 
mouse

A pointer to a variable containing, on return, the current mouse location in global screen coordinates. You may pass NULL if you wish to ignore this value. The value will be (0, 0) if the drag is not yet used. After a drag completes, the drop location is returned.
globalPinnedMouse

A pointer to a variable containing, on return, the current pinned mouse location in global screen coordinates. You may pass NULL if you wish to ignore this value. The value will be (0, 0) if the drag is not yet used. After a drag completes, the drop location is returned.The pinned mouse location is the mouse location that is used to draw the drag region on the screen. The pinned mouse location is different from the mouse location when the cursor is being constrained in either dimension by a tracking handler.
Return Value

A result code. See “Drag Manager Result Codes”.
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/
OSErr GetDragMouse (
   DragRef theDrag,
   Point * mouse,
   Point * globalPinnedMouse
);

/*CountDragItems

Gets the number of drag items that are contained in a drag reference.

OSErr CountDragItems (
   DragRef theDrag,
   UInt16 * numItems
);
Parameters
theDrag

The drag reference. 
numItems

On return, a pointer to the number of drag items in the specified drag reference.
Return Value

A result code. See “Drag Manager Result Codes”.
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/

OSErr CountDragItems (
   DragRef theDrag,
   UInt16 * numItems
);

/*InstallReceiveHandler

Installs a receive handler function for one or all of your application’s windows.

OSErr InstallReceiveHandler (
   DragReceiveHandlerUPP receiveHandler,
   WindowRef theWindow,
   void * handlerRefCon
);
Parameters
receiveHandler

A pointer to a receive handler function. Installing a receive handler function allows your application to accept a drag by getting drag item flavor data from the Drag Manager when the user releases the mouse button while dragging over one of your application’s windows. 
theWindow

A reference to the window for which to install the receive handler. When a drop occurs over this window, the Drag Manager calls your receive handler function to allow your application to accept the drag. If you pass NULL, the receive handler function is installed in the default handler space for your application. Receive handler functions installed in this way are called when a drop occurs over any window that belongs to your application. You may install more than one receive handler function on a single window.
handlerRefCon

A pointer to a reference constant that will be forwarded to your receive handler function when it is called by the Drag Manager. Use this constant to pass any data you wish to forward to your drag receive handler. 
Return Value

A result code. See “Drag Manager Result Codes”.
Discussion

The Drag Manager sequentially calls all of the receive handler functions installed on a window when a drop occurs in that window. 
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/

OSErr InstallReceiveHandler (
   DragReceiveHandlerUPP receiveHandler,
   WindowRef theWindow,
   void * handlerRefCon
);

/*InstallTrackingHandler

Installs a tracking handler function for one or all of your application’s windows.

OSErr InstallTrackingHandler (
   DragTrackingHandlerUPP trackingHandler,
   WindowRef theWindow,
   void * handlerRefCon
);
Parameters
trackingHandler

A pointer to a tracking handler function. Installing a tracking handler function allows your application to track the user’s movements through your application’s windows during a drag. 
theWindow

A reference to the window for which to track and handle dragging. When the cursor moves into this window during a drag, the Drag Manager sends tracking messages to the tracking handler function. If you pass NULL, the tracking handler function is installed in the default handler space for your application. Tracking handler functions installed in this way are called when the user moves the mouse over any window that belongs to your application. You may install more than one drag tracking handler on a single window.
handlerRefCon

A pointer to a reference constant that will be forwarded to your tracking handler function when it is called by the Drag Manager. Use this constant to pass any data you wish to forward to your tracking handler function. 
Return Value

A result code. See “Drag Manager Result Codes”.
Discussion

The Drag Manager sequentially calls all of the tracking handler functions installed for a window when the user moves the cursor over that window during a drag. 
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/

OSErr InstallTrackingHandler (
   DragTrackingHandlerUPP trackingHandler,
   WindowRef theWindow,
   void * handlerRefCon
);

/*GetFlavorData

Gets all or part of the data for a specific flavor in a drag item.

OSErr GetFlavorData (
   DragRef theDrag,
   DragItemRef theItemRef,
   FlavorType theType,
   void * dataPtr,
   Size * dataSize,
   UInt32 dataOffset
);
Parameters
theDrag

A drag reference. 
theItemRef

The reference number of the drag item containing the flavor data. 
theType

The flavor type of the flavor to get the data from. 
dataPtr

A pointer to a data buffer. On return, the buffer contains the requested flavor data. Your application is responsible for allocating the memory for the flavor data and for setting the dataSize parameter to the number of bytes that you have allocated for the data.
dataSize

On input, a pointer to the size of the data (in bytes) that you have allocated memory for and wish to receive from the flavor. On return, a pointer to the actual number of bytes copied into the buffer specified by the dataPtr parameter. 

If you specify a dataSize that is smaller than the amount of data in the flavor, the data is copied into your buffer and the dataSize parameter is unchanged. If you specify a dataSize that is larger than the amount of data in the flavor, only the amount of data in the flavor is copied into your buffer and the dataSize parameter contains, on return, the actual number of bytes copied. You have reached the end of the flavor’s data when the dataSize parameter points to a number of bytes lower than you provided. 

If you wish to receive the flavor data in smaller pieces than the entire size of the data, you can set the dataSize parameter to be as large as your buffer and call the GetFlavorData function multiple times while incrementing the dataOffset parameter by the size of your buffer. If the dataOffset parameter is larger than the amount of data contained within the flavor, 0 (zero) will be returned in the number pointed to by the dataSize parameter indicating that no data was copied into your buffer.
dataOffset

A pointer to the offset (in bytes) within the flavor structure at which to begin copying data.
Return Value

A result code. See “Drag Manager Result Codes”.
Discussion

You can first determine the size of a flavor’s data by calling the GetFlavorDataSize function.

Note that calling the GetFlavorData function on a flavor that requires translation will force that translation to occur in order to return the data. 
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/
OSErr GetFlavorData (
   DragRef theDrag,
   DragItemRef theItemRef,
   FlavorType theType,
   void * dataPtr,
   Size * dataSize,
   UInt32 dataOffset
);


/*GetFlavorDataSize

Gets the size of the data for a specific flavor in a drag item.

OSErr GetFlavorDataSize (
   DragRef theDrag,
   DragItemRef theItemRef,
   FlavorType theType,
   Size * dataSize
);
Parameters
theDrag

A drag reference. 
theItemRef

The reference number of the drag item containing the flavor. 
theType

The flavor type for which to get the size of the data. 
dataSize

On return, a pointer to the size of the data for the specified drag item flavor.
Return Value

A result code. See “Drag Manager Result Codes”.
Discussion

Note that calling the GetFlavorDataSize function on a flavor that requires translation will force that translation to be performed in order to determine the data size. Since translation may require a significant amount of time and memory during processing, call the GetFlavorDataSize function only when necessary. 
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/

OSErr GetFlavorDataSize (
   DragRef theDrag,
   DragItemRef theItemRef,
   FlavorType theType,
   Size * dataSize
);

/*GetFlavorType

Gets the type of a specific flavor in a drag item.

OSErr GetFlavorType (
   DragRef theDrag,
   DragItemRef theItemRef,
   UInt16 index,
   FlavorType * theType
);
Parameters
theDrag

A drag reference. 
theItemRef

The reference number of the drag item containing the flavor. 
index

The index of the desired flavor.
theType

On return, a pointer to the type of the specified flavor. If a flavor is marked with the flavorSenderOnly flag, it is not returned to any application other than the sender. 
Return Value

A result code. See “Drag Manager Result Codes”. If index is 0 or larger than the number of flavors in the item, GetFlavorType returns the badDragFlavorErr result code.
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/
OSErr GetFlavorType (
   DragRef theDrag,
   DragItemRef theItemRef,
   UInt16 index,
   FlavorType * theType
);

/*GetDragItemReferenceNumber

Gets the reference number of a specific item in a drag reference.

OSErr GetDragItemReferenceNumber (
   DragRef theDrag,
   UInt16 index,
   DragItemRef * theItemRef
);
Parameters
theDrag

The drag reference. 
index

The index of an item in a drag for which to get the reference. 
theItemRef

On return, a pointer to the reference number of the item with the specified index. 
Return Value

A result code. See “Drag Manager Result Codes”. If index is 0 or larger than the number of items in the drag, GetDragItemReferenceNumber returns the badDragItemErr result code.
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/
OSErr GetDragItemReferenceNumber (
   DragRef theDrag,
   UInt16 index,
   DragItemRef * theItemRef
);

/*HideDragHilite

Removes highlighting created with the ShowDragHilite function.

OSErr HideDragHilite (
   DragRef theDrag
);
Parameters
theDrag

The drag reference that is currently showing a drag highlight. 
Return Value

A result code. See “Drag Manager Result Codes”.
Discussion

The HideDragHilite function assumes that the highlighting should be erased from the current port. Your application should make sure that the correct port is set before calling the HideDragHilite function. Also, highlighting erased by the HideDragHilite function is clipped to the current port. Make sure that the port’s clip region is appropriately sized to remove the highlighting. 
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/
OSErr HideDragHilite (
   DragRef theDrag
);

/*ShowDragHilite

Highlights an area of your window during a drag.

OSErr ShowDragHilite (
   DragRef theDrag,
   RgnHandle hiliteFrame,
   Boolean inside
);
Parameters
theDrag

The drag reference of the drag currently in progress. 
hiliteFrame

A QuickDraw region of the frame of the window, pane, or shape you wish to highlight, in the window’s local coordinate system. 
inside

Pass true to draw the highlighting inside the frame shape. Otherwise it will be drawn outside the frame shape. Note that in either case, the highlight will not include the boundary edge of the frame. This allows you to highlight inside a window frame or a pane, or to highlight outside of a container or object in your window.
Return Value

A result code. See “Drag Manager Result Codes”.
Discussion

The ShowDragHilite function creates a standard drag and drop highlight in your window. Your tracking handler function should call this if a drop is allowed at the current mouse position.

You can only have one highlight showing at a time, and if you call this function when a highlight is currently visible, the first one is removed before the newly requested highlight is shown.

The ShowDragHilite function uses a two pixel thick line when drawing the highlight.

The ShowDragHilite function assumes that the highlighting should be drawn in the current port. Your application should make sure that the correct port is set before calling the ShowDragHilite function. Also, highlighting drawn by the ShowDragHilite function is clipped to the current port. Make sure that the port’s clip region is appropriately sized to draw the highlighting.

The Drag Manager maintains the currently highlighted portion of your window if you use the HideDragHilite and UpdateDragHilite functions. If you intend to scroll the window that contains the highlighting, you can use the DragPreScroll and DragPostScroll functions to properly update the drag highlighting. 
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/

OSErr ShowDragHilite (
   DragRef theDrag,
   RgnHandle hiliteFrame,
   Boolean inside
);

/*GetFlavorFlags

Gets the flags for a specific flavor in a drag item. 

OSErr GetFlavorFlags (
   DragRef theDrag,
   DragItemRef theItemRef,
   FlavorType theType,
   FlavorFlags * theFlags
);
Parameters
theDrag

A drag reference. 
theItemRef

The reference number of the drag item containing the flavor. 
theType

The flavor type for which to get the attributes. 
theFlags

On return, a pointer to the attributes of the specified flavor. If a flavor is marked with the flavorSenderOnly flag, it is not returned to any application other than the sender. 
Return Value

A result code. See “Drag Manager Result Codes”.
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/

OSErr GetFlavorFlags (
   DragRef theDrag,
   DragItemRef theItemRef,
   FlavorType theType,
   FlavorFlags * theFlags
);

/*NewDragReceiveHandlerUPP

Creates a new universal procedure pointer (UPP) to a drag receive handler.

DragReceiveHandlerUPP NewDragReceiveHandlerUPP (
   DragReceiveHandlerProcPtr userRoutine
);
Parameters
userRoutine

A pointer to your drag receive handler.
Return Value

On return, a UPP to the drag receive handler. See the description of the DragReceiveHandlerUPP data type.
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/
DragReceiveHandlerUPP NewDragReceiveHandlerUPP (
   DragReceiveHandlerProcPtr userRoutine
);

/*NewDragTrackingHandlerUPP

Creates a new universal procedure pointer (UPP) to a drag tracking handler.

DragTrackingHandlerUPP NewDragTrackingHandlerUPP (
   DragTrackingHandlerProcPtr userRoutine
);
Parameters
userRoutine

A pointer to your drag tracking handler.
Return Value

On return, a UPP to the drag tracking handler. See the description of the DragTrackingHandlerUPP data type.
Availability
Available in Mac OS X v10.0 and later.
Declared In
Drag.h*/
DragTrackingHandlerUPP NewDragTrackingHandlerUPP (
   DragTrackingHandlerProcPtr userRoutine
);

#ifdef __cplusplus
}
#endif

#endif