#ifndef MACH_O_DYLD_H
#define MACH_O_DYLD_H

#if __cplusplus
extern "C" {
#endif 

#include <stddef.h>
#include <stdint.h>
#if __cplusplus

#else
 #include <stdbool.h>
#endif
#include <mach-o/loader.h>

#include <HYS/mach-o/dyld.h>
#include <AvailabilityMacros.h>

/*with the options as follows:
#define NSLINKMODULE_OPTION_NONE 0x0
#define NSLINKMODULE_OPTION_BINDNOW 0x1
#define NSLINKMODULE_OPTION_PRIVATE 0x2*/
#define NSLINKMODULE_OPTION_NONE 0x0
#define NSLINKMODULE_OPTION_BINDNOW 0x1
#define NSLINKMODULE_OPTION_PRIVATE 0x2

/*where the options are:
#define NSUNLINKMODULE_OPTION_NONE 0x0
#define NSUNLINKMODULE_OPTION_KEEP_MEMORY_MAPPED 0x1
#define NSUNLINKMODULE_OPTION_RESET_LAZY_REFERENCES 0x2*/
#define NSUNLINKMODULE_OPTION_NONE 0x0
#define NSUNLINKMODULE_OPTION_KEEP_MEMORY_MAPPED 0x1
#define NSUNLINKMODULE_OPTION_RESET_LAZY_REFERENCES 0x2

/*Mach-O Image Creation Return Codes

Potential return values when creating a Mach-O image.

typedef enum {
   NSObjectFileImageFailure,
   NSObjectFileImageSuccess,
   NSObjectFileImageInappropriateFile,
   NSObjectFileImageArch,
   NSObjectFileImageFormat,
   NSObjectFileImageAccess
} NSObjectFileImageReturnCode;
Constants
NSObjectFileImageSuccess

The image creation operation was completed successfully.
NSObjectFileImageFailure

The image creation operation was not successfully completed.

When this value is returned, an error message is printed to the standard error stream.
NSObjectFileImageInappropriateFile

The Mach-O file is not of a type the called function can operate upon.
NSObjectFileImageArch

The specified Mach-O file is for a different CPU architecture.
NSObjectFileImageFormat

The specified file or memory block does not appear to point to a Mach-O file.
NSObjectFileImageAccess

The access permissions for the specified file do not permit the creation of the image.
Discussion

These return values are returned from NSCreateObjectFileImageFromFile and NSCreateObjectFileImageFromMemory.
Declared In
mach-o/dyld.h*/
typedef enum {
   NSObjectFileImageFailure,
   NSObjectFileImageSuccess,
   NSObjectFileImageInappropriateFile,
   NSObjectFileImageArch,
   NSObjectFileImageFormat,
   NSObjectFileImageAccess
} NSObjectFileImageReturnCode;

/*NSCreateObjectFileImageFromMemory

Creates an image reference for a Mach-O file currently in memory.

NSObjectFileImageReturnCode NSCreateObjectFileImageFromMemory(
   const void* address,
   size_t size,
   NSObjectFileImage* objectFileImage);
Parameters
address

A pointer to the memory block containing the Mach-O file contents. 
size

The size of the memory block, in bytes.
objectFileImage

On output, a pointer to an NSObjectFileImage opaque data structure.
Return Value

See Mach-O Image Creation Return Codes.
Discussion

Given a pointer to a Mach-O file in memory, this function creates and returns an NSObjectFileImage reference. The current implementation works only with bundles, so you must build the Mach-O executable file using the -bundle linker option.

The memory block that address points to, must be allocated with vm_allocate (/usr/include/mach/vm_map.h).

See also NSDestroyObjectFileImage.
Availability
Available in Mac OS X v10.3 and later.*/
NSObjectFileImageReturnCode NSCreateObjectFileImageFromMemory(
   const void* address,
   size_t size,
   NSObjectFileImage* objectFileImage);

/*NSCreateObjectFileImageFromFile

Creates an image reference for a given Mach-O file.

NSObjectFileImageReturnCode NSCreateObjectFileImageFromFile(
   const char* pathName,
   NSObjectFileImage* objectFileImage);
Parameters
pathName

A C string. Pass the pathname to a Mach-O executable file. You must have previously built this file with the -bundle linker option; otherwise, this function returns an error.
objectFileImage

On output, a pointer to an NSObjectFileImage opaque data structure.
Return Value

See Mach-O Image Creation Return Codes.
Discussion

Given a pathname to a Mach-O executable, this function creates and returns a NSObjectFileImage reference. The current implementation works only with bundles, so you must build the Mach-O executable file using the -bundle linker option.

For portability and efficiency, consider using dlopen.*/
NSObjectFileImageReturnCode NSCreateObjectFileImageFromFile(
   const char* pathName,
   NSObjectFileImage* objectFileImage);

/*NSLinkModule

Links the given object file image as a module into the current program.

NSModule NSLinkModule(
   NSObjectFileImage objectFileImage,
   const char* moduleName,
   uint32_t options);
Parameters
objectFileImage

An object file image reference. Pass a reference created using the NSCreateObjectFileImageFromFile function.
moduleName

A C string. Pass the absolute path to the object file image. GDB uses this path to retrieve debug symbol information from the library. 
options

An unsigned long value. Pass one or more of the following bit masks or NSLINKMODULE_OPTION_NONE to specify no options:
NSLINKMODULE_OPTION_BINDNOW

The dynamic linker binds all undefined references immediately, rather than waiting until the references are actually used. All dependent libraries are also bound.
NSLINKMODULE_OPTION_PRIVATE

Do not add the global symbols from the module to the global symbol list. Instead, you must use the NSLookupSymbolInModule function to obtain symbols from this module.
NSLINKMODULE_OPTION_RETURN_ON_ERROR

If an error occurs while binding the module, return NULL. You can then use the function NSLinkEditError to retrieve information about the error.

Without this option, this function calls the linkEdit error handler you have installed using the NSInstallLinkEditErrorHandlers function. If you have not installed a link edit error handler, this function prints a message to the standard error stream and causes a breakpoint trap to end the program.
Return Value

A reference to the linked module.
Discussion

When you call this function, all libraries referenced by the given module are added to the library search list. Unless you pass the NSLINKMODULE_OPTION_PRIVATE, NSLinkModule adds all global symbols in the module to the global symbol list.

For portability and efficiency, consider using dlopen.

See “Building Mach-O Files” in “Mach-O Programming Topics” for more information about modules.*/
NSModule NSLinkModule(
   NSObjectFileImage objectFileImage,
   const char* moduleName,
   uint32_t options);

/*NSLookupSymbolInModule

Given a module reference, returns a reference to the symbol with the given name.

NSSymbol NSLookupSymbolInModule(
   NSModule module,
   const char* symbolName);
Parameters
module

A module reference. Pass the module that contains the symbol.
symbolname

A pointer to a C string. Pass the name of the symbol to look up.
Return Value

The symbol reference or NULL if the symbol cannot be found.
Discussion

For portability and efficiency, consider using dlsym.*/

NSSymbol NSLookupSymbolInModule(
   NSModule module,
   const char* symbolName);

/*NSAddressOfSymbol

Returns the address in the program’s address space of the data represented by the given symbol. The data may be a variable, a constant, or the first instruction of a function.

void* NSAddressOfSymbol(
   NSSymbol symbol);
Parameters
symbol

A symbol reference. Pass the symbol whose address you wish to obtain.
Return Value

A pointer to the data represented by the given symbol.
Discussion

For portability and efficiency, consider using dlsym.*/
void* NSAddressOfSymbol(
   NSSymbol symbol);

/*NSUnLinkModule

Unlinks the given module from the current program.

bool NSUnLinkModule(
   NSModule module,
   uint32_t options);
Parameters
module

A module reference. Pass a reference to a module that you have previously linked using the NSLinkModule function.
options

An unsigned long value. You can specify one or more of the following bit masks:
NSUNLINKMODULE_OPTION_NONE

Unlink the module and deallocate the memory it occupies.
NSUNLINKMODULE_OPTION_KEEP_MEMORY_MAPPED

Unlink the module, but do not deallocate the memory it occupies. Addresses that reside within the module remain valid. You cannot unmap this memory later; it’s released when the process exits or is terminated.
NSUNLINKMODULE_OPTION_RESET_LAZY_REFERENCES

Unlink the module and reset lazy references from other modules that are bound to the module. You can then link a new module that implements the same symbols, and the function call references are bound to the new module when accessed.
Discussion

For portability and efficiency, consider using dlopen in conduction with dlclose.

See “Building Mach-O Files” in “Mach-O Programming Topics” for more information about modules.
Special Considerations

In Mac OS X v10.2 and later, NSUNLINKMODULE_OPTION_RESET_LAZY_REFERENCES can be used only with PowerPC CPU executables.*/

bool NSUnLinkModule(
   NSModule module,
   uint32_t options);

/*NSDestroyObjectFileImage

Releases the given object file image.

bool NSDestroyObjectFileImage(
   NSObjectFileImage objectFileImage);
Parameters
objectFileImage

A reference to the object file image to destroy.
Return Value

TRUE if the image was successfully destroyed, FALSE if not.
Discussion

When this function is called, the dynamic loader calls vm_deallocate (/usr/include/mach/vm_map.h) on the memory pointed to by the objectFileImage parameter.

For portability and efficiency, consider using dlopen in conjunction with dlclose.

See also NSCreateObjectFileImageFromMemory.*/
bool NSDestroyObjectFileImage(
   NSObjectFileImage objectFileImage);

/*NSIsSymbolNameDefined

Returns TRUE if the given symbol is defined in the current program.

enum bool NSIsSymbolNameDefined(
   const char* symbolName);
Parameters
symbolName

A C string. Pass the name of the symbol whose definition status you wish to discover.
Return Value

TRUE when the symbol is defined by any image loaded in the current process; FALSE when the symbol cannot be found.
Discussion

Deprecated in Mac OS X v10.4. Use NSLookupSymbolInImage instead.

If you know the name of the library in which the symbol is likely to be located, you can use the NSIsSymbolNameDefinedWithHint function, which may be faster than this function. You should use the NSIsSymbolNameDefinedInImage function to perform a two-level namespace lookup*/
bool NSIsSymbolNameDefined(
   const char* symbolName);

/*NSLookupAndBindSymbol

Given a symbol name, returns the corresponding symbol from the global symbol table.

NSSymbol NSLookupAndBindSymbol(
   const char* symbolName);
Parameters
symbolName

A pointer to a C string. Pass the name of the symbol you wish to find.
Return Value

The symbol reference for the requested symbol.
Discussion

Deprecated in Mac OS X v10.4. Use NSLookupSymbolInImage instead.

On error, if you have installed a link edit error handler, it is called; otherwise, this function writes an error message to file descriptor 2 (usually the standard error stream, stderr) and causes a breakpoint trap to end the program.

If you know the name of the library in which the symbol is likely to be located, you can use the NSLookupAndBindSymbolWithHint function, which may be faster than this function. You should use the NSLookupSymbolInImage function to perform a two-level namespace lookup.*/
NSSymbol NSLookupAndBindSymbol(
   const char* symbolName);

#endif