#ifndef SCRAP_H
#define SCRAP_H

#ifndef CORESERVICES_H
#include <CoreServices/CoreServices.h>
#endif


#include <AvailabilityMacros.h>

#if PRAGMA_ONCE
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*Scrap Flavor Flags

Describe attributes of a scrap flavor.

enum {
   kScrapFlavorMaskNone = 0x00000000,
   kScrapFlavorMaskSenderOnly = 0x00000001,
   kScrapFlavorMaskTranslated = 0x00000002
};
typedef UInt32 ScrapFlavorFlags;
Constants
kScrapFlavorMaskNone

No flavors in the scrap.

Available in Mac OS X v10.0 and later.
kScrapFlavorMaskSenderOnly

The flavor is intended to be visible to the sender only. This is typically used to save a private flavor into the scrap so that other public promised flavors can be derived from it on demand. If another process put a flavor with this flag on the scrap, your process will never see the flavor, so you do not need to test for this flag.

Available in Mac OS X v10.0 and later.
kScrapFlavorMaskTranslated

The flavor has been translated (or will be translated when the promise is kept) from some other flavor in the scrap, either by the sender or by the Translation Manager.

Available in Mac OS X v10.0 and later.
Discussion

To determine the attributes of a scrap flavor, call the GetScrapFlavorFlags function.*/
enum {
   kScrapFlavorMaskNone = 0x00000000,
   kScrapFlavorMaskSenderOnly = 0x00000001,
   kScrapFlavorMaskTranslated = 0x00000002
};
typedef UInt32 ScrapFlavorFlags;

/*ScrapRef

Defines a reference to a scrap.

typedef struct OpaqueScrapRef * ScrapRef;
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct OpaqueScrapRef * ScrapRef;

/*ScrapFlavorType

Defines a scrap flavor type.

typedef FourCharCode ScrapFlavorType;
Discussion

Some values for scrap flavor types are shown in “Scrap Flavor Types”. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef FourCharCode ScrapFlavorType;

//functions

/*ClearCurrentScrap

Clears the current scrap.

OSStatus ClearCurrentScrap ();
Return Value

A result code. See “Scrap Manager Result Codes”.
Discussion

Call ClearCurrentScrap immediately when the user requests a Copy or Cut operation, even if you maintain a private scrap. You should not wait until receiving a suspend event to call ClearCurrentScrap. You don't need to put any data on the scrap immediately, although it's perfectly fine to do so. You do need to call GetCurrentScrap after ClearCurrentScrap so you'll have a valid scrap reference to pass to other functions. 
Availability
Available in CarbonLib 1.0 and later.
Available in Mac OS X 10.0 and later.
Declared In
Scrap.h*/
OSStatus ClearCurrentScrap ();

/*PutScrap

Writes data to the current location of the scrap.

Unsupported

OSStatus PutScrap (
   SInt32 length,
   ResType theType,
   const void *source
);
Parameters
length

The amount (in bytes) of data to write to the scrap.
theType

The scrap format type of the data you wish to write to the scrap. The scrap format type is a four-character sequence that refers to the particular data format, such as 'TEXT', 'PICT', 'styl', 'snd ', or 'movv'.
source

A pointer to the data that the function should write to the scrap.
Return Value

A result code. See “Scrap Manager Result Codes”. Returns noScrapErr if the scrap at source does not already exist (in memory or on the disk).
Discussion

If your application uses a private scrap, it should copy data from its private scrap to the scrap using this function whenever it receives a suspend event. 

Your application can use this function multiple times to write different formats of the same data to the scrap. Whenever your application needs to write data in one or more formats to the scrap, you should call the ZeroScrap function before the first time you call this function. If your application writes multiple formats to the scrap, you should write your application’s preferred scrap format type first. For example, if the SurfWriter application’s preferred scrap format type is a private scrap format type called 'SURF' and SurfWriter also supports the scrap format types 'TEXT' and 'PICT', then SurfWriter should write the data to the scrap using the 'SURF' scrap format type first, and then write any other scrap format types that it supports in subsequent order of preference.

Do not write data to the scrap that has the same scrap format type as any data already in the scrap. If you do so, the new data is appended to the scrap. Note that when you request data from the scrap using the GetScrap function, GetScrap returns the first data that it finds with the requested scrap format type; thus you cannot retrieve any appended data of the same format type using GetScrap.

If your application uses TextEdit to handle text in its documents, use TextEdit functions to implement cut and copy operations and to write the TextEdit scrap to the scrap. If your application uses the Dialog Manager to handle editable text in your application’s dialog boxes and a dialog box is the frontmost window, use the Dialog Manager function DialogCut or DialogCopy to copy the data from the current editable text item to the scrap. 
Special Considerations

This function may move or purge memory blocks in the application heap. Your application should not call this function at interrupt time.
Availability
Not available in CarbonLib. Not available in Mac OS X.
Carbon Porting Notes

Use the function PutScrapFlavor instead.
Declared In
Scrap.h*/

OSStatus PutScrap (
   SInt32 length,
   ResType theType,
   const void *source
);

/*GetCurrentScrap

Obtains a reference to the current scrap.

OSStatus GetCurrentScrap (
   ScrapRef * scrap
);
Parameters
scrap

A pointer to a scrap reference. On return, this scrap reference refers to the current scrap. 
Return Value

A result code. See “Scrap Manager Result Codes”.
Discussion

Your application can determine if the scrap contents have changed by storing the scrap reference returned by GetCurrentScrap and comparing it against the scrap reference returned by GetCurrentScrap at a later time. If the two scrap references are different, the scrap has changed.

Carbon applications should use GetCurrentScrap instead of checking the convertClipboardFlag in the EventRecord.

The ScrapRef obtained via GetCurrentScrap becomes invalid and unusable after the scrap is cleared. That is, the scrap reference is valid until a Carbon client calls ClearCurrentScrap, a Classic client calls ZeroScrap , or a Cocoa client calls declareTypes:owner:. 
Availability
Available in CarbonLib 1.0 and later.
Available in Mac OS X 10.0 and later.
Declared In
Scrap.h*/
OSStatus GetCurrentScrap (
   ScrapRef * scrap
);

/*PutScrapFlavor

Puts data on or promises data to the specified scrap.

OSStatus PutScrapFlavor (
   ScrapRef scrap,
   ScrapFlavorType flavorType,
   ScrapFlavorFlags flavorFlags,
   Size flavorSize,
   const void * flavorData
);
Parameters
scrap

A reference to the scrap to supply data or promises to. 
flavorType

The flavor type to supply or promise the data for. Some flavor types are described in “Scrap Flavor Types”.
flavorFlags

A variable of type ScrapFlavorFlags that you use to supply information about the flavor specified by the flavorType parameter. See “Scrap Flavor Flags” for a description of the values you can use in this parameter.
flavorSize

The size of the data you are supplying or promising, in bytes. If you don't know the size, pass kScrapFlavorSizeUnknown to place a promise for data of undetermined size on the scrap. If you pass 0 in this parameter, a flavor with no expected data—not a promise—is placed on the scrap, and the value of the flavorData parameter is ignored.
flavorData

A pointer to a buffer, local variable, or other storage location, created and disposed of by your application. Before calling PutScrapFlavor to put flavor data on the scrap, store the data in this buffer. For information on the number of bytes of data, see the description of the flavorSize parameter.

Pass NULL for this parameter to indicate you will provide data through a subsequent call to PutScrapFlavor, either later in the same code flow or during execution of your ScrapPromiseKeeperProcPtr callback.

The last time you can provide scrap flavor data is when your scrap promise-keeper function gets called. It is not possible to call PutScrapFlavor while handling a suspend event; suspend events under Carbon don't work the way they do under Mac OS 8 and 9.
Return Value

A result code. See “Scrap Manager Result Codes”.
Discussion

PutScrapFlavor is different than PutScrap in that it includes a ScrapRef parameter and it supports promising a flavor for later delivery, rather than supplying it immediately. 
Availability
Available in CarbonLib 1.0 and later.
Available in Mac OS X 10.0 and later.
Declared In
Scrap.h*/

OSStatus PutScrapFlavor (
   ScrapRef scrap,
   ScrapFlavorType flavorType,
   ScrapFlavorFlags flavorFlags,
   Size flavorSize,
   const void * flavorData
);


#ifdef __cplusplus
}
#endif

#endif