#include <SearchKit/SearchKit.h>

/*SKIndexCreateWithMutableData

Creates a named index stored in a CFMutableData object.

SKIndexRef SKIndexCreateWithMutableData (
   CFMutableDataRef    inData,
   CFStringRef         inIndexName,
   SKIndexType         inIndexType,
   CFDictionaryRef     inAnalysisProperties
);
Parameters
inData

An empty CFMutableData object to contain the index being created.
inIndexName

The name of the index. If you call this function with inIndexName set to NULL, Search Kit assigns the index the default index name IADefaultIndex. If you then attempt to create a second index in the same file without assigning a name, no second index is created and this function returns NULL. Search Kit does not currently support retrieving index names from an index.
inIndexType

The index type. See SKIndexType.
inAnalysisProperties

The text analysis properties dictionary, which optionally sets the minimum term length, stopwords, term substitutions, maximum unique terms to index, and proximity support (for phrase-based searches) when creating the index. See Text Analysis Keys. The inAnalysisProperties parameter can be NULL, in which case Search Kit applies the default dictionary, which is NULL.
Return Value

The newly created index.
Discussion

SKIndexCreateWithMutableData creates an index in memory as a CFMutableData object. Search Kit indexes are initially empty. A memory-based index is useful for quick searching and when your application doesn’t need persistent storage. To create a disk-based, persistent index, use SKIndexCreateWithURL.

Search Kit is thread-safe. You can use separate indexing and searching threads. Your application is responsible for ensuring that no more than one process is open at a time for writing to an index.

When your application no longer needs the index, dispose of it by calling SKIndexClose.
Availability
Available in Mac OS X v10.3 and later.
Declared In
SKIndex.h*/

SKIndexRef SKIndexCreateWithMutableData (
   CFMutableDataRef    inData,
   CFStringRef         inIndexName,
   SKIndexType         inIndexType,
   CFDictionaryRef     inAnalysisProperties
){};


/*SKDocumentCreate

Creates a document URL object based on a scheme, parent, and name.

SKDocumentRef SKDocumentCreate (
CFStringRef     inScheme,
SKDocumentRef   inParent,
CFStringRef     inName
);
Parameters
inScheme

The scheme to use—analogous to the scheme of a URL. Only documents referenced with the “file” scheme can be read by the SKIndexAddDocument function. The scheme can be anything you like if you use the SKIndexAddDocumentWithText function. The scheme can be NULL, in which case it will be set to be the same scheme as the document URL object’s (SKDocumentRef’s) parent. For more information on schemes, see http://www.iana.org/assignments/uri-schemes.html.
inParent

The document URL object one step up in the document hierarchy. Can be NULL.
inName

The name of the document that you’re creating a document URL object for. For the “file” scheme, it is the name of the file or the container, not its path. The path can be constructed by following parent links. The maximum length for a document name is 256 bytes.
Return Value

The new document URL object, or NULL on failure.
Discussion

The new document URL object’s (SKDocumentRef’s) parent can be NULL, but you must specify either a scheme or a parent. When your application no longer needs the document URL object, dispose of it by calling CFRelease.
Availability
Available in Mac OS X v10.3 and later.
Declared In
SKDocument.h*/
SKDocumentRef SKDocumentCreate (
CFStringRef     inScheme,
SKDocumentRef   inParent,
CFStringRef     inName
){};


/*SKIndexAddDocumentWithText

Adds a document URL object, and the associated document’s textual content, to an index.

Boolean SKIndexAddDocumentWithText (
   SKIndexRef      inIndex,
   SKDocumentRef   inDocument,
   CFStringRef     inDocumentText,
   Boolean         inCanReplace
);
Parameters
inIndex

The index to which you are adding the document URL object (SKDocumentRef).
inDocument

The document URL object to add.
inDocumentText

The document text. Can be NULL.
inCanReplace

A Boolean value specifying whether Search Kit will overwrite a document’s index entry (true, indicated by 1 or kCFBooleanTrue), or retain the entry if it exists (false, indicated by 0 or kCFBoolenFalse).
Return Value

A Boolean value of true on success, or false on failure. Also returns false if the document has an entry in the index and inCanReplace is set to false.
Discussion

Use this function to add the textual contents of arbitrary document types to an index. With this function, your application takes responsibility for getting textual content and handing it to the index as a CFString object. Because of this, your application can define what it considers to be a document—a database record, a tagged field in an XML document, an object in memory, a text file, and so on.

Search Kit will index any size text string that you give it, up to its 4 GB index file size limit.

To add the textual content of file-based documents to a Search Kit index, you can use this function or take advantage of Search Kit’s ability to locate and read certain on-disk, file-based document types—see SKIndexAddDocument.

Search Kit is thread-safe. You can use separate indexing and searching threads. Your application is responsible for ensuring that no more than one process is open at a time for writing to an index.

A single Search Kit index file can be up to 4 GB in size.
Special Considerations

In Mac OS X v10.3, some functions do not provide expected results unless you follow a call to SKIndexAddDocumentWithText with a call to SKIndexFlush. The affected functions include SKIndexGetDocumentCount, SKIndexGetDocumentTermCount, SKIndexGetDocumentTermFrequency, and SKIndexGetTermDocumentCount. However, in typical use this won’t be an issue, because applications call these functions after a search, and you must call SKIndexFlush before a search.
Availability
Available in Mac OS X v10.3 and later.
Declared In
SKIndex.h*/

Boolean SKIndexAddDocumentWithText (
   SKIndexRef      inIndex,
   SKDocumentRef   inDocument,
   CFStringRef     inDocumentText,
   Boolean         inCanReplace
){};


/*SKIndexFlush

Invokes all pending updates associated with an index and commits them to backing store.

Boolean SKIndexFlush (
   SKIndexRef  inIndex
);
Parameters
inIndex

The index you want to update and commit to backing store.
Return Value

A Boolean value of true on success, or false on failure.
Discussion

An on-disk or memory-based index becomes stale when your application updates it by adding or removing a document entry. A search on an index in such a state won’t have access to the nonflushed updates. The solution is to call this function before searching. SKIndexFlush flushes index-update information and commits memory-based index caches to disk, in the case of an on-disk index, or to a memory object, in the case of a memory-based index. In both cases, calling this function makes the state of the index consistent.

Before searching an index, always call SKIndexFlush, even though the flush process may take up to several seconds. If there are no updates to commit, a call to SKIndexFlush does nothing and takes minimal time.

A new Search Kit index does not have term IDs until it is flushed.

Search Kit is thread-safe. You can use separate indexing and searching threads. Your application is responsible for ensuring that no more than one process is open at a time for writing to an index.
Availability
Available in Mac OS X v10.3 and later.
Declared In
SKIndex.h*/
Boolean SKIndexFlush (
   SKIndexRef  inIndex
){};


/*SKIndexClose

Closes an index.

void SKIndexClose (
   SKIndexRef      inIndex
);
Parameters
inIndex

The index to close.
Discussion

When your application no longer needs an index that it has opened or created, call SKIndexClose. Calling this function is equivalent to calling CFRelease on an index.

Search Kit is thread-safe. You can use separate indexing and searching threads. Your application is responsible for ensuring that no more than one process is open at a time for writing to an index.
Availability
Available in Mac OS X v10.4 and later.
Declared In
SKIndex.h*/


void SKIndexClose (
   SKIndexRef      inIndex
){};


/*SKIndexRemoveDocument

Removes a document URL object and its children, if any, from an index.

Boolean SKIndexRemoveDocument (
   SKIndexRef      inIndex,
   SKDocumentRef   inDocument
);
Parameters
inIndex

The index from which you want to remove the document URL object (SKDocumentRef).
inDocument

The document URL object to remove.
Return Value

A Boolean value of true on success, or false on failure.
Discussion

When your application deletes a document, use this function to update the index to reflect the change.

Search Kit is thread-safe. You can use separate indexing and searching threads. Your application is responsible for ensuring that no more than one process is open at a time for writing to an index.
Availability
Available in Mac OS X v10.3 and later.
Declared In
SKIndex.h*/

Boolean SKIndexRemoveDocument (
   SKIndexRef      inIndex,
   SKDocumentRef   inDocument
){};


/*SKSearchCreate

Creates an asynchronous search object for querying an index, and initiates search.

SKSearchRef SKSearchCreate (
   SKIndexRef              inIndex,
   CFStringRef             inQuery,
   SKSearchOptions         inSearchOptions
);
Parameters
inIndex

The index to query.
inQuery

The query string to search for.
inSearchOptions

The search options. May be NULL. See the SKSearchOptions enumeration for a description of the available options.
Return Value

A search object.
Discussion

This function creates an asynchronous search object for querying the document contents in an index. It also initiates the search on a separate thread.

After you create the search object, call SKSearchFindMatches to retrieve results. You can call SKSearchFindMatches immediately. To cancel a search, call SKSearchCancel.

For normal (non-similarity-based) queries, Search Kit discerns the type of query—Boolean, prefix, phrase, and so on—from the syntax of the query itself. Moreover, Search Kit supports multiple query types within a single search. For example, the following query includes Boolean, prefix, and suffix searching:appl* OR *ing


This query will return documents containing words that begin with “appl” as well as documents that contain words that end with “ing”.

For similarity searches, specified with the kSKSearchOptionFindSimilar flag in the inSearchOptions parameter, SKSearchCreate ignores all query operators.

The query operators that SKSearchCreate recognizes for non-similarity searching are:
Table 1  Search Kit query operators for non-similarity searches
Operator	
meaning

AND	
Boolean AND

&	
Boolean AND

<space>	
Boolean AND by default when no other operator is present, or Boolean OR if specified by kSKSearchOptionSpaceMeansOR.

OR	
Boolean inclusive OR

|	
Boolean inclusive OR

NOT	
Boolean NOT (see Special Considerations)

!	
Boolean NOT (see Special Considerations)

*	
Wildcard for prefix or suffix; surround term with wildcard characters for substring search. Ignored in phrase searching.

(	
Begin logical grouping

)	
End logical grouping

"	
delimiter for phrase searching


The operators AND, OR, and NOT are case sensitive.

Search Kit performs Unicode normalization on query strings and on the text placed into indexes. It uses Unicode Normalization Form KC (NFKC, compatibility decomposition followed by canonical composition) as documented in Unicode Standard Annex #15. For example, the a-grave character, ‘à’, can be written as the two Unicode characters (0x0061, 0x0300) or as the single Unicode character 0x00E0. Search Kit will normalize (0x0061, 0x0300) to 0x00E0. For more information on Unicode normalization, see http://unicode.org/reports/tr15.

Search Kit further normalizes query strings and indexes by stripping diacritical marks and by forcing characters to lowercase. For example, Search Kit normalizes each of the following characters to ‘a’: ‘a’, ‘à’, ‘A’, and ‘À’.

Search Kit is thread-safe. You can use separate indexing and searching threads. Your application is responsible for ensuring that no more than one process is open at a time for writing to an index.

When your application no longer needs the search object, dispose of it by calling CFRelease.
Special Considerations

Search Kit supports logical exclusion. The NOT and ! operators behave as though they were EXCLUDE operators. For example, a search for ‘red NOT blue’ returns all documents that contain the word ‘red’ and do not contain the word ‘blue’.

Unary Boolean operators, however, are not currently implemented in Search Kit. A search, for example, for ‘NOT blue’, returns zero documents no matter what their content. 
Version Notes

Mac OS X version 10.4 uses a completely revised, and far more powerful, query approach than did earlier versions of Mac OS X. Refer to the Discussion in this function for details. Refer to SKSearchResultsCreateWithQuery (deprecated) for a description of Search Kit’s behavior in earlier versions of Mac OS X.

In versions of Mac OS X prior to version 10.4, Search Kit did not perform Unicode normalization, and did not remove diacritical marks.
Availability
Available in Mac OS X v10.4 and later.
Declared In
SKSearch.h*/
SKSearchRef SKSearchCreate (
   SKIndexRef              inIndex,
   CFStringRef             inQuery,
   SKSearchOptions         inSearchOptions
){};

/*SKSearchFindMatches

Extracts search result information from a search object.

Boolean SKSearchFindMatches (
   SKSearchRef         inSearch,
   CFIndex             inMaximumCount,
   SKDocumentID        *outDocumentIDsArray,
   float               *outScoresArray,
   CFTimeInterval      maximumTime
   CFIndex             *outFoundCount
);
Parameters
inSearch

A reference to a search object (SKSearchRef) previously created with SKSearchCreate.
inMaximumCount

The maximum number of items to find. For each item found, SKSearchFindMatches places the associated document ID into the outDocumentIDsArray array. Specify an inMaximumCount of 0 to find as many items as possible within maximumTime.
outDocumentIDsArray

On input, a pointer to an array for document IDs. On output, points to points to the previously allocated array, which now contains the found document IDs. The size of this array must be equal to inMaximumCount.
outScoresArray

On input, a pointer to an array for scores. On output, points to the previously allocated array, which now contains relevance scores for the found items. The size of this array, if not NULL, must be equal to inMaximumCount. Can be NULL on input, provided that your application doesn’t need this information. Search Kit does not normalize relevance scores, so they can be very large.
maximumTime

The maximum number of seconds before this function returns, whether or not inMaximumCount items have been found. Setting maximumTime to 0 tells the search to return quickly
outFoundCount

On input, a pointer to a CFIndex object that will hold the number of items found. On output, points to the CFIndex object that now contains the actual number of items found.
Return Value

A logical value indicating whether the search is still in progress. Returns false when the search is exhausted.
Discussion

The SKSearchFindMatches extracts results from a find operation initiated by a search object (SKSearchRef).

This function provides results to its output parameters simply in the order in which they are found. This reduces latency to support search-as-you-type functionality. Larger scores mean greater relevance.

You can call this function on a search object repeatedly to get additional sets of search results. For example, if you call this function twice with an inMaximumCount value of 10, the first call will put the first 10 items found into the output arrays and the second call will put the second 10 items found into the output arrays.

Applications are free to determine relevance in any appropriate manner. One simple way is to divide each relevance score by the largest number returned to get relevance numbers scaled linearly from 0.0 to 1.0. Search Kit does not scale the relevance scores for you, because you may want to combine the scores from several calls on a search object or the scores from calls to more than one search object.

Search Kit is thread-safe. You can use separate indexing and searching threads. Your application is responsible for ensuring that no more than one process is open at a time for writing to an index.

Before invoking a search, call SKIndexFlush on all indexes you will query to ensure that updates to the indexes have been flushed to disk.
Availability
Available in Mac OS X v10.4 and later.
Declared In
SKSearch.h*/
Boolean SKSearchFindMatches (
   SKSearchRef         inSearch,
   CFIndex             inMaximumCount,
   SKDocumentID        *outDocumentIDsArray,
   float               *outScoresArray,
   CFTimeInterval      maximumTime,
   CFIndex             *outFoundCount
){};

/*SKIndexCopyInfoForDocumentIDs

Gets document names and parent IDs based on document IDs.

void SKIndexCopyInfoForDocumentIDs (
   SKIndexRef      inIndex,
   CFIndex         inCount,
   SKDocumentID    *inDocumentIDsArray,
   CFStringRef     *outNamesArray,
   SKDocumentID    *outParentIDsArray
);
Parameters
inIndex

The index containing the document information.
inCount

The number of document IDs in inDocumentIDsArray.
inDocumentIDsArray

Points to an array of document IDs representing the documents whose names and parent IDs you want.
outNamesArray

On input, a pointer to an array for document names. On output, points to the previously allocated array, which now contains the document names corresponding to the document IDs in inDocumentIDsArray. May be NULL on input if you don’t want to get the document names.

When finished with the names array, dispose of it by calling CFRelease on each array element.
outParentIDsArray

On input, a pointer to an array for parent document IDs. On output, points to the previously allocated array, which now contains document IDs representing the parents of the documents whose IDs are in inDocumentIDsArray. May be NULL on input if you don’t want to get the parent document IDs.
Discussion

The SKIndexCopyInfoForDocumentIDs function lets you get a batch of document names and parent document IDs in one step, based on a list of document IDs.
Availability
Available in Mac OS X v10.4 and later.
Declared In
SKSearch.h*/
void SKIndexCopyInfoForDocumentIDs (
   SKIndexRef      inIndex,
   CFIndex         inCount,
   SKDocumentID    *inDocumentIDsArray,
   CFStringRef     *outNamesArray,
   SKDocumentID    *outParentIDsArray
){};







