
#ifndef  DVDPLAYBACK_H
#define DVDPLAYBACK_H

#include 	<AvailabilityMacros.h>
#include	<ApplicationServices/ApplicationServices.h>
#include	<Security/Authorization.h>

#include	<HYS/DVDPlayback/DVDPlayback.h>

#ifdef __cplusplus
extern "C" {
#endif

/*DVDPlatback Result Codes

The table below lists the result codes defined for the DVD Playback Services API.
Result Code	Value	Description
kDVDErrorUnknown	-70001	
Unspecified error.
kDVDErrorInitializingLib	-70002	
An error occurred while initializing a DVD-Video playback session. DVD Playback Services is probably being used in another process.
kDVDErrorUninitializedLib	-70003	
A DVD-Video playback session has not been initialized. Any function except DVDInitialize could return this value. 
kDVDErrorNotAllowedDuringPlayback	-70004	
The operation is not permitted while DVD-Video media is playing. 
kDVDErrorUnassignedGrafPort	-70005	
A graphics port for DVD-Video playback has not been set.
kDVDErrorAlreadyPlaying	-70006	
See kDVDErrorNotAllowedDuringPlayback.
kDVDErrorNoFatalErrCallBack	-70007	
No callback has been registered to handle unrecoverable errors during playback. 
kDVDErrorIsAlreadySleeping	-70008	
DVDSleep was called twice without an intervening call to DVDWakeUp. 
kDVDErrorDontNeedWakeup	-70009	
DVDWakeUp was called without first calling DVDSleep.
kDVDErrorTimeOutOfRange	-70010	
The time code is outside the valid range for the current title.
kDVDErrorUserActionNoOp	-70011	
The operation is not permitted at this time.
kDVDErrorMissingDrive	-70012	
A DVD drive is not available.
kDVDErrorNotSupportedConfiguration	-70013	
The current system configuration does not support DVD-Video playback.
kDVDErrorNotSupportedFunction	-70014	
DVD Playback Services does not support this operation. 
kDVDErrorNoValidMedia	-70015	
A VIDEO_TS media folder is not open for playback. 
kDVDErrorWrongParam	-70016	
An argument is invalid. 
kDVDErrorMissingGraphicsDevice	-70017	
A graphics device is not available.
kDVDErrorGraphicsDevice	-70018	
A graphics device error was encountered.
kDVDErrorPlaybackOpen	-70019	
A VIDEO_TS media folder is already open for playback. 
kDVDErrorInvalidRegionCode	-70020	
Region code is not valid. 
kDVDErrorRgnMgrInstall	-70021	
Not used.
kDVDErrorMismatchedRegionCode	-70022	
DVD-Video disc region code does not match the region code currently assigned to the DVD drive.
kDVDErrorNoMoreRegionSets	-70023	
Changing the region code assigned to the DVD drive is no longer permitted.
kDVDErrordRegionCodeUninitialized	-70024	
No region code has been assigned to the DVD drive.
kDVDErrorAuthentification	-70025	
An attempt to change the DVD drive region code could not be authenticated.
kDVDErrorOutOfVideoMemory	-70026	
The video driver does not have enough video memory available for playback.
kDVDErrorNoAudioOutputDevice	-70027	
An appropriate audio output device could not be found.
kDVDErrorSystem	-70028	
A system error occurred.
kDVDErrorNavigation	-70029	
The user has made a selection not supported in the current menu.*/
enum {
DVDErrorUnknown	=-70001	,
kDVDErrorInitializingLib	=-70002	,
kDVDErrorUninitializedLib	=-70003	,
kDVDErrorNotAllowedDuringPlayback	=-70004	,
kDVDErrorUnassignedGrafPort	=-70005	,
kDVDErrorAlreadyPlaying	=-70006	,
kDVDErrorNoFatalErrCallBack=	-70007	,
kDVDErrorIsAlreadySleeping	=-70008	,
kDVDErrorDontNeedWakeup	=-70009	,
kDVDErrorTimeOutOfRange	=-70010	,
kDVDErrorUserActionNoOp	=-70011	,
kDVDErrorMissingDrive	=-70012	,
kDVDErrorNotSupportedConfiguration	=-70013	,
kDVDErrorNotSupportedFunction	=-70014	,
kDVDErrorNoValidMedia	=-70015	,
kDVDErrorWrongParam	=-70016	,
kDVDErrorMissingGraphicsDevice	=-70017	,
kDVDErrorGraphicsDevice=	-70018	,
kDVDErrorPlaybackOpen	=-70019	,
kDVDErrorInvalidRegionCode	=-70020	,
kDVDErrorRgnMgrInstall	=-70021	,
kDVDErrorMismatchedRegionCode=	-70022	,
kDVDErrorNoMoreRegionSets	=-70023	,
kDVDErrordRegionCodeUninitialized =-70024	,
kDVDErrorAuthentification	=-70025	,
kDVDErrorOutOfVideoMemory	=-70026	,
kDVDErrorNoAudioOutputDevice	=-70027	,
kDVDErrorSystem	=-70028	,
kDVDErrorNavigation	=-70029	
};

/*Scan Directions

Constants that specify the direction of play. 

enum {
   kDVDScanDirectionForward,
   kDVDScanDirectionBackward
};
typedef SInt8 DVDScanDirection;
Constants
kDVDScanDirectionForward

Scan in the forward direction. 

Available in Mac OS X v10.3 and later.
kDVDScanDirectionBackward

Scan in the reverse direction. 

Available in Mac OS X v10.3 and later.
Discussion

These constants are used along with the scan rate to specify the direction and speed of play. See the functions DVDScan, DVDGetScanRate, and DVDStepFrame. 
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
enum {
   kDVDScanDirectionForward,
   kDVDScanDirectionBackward
};
typedef SInt8 DVDScanDirection;

/*Playback States

Constants that specify the current play state of the framework.

enum {
   kDVDStateUnknown,
   kDVDStatePlaying,
   kDVDStatePlayingStill,
   kDVDStatePaused,
   kDVDStateStopped,
   kDVDStateScanning,
   kDVDStateIdle,
   kDVDStatePlayingSlow
};
typedef OSStatus DVDState;
Constants
kDVDStateUnknown

The initial state of a new DVD-Video playback session.

Available in Mac OS X v10.3 and later.
kDVDStatePlaying

DVD-Video media is open and playing at normal rate (1x scan rate).

Available in Mac OS X v10.3 and later.
kDVDStatePlayingStill

DVD-Video media is open and playing a single frame.

Available in Mac OS X v10.3 and later.
kDVDStatePaused

DVD-Video media is open and paused or frame-stepping.

Available in Mac OS X v10.3 and later.
kDVDStateStopped

DVD-Video media is open and stopped (not playing). 

Available in Mac OS X v10.3 and later.
kDVDStateScanning

DVD-Video media is open and playing in fast motion (greater than 1x scan rate).

Available in Mac OS X v10.3 and later.
kDVDStateIdle

Not used. 

Available in Mac OS X v10.3 and later.
kDVDStatePlayingSlow

DVD-Video media is open and playing in slow motion (less than 1x scan rate).

Available in Mac OS X v10.4 and later.
Discussion

These constants are passed back to your application when calling the function DVDGetState or when handling the kDVDEventPlayback event. 
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
enum {
   kDVDStateUnknown,
   kDVDStatePlaying,
   kDVDStatePlayingStill,
   kDVDStatePaused,
   kDVDStateStopped,
   kDVDStateScanning,
   kDVDStateIdle,
   kDVDStatePlayingSlow
};
typedef OSStatus DVDState;

/*Aspect Ratios

Constants that specify the current aspect ratio.

enum {
   kDVDAspectRatioUninitialized,
   kDVDAspectRatio4x3,
   kDVDAspectRatio4x3PanAndScan,
   kDVDAspectRatio16x9,
   kDVDAspectRatioLetterBox
};
typedef SInt16 DVDAspectRatio;
Constants
kDVDAspectRatioUninitialized

An unspecified aspect ratio.

Available in Mac OS X v10.3 and later.
kDVDAspectRatio4x3

The process of displaying a 4:3 video frame on a screen with the same aspect ratio.

Available in Mac OS X v10.3 and later.
kDVDAspectRatio4x3PanAndScan

The process of choosing a sample area of a 16:9 video frame for display on a 4:3 screen, using center-of-interest information stored on the DVD.

Available in Mac OS X v10.3 and later.
kDVDAspectRatio16x9

The process of displaying a 16:9 video frame on a screen with the same aspect ratio.

Available in Mac OS X v10.3 and later.
kDVDAspectRatioLetterBox

The process of displaying the entire 16:9 video frame on a 4:3 screen by shrinking the frame and adding mattes or black bars above and below the frame.

Available in Mac OS X v10.3 and later.
Discussion

These constants are used to specify an aspect ratio when calling the function DVDGetAspectRatio and when handling the kDVDEventDisplayMode event. 
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
enum {
   kDVDAspectRatioUninitialized,
   kDVDAspectRatio4x3,
   kDVDAspectRatio4x3PanAndScan,
   kDVDAspectRatio16x9,
   kDVDAspectRatioLetterBox
};
typedef SInt16 DVDAspectRatio;

/*Scan Rates

Constants that specify the rate at which to scan. 

enum {
   kDVDScanRateOneEigth    = -8,
   kDVDScanRateOneFourth   = -4,
   kDVDScanRateOneHalf     = -2,
   kDVDScanRate1x          = 1,
   kDVDScanRate2x          = 2,
   kDVDScanRate4x          = 4,
   kDVDScanRate8x          = 8,
   kDVDScanRate16x         = 16,
   kDVDScanRate32x         = 32
};
typedef SInt16 DVDScanRate;
Discussion

These constants are used along with the scan direction to specify the speed and direction of play. The constant kDVDScanRate1x represents the normal playback speed; the slower and faster playback speeds are multiples of the normal speed. See the functions DVDScan and DVDGetScanRate. 
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
enum {
   kDVDScanRateOneEigth    = -8,
   kDVDScanRateOneFourth   = -4,
   kDVDScanRateOneHalf     = -2,
   kDVDScanRate1x          = 1,
   kDVDScanRate2x          = 2,
   kDVDScanRate4x          = 4,
   kDVDScanRate8x          = 8,
   kDVDScanRate16x         = 16,
   kDVDScanRate32x         = 32
};
typedef SInt16 DVDScanRate;


/*DVDDiscID

Defines a unique identifier for a DVD-Video media folder.

typedef UInt8 DVDDiscID[8];
Discussion

The purpose of this 64-bit identifier is to help you distinguish between different media folders. You can obtain the identifier for a media folder by calling the function DVDGetMediaUniqueID.
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
typedef UInt8 DVDDiscID[8];

/*Playback Menus

Constants that specify input menus for a DVD-Video volume.

enum {
   kDVDMenuTitle,
   kDVDMenuRoot,
   kDVDMenuSubPicture,
   kDVDMenuAudio,
   kDVDMenuAngle,
   kDVDMenuPTT,
   kDVDMenuNone
};
typedef UInt32 DVDMenu;
Constants
kDVDMenuTitle

A title or top menu.

Available in Mac OS X v10.3 and later.
kDVDMenuRoot

A root menu.

Available in Mac OS X v10.3 and later.
kDVDMenuSubPicture

A menu used to select subpictures. 

Available in Mac OS X v10.3 and later.
kDVDMenuAudio

An audio menu.

Available in Mac OS X v10.3 and later.
kDVDMenuAngle

A viewing angle menu.

Available in Mac OS X v10.3 and later.
kDVDMenuPTT

A part-of-title (PTT) or chapter menu. 

Available in Mac OS X v10.3 and later.
kDVDMenuNone

Menu not defined.

Available in Mac OS X v10.3 and later.
Discussion

These constants are used when calling DVDGoToMenu and other menu-related functions, or when handling the kDVDEventMenuCalled event. 
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
enum {
   kDVDMenuTitle,
   kDVDMenuRoot,
   kDVDMenuSubPicture,
   kDVDMenuAudio,
   kDVDMenuAngle,
   kDVDMenuPTT,
   kDVDMenuNone
};
typedef UInt32 DVDMenu;

/*User Navigation Directions

Constants that specify keyboard operations used to select a menu button and execute the action or command associated with a button.

enum {
   kDVDUserNavigationMoveUp = 1,
   kDVDUserNavigationMoveDown,
   kDVDUserNavigationMoveLeft,
   kDVDUserNavigationMoveRight,
   kDVDUserNavigationEnter
};
typedef UInt32 DVDUserNavigation;
Constants
kDVDUserNavigationMoveUp

Move to and select the button above the current location. 

Available in Mac OS X v10.3 and later.
kDVDUserNavigationMoveDown

Move to and select the button below the current location. 

Available in Mac OS X v10.3 and later.
kDVDUserNavigationMoveLeft

Move to and select the button to the left of the current location. 

Available in Mac OS X v10.3 and later.
kDVDUserNavigationMoveRight

Move to and select the button to the right of the current location. 

Available in Mac OS X v10.3 and later.
kDVDUserNavigationEnter

Execute the action of the selected button.

Available in Mac OS X v10.3 and later.
Discussion

These constants are used when calling the function DVDDoUserNavigation.
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
enum {
   kDVDUserNavigationMoveUp = 1,
   kDVDUserNavigationMoveDown,
   kDVDUserNavigationMoveLeft,
   kDVDUserNavigationMoveRight,
   kDVDUserNavigationEnter
};
typedef UInt32 DVDUserNavigation;

/*Region Codes

Constants that specify marketing regions for DVD discs and playback devices. 

enum {
   kDVDRegionCodeUninitialized = 0xff,
   kDVDRegionCode1 = 0xfe,
   kDVDRegionCode2 = 0xfd,
   kDVDRegionCode3 = 0xfb,
   kDVDRegionCode4 = 0xf7,
   kDVDRegionCode5 = 0xef,
   kDVDRegionCode6 = 0xdf,
   kDVDRegionCode7 = 0xbf,
   kDVDRegionCode8 = 0x7f
   
};
typedef UInt32 DVDRegionCode;
Constants
kDVDRegionCodeUninitialized

The default region code for an uninitialized DVD drive.

Available in Mac OS X v10.3 and later.
kDVDRegionCode1

Region 1: North America.

Available in Mac OS X v10.3 and later.
kDVDRegionCode2

Region 2: Japan, Europe, South Africa.

Available in Mac OS X v10.3 and later.
kDVDRegionCode3

Region 3: Southeast Asia.

Available in Mac OS X v10.3 and later.
kDVDRegionCode4

Region 4: Australia, New Zealand, Central & South America.

Available in Mac OS X v10.3 and later.
kDVDRegionCode5

Region 5: Northwest Asia and North Africa.

Available in Mac OS X v10.3 and later.
kDVDRegionCode6

Region 6: China.

Available in Mac OS X v10.3 and later.
kDVDRegionCode7

Region 7: Unassigned.

Available in Mac OS X v10.3 and later.
kDVDRegionCode8

Region 8: Special venues (airplanes, hotels, cruise ships).

Available in Mac OS X v10.3 and later.
Discussion

A DVD Playback region code is a bitfield in which a zero in bit n is used to specify region n. The bits are numbered from right to left, and only the least significant 8 bits are used. For example, the region code 11101111 or 0xef specifies region 5. A region code can also specify more than one region; the region code 11110110 or 0xf6, for example, specifies regions 1 and 4.

Most DVD drives are designed to play only those discs that are authorized for use in a specific region. Region codes are used to:
Assign a new region to a DVD drive when calling the function DVDSetDriveRegionCode. 
Specify the region currently assigned to a DVD drive when calling the function DVDGetDriveRegionCode.
Specify one or more regions assigned to a DVD disc when calling the function DVDGetDiscRegionCode.
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
enum {
   kDVDRegionCodeUninitialized = 0xff,
   kDVDRegionCode1 = 0xfe,
   kDVDRegionCode2 = 0xfd,
   kDVDRegionCode3 = 0xfb,
   kDVDRegionCode4 = 0xf7,
   kDVDRegionCode5 = 0xef,
   kDVDRegionCode6 = 0xdf,
   kDVDRegionCode7 = 0xbf,
   kDVDRegionCode8 = 0x7f
   
};
typedef UInt32 DVDRegionCode;

/*Event Codes

Constants that specify events for which applications can register to get notified. 

enum {
   kDVDEventTitle              = 1,
   kDVDEventPTT                = 2,
   kDVDEventValidUOP           = 3,
   kDVDEventAngle              = 4,
   kDVDEventAudioStream        = 5,
   kDVDEventSubpictureStream   = 6,
   kDVDEventDisplayMode        = 7,
   kDVDEventDomain             = 8,
   kDVDEventBitrate            = 9,
   kDVDEventStill              = 10,
   kDVDEventPlayback           = 11,
   kDVDEventVideoStandard      = 12,
   kDVDEventStreams            = 13,
   kDVDEventScanSpeed          = 14,
   kDVDEventMenuCalled         = 15,
   kDVDEventParental           = 16,
   kDVDEventPGC                = 17,
   kDVDEventGPRM               = 18,
   kDVDEventRegionMismatch     = 19,
   kDVDEventTitleTime          = 20,
   kDVDEventSubpictureStreamNumbers = 21,
   kDVDEventAudioStreamNumbers = 22,
   kDVDEventAngleNumbers       = 23,
   kDVDEventError              = 24,
   kDVDEventCCInfo             = 25,
   kDVDEventChapterTime        = 26
};
typedef UInt32 DVDEventCode;
Discussion

Table 1 lists the constants you use to register a callback function as an event handler for one or more DVD playback events. To learn how to register an event handler, see the function “DVDRegisterEventCallBack”. 

Unlike Carbon events, you do not need to call a function to retrieve the data associated with a playback event. When your event callback is invoked, the associated data is contained in one or both event value parameters. For more information, see DVDEventCallBackFunctionPtr. 
Table 1  DVD Playback events
Event ID	
Event description	
Event value 1	
Event value 2

kDVDEventTitle	
Title has changed.	
Title ID	
-

kDVDEventPTT	
Chapter has changed.	
Chapter ID	
-

kDVDEventValidUOP	
User operations that are disabled.	
DVDUOPCode	
-

kDVDEventAngle	
Camera angle has changed.	
Stream ID	
-

kDVDEventAudioStream	
Audio stream has changed.	
Stream ID	
-

kDVDEventSubpictureStream	
Subpicture has changed.	
Stream ID	
isVisible

kDVDEventDisplayMode	
Display mode has changed. 	
DVDAspectRatio	
-

kDVDEventDomain	
Domain has changed.	
DVDDomainCode	
-

kDVDEventBitrate	
Bit rate has changed.	
Bits per second	
-

kDVDEventStill	
Still picture on or off.	
On (1) - Off (0)	
-

kDVDEventPlayback	
Playback state has changed.	
DVDState	
-

kDVDEventVideoStandard	
Video format has changed	
DVDFormat	
-

kDVDEventStreams	
Stream has changed (audio, subpicture, or angle).	
-	
-

kDVDEventScanSpeed	
Scan rate has changed.	
Speed (1x, 2x, 3x, and so on)	
-

kDVDEventMenuCalled	
Menu has changed.	
DVDMenu	
-

kDVDEventParental	
Parental level has changed.	
Level number	
-

kDVDEventPGC	
Program chain has changed.	
PGC number	
-

kDVDEventGPRM	
GPRM data has changed.	
GPRM index	
Data

kDVDEventRegionMismatch	
Region mismatch between disc and playback device.	
Disc region	
-

kDVDEventTitleTime	
Title time has changed.	
Elapsed time in milliseconds	
Duration in milliseconds

kDVDEventSubpictureStreamNumbers	
The number of subpicture streams has changed. 	
The number of subpicture streams in title	
-

kDVDEventAudioStreamNumbers	
The number of audio streams has changed.	
The number of audio streams in title	
-

kDVDEventAngleNumbers	
The number of angles has changed.	
The number of angles in title	
-

kDVDEventError	
Hardware error. 	
DVDErrorCode	
-

kDVDEventCCInfo	
Closed caption has changed.	
Opcode	
Data

kDVDEventChapterTime	
Chapter time has changed.	
Elapsed time in milliseconds	
Duration in milliseconds

Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
enum {
   kDVDEventTitle              = 1,
   kDVDEventPTT                = 2,
   kDVDEventValidUOP           = 3,
   kDVDEventAngle              = 4,
   kDVDEventAudioStream        = 5,
   kDVDEventSubpictureStream   = 6,
   kDVDEventDisplayMode        = 7,
   kDVDEventDomain             = 8,
   kDVDEventBitrate            = 9,
   kDVDEventStill              = 10,
   kDVDEventPlayback           = 11,
   kDVDEventVideoStandard      = 12,
   kDVDEventStreams            = 13,
   kDVDEventScanSpeed          = 14,
   kDVDEventMenuCalled         = 15,
   kDVDEventParental           = 16,
   kDVDEventPGC                = 17,
   kDVDEventGPRM               = 18,
   kDVDEventRegionMismatch     = 19,
   kDVDEventTitleTime          = 20,
   kDVDEventSubpictureStreamNumbers = 21,
   kDVDEventAudioStreamNumbers = 22,
   kDVDEventAngleNumbers       = 23,
   kDVDEventError              = 24,
   kDVDEventCCInfo             = 25,
   kDVDEventChapterTime        = 26
};
typedef UInt32 DVDEventCode;

/*DVDErrorCode

Defines a non-zero result code passed to a callback function. 

typedef OSStatus DVDErrorCode;
Discussion

This data type is used to pass a result code to the following types of callback functions:
An event callback function that’s registered for the kDVDEventError event. See “Event Codes”.
A fatal error callback function. See “DVDFatalErrCallBackFunctionPtr”.

For information about the result codes defined for DVD Playback Services, see “DVD Playback Services Result Codes”. 
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
typedef OSStatus DVDErrorCode;

/*DVDFatalErrCallBackFunctionPtr

Defines a pointer to an application-defined callback function that’s called when an unrecoverable error occurs during a DVD-Video playback session.

typedef void (*DVDFatalErrCallBackFunctionPtr) (
   DVDErrorCode inError,
   UInt32 inRefCon
);

If you name your function MyDVDFatalErrCallBack, you would declare it like this:

void MyDVDFatalErrCallBack (
   DVDErrorCode inError,
   UInt32 inRefCon
);
Parameters
inError

A code that identifies the fatal error. See “DVD Playback Services Result Codes” for a list of possible values.
inRefCon

Optional application-defined data. This is the same data that you pass in when you register this callback using the function DVDSetFatalErrorCallBack. 
Discussion

When an unrecoverable error occurs, the DVD-Video playback session cannot continue and your callback function is invoked to handle the error. Your function should report the fatal error to the user, perform any necessary cleanup, and call the function DVDDispose.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
typedef void (*DVDFatalErrCallBackFunctionPtr) (
   DVDErrorCode inError,
   UInt32 inRefCon
);


/*DVDEventCallBackFunctionPtr

Defines a pointer to an application-defined callback function that’s called when a registered event occurs. 

typedef void (*DVDEventCallBackFunctionPtr) (
   DVDEventCode inEventCode,
   UInt32 inEventValue1,
   UInt32 inEventValue2,
   UInt32 inRefCon
);

If you name your function MyDVDEventCallBack, you would declare it like this:

void MyDVDEventCallBack (
   DVDEventCode inEventCode,
   UInt32 inEventValue1,
   UInt32 inEventValue2,
   UInt32 inRefCon
);
Parameters
inEventCode

A code that identifies the event. For a list of event codes, see “Event Codes”.
inEventValue1

The first event-specific parameter value. For a list of one or both parameters associated with each event, see “Event Codes”. 
inEventValue2

The second event-specific parameter value. For a list of one or both parameters associated with each event, see “Event Codes”. 
inRefCon

Optional application-defined data. This is the same data you pass in when you register the callback using the function DVDRegisterEventCallBack. 
Discussion

Your callback function is called from a different thread than the main application thread. You should not attempt to draw inside this function. Instead, cache any necessary information and do the actual drawing when you are back in the main thread for your application.

If your callback function is registered for multiple events, the function is called separately for each different event. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/

typedef void (*DVDEventCallBackFunctionPtr) (
   DVDEventCode inEventCode,
   UInt32 inEventValue1,
   UInt32 inEventValue2,
   UInt32 inRefCon
);


/*DVDSetTimeEventRate

Sets the rate of the DVD Playback Services time event. 

OSStatus DVDSetTimeEventRate (
   UInt32 inMilliseconds
);
Parameters
inMilliseconds

The new interval between time events, specified in milliseconds. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

The default interval between DVD Playback Services time events is 900 milliseconds. You can use this function to lengthen or shorten the interval between time events. This function is relevant when you have used the function DVDRegisterEventCallBack to register a callback for one or both of the time events: kDVDEventTitleTime and kDVDEventChapterTime . Time event callbacks are typically used for tasks such as updating the elapsed time or remaining time displayed in your user interface.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDSetTimeEventRate (
   UInt32 inMilliseconds
);

/*DVDGetDriveRegionCode

Gets the region code assigned to a DVD drive, and finds how many additional region code changes are permitted.

OSStatus DVDGetDriveRegionCode (
   DVDRegionCode *outCode,
   SInt16 *outNumberChangesLeft
);
Parameters
outCode

A pointer to a DVDRegionCode variable. On return, the variable specifies the region code that’s assigned to the DVD drive currently in use. If the drive was never initialized, the region code is kDVDRegionCodeUninitialized. For more information about region codes, see “Region Codes”. 
outNumberChangesLeft

A pointer to an integer. On return, the integer contains the number of region code changes remaining for this drive. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function passes back the region code assigned to the DVD drive that’s currently in use. Before calling this function, you need to call the function DVDOpenMediaVolume to open the media folder on a disc that’s mounted in the drive.

To test whether region n is assigned to a drive, you need to compute the bitwise AND of outCode with region code n. If the result is equal to region code n, the drive can play discs that are authorized for playback in region n. For example, this source code shows how to test whether region 1 is assigned to a drive:const DVDRegionCode myRegionCode1 = kDVDRegionCode1;
DVDRegionCode outCode;
SInt16 outNumberChangesLeft;
DVDGetDriveRegionCode (&outCode, &outNumberChangesLeft);
if ((~outCode & ~myRegionCode1) == ~myRegionCode1) {
    // media supports region 1
}

Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGetDriveRegionCode (
   DVDRegionCode *outCode,
   SInt16 *outNumberChangesLeft
);

/*DVDHasMedia

Determines whether DVD-Video media is open for playback.

OSStatus DVDHasMedia (
   Boolean *outHasMedia
);
Parameters
outHasMedia

A pointer to a Boolean variable. On return, a value of TRUE indicates that DVD-Video media is open for playback. A value of FALSE indicates no media is currently open.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function checks to see if DVD-Video media is currently open as a result of successfully calling DVDOpenMediaFile or DVDOpenMediaVolume.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDHasMedia (
   Boolean *outHasMedia
);

/*DVDPlay

Starts playing the media, if currently stopped. 

OSStatus DVDPlay ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

Before calling this function, DVD-Video media needs to be open. If you call this function after calling DVDStop, video playback resumes at the stop position.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDPlay ();

/*DVDPause

Pauses the media, if currently playing.

OSStatus DVDPause ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function pauses the video playback and freezes the video. Before calling this function, DVD-Video media needs to be open and video playback started. After calling this function, the media is still considered to be in video playback mode—that is, you may call DVDStop without first calling DVDResume.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDPause ();

/*DVDStop

Stops playing the current title. If video playback is already stopped, the current video playback position is cleared.

OSStatus DVDStop ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

If you call this function while a title is playing, video playback is stopped, the current video playback position is saved, and the video area is cleared to the color black. 

If you call this function a second time with no intervening play commands, the video playback position is cleared and the playback position is now the beginning of the disc. This is equivalent to calling the function DVDClearLastPlayBookmark.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDStop ();

/*DVDPreviousChapter

Plays the previous chapter in the current title.

OSStatus DVDPreviousChapter ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

Before calling this function, DVD-Video media needs to be open and video playback started. This function finds and begins to play the previous chapter. If there is no previous chapter, this function rewinds to the beginning of the current chapter and begins playing. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDPreviousChapter ();

/*DVDMute

Toggles the mute setting on or off.

OSStatus DVDMute (
   Boolean inMute
);
Parameters
inMute

A Boolean value that specifies whether to turn the mute setting on or off. To mute the video playback volume, pass TRUE. To restore the previous volume setting, pass FALSE. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDMute (
   Boolean inMute
);

/*DVDWakeUp

Restores video playback state information and resumes playback.

OSStatus DVDWakeUp ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function restores the video playback information saved by a previous call to the function DVDSleep. You should call this function whenever the operating system wakes up the machine and you have already called DVDSleep. To find out when the machine is waking up from sleep:
Cocoa applications can add an observer to the workspace notification center for the notification NSWorkspaceDidWakeNotification.
Carbon applications can use the procedure described in Technical Q&A QA1340. 

You should also call this function if the user changes the display configuration or resolution during video playback. To learn how to receive a notification that the display configuration has changed, see Quartz Services Reference. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDWakeUp ();

/*DVDCloseMediaVolume

Closes a previously opened VIDEO_TS media folder on a DVD-Video disc.

OSStatus DVDCloseMediaVolume ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function closes a VIDEO_TS media folder that was previously opened using the function DVDOpenMediaVolume. If necessary, this function stops playback before closing the folder. You must call this function before attempting to open another media folder. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDCloseMediaVolume ();

/*DVDGetAudioVolumeInfo

Gets video playback volume information (minimum, maximum, and current volume).

OSStatus DVDGetAudioVolumeInfo (
   UInt16 *outMinVolume,
   UInt16 *outCurVolume,
   UInt16 *outMaxVolume
);
Parameters
outMinVolume

A pointer to an integer. On return, the integer contains the minimum video playback volume. This parameter is optional; you may specify NULL.
outCurVolume

A pointer to an integer. On return, the integer contains the current video playback volume. This parameter is optional; you may specify NULL.
outMaxVolume

A pointer to an integer. On return, the integer contains the maximum video playback volume.This parameter is optional; you may specify NULL. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

Currently the minimum and maximum volumes are always 0 and 255.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGetAudioVolumeInfo (
   UInt16 *outMinVolume,
   UInt16 *outCurVolume,
   UInt16 *outMaxVolume
);

/*DVDInitialize

Starts a new DVD-Video playback session.

OSStatus DVDInitialize ();
Return Value

A result code. If a new DVD-Video playback session is successfully initiated, this function returns noErr. If a session already exists, this function returns kDVDErrorInitializingLib. For other possible result codes, see “DVD Playback Services Result Codes”. 
Discussion

When preparing to play a DVD, you must call this function to initialize a new DVD-Video playback session. Only one process at a time can use DVD Playback Services. When you are finished using DVD Playback Services, you should call the function DVDDispose to end the session. 
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
OSStatus DVDInitialize ();

/*DVDRegisterEventCallBack

Registers a callback function to get notified when state changes in DVD Playback Services occur.

OSStatus DVDRegisterEventCallBack (
   DVDEventCallBackFunctionPtr inCallBackProc,
   DVDEventCode *inCode,
   UInt32 inCodeCount,
   UInt32 inRefCon,
   UInt32 *outCallBackID
);
Parameters
inCallBackProc

A callback function that handles one or more DVD Playback Services events.
inCode

An array of one or more events for which you are registering this callback. For a list of possible events, see “Event Codes”.
inCodeCount

The number of events for which you are registering.
inRefCon

Application-defined data that’s passed to your callback. If you decide not to use this parameter, you should pass NULL. 
outCallBackID

A pointer to an integer. On return, the integer contains the unique identifier for this registration. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function is optional. You may register more than one callback, and you may register multiple events for a single callback. For every call to this function, there should be a corresponding call to the function DVDUnregisterEventCallBack using the identifier for this registration. You should unregister your callbacks before calling the function DVDDispose to end the current DVD-Video playback session.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDRegisterEventCallBack (
   DVDEventCallBackFunctionPtr inCallBackProc,
   DVDEventCode *inCode,
   UInt32 inCodeCount,
   UInt32 inRefCon,
   UInt32 *outCallBackID
);

/*DVDSetFatalErrorCallBack

Registers a callback function to handle unrecoverable errors.

OSStatus DVDSetFatalErrorCallBack (
   DVDFatalErrCallBackFunctionPtr inCallBackProc,
   UInt32 inRefCon
);
Parameters
inCallBackProc

A pointer to the callback function you provide to handle unrecoverable errors. For infomation about the requirements for such a function, see DVDFatalErrCallBackFunctionPtr.
inRefCon

Application-defined data that is passed to your callback function when it is called to handle a fatal error. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function registers a callback that you write to handle unrecoverable errors during video playback. You should call this function immediately after calling the function DVDInitialize. If you don’t register a callback to handle unrecoverable errors, your application may crash when a fatal error occurs. For information about how to write such a callback function, see DVDFatalErrCallBackFunctionPtr. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDSetFatalErrorCallBack (
   DVDFatalErrCallBackFunctionPtr inCallBackProc,
   UInt32 inRefCon
);

/*DVDIsValidMediaRef

Determines whether a VIDEO_TS media folder has a valid layout.

OSStatus DVDIsValidMediaRef (
   FSRef *inFile,
   Boolean *outIsValid
);
Parameters
inFile

A pointer to a VIDEO_TS media folder on a DVD-Video disc or another mass storage device. To convert a path into an FSRef structure, use the function FSPathMakeRef. 
outIsValid

A pointer to a Boolean variable. On return, a value of TRUE indicates the folder layout is valid. A value of FALSE indicates the layout is not valid and the folder cannot be opened for playback.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

Before attempting to open a DVD-Video media folder, you should call this function to verify that the folder is valid. If another media folder is open, before calling this function you should call DVDCloseMediaFile or DVDCloseMediaVolume to close the other folder. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDIsValidMediaRef (
   FSRef *inFile,
   Boolean *outIsValid
);

/*DVDGetNumAngles

Gets the number of camera angles currently available.

OSStatus DVDGetNumAngles (
   UInt16 *outNumAngles
);
Parameters
outNumAngles

On return, a pointer to the number of different camera angles in the current title. DVD-Video media supports up to 9 camera angles.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGetNumAngles (
   UInt16 *outNumAngles
);

/*DVDSetAngle

Sets the camera angle to display.

OSStatus DVDSetAngle (
   UInt16 inAngleNum
);
Parameters
inAngleNum

A 1-based index that specifies the camera angle to display. The angle number of the main angle is 1. DVD-Video media supports up to nine camera angles.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

Before calling this function, DVD-Video media needs to be open and video playback started. The DVD-Video author decides how many camera angles are used. If the specified angle does not exist, this function does nothing. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDSetAngle (
   UInt16 inAngleNum
);

/*DVDGetAngle

Gets the camera angle displayed.

OSStatus DVDGetAngle (
   UInt16 *outAngleNum
);
Parameters
outAngleNum

A pointer to an integer. On return, the integer contains the 1-based index of the current camera angle. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

Before calling this function, DVD-Video media needs to be open and video playback started. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGetAngle (
   UInt16 *outAngleNum
);

/*DVDDoUserNavigation

Allows the user to navigate between menu buttons. 

OSStatus DVDDoUserNavigation (
   DVDUserNavigation inNavigation
);
Parameters
inNavigation

A user-navigation constant that specifies the direction the user is navigating in the menu. For a list of possible values, see “User Navigation Directions”. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

You should call this function whenever the user navigates between buttons in the menu using the keyboard. This function moves the focus to the button. If you pass in kDVDUserNavigationEnter, the button action is executed.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDDoUserNavigation (
   DVDUserNavigation inNavigation
);


/*DVDGoToMenu

Jumps to a specified menu.

OSStatus DVDGoToMenu (
   DVDMenu inMenu
);
Parameters
inMenu

A constant that specifies the desired menu. For a list of possible values, see “Playback Menus”. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

Before calling this function, DVD-Video media needs to be open and video playback started. Not every DVD has every menu type. If you attempt to jump to a non-existent menu, this function returns an error. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGoToMenu (
   DVDMenu inMenu
);

/*DVDSleep

Saves video playback state information and suspends playback.

OSStatus DVDSleep ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

You should call this function when the operating system is putting the machine to sleep, to save video playback information. To find out when the machine is about to sleep:
Cocoa applications can add an observer to the workspace notification center for the notification NSWorkspaceWillSleepNotification.
Carbon applications can use the procedure described in Technical Q&A QA1340. 

You should also call this function if the user changes the display configuration or resolution during video playback. To learn how to receive a notification that the display configuration is about to change, see Quartz Services Reference. 

Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDSleep ();

/*DVDSetAudioVolume

Sets the playback audio volume setting (0 - 255).

OSStatus DVDSetAudioVolume (
   UInt16 inVolume
);
Parameters
inVolume

An integer in the range 0 to 255 that specifies the new video playback volume setting. Larger values are clamped to 255.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function changes the current playback audio volume setting, not the system audio volume setting. The actual audio volume experienced by the user is a function of these two settings: actual audio volume = system volume * playback audio volume / 255


For example, if you set the playback volume to 127, the actual volume is approximately one-half the current system volume.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDSetAudioVolume (
   UInt16 inVolume
);

/*DVDReturnToTitle

Returns from a menu to the current position within the title.

OSStatus DVDReturnToTitle ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function returns to the last played position in the current title. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDReturnToTitle ();

/*DVDNextChapter

Plays the next chapter in the current title.

OSStatus DVDNextChapter ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

Before calling this function, DVD-Video media needs to be open and video playback started. This function finds and begins to play the next chapter. If there is no next chapter, this function continues playing the current chapter. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDNextChapter ();

/*DVDUnregisterEventCallBack

Unregisters an event callback.

OSStatus DVDUnregisterEventCallBack (
   UInt32 inCallBackID
);
Parameters
inCallBackID

The unique identifier for an event callback registration. See DVDRegisterEventCallBack.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function should be called once for each call to the function DVDRegisterEventCallBack. You should unregister all event callbacks before calling the function DVDDispose to end the current DVD-Video playback session.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDUnregisterEventCallBack (
   UInt32 inCallBackID
);

/*DVDIsOnMenu

Determines whether a menu is being displayed, and if so, which one.

OSStatus DVDIsOnMenu (
   Boolean *outOnMenu,
   DVDMenu *outMenu
);
Parameters
outOnMenu

A pointer to a Boolean variable. On return, a value of TRUE indicates a menu with one or more buttons is displayed. A value of FALSE indicates no menu is being displayed. 
outMenu

A pointer to a menu code. On return, the code’s value is the current menu or kDVDMenuNone. For a list of all possible values, see “Playback Menus”. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

Before calling this function, DVD-Video media needs to be open and video playback started. There can be only one menu displayed at a time. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/

OSStatus DVDIsOnMenu (
   Boolean *outOnMenu,
   DVDMenu *outMenu
);

/*DVDOpenMediaVolume

Opens a VIDEO_TS media folder located on a DVD-Video disc.

OSStatus DVDOpenMediaVolume (
   FSRef *inVolume
);
Parameters
inVolume

A pointer to the VIDEO_TS media folder located at the root level of a DVD-Video disc. To convert a path into an FSRef structure, use the function FSPathMakeRef. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function opens the VIDEO_TS media folder on a DVD-Video disc. To open a media folder on a mass storage device such as a hard drive, you should use the function DVDOpenMediaFile instead.

Before calling this function:
If another media folder is open, you should close the other folder by calling DVDCloseMediaFile or DVDCloseMediaVolume. 
You should call the function DVDIsValidMediaRef to verify that this media folder is valid.

After you are finished playing the media on the disc, you should call the function DVDCloseMediaVolume to close the folder.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDOpenMediaVolume (
   FSRef *inVolume
);

/*DVDIsMuted

Determines whether the video playback volume is currently muted.

OSStatus DVDIsMuted (
   Boolean *outIsMuted
);
Parameters
outIsMuted

A pointer to a Boolean variable. On return, a value of TRUE indicates the video playback volume is currently muted. A value of FALSE indicates the volume is not muted.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDIsMuted (
   Boolean *outIsMuted
);

/*DVDOpenMediaFile

Opens a VIDEO_TS media folder located on a mass storage device.

OSStatus DVDOpenMediaFile (
   FSRef *inFile
);
Parameters
inFile

A pointer to a VIDEO_TS media folder on a mass storage device other than a DVD-Video disc. To convert a path into an FSRef structure, use the function FSPathMakeRef. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function opens a VIDEO_TS media folder on a mass storage device such as a hard drive. To open the media folder on a DVD-Video disc, you should use the function DVDOpenMediaVolume instead.

Before calling this function:
If another media folder is open, you should close the other folder by calling DVDCloseMediaFile or DVDCloseMediaVolume. 
You should call the function DVDIsValidMediaRef to verify that this media folder is valid.

After you are finished playing the DVD-Video media, you should call the function DVDCloseMediaFile to close the folder.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h */

OSStatus DVDOpenMediaFile (
   FSRef *inFile
);

/*DVDGetMediaUniqueID

Gets a unique identifier for a media folder. 

OSStatus DVDGetMediaUniqueID (
   DVDDiscID outDiscID
);
Parameters
outDiscID

An 8-byte array of type DVDDiscID. On return, this array contains the unique identifier of the current media. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function passes back a 64-bit identifier that can help you distinguish between different media folders. While the identifiers generated by this function are not guaranteed to be unique (see below), duplicate identifiers are extremely rare.

There are two known limitations of this function:
Two copies of the same DVD that are produced at different times do not necessarily have the same identifier. This can happen when the two media folders have minor differences. 
Two different DVDs with similar content—collections of episodes from a television series, for example—do not always have different identifiers. This can happen when the two media folders are identical. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGetMediaUniqueID (
   DVDDiscID outDiscID
);

/*DVDScan

Fast forwards or rewinds at the specified scan rate.

OSStatus DVDScan (
   DVDScanRate inRate,
   DVDScanDirection inDirection
);
Parameters
inRate

An integer that specifies the rate at which to scan. For a list of possible values, see “Scan Rates”.
inDirection

An integer that specifies the direction in which to scan. For a list of possible values, see “Scan Directions”. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function advances the media in the specified direction at the specified rate. When you scan at a rate other than kDVDScanRate1x (normal speed), the audio channel is muted and subtitles are not displayed. If the scan rate is normal and the direction is forward, this function is equivalent to DVDPlay. This function does not support video playback at normal speed in the reverse direction. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/

OSStatus DVDScan (
   DVDScanRate inRate,
   DVDScanDirection inDirection
);

/*DVDGetDiscRegionCode

Gets the region codes available on a DVD-Video disc. 

OSStatus DVDGetDiscRegionCode (
   DVDRegionCode *outCode
);
Parameters
outCode

A pointer to a DVDRegionCode variable. On return, the variable contains a bitfield that specifies one or more region codes. For more information about region codes, see “Region Codes”. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function passes back the region codes assigned to the DVD-Video disc that’s currently in use. Before calling this function, you need to call the function DVDOpenMediaVolume to open the media folder on the disc. 

To test whether a disc is authorized for playback in region n, you need to compute the bitwise AND of outCode with region code n. If the result is equal to region code n, the disc is authorized for playback in region n. For example, this source code shows how to test whether a disc is authorized for playback in region 1:const DVDRegionCode myRegionCode1 = kDVDRegionCode1;
DVDRegionCode outCode;
DVDGetDiscRegionCode (&outCode);
if ((~outCode & ~myRegionCode1) == ~myRegionCode1) {
    // media supports region 1
}

Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGetDiscRegionCode (
   DVDRegionCode *outCode
);

/*DVDDispose

Ends the current DVD-Video playback session.

OSStatus DVDDispose ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

You call this function when you’re finished using DVD Playback Services, usually when your application quits. 
Availability
Available in Mac OS X version 10.3 and later.
Declared In
DVDPlayback.h*/
OSStatus DVDDispose ();

/*DVDCloseMediaFile

Closes a previously opened VIDEO_TS media folder on a mass storage device.

OSStatus DVDCloseMediaFile ();
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function closes a VIDEO_TS media folder that was previously opened using the function DVDOpenMediaFile. If necessary, this function stops playback before closing the folder. You must call this function before attempting to open another media folder. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDCloseMediaFile ();

/*DVDSetVideoBounds

Sets the bounds in which to display the video. 

OSStatus DVDSetVideoBounds (
   Rect *inPortRect
);
Parameters
inPortRect

A pointer to a rectangle that contains the desired bounds in window local coordinates.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function is used to set the area inside the current window in which to display the video. The video area is not required to fill the entire window. Generally you should set the video area to be smaller than the window whenever the aspect ratio of the current title and window are different. To find the aspect ratio of the current title, use the function DVDGetAspectRatio. 

Because the aspect ratios of the titles in a DVD-Video media folder are not always the same, you may need to call this function repeatedly to reset the video area as the user makes different viewing choices. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDSetVideoBounds (
   Rect *inPortRect
);

/*DVDGetAspectRatio

Gets the aspect ratio of the current title or menu. 

OSStatus DVDGetAspectRatio (
   DVDAspectRatio *outRatio
);
Parameters
outRatio

A pointer to a DVDAspectRatio variable. On return, the variable specifies the aspect ratio of the current title. For a list of possible aspect ratios, see “Aspect Ratios”.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

The aspect ratio of the video can change whenever a different title or menu is displayed. You can use this function together with DVDGetNativeVideoSize to calculate the bounds of the video area in a window. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGetAspectRatio (
   DVDAspectRatio *outRatio
);

/*DVDSwitchToDisplay

If a specified graphics display supports DVD-Video playback, sets the display for the current playback session. 

OSStatus DVDSwitchToDisplay (
   CGDirectDisplayID newDisplay,
   Boolean *outSupported
);
Parameters
newDisplay

The Quartz display ID for the graphics display you want to use for video playback. For information about how to get a display ID, see Quartz Services Reference.
outSupported

A pointer to a Boolean variable. On return, a value of TRUE indicates that the specified graphics display can be used for DVD-Video playback. A value of FALSE indicates the display cannot be used for this purpose. If the value is FALSE, the current video playback display remains unchanged. 
Return Value

A result code. If the new display is not supported, returns an error and maintains the current display. See “DVD Playback Services Result Codes”. 
Discussion

This convenience function is equivalent to calling the function DVDIsSupportedDisplay followed by a call to DVDSetVideoDisplay. This function works correctly even if there is no current video playback display. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDSwitchToDisplay (
   CGDirectDisplayID newDisplay,
   Boolean *outSupported
);

/*DVDDoMenuClick

If a menu button contains a specified point, the button’s action is executed and the button’s index is passed back.

OSStatus DVDDoMenuClick (
   Point inPortPt,
   SInt32 *outIndex
);
Parameters
inPortPt

A point specified using window local coordinates. 
outIndex

A pointer to an integer. On return, if the specified point lies inside the clickable area of a menu button, the integer contains the 1-based index of the button. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function checks to see if a specified point lies inside one of the buttons in the current menu. If so, this function executes the button’s action and passes back its index. Note that the bounding rectangle of a button is not necessarily the same as the clickable area; not all buttons even have a clickable area. See also DVDDoButtonActivate. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/

OSStatus DVDDoMenuClick (
   Point inPortPt,
   SInt32 *outIndex
);

/*.DVDDoMenuMouseOver

If a specified point lies inside a menu button, focuses the button and passes back its index.

OSStatus DVDDoMenuMouseOver (
   Point inPortPt,
   SInt32 *outIndex
);
Parameters
inPortPt

A point specified using window local coordinates. 
outIndex

A pointer to an integer. On return, if the point lies inside the button in focus, the integer contains the 1-based index of the button. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

See also DVDDoUserNavigation.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDDoMenuMouseOver (
   Point inPortPt,
   SInt32 *outIndex
);

/*DVDGetNativeVideoSize

Gets the native width and height in pixels of the current title.

OSStatus DVDGetNativeVideoSize (
   UInt16 *outWidth,
   UInt16 *outHeight
);
Parameters
outWidth

A pointer to an integer. On return, the integer contains the native width of the current title. 
outHeight

A pointer to an integer. On return, the integer contains the native height of the current title. 
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

The native dimensions of a title are specified by the author of the DVD-Video media. The width and height are typically 720 x 480 pixels for the NTSC video format and 720 x 576 pixels for the PAL video format. 
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGetNativeVideoSize (
   UInt16 *outWidth,
   UInt16 *outHeight
);

/*DVDSetVideoWindowID

Enables or disables DVD-Video playback in a Cocoa window. 

OSStatus DVDSetVideoWindowID (
   UInt32 inVidWindowID
);
Parameters
inVidWindowID

The window ID associated with a Cocoa window, or NULL. For information about when to pass NULL, see the Discussion below.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

If you’re using a Cocoa window for DVD-Video playback, you need to use this function in two different situations:
Immediately after calling the function DVDInitialize to start a new DVD-Video playback session, you should call this function and pass the window ID associated with the Cocoa window. 
When video is not playing and you want to draw into the area of the window you set by calling the function DVDSetVideoBounds, you should call this function and pass NULL in the inVidWindowID parameter. 

To learn how to obtain the window number for a Cocoa window, see the description of the windowNumber method in the NSWindowclass.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDSetVideoWindowID (
   UInt32 inVidWindowID
);

/*DVDGetBookmark

Requests a bookmark to the current play position.

OSStatus DVDGetBookmark (
   void *outBookmarkData,
   UInt32 *ioBookmarkDataSize
);
Parameters
outBookmarkData

A generic pointer to memory for a bookmark in your calling program, or NULL. On return, this memory contains the new bookmark. 

The size of this memory should be equal to or greater than the actual size of a bookmark in bytes. To determine the actual size of a bookmark, see the Discussion below. 
ioBookmarkDataSize

On entry, a pointer to an integer that contains the size of your bookmark memory. If you haven’t allocated memory for a bookmark, set this size to zero. On return, the integer’s value is the actual size of a new bookmark. See the Discussion below.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function is used when playing media on a DVD-Video disc. Before calling this function, the media needs to be open and playing. This function passes back a bookmark to the current play position in the current DVD-Video playback session.

To request a play bookmark, you need to call this function twice. The first call determines the minimum required size for the bookmark:UInt32 size = 0;
DVDGetBookmark (NULL, &size);


After you allocate sufficient memory for the bookmark, call this function again to create the bookmark, passing it the location and size of your bookmark memory:void *bookmark = malloc (size);
OSStatus err = DVDGetBookmark (bookmark, &size);


For information about requesting a bookmark when the media is not playing, see DVDGetLastPlayBookmark.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGetBookmark (
   void *outBookmarkData,
   UInt32 *ioBookmarkDataSize
);

/*DVDGotoBookmark

Sets the video playback position using a bookmark, and resumes play.

OSStatus DVDGotoBookmark (
   void *inBookmarkData,
   UInt32 inBookmarkDataSize
);
Parameters
inBookmarkData

A generic pointer to memory for a bookmark. To learn how to obtain a bookmark, see DVDGetBookmark or DVDGetLastPlayBookmark.
inBookmarkDataSize

The actual size of the specified bookmark in bytes.
Return Value

A result code. See “DVD Playback Services Result Codes”. 
Discussion

This function is used when playing media on a DVD-Video disc. Before calling this function, the media needs to be open. This function uses the specified bookmark to set the video playback position, and begins playing the media. See also DVDSetLastPlayBookmark.
Availability
Available in Mac OS X version 10.3 and later. 
Declared In
DVDPlayback.h*/
OSStatus DVDGotoBookmark (
   void *inBookmarkData,
   UInt32 inBookmarkDataSize
);

#ifdef __cplusplus
}
#endif

#endif 

