#ifndef ATSUNICODEOBJECTS_H
#define ATSUNICODEOBJECTS_H

#ifndef ATSUNICODETYPES_H
#include <QD/ATSUnicodeTypes.h>
#endif


#include <AvailabilityMacros.h>

#if PRAGMA_ONCE
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*ATSUDisposeStyle

Disposes of the memory associated with a style object.

OSStatus ATSUDisposeStyle (
   ATSUStyle iStyle
);
Parameters
iStyle

An ATSUStyle value specifying the style object to dispose of.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUDisposeStyle function frees the memory associated with the specified style object and its internal structures, including style run attributes. It does not dispose of the memory pointed to by application-defined style run attributes or reference constants. You are responsible for doing so.

You should call this function after calling the function ATSUDisposeTextLayout to dispose of any text layout objects associated with the style object.

For best performance, once you create a style object, you should keep it and use it as often as needed. You should dispose of the style object only when it is no longer needed in your application.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUDisposeStyle (
   ATSUStyle iStyle
);

/*ATSUDisposeTextLayout

Disposes of the memory associated with a text layout object.

OSStatus ATSUDisposeTextLayout (
   ATSUTextLayout iTextLayout
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object to dispose of. 
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUDisposeTextLayout function frees the memory associated with the specified text layout object and its internal structures, including line and layout control attributes, style runs, and soft line breaks. ATSUDisposeTextLayout does not dispose of any memory that may be allocated for reference constants or style objects associated with the text layout object. You are responsible for doing so.

For best performance, text layout objects are readily reusable and should be cached for later use, if possible. You can reuse a text layout object even if the text associated with it is altered. Call the functions ATSUSetTextPointerLocation, ATSUTextDeleted, or ATSUTextInserted to manage the altered text, rather than disposing of the text layout object and creating a new one.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUDisposeTextLayout (
   ATSUTextLayout iTextLayout
);

/*ATSUCreateAndCopyStyle

Creates a copy of a style object.

OSStatus ATSUCreateAndCopyStyle (
   ATSUStyle iStyle,
   ATSUStyle *oStyle
);
Parameters
iStyle

An ATSUStyle value specifying the style object to copy.
oStyle

A pointer to an ATSUStyle value. On return, the pointer refers to a newly created style object. This style object contains the same values for style attributes, font features, and font variations as those of the style object passed in the iStyle parameter.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUCreateAndCopyStyle function creates a new style object with values obtained from the source style object’s style attributes, font features, and font variations. ATSUCreateAndCopyStyle does not copy reference constants.

To create a new style object without copying a source object, you can call the function ATSUCreateStyle. Alternately, to copy the contents of a source style object into an existing style object, call the function ATSUCopyAttributes.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUCreateAndCopyStyle (
   ATSUStyle iStyle,
   ATSUStyle *oStyle
);

/*ATSUCreateTextLayoutWithTextPtr

Creates an opaque text layout object containing default text layout attributes as well as associated text and text styles.

OSStatus ATSUCreateTextLayoutWithTextPtr (
   ConstUniCharArrayPtr iText,
   UniCharArrayOffset iTextOffset,
   UniCharCount iTextLength,
   UniCharCount iTextTotalLength,
   ItemCount iNumberOfRuns,
   const UniCharCount *iRunLengths,
   ATSUStyle *iStyles,
   ATSUTextLayout *oTextLayout
);
Parameters
iText

A pointer of type ConstUniCharArrayPtr, referring to a text buffer containing UTF-16–encoded text. ATSUI associates this buffer with the new text layout object and analyzes the complete text of the buffer when obtaining the layout context for the current text range. Thus, for paragraph-format text, if you specify a buffer containing less than a complete paragraph, some of ATSUI’s layout results are not guaranteed to be accurate. For example, with a buffer of less than a full paragraph, ATSUI can neither reliably obtain the context for bidirectional processing nor reliably generate accent attachments and ligature formations for Roman text. 
iTextOffset

A UniCharArrayOffset value specifying the offset from the beginning of the text buffer to the first character of the range to include in the layout. To indicate that the specified text range starts at the beginning of the text buffer, you can pass the constant kATSUFromTextBeginning. To specify the entire text buffer, pass kATSUFromTextBeginning in this parameter and kATSUToTextEnd in the iTextLength parameter. 
iTextLength

A UniCharCount value specifying the length of the text range. Note that iTextOffset + iTextLength must be less than or equal to the value of the iTextTotalLength parameter. If you want the range of text to extend to the end of the text buffer, you can pass the constant kATSUToTextEnd.
iTextTotalLength

A UniCharCount value specifying the length of the entire text buffer. This value should be greater than or equal to the range of text defined by the iTextLength parameter.
iNumberOfRuns

An ItemCount value specifying the number of text style runs you want to define within the overall text range. The number of style objects and style run lengths passed in the iStyles and iRunLengths parameters, respectively, should be equal to the number of runs specified here.
iRunLengths

A pointer to the first element in a UniCharCount array. This array provides ATSUI with the lengths of each of the text’s style runs. You can pass kATSUToTextEnd for the last style run length if you want the style run to extend to the end of the text range. If the sum of the style run lengths is less than the total length of the text range, the remaining characters are assigned to the last style run.
iStyles

A pointer to the first element in an ATSUStyle array. Each element in the array must contain a valid style object that corresponds to a style run defined by the iRunLengths array. 
oTextLayout

A valid pointer to an ATSUTextLayout value. On return, the value refers to the newly created text layout object. 
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUCreateTextLayoutWithTextPtr function creates a text layout object associated with style objects and text and containing the default text layout attributes described in “Attribute Tags”. To provide nondefault line or layout attributes for a text layout object, you can call the functions ATSUSetLineControls or ATSUSetLayoutControls. After setting text attributes, call ATSUDrawText to draw the text.

Because the only way that ATSUI interacts with text is via the memory references you associate with a text layout object, you are responsible for keeping these references updated, as in the following cases:
When the user deletes or inserts a subrange within a text buffer (but the buffer itself is not relocated), you should call the functions ATSUTextDeleted and ATSUTextInserted, respectively.
When you relocate the entire text buffer (but no other changes have occurred that would affect the buffer’s current subrange), you should call the function ATSUTextMoved.
When both the buffer itself is relocated and a subrange of the buffer’s text is deleted or inserted (that is, a combination of cases 1 and 2, above), you must use the function ATSUSetTextPointerLocation to inform ATSUI.
When you are associating an entirely different buffer with a text layout object, you must call the function ATSUSetTextPointerLocation.

Note that, because ATSUI objects retain state information, doing superfluous calling can degrade performance. For example, you could call ATSUSetTextPointerLocation rather than ATSUTextInserted when the user inserts text, but there would be a performance penalty, as all the layout caches are flushed when you call ATSUSetTextPointerLocation, rather than just the affected ones.

Text layout objects are readily reusable and should themselves be cached for later use, if possible. Text objects are thread-safe starting with ATSUI version 2.4.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUCreateTextLayoutWithTextPtr (
   ConstUniCharArrayPtr iText,
   UniCharArrayOffset iTextOffset,
   UniCharCount iTextLength,
   UniCharCount iTextTotalLength,
   ItemCount iNumberOfRuns,
   const UniCharCount *iRunLengths,
   ATSUStyle *iStyles,
   ATSUTextLayout *oTextLayout
);

/*ATSUSetTransientFontMatching

Turns automatic font substitution on or off for a text layout object.

OSStatus ATSUSetTransientFontMatching (
   ATSUTextLayout iTextLayout,
   Boolean iTransientFontMatching
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object for which to set automatic font substitution on or off. 
iTransientFontMatching

A Boolean value indicating whether ATSUI is to perform automatic font substitution for the text layout object. If you pass true, ATSUI performs automatic font substitution for the text range associated with the text layout object.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

Calling the ATSUSetTransientFontMatching function sets ATSUI’s automatic font substitution to on or off for a given text layout object. When automatic font substitution is on, ATSUI scans the text range associated with specified text layout object looking for undrawable characters whenever a layout is performed, for example, when text is measured or drawn. When ATSUI finds a character that cannot be drawn with the currently assigned font, it identifies a valid font for the character and draws the character. ATSUI continues scanning the text range for characters in need of substitute fonts, replacing the font and redrawing the characters as needed. ATSUI stops scanning when it reaches the end of the text range associated with the text layout object.

ATSUI’s default behavior for finding a substitute font is to use the first valid font that it finds when sequentially scanning the fonts in the user’s system. However, you can alter this behavior by calling the function ATSUCreateFontFallbacks and defining your own font fallback settings for the text layout object. If ATSUI cannot find any suitable replacement fonts, it substitutes the missing-character glyph—that is, a glyph representing an empty box—to indicate to the user that a valid font is not installed on their system.

Note that when ATSUSetTransientFontMatching performs font substitution, it does not change the font attribute in the associated style object. That is, the font attribute for the style object associated with the redrawn character(s) remains set to the invalid font—not the valid substitute font— just as it was prior to calling ATSUSetTransientFontMatching.

If you want ATSUI to identify a substitute font, but you do not want ATSUI to automatically perform the font substitution, you can call the function ATSUMatchFontsToText.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUSetTransientFontMatching (
   ATSUTextLayout iTextLayout,
   Boolean iTransientFontMatching
);

/*ATSUSetLayoutControls

Sets layout control attribute values in a text layout object.

OSStatus ATSUSetLayoutControls (
   ATSUTextLayout iTextLayout,
   ItemCount iAttributeCount,
   const ATSUAttributeTag *iTag,
   const ByteCount *iValueSize,
   const ATSUAttributeValuePtr *iValue
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object for which to set layout control attributes. 
iAttributeCount

An ItemCount value specifying the number of attributes to set. This value should correspond to the number of elements in the iTag and iValueSize arrays.
iTag

A pointer to the initial ATSUAttributeTag value in an array of layout control attribute tags. Each element in the array must contain a valid tag that corresponds to the layout control attribute to set. See “Attribute Tags” for a description of the Apple-defined layout control attribute tag constants. 
iValueSize

A pointer to the initial ByteCount value in an array of attribute value sizes. Each element in the array must contain the size (in bytes) of the corresponding layout control attribute being set. ATSUSetLayoutControls sets layout attributes after confirming the sizes in the array.
iValue

A pointer to the initial ATSUAttributeValuePtr value in an array of attribute value pointers. Each value in the array must correspond to a tag in the iTag array and be a legal value for that tag. 
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

When you use ATSUI to image your text, you can control the text’s display and formatting at a number of different levels.

One level is that of the entire text range associated with your text layout object, also known as the “layout level.” To affect display and formatting on this level, you can specify various layout control attributes using the ATSUSetLayoutControls function. These attributes affect the width of the text area from margin to margin, the alignment of the text, its justification, rotation, and direction, as well as other layout options.

Another level is that of a single line of text, that is, the “line level.” To affect display and formatting on this level, you specify various line control attributes via the function ATSUSetLineControls. These attributes are similar to those that you can apply on a full-layout basis, but each affects only an individual text line.

Given that ATSUI allows you to control similar aspects of the display and formatting of your text at either the line level or the layout level (or both, or neither), it is up to you to decide how much layout control to take. However, you should note the following:
Setting layout control attributes overrides the corresponding default layout-level settings for a text layout object. Any layout attributes that you do not set retain the default values described in “Attribute Tags”.
Setting line control attributes overrides the corresponding layout-level settings (whether set or at default values) for a text layout object. This is true even if you set the layout-level attributes subsequently to the line-level ones.
From a performance standpoint, it is preferable to work from the layout level and not specify layout line by line unless necessary.

Finally, it is also possible to control the display and formatting of your text at the level of an individual character or “run” of characters. At this level, you customize layout by manipulating style settings in a style object. Among the character-level aspects you can control are style attributes (such as font size and color), font features (such as ligatures), and font variations (such as continually varying font weights or widths). However, there are certain line control attributes (specified via the ATSLineLayoutOptions flags) that can override style attributes applied to the same text.

Similarly to style attributes, you use a “triple” to specify a line or layout control attribute. That is, an attribute tag, the value of the attribute it sets, and the size (in bytes) of the attribute value. Attribute tags are constants supplied by ATSUI. Attribute values may be a scalar, a structure, or a pointer. And as with style attributes, you can also create a custom attribute for a line or layout attribute for which ATSUI does not provide a tag.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUSetLayoutControls (
   ATSUTextLayout iTextLayout,
   ItemCount iAttributeCount,
   const ATSUAttributeTag *iTag,
   const ByteCount *iValueSize,
   const ATSUAttributeValuePtr *iValue
);

/*ATSUSetAttributes

Sets style attribute values in a style object.

OSStatus ATSUSetAttributes (
   ATSUStyle iStyle,
   ItemCount iAttributeCount,
   const ATSUAttributeTag *iTag,
   const ByteCount *iValueSize,
   const ATSUAttributeValuePtr *iValue
);
Parameters
iStyle

An ATSUStyle value specifying the style object for which to set attributes.
iAttributeCount

An ItemCount value specifying the number of attributes to set. This value should correspond to the number of elements in the iTag and iValueSize arrays.
iTag

A pointer to the initial ATSUAttributeTag value in an array of attribute tags. Each element in the array must contain a valid style attribute tag that corresponds to the style attribute value to set. Note that an attribute tag cannot be used in versions of the Mac OS that are earlier than the version in which the tag was introduced. For example, a tag available in Mac OS version 10.2 cannot be used in Mac OS version 10.1 or earlier. You can call the function Gestalt to check version information for ATSUI. See “Attribute Tags” for a description of the Apple-defined style attribute tag constants and for availability information.
iValueSize

A pointer to the initial ByteCount value in an array of attribute value sizes. Each element in the array must contain the size (in bytes) of the corresponding style run attribute value being set. ATSUSetAttributes sets style attributes after confirming the sizes in the array.
iValue

A pointer to the initial ATSUAttributeValuePtr value in an array of attribute value pointers. Each pointer in the array must reference an attribute value corresponding to a tag in the iTag array. The value referenced by the pointer must be legal for that tag.
Return Value

A result code. See “ATSUI Result Codes”. If there is a function error, ATSUSetAttributes does not set any attributes in the style object.
Discussion

Style attributes are a collection of values and settings that override the font-specified behavior for displaying and formatting text in a style run. To specify a style attribute, ATSUI uses a “triple” consisting of (1) an attribute tag, (2) a value for that tag, and (3) the size of the value.

The ATSUSetAttributes function enables you to set multiple style attribute values for a style object. When you call ATSUSetAttributes, any style attributes that you do not set retain their previous values. To set font features and font variations, call the functions ATSUSetFontFeatures and ATSUSetVariations, respectively.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/

OSStatus ATSUSetAttributes (
   ATSUStyle iStyle,
   ItemCount iAttributeCount,
   const ATSUAttributeTag *iTag,
   const ByteCount *iValueSize,
   const ATSUAttributeValuePtr *iValue
);

/*ATSUCreateStyle

Creates an opaque style object containing only default style attributes, font features, and font variations.

OSStatus ATSUCreateStyle (
   ATSUStyle *oStyle
);
Parameters
oStyle

A pointer to an ATSUStyle value. On return, the pointer refers to an empty style object.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUCreateStyle function creates a style object containing only default values for style attributes, font features, and font variations. The default values for the font features and variations are assigned by the font. The default style attribute values are described in “Attribute Tags”.

To make changes to the default style attribute values, you can call the function ATSUSetAttributes. To set font features and font variations, call the functions ATSUSetFontFeatures and ATSUSetVariations, respectively. You can also use the function ATSUCreateAndCopyStyle to create a new style object by copying all the settings from an existing one.

For ATSUI to apply your selected character-style information, you must associate the style object with a text run in a text layout object. A text run consists of one or more characters that are contiguous in memory. If you associate these characters with a distinct style, you define a style run. You can use the function ATSUSetRunStyle to define a style run by associating a style object with a run of text in a text layout object. Or, to create a text layout object and associate style objects with it at the same time, you can call the function ATSUCreateTextLayoutWithTextPtr. In either case, each text run in a text layout object must be assigned a style object, which may or may not differ from other style objects assigned to other text runs in the text layout object.

Style objects are readily reusable and should be cached for later use, if possible. You can create a style object once and then use it for as many text layout objects as appropriate. Style objects are thread-safe starting with ATSUI version 2.3.

Note that you are responsible for disposing of the memory allocated for the style object. However, you should dispose of any text layout objects with which the style object is associated prior to disposing of the style object itself. To dispose of a style object, call the function ATSUDisposeStyle.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUCreateStyle (
   ATSUStyle *oStyle
);

/*ATSUSetTextPointerLocation

Associates text with a text layout object or updates previously associated text.

OSStatus ATSUSetTextPointerLocation (
   ATSUTextLayout iTextLayout,
   ConstUniCharArrayPtr iText,
   UniCharArrayOffset iTextOffset,
   UniCharCount iTextLength,
   UniCharCount iTextTotalLength
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object for which to set text. 
iText

A pointer of type ConstUniCharArrayPtr, referring to a text buffer containing UTF-16–encoded text. ATSUI associates this buffer with the text layout object and analyzes the complete text of the buffer when obtaining the layout context for the current text range. Thus, for paragraph-format text, if you specify a buffer containing less than a complete paragraph, some of ATSUI’s layout results are not guaranteed to be accurate. For example, with a buffer of less than a full paragraph, ATSUI can neither reliably obtain the context for bidirectional processing nor reliably generate accent attachments and ligature formations. 
iTextOffset

A UniCharArrayOffset value specifying the offset from the beginning of the text buffer to the first character of the range to include in the layout. To indicate that the specified text range starts at the beginning of the text buffer, you can pass the constant kATSUFromTextBeginning. To specify the entire text buffer, pass kATSUFromTextBeginning in this parameter and kATSUToTextEnd in the iTextLength parameter. 
iTextLength

A UniCharCount value specifying the length of the text range. Note that iTextOffset + iTextLength must be less than or equal to the value of the iTextTotalLength parameter. If you want the range of text to extend to the end of the text buffer, you can pass the constant kATSUToTextEnd.
iTextTotalLength

A UniCharCount value specifying the length of the entire text buffer. This value should be greater than or equal to the range of text defined by the iTextLength parameter.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

For ATSUI to render your text, you must associate the text with both a text layout object and style information. Some functions, such as ATSUCreateTextLayoutWithTextPtr, create a text layout object and associate text with it concurrently. However, if you use the function ATSUCreateTextLayout to create a text layout object, you must assign text to the object prior to attempting most ATSUI operations.

You can use the function ATSUSetTextPointerLocation or to associate text with a text layout object. When you call this function, you are both assigning a text buffer to a text layout object and specifying the current text subrange within the buffer to include in the layout.

If there is already text associated with a text layout object, calling ATSUSetTextPointerLocation overrides the previously associated text, as well as clearing the object’s layout caches. You would typically only call this function for a text layout object with existing associated text if either (a) both the buffer itself is relocated and a subrange of the buffer’s text is deleted or inserted or (b) when associating an entirely different buffer with a text layout object.

Note that, because ATSUI objects retain state, doing superfluous calling can degrade performance. For example, you could call ATSUSetTextPointerLocation rather than ATSUTextInserted when the user simply inserts a subrange of text within a text buffer, but there would be a performance penalty, as all the layout caches are flushed by ATSUSetTextPointerLocation, rather than just the affected ones.

Similarly, you should not call ATSUSetTextPointerLocation, when an entire text buffer associated with a text layout object is relocated, but no other changes have occurred that would affect the buffer’s current subrange. Instead, you should call ATSUTextMoved, which is a more focused function and therefore more efficient.

After associating text with a text layout object, use ATSUSetRunStyle to associate style information with the text. You can then call the function ATSUDrawText to display the text or a subrange of the text.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUSetTextPointerLocation (
   ATSUTextLayout iTextLayout,
   ConstUniCharArrayPtr iText,
   UniCharArrayOffset iTextOffset,
   UniCharCount iTextLength,
   UniCharCount iTextTotalLength
);

/*ATSUTextInserted

Informs ATSUI of the location and length of a text insertion.

OSStatus ATSUTextInserted (
   ATSUTextLayout iTextLayout,
   UniCharArrayOffset iInsertionLocation,
   UniCharCount iInsertionLength
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object containing the inserted text. 
iInsertionLocation

A UniCharArrayOffset value specifying the offset from the beginning of the text buffer to the memory location of the inserted text. To specify an insertion point at the beginning of the text buffer, you can pass the constant kATSUFromTextBeginning. 
iInsertionLength

A UniCharCount value specifying the length of the inserted text.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

When you call the ATSUTextInserted function to inform ATSUI of a text insertion, it extends the style run containing the insertion point by the amount of the inserted text. If the insertion point is between two style runs, the first style run is extended to include the new text.

The ATSUTextInserted function also extends the total length of the text buffer containing the inserted text by the amount of the inserted text. That is, it shifts the memory location of the text following the inserted text by iInsertionLength. ATSUTextInserted then updates drawing caches.

Note that the ATSUTextInserted function does not change the actual memory location of the inserted text. You are responsible for placing the inserted text into the text buffer at the appropriate location.

The ATSUTextInserted function does not insert style runs or line breaks; to do so, call the functions ATSUSetRunStyle and ATSUSetSoftLineBreak, respectively. Break line operations should be redone after you call ATSUTextInserted.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/

OSStatus ATSUTextInserted (
   ATSUTextLayout iTextLayout,
   UniCharArrayOffset iInsertionLocation,
   UniCharCount iInsertionLength
);

/*ATSUGetAttribute

Obtains a style attribute value for a style object.

OSStatus ATSUGetAttribute (
   ATSUStyle iStyle,
   ATSUAttributeTag iTag,
   ByteCount iExpectedValueSize,
   ATSUAttributeValuePtr oValue,
   ByteCount *oActualValueSize
);
Parameters
iStyle

An ATSUStyle value specifying the style object for which to obtain an attribute value.
iTag

An ATSUAttributeTag constant identifying the attribute value to obtain. See “Attribute Tags” for a description of the Apple-defined style attribute tag constants.
iExpectedValueSize

The expected size (in bytes) of the value to obtain. To determine the size of an application-defined style attribute value, see the Discussion.
oValue

An ATSUAttributeValuePtr value, identifying the memory you have allocated for the attribute value. If you are uncertain of how much memory to allocate, see the Discussion. On return, oValue contains a valid pointer to the actual attribute value.
oActualValueSize

A pointer to a ByteCount value. On return, the value contains the actual size (in bytes) of the attribute value. You should examine this parameter if you are unsure of the size of the attribute value being obtained, as in the case of custom style run attributes.
Return Value

A result code. See “ATSUI Result Codes”. Note that if the attribute value you want to obtain is not set, ATSUGetAttribute produces the default value in the oValue parameter and returns the result code kATSUNotSetErr.
Discussion

The ATSUGetAttribute function obtains the value of a specified style attribute for a given style object.

Before calling ATSUGetAttribute, you should call the function ATSUGetAllAttributes to obtain an array of nondefault style attribute tags and value sizes for the style object. You can then pass ATSUGetAttribute the tag and value size for the attribute value to obtain.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUGetAttribute (
   ATSUStyle iStyle,
   ATSUAttributeTag iTag,
   ByteCount iExpectedValueSize,
   ATSUAttributeValuePtr oValue,
   ByteCount *oActualValueSize
);

/*ATSUGetRunStyle

Obtains style run information for a character offset in a run of text.

OSStatus ATSUGetRunStyle (
   ATSUTextLayout iTextLayout,
   UniCharArrayOffset iOffset,
   ATSUStyle *oStyle,
   UniCharArrayOffset *oRunStart,
   UniCharCount *oRunLength
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object for which to obtain style run information. 
iOffset

A pointer to a UniCharArrayOffset value. This value should specify the offset from the beginning of the text buffer to the character for which to obtain style run information. To specify the beginning of the text buffer, you can pass the constant kATSUFromTextBeginning. 
oStyle

A pointer to an ATSUStyle value. On return, the value specifies the style object assigned to the range of text containing the character at iOffset. Note that if you pass an offset in the iOffset parameter that is at a style run boundary, ATSUGetRunStyle produces style run information for the following, not preceding, style run. 
oRunStart

A pointer to a UniCharArrayOffset value. On return, the value specifies the offset from the beginning of the text buffer to the first character of the style run containing the character at iOffset. Note that the entire style run does not necessarily share the same unset attribute values as the character at iOffset. 
oRunLength

A pointer to a UniCharCount value. On return, the value specifies the length of the style run containing the character at iOffset.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

You can use the ATSUGetRunStyle function to obtain the style object assigned to a given text offset. ATSUGetRunStyle also produces the encompassing text range that shares the style object with the offset.

Note that the style object contains those previously set style attributes, font features, and font variations that are continuous for the range of text that includes the specified text offset. If you want to obtain all shared style information for a style run, including any unset attributes, call the function ATSUGetContinuousAttributes instead.

If only one style run is set in the text layout object, and it does not cover the entire text layout object, ATSUGetRunStyle uses the style run information for the iOffset parameter to set the style run information for the remaining text.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUGetRunStyle (
   ATSUTextLayout iTextLayout,
   UniCharArrayOffset iOffset,
   ATSUStyle *oStyle,
   UniCharArrayOffset *oRunStart,
   UniCharCount *oRunLength
);

/*ATSUSetLineControls

Sets layout control attribute values for a single line in a text layout object.

OSStatus ATSUSetLineControls (
   ATSUTextLayout iTextLayout,
   UniCharArrayOffset iLineStart,
   ItemCount iAttributeCount,
   const ATSUAttributeTag *iTag,
   const ByteCount *iValueSize,
   const ATSUAttributeValuePtr *iValue
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object for which to set line control attribute values. 
iLineStart

A UniCharArrayOffset value specifying the offset from the beginning of the text layout object’s text buffer to the first character of the line for which to set control attributes. 
iAttributeCount

An ItemCount value specifying the number of attributes to set. This value should correspond to the number of elements in the iTag and iValueSize arrays.
iTag

A pointer to the initial ATSUAttributeTag value in an array of line control attribute tags. Each element in the array must contain a valid tag that corresponds to the line control attribute to set. See “Attribute Tags” for a description of the Apple-defined line control attribute tag constants. 
iValueSize

A pointer to the initial ByteCount value in an array of attribute value sizes. Each element in the array must contain the size (in bytes) of the corresponding line control attribute being set. ATSUSetLineControls sets line attributes after confirming the sizes in the array.
iValue

A pointer to the initial ATSUAttributeValuePtr value in an array of attribute value pointers. Each value in the array must correspond to a tag in the iTag array and be a legal value for that tag. 
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

When you use ATSUI to image your text, you can control the text’s display and formatting at a number of different levels. One level is that of the entire text range associated with your text layout object, also known as the “layout level.” To affect display and formatting on this level, you can specify various layout control attributes using the ATSUSetLayoutControls function. These attributes affect the width of the text area from margin to margin, the alignment of the text, its justification, rotation, and direction, as well as other layout options.

Another level is that of a single line of text, that is, the “line level.” To affect display and formatting on the line level, you specify various line control attributes using the function ATSUSetLineControls. These attributes are similar to those that you can apply on a full-layout basis, but each affects only an individual text line.

You can break text into lines by calling the functions ATSUBatchBreakLines or ATSUBreakLine. You can define separate lines of text by specifying soft breaks either by 
calling the function ATSUBatchBreakLines
calling the function ATSUBreakLine with the iUseAsSoftBreak parameter set to true
specifying the soft line breaks using the function ATSUSetSoftLineBreak

Given that ATSUI allows you to control similar aspects of the display and formatting of your text at either the line level or the layout level (or both, or neither), it is up to you to decide how much layout control to take. However, you should note the following:
Setting layout control attributes overrides the corresponding default layout-level settings for a text layout object. Any layout attributes that you do not set retain the default values described in “Attribute Tags”.
Setting line control attributes overrides the corresponding layout-level settings (whether set or at default values) for a text layout object. This is true even if you set the layout-level attributes subsequently to the line-level ones. Any line attributes that you do not set retain their default values.
From a performance standpoint, it is preferable to work from the layout level and not specify layout line by line unless necessary.

Finally, it is also possible to control the display and formatting of your text at the level of an individual character or “run” of characters. At this level, you customize layout by manipulating style settings in a style object. Among the character-level aspects you can control are style attributes (such as font size and color), font features (such as ligatures), and font variations (such as continually varying font weights or widths). However, there are certain line control attributes (specified via the ATSLineLayoutOptions flags) that can override style attributes applied to the same text.

Similarly to style attributes, you use a “triple” to specify a line or layout control attribute. That is, an attribute tag, the value of the attribute it sets, and the size (in bytes) of the attribute value. Attribute tags are constants supplied by ATSUI. Attribute values may be a scalar, a structure, or a pointer. And as with style attributes, you can also create a custom attribute for a line or layout attribute for which ATSUI does not provide a tag.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/

OSStatus ATSUSetLineControls (
   ATSUTextLayout iTextLayout,
   UniCharArrayOffset iLineStart,
   ItemCount iAttributeCount,
   const ATSUAttributeTag *iTag,
   const ByteCount *iValueSize,
   const ATSUAttributeValuePtr *iValue
);

/*ATSUGetLayoutControl

Obtains a layout control attribute value for a text layout object.

OSStatus ATSUGetLayoutControl (
   ATSUTextLayout iTextLayout,
   ATSUAttributeTag iTag,
   ByteCount iExpectedValueSize,
   ATSUAttributeValuePtr oValue,
   ByteCount *oActualValueSize
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object for which to obtain a layout control attribute value. 
iTag

An ATSUAttributeTag constant identifying the attribute value to obtain. See “Attribute Tags” for a description of the Apple-defined attribute tag constants. 
iExpectedValueSize

The expected size (in bytes) of the value to obtain. To determine the size of an application-defined style attribute value, see the Discussion.
oValue

An ATSUAttributeValuePtr pointer, identifying the memory you have allocated for the attribute value. If you are uncertain of how much memory to allocate, see the Discussion. On return, oValue contains a valid pointer to the actual attribute value. If the value is unset, ATSUGetLayoutControl produces the default value in this parameter. 
oActualValueSize

A pointer to a ByteCount value. On return, the value contains the actual size (in bytes) of the attribute value. You should examine this parameter if you are unsure of the size of the attribute value being obtained, as in the case of custom layout control attributes.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUGetLayoutControl function obtains the value of a specified layout control attribute for a given text layout object.

Before calling ATSUGetLayoutControl, you should call the function ATSUGetAllLayoutControls to obtain an array of nondefault layout control attribute tags and value sizes for the text layout object. You can then pass the tag and value size for the attribute value to obtain to ATSUGetLayoutControl.

Typically you use the function ATSUGetLayoutControl by calling it twice, as follows:
Pass a reference to the text layout object to examine in the iTextLayout parameter, NULL for the oValue parameter, 0 for the iExpectedValueSize parameter. ATSUGetLayoutControl returns the actual size of the attribute value in the oActualValueSize parameter.
Allocate enough space for an array of the returned size, then call the ATSUGetLayoutControl function again, passing a valid pointer in the oValue parameter. On return, the pointer refers to the actual attribute value contained in the text layout object.

To obtain the value of a line control attribute value for a text layout object, call the function ATSUGetLineControl.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUGetLayoutControl (
   ATSUTextLayout iTextLayout,
   ATSUAttributeTag iTag,
   ByteCount iExpectedValueSize,
   ATSUAttributeValuePtr oValue,
   ByteCount *oActualValueSize
);

/*ATSUGetLineControl

Obtains a line control attribute value for a line in a text layout object.

OSStatus ATSUGetLineControl (
   ATSUTextLayout iTextLayout,
   UniCharArrayOffset iLineStart,
   ATSUAttributeTag iTag,
   ByteCount iExpectedValueSize,
   ATSUAttributeValuePtr oValue,
   ByteCount *oActualValueSize
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object for which to obtain a line control attribute value. 
iLineStart

A UniCharArrayOffset value specifying the offset from the beginning of the text layout object’s text buffer to the first character of the line for which to obtain a line control attribute value. 
iTag

An ATSUAttributeTag constant identifying the attribute value to obtain. See “Attribute Tags” for a description of the Apple-defined attribute tag constants. 
iExpectedValueSize

The expected size (in bytes) of the value to obtain.
oValue

An ATSUAttributeValuePtr pointer, identifying the memory you have allocated for the attribute value. If you are uncertain of how much memory to allocate, see the Discussion. On return, oValue contains a valid pointer to the actual attribute value. If the value is unset, ATSUGetLineControl produces the default value in this parameter. 
oActualValueSize

A pointer to a ByteCount value. On return, the value contains the actual size (in bytes) of the attribute value. You should examine this parameter if you are unsure of the size of the attribute value being obtained, as in the case of custom line control attributes.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUGetLineControl function obtains the value of a specified line control attribute for a given line of text in a text layout object.

Before calling ATSUGetLineControl, you should call the function ATSUGetAllLineControls to obtain an array of nondefault line control attribute tags and value sizes for the line. You can then pass the tag and value size for the attribute value to obtain to ATSUGetLineControl.

To obtain the value of a layout control attribute value for a text layout object, call the function ATSUGetLayoutControl.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUGetLineControl (
   ATSUTextLayout iTextLayout,
   UniCharArrayOffset iLineStart,
   ATSUAttributeTag iTag,
   ByteCount iExpectedValueSize,
   ATSUAttributeValuePtr oValue,
   ByteCount *oActualValueSize
);

/*ATSUClearStyle

Restores default values to a style object.

OSStatus ATSUClearStyle (
   ATSUStyle iStyle
);
Parameters
iStyle

An ATSUStyle value specifying the style object for which to restore default values.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUClearStyle function clears a style object of all style attributes (including any application-defined attributes), font features, and font variations and returns these values to their default settings. Default font variations and font features are defined by the font; default style attribute values are described in “Attribute Tags”. ATSUClearStyle does not remove reference constants.

To restore only default style attributes to a style object, you should call the function ATSUClearAttributes. To restore only font variations to a style object, call ATSUClearFontVariations. To restore only font features, call ATSUClearFontFeatures.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUClearStyle (
   ATSUStyle iStyle
);

/*ATSUGetTextLocation

Obtains information about the text associated with a text layout object.

OSStatus ATSUGetTextLocation (
   ATSUTextLayout iTextLayout,
   void **oText,
   Boolean *oTextIsStoredInHandle,
   UniCharArrayOffset *oOffset,
   UniCharCount *oTextLength,
   UniCharCount *oTextTotalLength
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object to examine. 
oText

A pointer to data of any type. On return, the pointer is set to either a pointer or a handle that refers to the text buffer for the specified text layout object.
oTextIsStoredInHandle

A pointer to a Boolean value. On return, the value is set to true if the text buffer in the oText parameter is accessed by a handle; if false, a pointer.
oOffset

A pointer to a UniCharArrayOffset value. On return, the value specifies the offset from the beginning of the text buffer to the first character of the layout’s current text range. 
oTextLength

A pointer to a UniCharCount value. On return, the value specifies the length of the text range.
oTextTotalLength

A pointer to a UniCharCount value. On return, the value specifies the length of the entire text buffer.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

When you call the ATSUGetTextLocation function for a given text layout object, ATSUI obtains the location of the text layout object’s associated text in physical memory, the length of the text range and its text buffer, and whether the text is accessed by a pointer or handle.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUGetTextLocation (
   ATSUTextLayout iTextLayout,
   void **oText,
   Boolean *oTextIsStoredInHandle,
   UniCharArrayOffset *oOffset,
   UniCharCount *oTextLength,
   UniCharCount *oTextTotalLength
);

/*ATSUCreateAndCopyTextLayout

Creates a copy of a text layout object.

OSStatus ATSUCreateAndCopyTextLayout (
   ATSUTextLayout iTextLayout,
   ATSUTextLayout *oTextLayout
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object to copy. 
oTextLayout

A pointer to an ATSUTextLayout value. On return, the pointer refers to a newly created text layout object containing the contents of the text layout object in the iTextLayout parameter. 
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUCreateAndCopyTextLayout function creates a copy of the source text layout object’s style runs (including references to the associated text buffer and style objects), line attributes, layout attributes, and layout caches. ATSUCreateAndCopyTextLayout does not copy reference constants.

To create a text layout object without copying a source object, you can the function ATSUCreateTextLayout or the function ATSUCreateTextLayoutWithTextPtr.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUCreateAndCopyTextLayout (
   ATSUTextLayout iTextLayout,
   ATSUTextLayout *oTextLayout
);

/*ATSUSetRunStyle

Defines a style run by associating style information with a run of text.

OSStatus ATSUSetRunStyle (
   ATSUTextLayout iTextLayout,
   ATSUStyle iStyle,
   UniCharArrayOffset iRunStart,
   UniCharCount iRunLength
);
Parameters
iTextLayout

An ATSUTextLayout value specifying a text layout object with an associated text buffer. ATSUSetRunStyle assigns a style object to a run of text in this buffer. 
iStyle

An ATSUStyle value specifying the style object to associate with the text run. 
iRunStart

A UniCharArrayOffset value specifying the offset from the beginning of the text buffer to the first character of the text run. 
iRunLength

A UniCharCount value specifying the length of the text run.
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

A text run consists of one or more characters that are contiguous in memory. If you associate these characters with a distinct style, you define a style run. You can use the ATSUSetRunStyle function to define a style run, by associating a style object with a run of text in a text layout object. There is a limit of 64K different styles for each ATSUI text layout object. Each text run must be assigned its own style object, which may or may not differ from other style objects assigned to other text runs in a given text layout object. 

You can create a new style object containing only default settings by calling the function ATSUCreateStyle. To make changes to the default style attributes, you can call the function ATSUSetAttributes. To set font features and font variations, call the functions ATSUSetFontFeatures and ATSUSetVariations, respectively.

Note that if you call ATSUSetRunStyle on a text run that is already associated with a style object, the style set by ATSUSetRunStyle overrides the previous style. Additionally, upon completion, ATSUSetRunStyle adjusts the lengths of any style runs on either side of the affected style run.

For example, you may currently have a run of text, 40 characters long, that is assigned a single style, styleA. If you call ATSUSetRunStyle, you can reassign characters at offset 10–29 to a new style, styleB. If you do so, you would then have three style runs, where there once was one: characters at offset 0–9 (styleA), 10–29 (styleB), and 30–39 (styleA).

After calling ATSUSetRunStyle, you can call the function ATSUDrawText to display the styled text. When you call ATSUDrawText, if you have not previously assigned styles to all the characters you request to be drawn, ATSUI automatically does so. Specifically, ATSUI extends the first style it locates immediately prior (in storage order) to the unstyled characters to include those unassigned characters. If the unstyled characters are at the beginning of the text stream, ATSUI finds the first style run in the stream and extends it backward to the first character.

You should call ATSUSetRunStyle whenever you create a new text layout object without any associated styles, as by using the function ATSUCreateTextLayout. You should also call ATSUSetRunStyle to assign a style to a text run in response to a user action, such as when the user selects a run of text and changes the font.

You do not need to call ATSUSetRunStyle when you change style attributes or text layout attributes. In such cases, ATSUI automatically updates the layout of the text as appropriate.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUSetRunStyle (
   ATSUTextLayout iTextLayout,
   ATSUStyle iStyle,
   UniCharArrayOffset iRunStart,
   UniCharCount iRunLength
);

/*ATSUCopyLineControls

Copies line control attribute settings from a line in a source text layout object to a line in a destination text layout object. 

OSStatus ATSUCopyLineControls (
   ATSUTextLayout iSourceTextLayout,
   UniCharArrayOffset iSourceLineStart,
   ATSUTextLayout iDestTextLayout,
   UniCharArrayOffset iDestLineStart
);
Parameters
iSourceTextLayout

An ATSUTextLayout value specifying the text layout object from which to copy line control attributes. 
iSourceLineStart

A UniCharArrayOffset value specifying the offset from the beginning of the text layout object’s text buffer to the first character of the line from which to copy control attributes. 
iDestTextLayout

An ATSUTextLayout value specifying the text layout object for which to set line control attributes. This can be the same text layout object passed in the iSourceTextLayout parameter if you want to copy line control attributes from one line to another within a text layout object. 
iDestLineStart

A UniCharArrayOffset value specifying the offset from the beginning of the text layout object’s text buffer to the first character of the line for which to set control attributes. 
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUCopyLineControls function copies all line control attribute values to a line in a destination text layout object from a line in a source text layout object, including any default (unset) values in the source line. Unset line control attributes are assigned the default values described in “Attribute Tags”. 

ATSUCopyLineControls does not copy the contents of memory referenced by pointers within custom line attributes or within reference constants. You are responsible for ensuring that this memory remains valid until the source text layout object is disposed.

To copy layout control attributes from one text layout object to another, call the function ATSUCopyLayoutControls.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUCopyLineControls (
   ATSUTextLayout iSourceTextLayout,
   UniCharArrayOffset iSourceLineStart,
   ATSUTextLayout iDestTextLayout,
   UniCharArrayOffset iDestLineStart
);

/*ATSUCopyLayoutControls

Copies all layout control attribute settings from a source text layout object to a destination text layout object.

OSStatus ATSUCopyLayoutControls (
   ATSUTextLayout iSourceTextLayout,
   ATSUTextLayout iDestTextLayout
);
Parameters
iSourceTextLayout

An ATSUTextLayout value specifying the text layout object from which to copy layout control attributes. 
iDestTextLayout

An ATSUTextLayout value specifying the text layout object for which to set layout control attributes. 
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUCopyLayoutControls function copies all layout control attribute values to a destination text layout object from a source text layout object, including any default (unset) values in the source object. Default values for unset layout control attributes are described in “Attribute Tags”. 

ATSUCopyLayoutControls does not copy the contents of memory referenced by pointers within custom layout attributes or within reference constants. You are responsible for ensuring that this memory remains valid until both the source and destination text layout objects are disposed.

To copy line control attribute values from one text layout object to another, call the function ATSUCopyLineControls.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUCopyLayoutControls (
   ATSUTextLayout iSourceTextLayout,
   ATSUTextLayout iDestTextLayout
);

/*ATSUClearLayoutControls

Restores default values to the specified layout control attributes of a text layout object.

OSStatus ATSUClearLayoutControls (
   ATSUTextLayout iTextLayout,
   ItemCount iTagCount,
   const ATSUAttributeTag *iTag
);
Parameters
iTextLayout

An ATSUTextLayout value specifying the text layout object for which to restore default layout control attribute values. 
iTagCount

An ItemCount value specifying the number of layout control attributes to restore to default values. This value should correspond to the number of elements in the iTag array. To restore all layout control attributes in the specified text layout object, pass the constant kATSUClearAll in this parameter. In this case, the value in the iTag parameter is ignored.
iTag

A pointer to the initial ATSUAttributeTag constant in an array of attribute tags. Each tag should identify a layout control attribute to restore to its default value. See “Attribute Tags” for a description of the Apple-defined layout control attribute tag constants. 
Return Value

A result code. See “ATSUI Result Codes”.
Discussion

The ATSUClearLayoutControls function removes those layout control attribute values identified by the tag constants in the iTag array and replaces them with the default values described in “Attribute Tags”. If you specify that any currently unset attribute values be removed, the function does not return an error.

To restore default values to line control attributes in a text layout object, call the function ATSUClearLineControls.
Availability
Available in Mac OS X v10.0 and later.
Declared In
ATSUnicodeObjects.h*/
OSStatus ATSUClearLayoutControls (
   ATSUTextLayout iTextLayout,
   ItemCount iTagCount,
   const ATSUAttributeTag *iTag
);

#ifdef __cplusplus
}
#endif

#endif
