
#ifndef AUTHORIZATION_H
#define AUTHORIZATION_H

#include <CarbonCore/MacTypes.h>
#include <stdio.h>

#if defined(__cplusplus)
extern "C" {
#endif

/*Empty Environment

Defines an empty environment.

#define kAuthorizationEmptyEnvironment NULL
Constants
kAuthorizationEmptyEnvironment

Indicates an empty environment. You should pass this constant in functions with an environment parameter if you have no environment data to provide.

Available in Mac OS X v10.0 and later.*/
#define kAuthorizationEmptyEnvironment NULL


/*Authorization Options

Define valid authorization options.

   
enum {
   kAuthorizationFlagDefaults = 0,
   kAuthorizationFlagInteractionAllowed = (1 << 0),
   kAuthorizationFlagExtendRights = (1 << 1),
   kAuthorizationFlagPartialRights = (1 << 2),
   kAuthorizationFlagDestroyRights = (1 << 3),
   kAuthorizationFlagPreAuthorize = (1 << 4),
   kAuthorizationFlagNoData = (1 << 20)
};
Constants
kAuthorizationFlagDefaults

If no bits are set, none of the following features are available.

Available in Mac OS X v10.0 and later.
kAuthorizationFlagInteractionAllowed

If the bit specified by this mask is set, you permit the Security Server to interact with the user when necessary.

Available in Mac OS X v10.0 and later.
kAuthorizationFlagExtendRights

If the bit specified by this mask is set, the Security Server attempts to grant the rights requested. Once the Security Server denies one right, it ignores the remaining requested rights. 

Available in Mac OS X v10.0 and later.
kAuthorizationFlagPartialRights

If the bit specified by this mask and the kAuthorizationFlagExtendRights mask are set, the Security Server grants or denies rights on an individual basis and all rights are checked.

Available in Mac OS X v10.0 and later.
kAuthorizationFlagDestroyRights

If the bit specified by this mask is set, the Security Server revokes authorization from the process as well as from any other process that is sharing the authorization. If the bit specified by this mask is not set, the Security Server revokes authorization from the process but not from other processes that share the authorization.

Available in Mac OS X v10.0 and later.
kAuthorizationFlagPreAuthorize

If the bit specified by this mask is set, the Security Server preauthorizes the rights requested.

Available in Mac OS X v10.0 and later.
kAuthorizationFlagNoData

Private bits. Do not use.

Available in Mac OS X v10.0 and later.
Discussion

The bits represented by these masks instruct the Security Server how to proceed with the function in which you pass them. Set all unused bits to 0 to allow for future expansion.*/
enum {
   kAuthorizationFlagDefaults = 0,
   kAuthorizationFlagInteractionAllowed = (1 << 0),
   kAuthorizationFlagExtendRights = (1 << 1),
   kAuthorizationFlagPartialRights = (1 << 2),
   kAuthorizationFlagDestroyRights = (1 << 3),
   kAuthorizationFlagPreAuthorize = (1 << 4),
   kAuthorizationFlagNoData = (1 << 20)
};


/*AuthorizationFlags

Represents a bit mask of authorization options.

typedef UInt32 AuthorizationFlags;
Discussion

See “Authorization Options” for a description of masks that you can use.
Availability
Available in Mac OS X v10.0 and later.*/
typedef UInt32 AuthorizationFlags;

/*AuthorizationString

Represents a zero-terminated string in UTF-8 encoding.

typedef const char* AuthorizationString;
Availability
Available in Mac OS X v10.0 and later.*/
typedef const char* AuthorizationString;


/*AuthorizationItem

Contains information about an authorization right or the authorization environment.

typedef struct {
AuthorizationString name;
UInt32 valueLength;
void *value;
UInt32 flags;
}AuthorizationItem;
Fields
name

The required name of the authorization right or environment data. The name of a right is something that you create. You should name rights in a style similar to Java package names. For example, "com.myOrganization.myProduct.myRight". Set this field to kAuthorizationRightExecute when requesting a right for use in the function AuthorizationExecuteWithPrivileges.

See the Security.framework/Headers/AuthorizationTags.h header file for possible values for environment data.
valueLength

An unsigned 32-bit integer that represents the number of bytes in the value field. Set the valueLength field to 0 if you set the value field to NULL.
value

A pointer to information pertaining to the name field. For example, if the name field is set to the value represented by the constant kAuthorizationRightExecute, then set the value field to the full POSIX pathname of the tool you want to execute. In most other cases, set this field to NULL.
flags

Reserved option bits. Set to 0. 
Discussion

When using an authorization item to contain a right, set the name field to the name of the right—for example, "com.myOrganization.myProduct.myRight", the valueLength and flags fields to 0, and the value field to NULL. For more information on naming rights, read Performing Privileged Operations With Authorization Services

When using an authorization item for the AuthorizationExecuteWithPrivileges function, set the name field to kAuthorizationRightExecute, and the flags field to 0. Set the value field to the full POSIX pathname of the tool to execute and the valueLength field to the byte length of the value in the value field.

When using an authorization item to contain environment data, set the name field to the name of the environment data—for example, kAuthorizationEnvironmentUsername—and the flags field to 0. Set the value field, in this case, to the actual user name and the valueLength field to the byte length of the value in the value field.*/
typedef struct {
AuthorizationString name;
UInt32 valueLength;
void *value;
UInt32 flags;
}AuthorizationItem;


/*AuthorizationItemSet

Represents a set of authorization items.

typedef struct {
UInt32 count;
AuthorizationItem *items;
}AuthorizationItemSet;
Fields
count

The number of elements in the items array.
items

A pointer to an array of authorization items. If count is greater than 1, items points to the first item in an array of such items. You should set this parameter to NULL if there are no items.*/
typedef struct {
UInt32 count;
AuthorizationItem *items;
}AuthorizationItemSet;


/*AuthorizationEnvironment

Represents a set of data about the environment, such as user name and other information gathered during evaluation of authorization.

typedef AuthorizationItemSet AuthorizationEnvironment;
Availability
Available in Mac OS X v10.0 and later.*/
typedef AuthorizationItemSet AuthorizationEnvironment;




/*AuthorizationRights

Represents a set of authorization rights.

typedef AuthorizationItemSet AuthorizationRights;
Availability
Available in Mac OS X v10.0 and later.*/
typedef AuthorizationItemSet AuthorizationRights;

/*AuthorizationRef

Represents a pointer to an opaque authorization reference structure.

typedef const struct AuthorizationOpaqueRef* AuthorizationRef;
Discussion

This data type points to a structure the Security Server uses to store information about the authorization session. Use the functions described in “Authorization Services Functions” to create, access, and free the authorization reference.
Availability
Available in Mac OS X v10.0 and later.*/
typedef const struct AuthorizationOpaqueRef* AuthorizationRef;

/*Result Codes

The most common result codes returned by the Security Server are listed in the table below.
Result Code	Value	Description
errAuthorizationSuccess	0	
The operation completed successfully.
errAuthorizationInvalidSet	-60001	
The set parameter is invalid.
errAuthorizationInvalidRef	-60002	
The authorization parameter is invalid.
errAuthorizationInvalidTag	-60003	
The tag parameter is invalid.
errAuthorizationInvalidPointer	-60004	
The authorizedRights parameter is invalid.
errAuthorizationDenied	-60005	
The Security Server denied authorization for one or more requested rights. This error is also returned if there was no definition found in the policy database, or a definition could not be created. 
errAuthorizationCanceled	-60006	
The user canceled the operation.
errAuthorizationInteractionNotAllowed	-60007	
The Security Server denied authorization because no user interaction is allowed.
errAuthorizationInternal	-60008	
An unrecognized internal error occurred.
errAuthorizationExternalizeNotAllowed	-60009	
The Security Server denied externalization of the authorization reference.
errAuthorizationInternalizeNotAllowed	-60010	
The Security Server denied internalization of the authorization reference.
errAuthorizationInvalidFlags	-60011	
The flags parameter is invalid.
errAuthorizationToolExecuteFailure	-60031	
The tool failed to execute.
errAuthorizationToolEnvironmentError	-60032	
The attempt to execute the tool failed to return a success or an error code.*/
enum {
errAuthorizationSuccess	=0	,
errAuthorizationInvalidSet	=-60001	,
errAuthorizationInvalidRef	=-60002	,
errAuthorizationInvalidTag=	-60003	,
errAuthorizationInvalidPointer=	-60004	,
errAuthorizationDenied	=-60005	,
errAuthorizationCanceled	=-60006	,
errAuthorizationInteractionNotAllowed=	-60007	,
errAuthorizationInternal	=-60008	,
errAuthorizationExternalizeNotAllowed	=-60009	,
errAuthorizationInternalizeNotAllowed	=-60010	,
errAuthorizationInvalidFlags	=-60011	,
errAuthorizationToolExecuteFailure=	-60031,
errAuthorizationToolEnvironmentError=	-60032
};
//functions

/*AuthorizationCopyRights

Authorizes and preauthorizes rights.

OSStatus AuthorizationCopyRights (
   AuthorizationRef authorization,
   const AuthorizationRights *rights,
   const AuthorizationEnvironment *environment,
   AuthorizationFlags flags,
   AuthorizationRights **authorizedRights
);
Parameters
authorization

An authorization reference referring to the authorization session. 
rights

A pointer to a set of authorization rights you create. Pass NULL if the application requires no rights at this time. 
environment

Data used when authorizing or preauthorizing rights. Not used in Mac OS X v10.2 and earlier. In Mac OS X v10.3 and later, you can pass icon or prompt data to be used in the authentication dialog box. In Mac OS X v10.4 and later, you can also pass a user name and password in order to authorize a user without displaying the authentication dialog box. Possible values for this parameter are listed in Security.framework/Headers/AuthorizationTags.h. The data passed in this parameter is not stored in the authorization reference; it is used only during authorization. If you are not passing any data in this parameter, pass the constant kAuthorizationEmptyEnvironment.
flags

A bit mask for specifying authorization options. Use the following option sets.
Pass the constant kAuthorizationFlagDefaults if no options are necessary.
Specify the kAuthorizationFlagExtendRights mask to request rights. You can also specify the kAuthorizationFlagInteractionAllowed mask to allow user interaction.
Specify the kAuthorizationFlagPartialRights and kAuthorizationFlagExtendRights masks to request partial rights. You can also specify the kAuthorizationFlagInteractionAllowed mask to allow user interaction.
Specify the kAuthorizationFlagPreAuthorize and kAuthorizationFlagExtendRights masks to preauthorize rights.
Specify the kAuthorizationFlagDestroyRights mask to prevent the Security Server from preserving the rights obtained during this call.
authorizedRights

A pointer to a newly allocated AuthorizationRights structure. On return, this structure contains the rights granted by the Security framework. If you do not require this information, pass NULL. If you specify the kAuthorizationFlagPreAuthorize mask in the flags parameter, the method returns all the requested rights, including those not granted, but the flags of the rights that could not be preauthorized include the kAuthorizationFlagCanNotPreAuthorize bit.

Free the memory associated with this set by calling the function AuthorizationFreeItemSet.
Return Value

A result code. See “Authorization Services Result Codes”.
Discussion

There are three main reasons to use this function. The first reason is to preauthorize rights by specifying the kAuthorizationFlagPreAuthorize, kAuthorizationFlagInteractionAllowed, and kAuthorizationFlagExtendRights masks as authorization options. Preauthorization is most useful when a right has a zero timeout. For example, you can preauthorize in the application and if it succeeds, call the helper tool and request authorization. This eliminates calling the helper tool if the Security Server cannot later authorize the specified rights.

The second reason to use this function is to authorize rights before performing a privileged operation by specifying the kAuthorizationFlagInteractionAllowed, and kAuthorizationFlagExtendRights masks as authorization options.

The third reason to use this function is to authorize partial rights. By specifying the kAuthorizationFlagPartialRights, kAuthorizationFlagInteractionAllowed, and kAuthorizationFlagExtendRights masks as authorization options, the Security Server grants all rights it can authorize. On return, the authorized set contains all the rights.

If you do not specify the kAuthorizationFlagPartialRights mask and the Security Server denies at least one right, then the status of this function on return is errAuthorizationDenied.

If you do not specify the kAuthorizationFlagInteractionAllowed mask and the Security Server requires user interaction, then the status of this function on return is errAuthorizationInteractionNotAllowed.

If you specify the kAuthorizationFlagInteractionAllowed mask and the user cancels the authentication process, then the status of this function on return is errAuthorizationCanceled.
Availability
Available in Mac OS X v10.0 and later.*/
OSStatus AuthorizationCopyRights (
   AuthorizationRef authorization,
   const AuthorizationRights *rights,
   const AuthorizationEnvironment *environment,
   AuthorizationFlags flags,
   AuthorizationRights **authorizedRights
);

/*AuthorizationCreate

Creates a new authorization reference and provides an option to authorize or preauthorize rights.

OSStatus AuthorizationCreate (
   const AuthorizationRights *rights,
   const AuthorizationEnvironment *environment,
   AuthorizationFlags flags,
   AuthorizationRef *authorization
);
Parameters
rights

A pointer to a set of authorization rights you create. Pass NULL if the application requires no rights at this time. 
environment

Data used when authorizing or preauthorizing rights. Not used in Mac OS X v10.2 and earlier. In Mac OS X v10.3 and later, you can pass icon or prompt data to be used in the authentication dialog box. In Mac OS X v10.4 and later, you can also pass a user name and password in order to authorize a user without user interaction. Possible values for this parameter are listed in Security.framework/Headers/AuthorizationTags.h. The data passed in this parameter is not stored in the authorization reference; it is used only during authorization. If you are not passing any data in this parameter, pass the constant kAuthorizationEmptyEnvironment. 
flags

A bit mask for specifying authorization options. Use the following option sets.
Pass the constant kAuthorizationFlagDefaults if no options are necessary.
Specify the kAuthorizationFlagExtendRights mask to request rights. You can also specify the kAuthorizationFlagInteractionAllowed mask to allow user interaction.
Specify the kAuthorizationFlagPartialRights and kAuthorizationFlagExtendRights masks to request partial rights. You can also specify the kAuthorizationFlagInteractionAllowed mask to allow user interaction.
Specify the kAuthorizationFlagPreAuthorize and kAuthorizationFlagExtendRights masks to preauthorize rights.
Specify the kAuthorizationFlagDestroyRights mask to prevent the Security Server from preserving the rights obtained during this call.
authorization

A pointer to an authorization reference. On return, this parameter refers to the authorization session the Security Server creates. Pass NULL if you require a function result but no authorization reference.
Return Value

A result code. See “Authorization Services Result Codes”.
Discussion

The primary purpose of this function is to create the opaque authorization reference structure associated with the authorization reference. You use the authorization reference in other authorization functions.

You can use this function to authorize all or partial rights. Authorizing rights with this function is most useful for applications that require a one-time authorization. By passing NULL to the authorization parameter, the Security Server attempts to authorize the requested rights and returns the appropriate result code without actually granting the rights. If you are not going to call any other authorization functions, use this method to determine if a user has authorization without granting any rights.

You can also use this function to preauthorize rights by specifying the kAuthorizationFlagPreAuthorize mask. Preauthorization is most useful when a right has a zero timeout. For example, you can preauthorize in the application and if it succeeds, call the helper tool and request authorization. This eliminates calling the helper tool if the user cannot later authorize the specified rights.

If you do not specify the kAuthorizationFlagPartialRights mask and the Security Server denies at least one right, then the status of this function on return is errAuthorizationDenied.

If you do not specify the kAuthorizationFlagInteractionAllowed mask and the Security Server requires user interaction, then the status of this function on return is errAuthorizationInteractionNotAllowed.

If you specify the kAuthorizationFlagInteractionAllowed mask and the user cancels the authentication process, then the status of this function on return is errAuthorizationCanceled.

When your application no longer needs the authorization reference, use the function AuthorizationFree to free the memory associated with it.
Availability
Available in Mac OS X v10.0 and later.*/
OSStatus AuthorizationCreate (
   const AuthorizationRights *rights,
   const AuthorizationEnvironment *environment,
   AuthorizationFlags flags,
   AuthorizationRef *authorization
);

#if defined(__cplusplus)
}
#endif

#endif 
