#ifndef  DNS_SD_H
#define DNS_SD_H

#include <stdint.h>//hys
#include <APPLE/dns_sd.h>

#ifdef  __cplusplus
    extern "C" {
#endif

/*DNS Record Types

enum { 
    kDNSServiceType_A = 1, 
    kDNSServiceType_NS = 2, 
    kDNSServiceType_MD = 3, 
    kDNSServiceType_MF = 4, 
    kDNSServiceType_CNAME = 5, 
    kDNSServiceType_SOA = 6, 
    kDNSServiceType_MB = 7, 
    kDNSServiceType_MG = 8, 
    kDNSServiceType_MR = 9, 
    kDNSServiceType_NULL = 10, 
    kDNSServiceType_WKS = 11, 
    kDNSServiceType_PTR = 12, 
    kDNSServiceType_HINFO = 13, 
    kDNSServiceType_MINFO = 14, 
    kDNSServiceType_MX = 15, 
    kDNSServiceType_TXT = 16, 
    kDNSServiceType_RP = 17, 
    kDNSServiceType_AFSDB = 18, 
    kDNSServiceType_X25 = 19, 
    kDNSServiceType_ISDN = 20, 
    kDNSServiceType_RT = 21, 
    kDNSServiceType_NSAP = 22, 
    kDNSServiceType_NSAP_PTR = 23, 
    kDNSServiceType_SIG = 24, 
    kDNSServiceType_KEY = 25, 
    kDNSServiceType_PX = 26, 
    kDNSServiceType_GPOS = 27, 
    kDNSServiceType_AAAA = 28, 
    kDNSServiceType_LOC = 29, 
    kDNSServiceType_NXT = 30, 
    kDNSServiceType_EID = 31, 
    kDNSServiceType_NIMLOC = 32, 
    kDNSServiceType_SRV = 33, 
    kDNSServiceType_ATMA = 34, 
    kDNSServiceType_NAPTR = 35, 
    kDNSServiceType_KX = 36, 
    kDNSServiceType_CERT = 37, 
    kDNSServiceType_A6 = 38, 
    kDNSServiceType_DNAME = 39, 
    kDNSServiceType_SINK = 40, 
    kDNSServiceType_OPT = 41, 
    kDNSServiceType_TKEY = 249, 
    kDNSServiceType_TSIG = 250, 
    kDNSServiceType_IXFR = 251, 
    kDNSServiceType_AXFR = 252, 
    kDNSServiceType_MAILB = 253, 
    kDNSServiceType_MAILA = 254, 
    kDNSServiceType_ANY = 255 
};  
Constants
kDNSServiceType_A
Host address.
kDNSServiceType_NS
Authoritative server.
kDNSServiceType_MD
Mail destination.
kDNSServiceType_MF
Mail forwarder.
kDNSServiceType_CNAME
Canonical name.
kDNSServiceType_SOA
Start of authority zone.
kDNSServiceType_MB
Mailbox domain name.
kDNSServiceType_MG
Mail group member.
kDNSServiceType_MR
Mail rename name.
kDNSServiceType_NULL
Null resource record.
kDNSServiceType_WKS
Well known service.
kDNSServiceType_PTR
Domain name pointer.
kDNSServiceType_HINFO
Host information.
kDNSServiceType_MINFO
Mailbox information.
kDNSServiceType_MX
Mail routing information.
kDNSServiceType_TXT
Text strings.
kDNSServiceType_RP
Responsible person.
kDNSServiceType_AFSDB
AFS cell database.
kDNSServiceType_X25
X_25 calling address.
kDNSServiceType_ISDN
ISDN calling address.
kDNSServiceType_RT
Router.
kDNSServiceType_NSAP
NSAP address.
kDNSServiceType_NSAP_PTR
Reverse NSAP lookup (deprecated).
kDNSServiceType_SIG
Security signature.
kDNSServiceType_KEY
Security key.
kDNSServiceType_PX
X.400 mail mapping.
kDNSServiceType_GPOS
Geographical position (withdrawn).
kDNSServiceType_AAAA
Ip6 Address.
kDNSServiceType_LOC
Location Information.
kDNSServiceType_NXT
Next domain (security).
kDNSServiceType_EID
Endpoint identifier.
kDNSServiceType_NIMLOC
Nimrod Locator.
kDNSServiceType_SRV
Server Selection.
kDNSServiceType_ATMA
ATM Address
kDNSServiceType_NAPTR
Naming Authority PoinTeR
kDNSServiceType_KX
Key Exchange
kDNSServiceType_CERT
Certification record
kDNSServiceType_A6
IPv6 address (deprecates AAAA)
kDNSServiceType_DNAME
Non-terminal DNAME (for IPv6)
kDNSServiceType_SINK
Kitchen sink (experimentatl)
kDNSServiceType_OPT
EDNS0 option (meta-RR)
kDNSServiceType_TKEY
Transaction key
kDNSServiceType_TSIG
Transaction signature.
kDNSServiceType_IXFR
Incremental zone transfer.
kDNSServiceType_AXFR
Transfer zone of authority.
kDNSServiceType_MAILB
Transfer mailbox records.
kDNSServiceType_MAILA
Transfer mail agent records.
kDNSServiceType_ANY
Wildcard match.
Discussion 

RFC 1035 lists constants for DNS types, and each operating system makes the types available in its DNS header file, but not all operating systems have the same DNS header file containing the same constants, and the names of constants are not consistent. For example, BIND 8 uses T_A, BIND 9 uses ns_t_a, and Windows uses DNS_TYPE_A. DNS Service Discovery uses the names enumerated in this section so that the same code compiles on all supported platforms. 

Record types are also defined at http://www.iana.org/assignments/dns-parameters.*/
enum { 
    kDNSServiceType_A = 1, 
    kDNSServiceType_NS = 2, 
    kDNSServiceType_MD = 3, 
    kDNSServiceType_MF = 4, 
    kDNSServiceType_CNAME = 5, 
    kDNSServiceType_SOA = 6, 
    kDNSServiceType_MB = 7, 
    kDNSServiceType_MG = 8, 
    kDNSServiceType_MR = 9, 
    kDNSServiceType_NULL = 10, 
    kDNSServiceType_WKS = 11, 
    kDNSServiceType_PTR = 12, 
    kDNSServiceType_HINFO = 13, 
    kDNSServiceType_MINFO = 14, 
    kDNSServiceType_MX = 15, 
    kDNSServiceType_TXT = 16, 
    kDNSServiceType_RP = 17, 
    kDNSServiceType_AFSDB = 18, 
    kDNSServiceType_X25 = 19, 
    kDNSServiceType_ISDN = 20, 
    kDNSServiceType_RT = 21, 
    kDNSServiceType_NSAP = 22, 
    kDNSServiceType_NSAP_PTR = 23, 
    kDNSServiceType_SIG = 24, 
    kDNSServiceType_KEY = 25, 
    kDNSServiceType_PX = 26, 
    kDNSServiceType_GPOS = 27, 
    kDNSServiceType_AAAA = 28, 
    kDNSServiceType_LOC = 29, 
    kDNSServiceType_NXT = 30, 
    kDNSServiceType_EID = 31, 
    kDNSServiceType_NIMLOC = 32, 
    kDNSServiceType_SRV = 33, 
    kDNSServiceType_ATMA = 34, 
    kDNSServiceType_NAPTR = 35, 
    kDNSServiceType_KX = 36, 
    kDNSServiceType_CERT = 37, 
    kDNSServiceType_A6 = 38, 
    kDNSServiceType_DNAME = 39, 
    kDNSServiceType_SINK = 40, 
    kDNSServiceType_OPT = 41, 
    kDNSServiceType_TKEY = 249, 
    kDNSServiceType_TSIG = 250, 
    kDNSServiceType_IXFR = 251, 
    kDNSServiceType_AXFR = 252, 
    kDNSServiceType_MAILB = 253, 
    kDNSServiceType_MAILA = 254, 
    kDNSServiceType_ANY = 255 
};  


/*(Flag Constants

enum { 
    kDNSServiceFlagsMoreComing = 0x1, 
    kDNSServiceFlagsAdd = 0x2, 
    kDNSServiceFlagsDefault = 0x4, 
    kDNSServiceFlagsNoAutoRename = 0x8, 
    kDNSServiceFlagsShared = 0x10, 
    kDNSServiceFlagsUnique = 0x20, 
    kDNSServiceFlagsBrowseDomains = 0x40, 
    kDNSServiceFlagsRegistrationDomains = 0x80, 
    kDNSServiceFlagsLongLivedQuery = 0x100, 
    kDNSServiceFlagsAllowRemoteQuery = 0x200, 
    kDNSServiceFlagsForceMulticast = 0x400 
};  
Constants
kDNSServiceFlagsMoreComing
MoreComing indicates to a callback that at least one more result is queued and will be delivered following immediately after this one. Applications should not update their UI to display browse results when the MoreComing flag is set, because this would result in a great deal of ugly flickering on the screen. Applications should instead wait until until MoreComing is not set, and then update their UI. When MoreComing is not set, that doesn't mean there will be no more answers EVER, just that there are no more answers immediately available right now at this instant. If more answers become available in the future they will be delivered as usual. 

kDNSServiceFlagsAdd
kDNSServiceFlagsAdd is set in the flags parameter and passed to your DNSServiceBrowseReply, DNSServiceDomainEnumReply, DNSServiceQueryRecordReply callback functions to indicate that a domain should be added to the list of domain names that your application maintains. A flags parameter in which kDNSServiceFlagsAdd is not set indicates that the service, domain, or record is no longer available and should be removed from the list that your application maintains. 

kDNSServiceFlagsDefault
kDNSServiceFlagsDefault is set in the flags parameter and passed to your DNSServiceDomainEnumReply callback function to indicate that the enumerated domain is a default domain. When kDNSServiceFlagsDefault is set, kDNSServiceFlagsAdd is also set. 

kDNSServiceFlagsNoAutoRename
Flag for specifying renaming behavior on name conflict when registering non-shared records. By default, name conflicts are automatically handled by renaming the service. NoAutoRename overrides this behavior - with this flag set, name conflicts will result in a callback. The NoAutorename flag is only valid if a name is explicitly specified when registering a service (ie the default name is not used.) 

kDNSServiceFlagsShared
kDNSServiceFlagsShared is set in the flags parameter and passed to DNSServiceRegisterRecord when registering a resource record over a connection with the mDNSResponder daemon. Setting kDNSServiceFlagsShared indicates that multiple records having the same full domain name on the network are allowed. For example, you would set kDNSServiceFlagsShared when registering PTR records that have the same full domain name. 

kDNSServiceFlagsUnique
kDNSServiceFlagsUnique is set in the flags parameter and passed to DNSServiceRegisterRecord when registering a resource record whose name must be unique on the network. For example, you would set kDNSServiceFlagsUnique when registering a DNS-SD SRV record. 

kDNSServiceFlagsBrowseDomains
kDNSServiceFlagsBrowseDomains is set in the flags parameter and passed to DNSServiceEnumerateDomains to indicate that domains recommended for browsing are to be enumerated. 

kDNSServiceFlagsRegistrationDomains
kDNSServiceFlagsRegistrationDomains is set in the flags parameter and passed to DNSServiceEnumerateDomains to indicate that domains recommended for registration are to be enumerated. 

kDNSServiceFlagsLongLivedQuery
kDNSServiceFlagsLongLivedQuery is set in the flags parameter and passed to DNSServiceQueryRecord in order to create a long-lived unicast query in a non-local domain. When this bit is not set, unicast queries are "one-shot"; that is, only responses that are available at the time of the call are returned. When the kDNSServiceFlagsLongLivedQuery bit is set, your DNSServiceQueryRecordReply callback function will be called to deliver responses for add and remove events that occur after the initial call. You are always responsible for freeing memory and resources associated with the service discovery reference and terminating the query. This flag has no effect on link-local multicast queries. 

kDNSServiceFlagsAllowRemoteQuery
kDNSServiceFlagsAllowRemoteQuery is set in the flags parameter and passed to DNSServiceRegister and DNSServiceRegisterRecord to create a record for which mDNSResponder will respond to remote queries from hosts that are not directly connected to the local link (that is, more than one hop away). 

kDNSServiceFlagsForceMulticast
kDNSServiceFlagsForceMulticast is set in the flags parameter and passed to DNSServiceQueryRecord and DNSServiceRegisterRecord to specify that a query or registration is to be performed by multicast DNS, even for a name in a domain that would normally be performed by unicast DNS.
Discussion 

These constants are used to set bits in the flags parameter to many DNS Service Discovery functions and callbacks.*/
enum { 
    kDNSServiceFlagsMoreComing = 0x1, 
    kDNSServiceFlagsAdd = 0x2, 
    kDNSServiceFlagsDefault = 0x4, 
    kDNSServiceFlagsNoAutoRename = 0x8, 
    kDNSServiceFlagsShared = 0x10, 
    kDNSServiceFlagsUnique = 0x20, 
    kDNSServiceFlagsBrowseDomains = 0x40, 
    kDNSServiceFlagsRegistrationDomains = 0x80, 
    kDNSServiceFlagsLongLivedQuery = 0x100, 
    kDNSServiceFlagsAllowRemoteQuery = 0x200, 
    kDNSServiceFlagsForceMulticast = 0x400 
};  


/*DNSServiceFlags

typedef uint32_t DNSServiceFlags;  */
typedef uint32_t DNSServiceFlags;  

/*Errors


Possible error code values. 
enum { 
    kDNSServiceErr_NoError = 0, 
    kDNSServiceErr_Unknown = -65537, 
    kDNSServiceErr_NoSuchName = -65538, 
    kDNSServiceErr_NoMemory = -65539, 
    kDNSServiceErr_BadParam = -65540, 
    kDNSServiceErr_BadReference = -65541, 
    kDNSServiceErr_BadState = -65542, 
    kDNSServiceErr_BadFlags = -65543, 
    kDNSServiceErr_Unsupported = -65544, 
    kDNSServiceErr_NotInitialized = -65545, 
    kDNSServiceErr_AlreadyRegistered = -65547, 
    kDNSServiceErr_NameConflict = -65548, 
    kDNSServiceErr_Invalid = -65549, 
    kDNSServiceErr_Firewall = -65550, 
    kDNSServiceErr_Incompatible = -65551, 
    kDNSServiceErr_BadInterfaceIndex = -65552, 
    kDNSServiceErr_Refused = -65553, 
    kDNSServiceErr_NoSuchRecord = -65554, 
    kDNSServiceErr_NoAuth = -65555, 
    kDNSServiceErr_NoSuchKey = -65556, 
    kDNSServiceErr_NATTraversal = -65557, 
    kDNSServiceErr_DoubleNAT = -65558, 
    kDNSServiceErr_BadTime = -65559 
};  
Constants
kDNSServiceErr_NoError
kDNSServiceErr_Unknown
0xFFFE FFFF
kDNSServiceErr_NoSuchName
kDNSServiceErr_NoMemory
kDNSServiceErr_BadParam
kDNSServiceErr_BadReference
kDNSServiceErr_BadState
kDNSServiceErr_BadFlags
kDNSServiceErr_Unsupported
kDNSServiceErr_NotInitialized
kDNSServiceErr_AlreadyRegistered
kDNSServiceErr_NameConflict
kDNSServiceErr_Invalid
kDNSServiceErr_Incompatible
Client library incompatible with daemon.
kDNSServiceErr_BadInterfaceIndex
kDNSServiceErr_Refused
kDNSServiceErr_NoSuchRecord
kDNSServiceErr_NoAuth
Discussion 

mDNS Error codes are in the range FFFE FF00 (-65792) to FFFE FFFF (-65537).*/
enum { 
    kDNSServiceErr_NoError = 0, 
    kDNSServiceErr_Unknown = -65537, 
    kDNSServiceErr_NoSuchName = -65538, 
    kDNSServiceErr_NoMemory = -65539, 
    kDNSServiceErr_BadParam = -65540, 
    kDNSServiceErr_BadReference = -65541, 
    kDNSServiceErr_BadState = -65542, 
    kDNSServiceErr_BadFlags = -65543, 
    kDNSServiceErr_Unsupported = -65544, 
    kDNSServiceErr_NotInitialized = -65545, 
    kDNSServiceErr_AlreadyRegistered = -65547, 
    kDNSServiceErr_NameConflict = -65548, 
    kDNSServiceErr_Invalid = -65549, 
    kDNSServiceErr_Firewall = -65550, 
    kDNSServiceErr_Incompatible = -65551, 
    kDNSServiceErr_BadInterfaceIndex = -65552, 
    kDNSServiceErr_Refused = -65553, 
    kDNSServiceErr_NoSuchRecord = -65554, 
    kDNSServiceErr_NoAuth = -65555, 
    kDNSServiceErr_NoSuchKey = -65556, 
    kDNSServiceErr_NATTraversal = -65557, 
    kDNSServiceErr_DoubleNAT = -65558, 
    kDNSServiceErr_BadTime = -65559 
};  


/*DNSServiceErrorType

typedef int32_t DNSServiceErrorType;  */
typedef int32_t DNSServiceErrorType;


/*DNSServiceRef

typedef struct _DNSServiceRef_t *DNSServiceRef;  
Discussion 

Opaque internal data type. Note: client is responsible for serializing access to these structures if they are shared between concurrent threads.*/
typedef struct _DNSServiceRef_t *DNSServiceRef;  


/*DNSServiceQueryRecordReply


Your application implements this callback function to receive replies from the mDNSResponder daemon in response to calling DNSServiceQueryRecord.
typedef void ( *DNSServiceQueryRecordReply) ( 
    DNSServiceRef DNSServiceRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    DNSServiceErrorType errorCode, 
    const char *fullname, 
    uint16_t rrtype, 
    uint16_t rrclass, 
    uint16_t rdlen, 
    const void *rdata, 
    uint32_t ttl, 
    void *context );  
Parameters
DNSServiceRef
The DNSServiceRef initialized by DNSServiceQueryRecord(). 

flags
Possible values are:
kDNSServiceFlagsMoreComing 
kDNSServiceFlagsAdd


interfaceIndex
The interface on which the query was resolved (the index for a given interface is determined via the if_nametoindex() family of calls). 

errorCode
Will be kDNSServiceErr_NoError on success, otherwise will indicate the failure that occurred. Other parameters are undefined if errorCode is non-zero. 

fullname
The resource record's full domain name. 

rrtype
The resource record's type (e.g. PTR, SRV,...) as defined in nameser.h. 

rrclass
The class of the resource record, as defined in nameser.h (usually 1). 

rdlen
The length, in bytes, of the resource record rdata. 

rdata
The raw rdata of the resource record. 

ttl
The resource record's time to live, in seconds. 

context
The context pointer that was passed to the callout. 

Discussion 

This is a prototype for your callback function that handles replies from DNSServiceQueryRecord. If a matching record was found, your callback function will be called and passed the matching record's resource record data, the length of that data in bytes, and the record's time-to-live value. 

If kDNSServiceMoreComing is set in flags, your callback function will be called with additional query results within the next 50 milliseconds. Applications should not update their user interface with search results until their callback function is called with a value of flags in which kDNSServiceMoreComing is not set. 

Your application is responsible for stopping the query. Remove sdRef from the run or select() loop, and call DNSServiceRefDeallocate to deallocate sdRef, which stops the query, terminates the connection with the mDNSResponder daemon, and frees the memory associated with sdRef.*/
typedef void ( *DNSServiceQueryRecordReply) ( 
    DNSServiceRef DNSServiceRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    DNSServiceErrorType errorCode, 
    const char *fullname, 
    uint16_t rrtype, 
    uint16_t rrclass, 
    uint16_t rdlen, 
    const void *rdata, 
    uint32_t ttl, 
    void *context );  

/*DNSServiceBrowseReply


Browse for instances of a service.
typedef void ( *DNSServiceBrowseReply) ( 
    DNSServiceRef sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    DNSServiceErrorType errorCode, 
    const char *serviceName, 
    const char *regtype, 
    const char *replyDomain, 
    void *context );  
Parameters
sdRef
The DNSServiceRef initialized by DNSServiceBrowse(). 

flags
Possible values are kDNSServiceFlagsMoreComing and kDNSServiceFlagsAdd. See flag definitions for details. 

interfaceIndex
The interface on which the service is advertised. This index should be passed to DNSServiceResolve() when resolving the service. 

errorCode
Will be kDNSServiceErr_NoError (0) on success, otherwise will indicate the failure that occurred. Other parameters are undefined if the errorCode is nonzero. 

serviceName
The discovered service name. This name should be displayed to the user, and stored for subsequent use in the DNSServiceResolve() call. 

regtype
The service type, which is usually (but not always) the same as was passed to DNSServiceBrowse(). One case where the discovered service type may not be the same as the requested service type is when using subtypes: The client may want to browse for only those ftp servers that allow anonymous connections. The client will pass the string "_ftp._tcp,_anon" to DNSServiceBrowse(), but the type of the service that's discovered is simply "_ftp._tcp". The regtype for each discovered service instance should be stored along with the name, so that it can be passed to DNSServiceResolve() when the service is later resolved. 

replyDomain
The domain of the discovered service instance. This may or may not be the same as the domain that was passed to DNSServiceBrowse(). The domain for each discovered service instance should be stored along with the name, so that it can be passed to DNSServiceResolve() when the service is later resolved. 

context
The context pointer that was passed to the callout. 

Discussion 

DNSServiceBrowseReply is a prototype for your callback function that handles replies from the mDNSResponder daemon made in response to DNSServiceBrowse calls. Your run or select() loop will indicate when the mDNSResponder daemon finds each service of the specified type found in the specified domain (or in the default domain(s) if no domain was specified when DNSServiceBrowse was called). For each response, call DNSServiceProcessResult, which will cause your callback function to be called. 

The mDNSResponder daemon provides information on currently available services, new services that are added to a domain, and services that are removed from a domain. The sum of the information from these calls is the current list of services. Your application is responsible for building and maintaining its own list of services. 

If multiple services are found, the kDNSServiceFlagsMoreComing flag will be set until your callback function has been called for each of them. Do not update the user interface (list of services) while this flag is set, and do not disable the user interface or change the cursor while waiting for data. That way, the list of available services will always be up to date. 

To use a service found by this function, obtain the name of the host that provides the service by calling DNSServiceResolve. 

Your application is responsible for terminating the search. Call DNSServiceRefDeallocate, which terminates the search, terminates the connection with the daemon, and frees the memory associated with sdRef.*/

typedef void ( *DNSServiceBrowseReply) ( 
    DNSServiceRef sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    DNSServiceErrorType errorCode, 
    const char *serviceName, 
    const char *regtype, 
    const char *replyDomain, 
    void *context );  

/*DNSServiceDomainEnumReply


Your application implements this callback function to receive replies from the mDNSResponder daemon in response to calling DNSServiceEnumerateDomains.
typedef void ( *DNSServiceDomainEnumReply) ( 
    DNSServiceRef sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    DNSServiceErrorType errorCode, 
    const char *replyDomain, 
    void *context );  
Parameters
sdRef
The DNSServiceRef initialized by DNSServiceEnumerateDomains(). 

flags
Possible values are: kDNSServiceFlagsMoreComing kDNSServiceFlagsAdd kDNSServiceFlagsDefault 

interfaceIndex
Specifies the interface on which the domain exists. (The index for a given interface is determined via the if_nametoindex() family of calls.) 

errorCode
Will be kDNSServiceErr_NoError (0) on success, otherwise indicates the failure that occurred (other parameters are undefined if errorCode is nonzero). 

replyDomain
The name of the domain. 

context
The context pointer passed to DNSServiceEnumerateDomains. 

Discussion 

DNSServiceDomainEnumReply is a prototype for your callback function that handles replies from the mDNSResponder daemon made in response to DNSServiceEnumerateDomains calls. Your run or select() loop will indicate when the mDNSResponder daemon has a response; it will have a response for every domain that it finds. Each time your run or select() loop indicates that there is a response, call DNSServiceProcessResult, which will cause your callback function to be called. Your run or select() loop will also be called if a domain is added or removed, or if the default domain changes. The sum of these replies is the current list of recommended domains. 

You should leave domain enumeration running the entire time you are browsing for services. To stop domain enumeration, remove sdRef from the run or select() loop, and call DNSServiceRefDeallocate to deallocate sdRef, which terminates the connection with the mDNSResponder daemon and frees the memory associated with sdRef.*/
typedef void ( *DNSServiceDomainEnumReply) ( 
    DNSServiceRef sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    DNSServiceErrorType errorCode, 
    const char *replyDomain, 
    void *context );  

/*DNSRecordRef

typedef struct _DNSRecordRef_t *DNSRecordRef;  
Discussion 

Opaque internal data type. Note: client is responsible for serializing access to these structures if they are shared between concurrent threads.*/
typedef struct _DNSRecordRef_t *DNSRecordRef;  

/*DNSServiceResolveReply


Resolve a service name discovered via DNSServiceBrowse() to a target host name, port number, and txt record.
typedef void ( *DNSServiceResolveReply) ( 
    DNSServiceRef sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    DNSServiceErrorType errorCode, 
    const char *fullname, 
    const char *hosttarget, 
    uint16_t port, 
    uint16_t txtLen, 
    const char *txtRecord, 
    void *context );  
Parameters
sdRef
The DNSServiceRef initialized by DNSServiceResolve(). 

flags
Currently unused, reserved for future use. 

interfaceIndex
The interface on which the service was resolved. 

errorCode
Will be kDNSServiceErr_NoError (0) on success, otherwise will indicate the failure that occurred. Other parameters are undefined if the errorCode is nonzero. 

fullname
The full service domain name, in the form [servicename].[protocol].[domain]. (This name is escaped following standard DNS rules, making it suitable for passing to standard system DNS APIs such as res_query(), or to the special-purpose functions included in this API that take fullname parameters. See "Notes on DNS Name Escaping" earlier in this file for more details.) 

hosttarget
The target hostname of the machine providing the service. This name can be passed to functions like gethostbyname() to identify the host's IP address. 

port
The port, in network byte order, on which connections are accepted for this service. 

txtLen
The length of the txt record, in bytes. 

txtRecord
The service's primary txt record, in standard txt record format. 

context
The context pointer that was passed to the callout. 

Discussion 

DNSServiceResolveReply is a prototype for your callback function that handles replies from the mDNSResponder daemon made in response to DNSServiceResolve calls. Your implementation of DNSServiceResolveReply will be called when information about the service has been obtained. When your callback function is called, you should call DNSServiceRefDeallocate to terminate the resolution. 


Your application is responsible for stopping the resolution. Remove the sdRef from the run or select() loop and call DNSServiceRefDeallocate to deallocate sdRef, which stops the resolution, terminates the connection with the daemon, and frees the memory associated with sdRef. 


If you are registering a service on a particular network interface, pass interfaceIndex to DNSServiceQueryRecord to query for the address record for that interface. 


The fullname parameter is in a format that is suitable for passing to standard DNS functions such as res_query() and the functions described in "Special Purpose Functions" that take a full domain name as a parameter. 

Note: Applications should NOT use DNSServiceResolve() solely for txt record monitoring - use DNSServiceQueryRecord() instead, as it is more efficient for this task. 

Note: When the desired results have been returned, the client MUST terminate the resolve by calling DNSServiceRefDeallocate(). 

Note: DNSServiceResolve() behaves correctly for typical services that have a single SRV record and a single TXT record (the TXT record may be empty.) To resolve non-standard services with multiple SRV or TXT records, DNSServiceQueryRecord() should be used.*/
typedef void ( *DNSServiceResolveReply) ( 
    DNSServiceRef sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    DNSServiceErrorType errorCode, 
    const char *fullname, 
    const char *hosttarget, 
    uint16_t port, 
    uint16_t txtLen, 
    const char *txtRecord, 
    void *context );  

/*DNSServiceRegisterReply


Register a service that is discovered via Browse() and Resolve() calls.
typedef void ( *DNSServiceRegisterReply) ( 
    DNSServiceRef sdRef, 
    DNSServiceFlags flags, 
    DNSServiceErrorType errorCode, 
    const char *name, 
    const char *regtype, 
    const char *domain, 
    void *context );  
Parameters
sdRef
The DNSServiceRef initialized by DNSServiceRegister(). 

flags
Currently unused, reserved for future use. 

errorCode
Will be kDNSServiceErr_NoError on success, otherwise will indicate the failure that occurred (including name conflicts, if the kDNSServiceFlagsNoAutoRename flag was passed to the callout.) Other parameters are undefined if errorCode is nonzero. 

name
The service name registered (if the application did not specify a name in DNSServiceRegister(), this indicates what name was automatically chosen). 

regtype
The type of service registered, as it was passed to the callout. 

domain
The domain on which the service was registered (if the application did not specify a domain in DNSServiceRegister(), this indicates the default domain on which the service was registered). 

context
The context pointer that was passed to the callout. 

Discussion 

DNSServiceRegisterReply is a prototype for your callback function that handles replies from the mDNSResponder daemon made in response to DNSServiceRegister calls. 

Your callback function should check errorCode to see if its value is kDNSServiceErr_NoError, which indicates that registration was successful. In this case, sdRef is initialized and can be passed to other DNS Service Discovery API functions, and name, regtype, and domain point to valid values. The registration consists of an SRV record and a TXT record for your service. If you specified NULL as the value of txtRecord when you called DNSServiceRegister, the TXT record is empty. 

If you specified kDNSServiceFlagsAutoRename in the flags parameter when you called DNSServiceRegister, name may point to a name that has been automatically modified by appending (n) to the service name, where n is a number that has been incremented until a locally unique DNS name could be registered. 

If you specified kDNSServiceFlagsNoAutoRename in the flags parameter when you called DNSServiceRegister, the value of errorCode may be kDNSServiceErr_NameConflict, which indicates that the requested DNS name is already in use. For example, your default service name may be in use by another copy of your service on the same network, such as an identical printer. Call DNSServiceRefDeallocate to deallocate sdRef, choose another name and call DNSServiceRegister again. 

If your service has no user interface for naming, it is good practice to append a number to your default name and retry, incrementing the number as necessary to obtain a unique name. 

To deregister a service that has been successfully registered, call DNSServiceRefDeallocate, which deregisters the service, terminates the connection with the mDNSResponder daemon, and frees the memory associated with sdRef.*/
typedef void ( *DNSServiceRegisterReply) ( 
    DNSServiceRef sdRef, 
    DNSServiceFlags flags, 
    DNSServiceErrorType errorCode, 
    const char *name, 
    const char *regtype, 
    const char *domain, 
    void *context );  

//func

/*DNSServiceRefDeallocate


Deallocates a service discovery reference.
void DNSServiceRefDeallocate(
    DNSServiceRef sdRef);  
Parameters
sdRef
A DNSServiceRef initialized by any of the DNSService calls. 

Discussion 

This call is to be used only with the DNSServiceRef defined by this API. It is not compatible with dns_service_discovery_ref objects defined in the legacy Mach-based DNSServiceDiscovery.h API.*/
void DNSServiceRefDeallocate(
    DNSServiceRef sdRef);  

/*DNSServiceProcessResult


Gets a response from the mDNSResponder daemon and calls the appropriate callback function.
DNSServiceErrorType DNSServiceProcessResult(
    DNSServiceRef sdRef);  
Parameters
sdRef
A DNSServiceRef initialized by any of the DNSService calls that take a callback parameter. 

Return Value

Returns kDNSServiceErr_NoError on success, otherwise returns an error code indicating the specific failure that occurred. 
Discussion 

Read a reply from the daemon, calling the appropriate application callback. This call will block until the daemon's response is received. Use DNSServiceRefSockFD() in conjunction with a run loop or select() to determine the presence of a response from the server before calling this function to process the reply without blocking. Call this function at any point if it is acceptable to block until the daemon's response arrives. Note that the client is responsible for ensuring that DNSServiceProcessResult() is called whenever there is a reply from the daemon - the daemon may terminate its connection with a client that does not process the daemon's responses.*/
DNSServiceErrorType DNSServiceProcessResult(
    DNSServiceRef sdRef);  

/*DNSServiceRefSockFD


Returns the socket descriptor for a service discovery reference.
int DNSServiceRefSockFD(
    DNSServiceRef sdRef);  
Parameters
sdRef
A DNSServiceRef initialized by any of the DNSService calls. 

Return Value

The DNSServiceRef's underlying socket descriptor, or -1 on error. 
Discussion 

Access underlying Unix domain socket for an initialized DNSServiceRef. The DNS Service Discovery implementation uses this socket to communicate between the client and the mDNSResponder daemon. The application MUST NOT directly read from or write to this socket. Access to the socket is provided so that it can be used as a run loop source, or in a select() loop: when data is available for reading on the socket, DNSServiceProcessResult() should be called, which will extract the daemon's reply from the socket, and pass it to the appropriate application callback. By using a run loop or select(), results from the daemon can be processed asynchronously. Without using these constructs, DNSServiceProcessResult() will block until the response from the daemon arrives. The client is responsible for ensuring that the data on the socket is processed in a timely fashion - the daemon may terminate its connection with a client that does not clear its socket buffer.*/
int DNSServiceRefSockFD(
    DNSServiceRef sdRef);  

/*DNSServiceQueryRecord


Gets the resource data from the specified resource record.
DNSServiceErrorType DNSServiceQueryRecord ( 
    DNSServiceRef *sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    const char *fullname, 
    uint16_t rrtype, 
    uint16_t rrclass, 
    DNSServiceQueryRecordReply callBack, 
    void *context / may be NULL *
    );  
Parameters
sdRef
A pointer to an uninitialized DNSServiceRef. If the call succeeds then it initializes the DNSServiceRef, returns kDNSServiceErr_NoError, and the query operation will run indefinitely until the client terminates it by passing this DNSServiceRef to DNSServiceRefDeallocate(). 

flags
Pass kDNSServiceFlagsLongLivedQuery to create a "long-lived" unicast query in a non-local domain. Without setting this flag, unicast queries will be one-shot - that is, only answers available at the time of the call will be returned. By setting this flag, answers (including Add and Remove events) that become available after the initial call is made will generate callbacks. This flag has no effect on link-local multicast queries. 

interfaceIndex
If non-zero, specifies the interface on which to issue the query (the index for a given interface is determined via the if_nametoindex() family of calls.) Passing 0 causes the name to be queried for on all interfaces. See "Constants for specifying an interface index" for more details. 

fullname
The full domain name of the resource record to be queried for. 

rrtype
The numerical type of the resource record to be queried for (e.g. kDNSServiceType_PTR, kDNSServiceType_SRV, etc) 

rrclass
The class of the resource record (usually kDNSServiceClass_IN). 

callBack
The function to be called when a result is found, or if the call asynchronously fails. 

context
An application context pointer which is passed to the callback function (may be NULL). 

Return Value

Returns kDNSServiceErr_NoError on success (any subsequent, asynchronous errors are delivered to the callback), otherwise returns an error code indicating the error that occurred (the callback is never invoked and the DNSServiceRef is not initialized.) 
Discussion 

This function queries the specified interfaces (if interfaceIndex is non-zero) or all interfaces (if interfaceIndex is zero) for a resource record that matches the values specified by the fullName, rrtype, and rrclass parameters. If your application needs to monitor TXT record changes, you should use this function. 

If DNSServiceQueryRecord returns kDNSServiceErr_NoError, the query was started, and sdRef is initialized. Call DNSServiceRefSockFD and pass sdRef to it to get the descriptor for the underlying socket. Use the descriptor that DNSServiceRefSockFD returns to set up a run loop or a select() loop. When the loop indicates that a response is available, call DNSServiceProcessResult and pass sdRef to it. DNSServiceProcessResult will call the callback function specified by callBack and pass to it the response from the mDNSResponder daemon. 

If you don't want to set up a run or select() loop, you can call DNSServiceProcessResult immediately after calling DNSServiceQueryRecord. In this case, DNSServiceProcessResult will block until a response from the mDNSResponder daemon becomes available. When a response becomes available, DNSServiceProcessResult will call your callback function and pass to it the response from the mDNSResponder daemon. 

Regardless of the value of the flags parameter, you always need to call DNSServiceRefDeallocate on sdRef to free the memory and resources associated with a query and terminate the query. 

For information about the callback function for DNSServiceQueryRecord, see DNSServiceQueryRecordReply.*/

DNSServiceErrorType DNSServiceQueryRecord ( 
    DNSServiceRef *sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    const char *fullname, 
    uint16_t rrtype, 
    uint16_t rrclass, 
    DNSServiceQueryRecordReply callBack, 
    void *context /* may be NULL */
    );  

/*DNSServiceBrowse


Browse for instances of a service.
DNSServiceErrorType DNSServiceBrowse ( 
    DNSServiceRef *sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    const char *regtype, 
    const char *domain, // may be NULL 
    DNSServiceBrowseReply callBack, 
    void *context // may be NULL 
    );  
Parameters
sdRef
A pointer to an uninitialized DNSServiceRef. If the call succeeds then it initializes the DNSServiceRef, returns kDNSServiceErr_NoError, and the browse operation will run indefinitely until the client terminates it by passing this DNSServiceRef to DNSServiceRefDeallocate(). 

flags
Currently ignored, reserved for future use. 

interfaceIndex
If non-zero, specifies the interface on which to browse for services (the index for a given interface is determined via the if_nametoindex() family of calls.) Most applications will pass 0 to browse on all available interfaces. See "Constants for specifying an interface index" for more details. 

regtype
The service type being browsed for followed by the protocol, separated by a dot (e.g. "_ftp._tcp"). The transport protocol must be "_tcp" or "_udp". 

domain
If non-NULL, specifies the domain on which to browse for services. Most applications will not specify a domain, instead browsing on the default domain(s). 

callBack
The function to be called when an instance of the service being browsed for is found, or if the call asynchronously fails. 

context
An application context pointer which is passed to the callback function (may be NULL). 

Return Value

Returns kDNSServiceErr_NoError on success (any subsequent, asynchronous errors are delivered to the callback), otherwise returns an error code indicating the error that occurred (the callback is not invoked and the DNSServiceRef is not initialized.) 
Discussion 

Call this function to browse for available services of a given type in a given domain. 

If DNSServiceBrowse returns kDNSServiceErr_NoError, the search was started, and sdRef is initialized. Call DNSServiceRefSockFD and pass sdRef to it to get the descriptor for the underlying socket. Use the descriptor that DNSServiceRefSockFD returns to set up a run loop or a select() loop. When the loop indicates that a response is available, call DNSServiceProcessResult and pass sdRef to it. DNSServiceProcessResult will call the callback function specified by callBack and pass to it the response from the mDNSResponder daemon. Your callback function will be called for each service that matches the browsing criteria. 

If you don't want to set up a run or select() loop, you can call DNSServiceProcessResult immediately after calling DNSServiceBrowse. In this case, DNSServiceProcessResult will block until a response from the mDNSResponder daemon becomes available. When a response becomes available, DNSServiceProcessResult will call your callback function and pass to it the response from the mDNSResponder daemon. 

For information about the callback function for DNSServiceBrowse, see DNSServiceBrowseReply. 

To search for multiple service types, or to search multiple domains, make multiple calls to DNSServiceBrowse.*/
DNSServiceErrorType DNSServiceBrowse ( 
    DNSServiceRef *sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    const char *regtype, 
    const char *domain, /* may be NULL */
    DNSServiceBrowseReply callBack, 
    void *context /* may be NULL */
    );  

/*DNSServiceEnumerateDomains


Asynchronously enumerate domains available for browsing and registration.
DNSServiceErrorType DNSServiceEnumerateDomains ( 
    DNSServiceRef *sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    DNSServiceDomainEnumReply callBack, 
    void *context // may be NULL 
    );  
Parameters
sdRef
A pointer to an uninitialized DNSServiceRef. If the call succeeds then it initializes the DNSServiceRef, returns kDNSServiceErr_NoError, and the enumeration operation will run indefinitely until the client terminates it by passing this DNSServiceRef to DNSServiceRefDeallocate(). 

flags
Possible values are: kDNSServiceFlagsBrowseDomains to enumerate domains recommended for browsing. kDNSServiceFlagsRegistrationDomains to enumerate domains recommended for registration. 

interfaceIndex
If non-zero, specifies the interface on which to look for domains. (the index for a given interface is determined via the if_nametoindex() family of calls.) Most applications will pass 0 to enumerate domains on all interfaces. See "Constants for specifying an interface index" for more details. 

callBack
The function to be called when a domain is found or the call asynchronously fails. 

context
An application context pointer which is passed to the callback function (may be NULL). 

Return Value

Returns kDNSServiceErr_NoError on success (any subsequent, asynchronous errors are delivered to the callback), otherwise returns an error code indicating the error that occurred (the callback is not invoked and the DNSServiceRef is not initialized.) 
Discussion 

Call this function to build a list of recommended domains in which to register your service (if flags is set to kDNSServiceFlagsRegistrationDomains) or a list of recommended domains in which to browse (if flags is set to kDNSServiceFlagsBrowseDomains). Currently, the only domain returned is local., but other domains will be returned in the future. 

If DNSServiceEnumerateDomains returns kDNSServiceErr_NoError, the enumeration was started successfully, and sdRef is initialized. Call DNSServiceRefSockFD and pass sdRef to it to get the descriptor for the underlying socket. Use the descriptor that DNSServiceRefSockFD returns to set up a run loop or a select() loop. When the loop indicates that a response is available, call DNSServiceProcessResult and pass sdRef to it. DNSServiceProcessResult will call the callback function specified by callBack and pass to it the response from the mDNSResponder daemon. 

If you don't want to set up a run or select() loop, you can call DNSServiceProcessResult immediately after calling DNSServiceEnumerateDomains. In this case, DNSServiceProcessResult will block until a response from the mDNSResponder daemon becomes available. When a response becomes available, DNSServiceProcessResult will call your callback function and pass to it the response from the mDNSResponder daemon. 

For information about the callback function for DNSServiceEnumerateDomains, see DNSServiceDomainEnumReply. 

If you simply want to register in or search the default domain(s), pass NULL as the domain parameter when calling DNSServiceRegister or DNSServiceBrowse, respectively, and they will use the default domain without requiring you to look it up.*/
DNSServiceErrorType DNSServiceEnumerateDomains ( 
    DNSServiceRef *sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    DNSServiceDomainEnumReply callBack, 
    void *context // may be NULL 
    );  

/*DNSServiceUpdateRecord


Update a registered resource record. The record must either be: The primary txt record of a service registered via DNSServiceRegister(). A record added to a registered service via DNSServiceAddRecord(). An individual record registered by DNSServiceRegisterRecord().
DNSServiceErrorType DNSServiceUpdateRecord ( 
    DNSServiceRef sdRef, 
    DNSRecordRef RecordRef, // may be NULL 
    DNSServiceFlags flags, 
    uint16_t rdlen, 
    const void *rdata, 
    uint32_t ttl );  
Parameters
sdRef
A DNSServiceRef that was initialized by DNSServiceRegister() or DNSServiceCreateConnection(). 

RecordRef
A DNSRecordRef initialized by DNSServiceAddRecord, or NULL to update the service's primary txt record. 

flags
Currently ignored, reserved for future use. 

rdlen
The length, in bytes, of the new rdata. 

rdata
The new rdata to be contained in the updated resource record. 

ttl
The time to live of the updated resource record, in seconds. 

Return Value

Returns kDNSServiceErr_NoError on success, otherwise returns an error code indicating the error that occurred. 
Discussion 

This function updates the resource record referenced by RecordRef for the service referenced by sdRef while the service is running. If RecordRef is NULL, this function updates the primary TXT record for the service referenced by sdRef. 

Most services rarely need to update a resource record. You might, for example, need to update a text record for a service.*/
DNSServiceErrorType DNSServiceUpdateRecord ( 
    DNSServiceRef sdRef, 
    DNSRecordRef RecordRef, // may be NULL 
    DNSServiceFlags flags, 
    uint16_t rdlen, 
    const void *rdata, 
    uint32_t ttl );  

/*DNSServiceResolve


Gets the port number on which the service accepts connections, the name of the host that provides the service, and the service's primary TXT record.
DNSServiceErrorType DNSServiceResolve ( 
    DNSServiceRef *sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    const char *name, 
    const char *regtype, 
    const char *domain, 
    DNSServiceResolveReply callBack, 
    void *context // may be NULL 
    );  
Parameters
sdRef
A pointer to an uninitialized DNSServiceRef. If the call succeeds then it initializes the DNSServiceRef, returns kDNSServiceErr_NoError, and the resolve operation will run indefinitely until the client terminates it by passing this DNSServiceRef to DNSServiceRefDeallocate(). 

flags
Currently ignored, reserved for future use. 

interfaceIndex
The interface on which to resolve the service. If this resolve call is as a result of a currently active DNSServiceBrowse() operation, then the interfaceIndex should be the index reported in the DNSServiceBrowseReply callback. If this resolve call is using information previously saved (e.g. in a preference file) for later use, then use interfaceIndex 0, because the desired service may now be reachable via a different physical interface. See "Constants for specifying an interface index" for more details. 

name
The name of the service instance to be resolved, as reported to the DNSServiceBrowseReply() callback. 

regtype
The type of the service instance to be resolved, as reported to the DNSServiceBrowseReply() callback. 

domain
The domain of the service instance to be resolved, as reported to the DNSServiceBrowseReply() callback. 

callBack
The function to be called when a result is found, or if the call asynchronously fails. 

context
An application context pointer which is passed to the callback function (may be NULL). 

Return Value

Returns kDNSServiceErr_NoError on success (any subsequent, asynchronous errors are delivered to the callback), otherwise returns an error code indicating the error that occurred (the callback is never invoked and the DNSServiceRef is not initialized.) 
Discussion 

Call this function to get the name of the host that provides a service. Before you calling this function, get the name, registration type, and domain of the service by calling DNSServiceBrowse. 

This function is appropriate for resolving services that have a single SRV record and a single TXT record (the TXT record may be empty). To resolve services that have multiple SRV or TXT records, call DNSServiceQueryRecord instead. 

If DNSServiceResolve returns kDNSServiceErr_NoError, the resolution was started successfully, and sdRef is initialized. Call DNSServiceRefSockFD and pass sdRef to it to get the descriptor for the underlying socket. Use the descriptor that DNSServiceRefSockFD returns to set up a run loop or a select() loop. When the loop indicates that a response is available, call DNSServiceProcessResult and pass sdRef to it. DNSServiceProcessResult will call the callback function specified by callBack and pass to it the response from the mDNSResponder daemon. Your callback function will be called when the service's host name has been resolved or to report an error that occurred asynchronously. 

If you don't want to set up a run or select() loop, you can call DNSServiceProcessResult immediately after calling DNSServiceResolve. In this case, DNSServiceProcessResult will block until a response from the mDNSResponder daemon becomes available. When a response becomes available, DNSServiceProcessResult will call your callback function and pass to it the response from the mDNSResponder daemon. 

For information about the callback function for DNSServiceResolve, see DNSServiceResolveReply. 

The IP address and local host name can change dynamically, and services can be removed or unplugged at any time, so call this function immediately before connecting to any network service, and call it each time you use the service.*/
DNSServiceErrorType DNSServiceResolve ( 
    DNSServiceRef *sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    const char *name, 
    const char *regtype, 
    const char *domain, 
    DNSServiceResolveReply callBack, 
    void *context // may be NULL 
    );  

/*DNSServiceRegister


Registers a service
DNSServiceErrorType DNSServiceRegister ( 
    DNSServiceRef *sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    const char *name, // may be NULL 
    const char *regtype, 
    const char *domain, // may be NULL 
    const char *host, // may be NULL 
    uint16_t port, 
    uint16_t txtLen, 
    const void *txtRecord, // may be NULL 
    DNSServiceRegisterReply callBack, // may be NULL 
    void *context // may be NULL 
    );  
Parameters
sdRef
A pointer to an uninitialized DNSServiceRef. If the call succeeds then it initializes the DNSServiceRef, returns kDNSServiceErr_NoError, and the registration will remain active indefinitely until the client terminates it by passing this DNSServiceRef to DNSServiceRefDeallocate(). 

interfaceIndex
If non-zero, specifies the interface on which to register the service (the index for a given interface is determined via the if_nametoindex() family of calls.) Most applications will pass 0 to register on all available interfaces. See "Constants for specifying an interface index" for more details. 

flags
Indicates the renaming behavior on name conflict (most applications will pass 0). See flag definitions above for details. 

name
If non-NULL, specifies the service name to be registered. Most applications will not specify a name, in which case the computer name is used (this name is communicated to the client via the callback). 

regtype
The service type followed by the protocol, separated by a dot (e.g. "_ftp._tcp"). The transport protocol must be "_tcp" or "_udp". New service types should be registered at http://www.dns-sd.org/ServiceTypes.html. 

domain
If non-NULL, specifies the domain on which to advertise the service. Most applications will not specify a domain, instead automatically registering in the default domain(s). 

host
If non-NULL, specifies the SRV target host name. Most applications will not specify a host, instead automatically using the machine's default host name(s). Note that specifying a non-NULL host does NOT create an address record for that host - the application is responsible for ensuring that the appropriate address record exists, or creating it via DNSServiceRegisterRecord(). 

port
The port, in network byte order, on which the service accepts connections. Pass 0 for a "placeholder" service (i.e. a service that will not be discovered by browsing, but will cause a name conflict if another client tries to register that same name). Most clients will not use placeholder services. 

txtLen
The length of the txtRecord, in bytes. Must be zero if the txtRecord is NULL. 

txtRecord
The txt record rdata. May be NULL. Note that a non-NULL txtRecord MUST be a properly formatted DNS TXT record, i.e. [length byte] [data] [length byte] [data] ... 

callBack
The function to be called when the registration completes or asynchronously fails. The client MAY pass NULL for the callback - The client will NOT be notified of the default values picked on its behalf, and the client will NOT be notified of any asynchronous errors (e.g. out of memory errors, etc.) that may prevent the registration of the service. The client may NOT pass the NoAutoRename flag if the callback is NULL. The client may still deregister the service at any time via DNSServiceRefDeallocate(). 

context
An application context pointer which is passed to the callback function (may be NULL). 

Return Value

Returns kDNSServiceErr_NoError on success (any subsequent, asynchronous errors are delivered to the callback), otherwise returns an error code indicating the error that occurred (the callback is never invoked and the DNSServiceRef is not initialized.) 

Discussion 

Call this function to register a service. If DNSServiceRegister returns kDNSServiceErr_NoError, the registration started successfully, and sdRef is initialized. 

Call DNSServiceRefSockFD and pass sdRef to it to get the descriptor for the underlying socket. Use the descriptor that DNSServiceRefSockFD returns to set up a run loop or a select() loop. When the loop indicates that a response is available, call DNSServiceProcessResult and pass sdRef to it. DNSServiceProcessResult will call the callback function specified by callBack and pass to it the response from the mDNSResponder daemon. 

If you don't want to set up a run or select() loop, you can call DNSServiceProcessResult immediately after calling DNSServiceRegister. In this case, DNSServiceProcessResult will block until a response from the mDNSResponder daemon becomes available. When a response becomes available, DNSServiceProcessResult will call your callback function (if one was specified) and pass to it the response from the mDNSResponder daemon. 

For information about the callback function for DNSServiceRegister, see DNSServiceRegisterReply.*/
DNSServiceErrorType DNSServiceRegister ( 
    DNSServiceRef *sdRef, 
    DNSServiceFlags flags, 
    uint32_t interfaceIndex, 
    const char *name, // may be NULL 
    const char *regtype, 
    const char *domain, // may be NULL 
    const char *host, // may be NULL 
    uint16_t port, 
    uint16_t txtLen, 
    const void *txtRecord, // may be NULL 
    DNSServiceRegisterReply callBack, // may be NULL 
    void *context // may be NULL 
    );  

/*TXTRecordGetItemAtIndex


Uses an index to get a pointer to the name of a key and its value.
DNSServiceErrorType TXTRecordGetItemAtIndex ( 
    uint16_t txtLen, 
    const void *txtRecord, 
    uint16_t index, 
    uint16_t keyBufLen, 
    char *key, 
    uint8_t *valueLen, 
    const void **value );  
Parameters
txtLen
The size of the received TXT Record. 

txtRecord
Pointer to the received TXT Record bytes. 

index
An index into the TXT Record. 

keyBufLen
The size of the string buffer being supplied. 

key
A string buffer used to store the key name. On return, the buffer contains a null-terminated C string giving the key name. DNS-SD TXT keys are usually 8 characters or less. To hold the maximum possible key name, the buffer should be 256 bytes long. 

valueLen
On output, will be set to the size of the "value" data. 

value
On output, *value is set to point to location within TXT Record bytes that holds the value data. 

Return Value

Returns kDNSServiceErr_NoError on success. Returns kDNSServiceErr_NoMemory if keyBufLen is too short. Returns kDNSServiceErr_Invalid if index is greater than TXTRecordGetCount()-1. 
Discussion 

This function uses an index into TXT record data (usually received by your applications DNSServiceResolveReply callback routine) to get a pointer to the name of a key and pointer to a value that points to the key's value. You can iterate through the keys in a TXT record by calling TXTRecordGetItemAtIndex repeatedly, starting with an index of zero and incrementing it until TXTRecordGetItemAtIndex returns kDNSServiceErr_Invalid. 

You can also call this function to iterate through keys in a TXT record reference that you built yourself. 

If a key has no value, on return, value is set to null and valueLen is zero. If the key has an empty value, on return, value points to a non-null value and valueLen is zero. If the key has a non-empty value, value points to a non-null value and valueLen points to a non-zero value.*/
DNSServiceErrorType TXTRecordGetItemAtIndex ( 
    uint16_t txtLen, 
    const void *txtRecord, 
    uint16_t index, 
    uint16_t keyBufLen, 
    char *key, 
    uint8_t *valueLen, 
    const void **value );  

/*TXTRecordGetCount


Gets the number of keys in TXT record data.
uint16_t TXTRecordGetCount ( 
    uint16_t txtLen, 
    const void *txtRecord );  
Parameters
txtLen
The size of the received TXT Record. 

txtRecord
Pointer to the received TXT Record bytes. 

Return Value

Returns the total number of keys in the TXT Record. 

Discussion 

This function gets the number of keys in TXT record data, which you can pass to TXTRecordGetItemAtIndex to iterate through the keys.*/
uint16_t TXTRecordGetCount ( 
    uint16_t txtLen, 
    const void *txtRecord );  

/*TXTRecordDeallocate


Releases resources associated with a TXT record reference.
void TXTRecordDeallocate ( 
    TXTRecordRef *txtRecord );  
Parameters
txtRecord
A TXTRecordRef initialized by calling TXTRecordCreate(). 

Discussion 

Releases any resources allocated in the course of preparing a TXT Record using TXTRecordCreate()/TXTRecordSetValue()/TXTRecordRemoveValue(). Ownership of the buffer provided in TXTRecordCreate() returns to the client.*/
void TXTRecordDeallocate (
    TXTRecordRef *txtRecord );

/*TXTRecordGetLength


Gets the length of the raw bytes containing key/value pairs in a TXT record reference.
uint16_t TXTRecordGetLength ( 
    const TXTRecordRef *txtRecord );  
Parameters
txtRecord
A TXTRecordRef initialized by calling TXTRecordCreate(). 

Return Value

Returns the size of the raw bytes inside a TXTRecordRef which you can pass directly to DNSServiceRegister() or to DNSServiceUpdateRecord(). Returns 0 if the TXTRecordRef is empty. 

Discussion 

This function gets the length of the raw bytes in a TXT record reference. You can pass the length directly to DNSServiceRegisterRecord to register the record or DNSServiceUpdateRecord to update the record along with a pointer to the raw bytes obtained by calling TXTRecordGetBytesPtr.*/
uint16_t TXTRecordGetLength ( 
    const TXTRecordRef *txtRecord );  

/*TXTRecordGetBytesPtr


Gets a pointer to the raw bytes containing key/value pairs in a TXT record reference.
const void * TXTRecordGetBytesPtr ( 
    const TXTRecordRef *txtRecord );  
Parameters
txtRecord
A TXTRecordRef initialized by calling TXTRecordCreate(). 

Return Value

Returns a pointer to the raw bytes inside the TXTRecordRef which you can pass directly to DNSServiceRegister() or to DNSServiceUpdateRecord(). 

Discussion 

This function gets a pointer to the raw bytes in a TXT record reference that you've constructed. You can pass the pointer directly to DNSServiceRegisterRecord to register the record or DNSServiceUpdateRecord to update a record that has already been registered along with the length of the raw bytes obtained by calling TXTRecordGetLength.*/
const void * TXTRecordGetBytesPtr ( 
    const TXTRecordRef *txtRecord );  

/*TXTRecordSetValue


Sets a key and an optional value in a TXT record reference.
DNSServiceErrorType TXTRecordSetValue ( 
    TXTRecordRef *txtRecord, 
    const char *key, 
    uint8_t valueSize, ///may be zero 
    const void *value // may be NULL /
    );  
Parameters
txtRecord
A TXTRecordRef initialized by calling TXTRecordCreate(). 

key
A null-terminated string which only contains printable ASCII values (0x20-0x7E), excluding '=' (0x3D). Keys should be 8 characters or less (not counting the terminating null). 

valueSize
The size of the value. 

value
Any binary value. For values that represent textual data, UTF-8 is STRONGLY recommended. For values that represent textual data, valueSize should NOT include the terminating null (if any) at the end of the string. If NULL, then "key" will be added with no value. If non-NULL but valueSize is zero, then "key=" will be added with empty value. 

Return Value

Returns kDNSServiceErr_NoError on success. Returns kDNSServiceErr_Invalid if the "key" string contains illegal characters. Returns kDNSServiceErr_NoMemory if adding this key would exceed the available storage. 
Discussion 

Adds a key (optionally with value) to a TXTRecordRef. If the "key" already exists in the TXTRecordRef, then the current value will be replaced with the new value. 

Keys may exist in four states with respect to a given TXT record:
Absent (key does not appear at all) 
Present with no value ("key" appears alone) 
Present with empty value ("key=" appears in TXT record) 
Present with non-empty value ("key=value" appears in TXT record)


For more details refer to Data Syntax for DNS-SD TXT Records at http://files.dns-sd.org/draft-cheshire-dnsext-dns-sd.txt*/
DNSServiceErrorType TXTRecordSetValue ( 
    TXTRecordRef *txtRecord, 
    const char *key, 
    uint8_t valueSize, //may be zero 
    const void *value // may be NULL /
    );  

/*TXTRecordCreate


Creates a new empty TXTRecordRef referencing the specified storage.
void TXTRecordCreate ( 
    TXTRecordRef *txtRecord, 
    uint16_t bufferLen, 
    void *buffer );  
Parameters
txtRecord
A pointer to an uninitialized TXTRecordRef. 

bufferLen
The size of the storage provided in the "buffer" parameter. 

buffer
The storage used to hold the TXTRecord data. This storage must remain valid for as long as the TXTRecordRef.
Discussion 

If the buffer parameter is NULL, or the specified storage size is not large enough to hold a key subsequently added using TXTRecordSetValue(), then additional memory will be added as needed using malloc(). 

On some platforms, when memory is low, malloc() may fail. In this case, TXTRecordSetValue() will return kDNSServiceErr_NoMemory, and this error condition will need to be handled as appropriate by the caller. 

You can avoid the need to handle this error condition if you ensure that the storage you initially provide is large enough to hold all the key/value pairs that are to be added to the record. The caller can precompute the exact length required for all of the key/value pairs to be added, or simply provide a fixed-sized buffer known in advance to be large enough. 
A no-value (key-only) key requires (1 + key length) bytes. 
A key with empty value requires (1 + key length + 1) bytes. 
A key with non-empty value requires (1 + key length + 1 + value length).
For most applications, DNS-SD TXT records are generally less than 100 bytes, so in most cases a simple fixed-sized 256-byte buffer will be more than sufficient. Recommended size limits for DNS-SD TXT Records are discussed in http://files.dns-sd.org/draft-cheshire-dnsext-dns-sd.txt 

Note: When passing parameters to and from these TXT record APIs, the key name does not include the '=' character. The '=' character is the separator between the key and value in the on-the-wire packet format; it is not part of either the key or the value.*/
void TXTRecordCreate ( 
    TXTRecordRef *txtRecord, 
    uint16_t bufferLen, 
    void *buffer );  

#ifdef  __cplusplus
    }
#endif


#endif  
