/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/


/*

	<HSTimer.c>	2005-03-13,10:51

*/

#include "HSTimer.h"





/*
 * Timer list management for killing timer.
 * timer list of lingered on
 * programer must init this list
 * to use HSTimer.
 */
static BOOL			gHSTimer = FALSE;
static NoLockList	gTimerList;
static HS_UINT		gTimerListId = 0;
static HSSemaphore	gSemaTimerList;

typedef struct
{
	HS_QID	mQid;
	HS_TID	mTid;
	HS_UINT	mSub;

	HS_UINT	mListId;
} TimerSet;





static TimerSet *newm_TimerSet(HS_QID pQid,HS_TID pTid,HS_UINT pSub);
static HS_RESULT delete_TimerSet(void *pObj);
static HS_RESULT deletem_TimerSet(TimerSet *pObj);
static HS_RESULT NoLockList_AttachTimerSet(NoLockList *pObj,TimerSet *pData);
static HS_RESULT NoLockList_DeleteTimerChain(NoLockList *pObj,ChainUnit *pData);
static ChainUnit *NoLockList_FindTimerSetByTimerListId(NoLockList *pObj,HS_UINT pListId);
static HS_RESULT NoLockList_DeleteTimerSetByTimerListId(NoLockList *pObj,HS_UINT pListId);
static ChainUnit *NoLockList_FindTimerSetByTimerId(NoLockList *pObj,HS_TID pTid,HS_UINT pSub);
static HS_RESULT NoLockList_DeleteTimerSetByTimerId(NoLockList *pObj,HS_TID pTid,HS_UINT pSub);





HS_RESULT OpenTimerList()
{
	if( gHSTimer==TRUE ) return HS_ERR;
	
	new_NoLockList(&gTimerList,delete_TimerSet);
	gHSTimer = TRUE;

	HSOpenSemaphore(gSemaTimerList);
	return HS_OK;
}


HS_RESULT CloseTimerList()
{
	if( gHSTimer==FALSE ) return HS_ERR;
	
	gHSTimer = FALSE;

	HSLockSemaphore(gSemaTimerList);
	delete_NoLockList(&gTimerList);
	HSUnlockSemaphore(gSemaTimerList);

	HSCloseSemaphore(gSemaTimerList);
	return HS_OK;
}





static TimerSet *newm_TimerSet(HS_QID pQid,HS_TID pTid,HS_UINT pSub)
{
	TimerSet *tResult = NULL;

	if( (tResult=(TimerSet*)HSMalloc(sizeof(TimerSet)))==NULL ) return NULL;

	tResult->mQid = pQid;
	tResult->mTid = pTid;
	tResult->mSub = pSub;
	tResult->mListId = gTimerListId++;

	return tResult;
}


static HS_RESULT delete_TimerSet(void *pObj)
{
	return HS_OK;
}


static HS_RESULT deletem_TimerSet(TimerSet *pObj)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	delete_TimerSet(pObj);
	HSFree(pObj);
	return HS_OK;
}


static HS_RESULT NoLockList_AttachTimerSet(NoLockList *pObj,TimerSet *pData)
{
	HS_RESULT ret;

	HSLockSemaphore(gSemaTimerList);
	ret = NoLockList_AttachData(pObj,pData);
	HSUnlockSemaphore(gSemaTimerList);

	return ret;
}


static HS_RESULT NoLockList_DeleteTimerChain(NoLockList *pObj,ChainUnit *pData)
{
	HS_RESULT ret;

	HSLockSemaphore(gSemaTimerList);
	ret = NoLockList_DeleteChain(pObj,pData);
	HSUnlockSemaphore(gSemaTimerList);

	return ret;
}


static ChainUnit *NoLockList_FindTimerSetByTimerListId(NoLockList *pObj,HS_UINT pListId)
{
	HS_UINT i;
	ChainUnit *ret = NULL;

	if( pObj==NULL ) return NULL;
	if( pObj->size==0 ) return NULL;

	HSLockSemaphore(gSemaTimerList);
	ret = pObj->units;
	for(i=0;i<pObj->size;i++)
	{
		if( ret==NULL )
			break;
		if( ret->data==NULL )
		{
			ret = NULL;
			break;
		}

		if( ((TimerSet*)(ret->data))->mListId == pListId ) break;
		ret = ret->next;
	}
	HSUnlockSemaphore(gSemaTimerList);

	return ret;
}


static HS_RESULT NoLockList_DeleteTimerSetByTimerListId(NoLockList *pObj,HS_UINT pListId)
{
	HS_UINT i;
	ChainUnit *ret = NULL;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->size==0 ) return HS_ERR_NO_EXIST;

	HSLockSemaphore(gSemaTimerList);
	ret = pObj->units;
	for(i=0;i<pObj->size;i++)
	{
		if( ret==NULL )
			break;
		if( ret->data==NULL )
		{
			ret = NULL;
			break;
		}

		if( ((TimerSet*)(ret->data))->mListId == pListId ) break;
		ret = ret->next;
	}
	if( ret != NULL )
		NoLockList_DeleteChain(pObj,ret);
	HSUnlockSemaphore(gSemaTimerList);

	return HS_OK;
}


static ChainUnit *NoLockList_FindTimerSetByTimerId(NoLockList *pObj,HS_TID pTid,HS_UINT pSub)
{
	HS_UINT i;
	ChainUnit *ret = NULL;
	TimerSet *tSet = NULL;

	if( pObj==NULL ) return NULL;
	if( pObj->size==0 ) return NULL;

	HSLockSemaphore(gSemaTimerList);
	ret = pObj->units;
	for(i=0;i<pObj->size;i++)
	{
		if( ret==NULL )
			break;
		if( ret->data==NULL )
		{
			ret = NULL;
			break;
		}

		tSet = (TimerSet*)(ret->data);
		if( tSet->mTid==pTid && tSet->mSub==pSub ) break;
		ret = ret->next;
	}
	HSUnlockSemaphore(gSemaTimerList);

	return ret;
}


static HS_RESULT NoLockList_DeleteTimerSetByTimerId(NoLockList *pObj,HS_TID pTid,HS_UINT pSub)
{
	HS_UINT i;
	ChainUnit *ret = NULL;
	TimerSet *tSet = NULL;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->size==0 ) return HS_ERR_NO_EXIST;

	HSLockSemaphore(gSemaTimerList);
	ret = pObj->units;
	for(i=0;i<pObj->size;i++)
	{
		if( ret==NULL )
			break;
		if( ret->data==NULL )
		{
			ret = NULL;
			break;
		}

		tSet = (TimerSet*)(ret->data);
		if( tSet->mTid==pTid && tSet->mSub==pSub ) break;
		ret = ret->next;
	}
	if( ret != NULL )
		NoLockList_DeleteChain(pObj,ret);
	HSUnlockSemaphore(gSemaTimerList);

	return HS_OK;
}





/*
 * HSTimerEvent member functions.
 * HSTimerEvent object : timer response object.
 * when timer expire, timer thread send queue message
 * to origin thread(calling thread) with this struct.
 */
HSTimerEvent *newm_HSTimerEvent(HS_TID pTid, HS_UINT pSub)
{
	HSTimerEvent *tResult = NULL;

	if( (tResult=(HSTimerEvent*)HSMalloc(sizeof(HSTimerEvent))) == NULL ) return NULL;

	tResult->mTid = pTid;
	tResult->mSub = pSub;

	return tResult;
}


HS_RESULT deletem_HSTimerEvent(HSTimerEvent *pObj)
{
	if( pObj == NULL ) return HS_ERR_NULL_PARAM;
	
	HSFree(pObj);
	return HS_OK;
}





/* HSTimer object
   (thread for timer)
*/
typedef struct
_thread_struct
(
	unsigned	mTime;	/*milli seconds*/
	HS_QID		mTarget;/*sender thread queue id*/

	HS_TID		mTid;	/*timer id*/
	HS_UINT		mSub;	/*sub timer id*/
	TimerType	mType;	/*once or forever*/

	HS_UINT		mListId;
) HSTimer;


HS_RESULT HSTimer_Demon(HS_TQ pQ,void *pObject,void *pArg)
{
	ChainUnit *tChain = NULL;
	HSTimerEvent *tEvent = NULL;
	HSTimer *pObj = (HSTimer*)pObject;

	if( pQ==HS_INVALID_TQ || pObj==NULL )
		return HSExitThread(pQ,HS_ERR_NULL_PARAM);

	if( pObj->mType==e_TimerType_Forever )
	{
		while(1)
		{
			HSSleep(pObj->mTime);
			if( (tChain=NoLockList_FindTimerSetByTimerListId(&gTimerList,pObj->mListId)) != NULL )
			{
				if( (tEvent=newm_HSTimerEvent(pObj->mTid,pObj->mSub)) != NULL )
					_HSThreadSendMessage(pObj->mTarget,HS_QM_TIMER,0,(HS_UINT)tEvent);
			}
			else break;
		}
	}
	else
	{
		HSSleep(pObj->mTime);
		if( (tChain=NoLockList_FindTimerSetByTimerListId(&gTimerList,pObj->mListId)) != NULL )
		{
			if( (tEvent=newm_HSTimerEvent(pObj->mTid,pObj->mSub)) != NULL )
				_HSThreadSendMessage(pObj->mTarget,HS_QM_TIMER,0,(HS_UINT)tEvent);

			NoLockList_DeleteTimerSetByTimerListId(&gTimerList,pObj->mListId);
		}
	}

	SubThreadQ(pObj->mQid);

	HSFree(pObj);
	return HSExitThread(pQ,HS_OK);
}



HS_RESULT HSSetTimer(HS_UINT pTime,HS_QID pQid,HS_TID pTid,HS_UINT pSub,TimerType pType)
{
	TimerSet *tSet = NULL;
	HSTimer *tObject = NULL;

	if( gHSTimer==FALSE ) return HS_ERR_INVALID;
	if( pTime==0 || pQid==HS_INVALID_QID ) return HS_ERR_NULL_PARAM;

	HSKillTimer(pTid,pSub);

	if( (tSet=newm_TimerSet(pQid,pTid,pSub))==NULL )
		return HS_ERR_MALLOC;
	if( (tObject=(HSTimer*)HSMalloc(sizeof(HSTimer)))==NULL )
	{
		deletem_TimerSet(tSet);
		return HS_ERR_MALLOC;
	}

	HS_THREAD_INIT(tObject,"timer");
	tObject->mTime   = pTime;
	tObject->mTarget = pQid;
	tObject->mTid  = pTid;
	tObject->mSub  = pSub;
	tObject->mType = pType;
	tObject->mListId = tSet->mListId;

	if( HSThreadStart(tObject,NULL,HSTimer_Demon)==HS_INVALID_QID )
	{
		HSFree(tObject);
		deletem_TimerSet(tSet);
		return HS_ERR;
	}

	NoLockList_AttachTimerSet(&gTimerList,tSet);
	return HS_OK;
}


HS_RESULT HSKillTimer(HS_TID pTid,HS_UINT pSub)
{
	if( gHSTimer==FALSE ) return HS_ERR_INVALID;
	return NoLockList_DeleteTimerSetByTimerId(&gTimerList,pTid,pSub);
}
