/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/


/*

	<Q931Message.c>		2005-03-13,16:53

*/

#include "Q931Message.h"


/*last Octet, National Number, E.164 ISDN/telephony Number*/
#define gDefaultPartyNumber								0xa1
/*Network congestion tone on*/
#define gDefaultSignal									0x03
/*last Octet, ITU-T Standard, speech*/
/*last Octet, 64k bps, circuit mode*/
/*last Octet, H.221 and H.242*/
unsigned char gDefaultBearerCapability[] = {0x80,0x90,0xa5};
/*Location:User, ITU-T Standard*/
/*In-band Information or an appropriate pattern is now available*/
unsigned char gDefaultProgressIndicator[] = {0x80,0x88};



#ifdef HS_DEBUG_ASN
const char *GetQ931ElementTypeName( Q931ElementType pType )
{
	switch( pType )
	{
		case e_Q931ElementType_BearerCapability:		return "BearerCapability";
		case e_Q931ElementType_Cause:					return "Cause";
		case e_Q931ElementType_ExtendedFacility:		return "ExtendedFacility";
		case e_Q931ElementType_CallIdentity:			return "CallIdentity";
		case e_Q931ElementType_CallState:				return "CallState";
		case e_Q931ElementType_ChannelIdentification:	return "ChannelIdentification";
		case e_Q931ElementType_Facility:				return "Facility";
		case e_Q931ElementType_ProgressIndicator:		return "ProgressIndicator";
		case e_Q931ElementType_NotificationIndicator:	return "NotificationIndicator";
		case e_Q931ElementType_Display:					return "Display";
		case e_Q931ElementType_DateTime:				return "DateTime";
		case e_Q931ElementType_KeypadFacility:			return "KeypadFacility";
		case e_Q931ElementType_Signal:					return "Signal";
		case e_Q931ElementType_ConnectedNumber:			return "ConnectedNumber";
		case e_Q931ElementType_ConnectedSubAddress:		return "ConnectedSubAddress";
		case e_Q931ElementType_CallingPartyNumber:		return "CallingPartyNumber";
		case e_Q931ElementType_CallingPartySubaddress:	return "CallingPartySubaddress";
		case e_Q931ElementType_CalledPartyNumber:		return "CalledPartyNumber";
		case e_Q931ElementType_CalledPartySubaddress:	return "CalledPartySubaddress";
		case e_Q931ElementType_UserToUser:				return "UserToUser";
		case e_Q931ElementType_SendingComplete:			return "SendingComplete";
	}

	return "Unknown";
}
#endif


/******************************************************************************/
/* Q931Element members
*/
/* new_Q931Element */
HS_RESULT new_Q931Element( void *pObject )
{
	Q931Element *pObj = (Q931Element *)pObject;
	
	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on new_Q931Element(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}

	pObj->discriminator = e_Q931ElementType_BearerCapability;
	pObj->nlen = 0;
	pObj->ilen = 0;
	pObj->value = HS_NULL;

	return HS_OK;
}


/* delete_Q931Element */
HS_RESULT delete_Q931Element( void *pObject )
{
	Q931Element *pObj = (Q931Element *)pObject;
	
	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on delete_Q931Element(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->value != HS_NULL )
	{
		HSFree( pObj->value );
		pObj->value = HS_NULL;
	}

	return HS_OK;
}


/* Q931Element_SetValue */
HS_RESULT Q931Element_SetValue( void *pObject, Q931ElementType pType, unsigned char *pValue, unsigned pLen )
{
	Q931Element *pObj = (Q931Element *)pObject;
	
	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on Q931Element_SetValue(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->value != HS_NULL )
	{
		HSFree(pObj->value);
		pObj->value = HS_NULL;
	}
	pObj->ilen=0;
	pObj->nlen=0;
	pObj->discriminator = pType;

	switch( pType )
	{
		case e_Q931ElementType_SendingComplete:
			return HS_OK;
		case e_Q931ElementType_BearerCapability:
			if( pValue == HS_NULL || pLen == 0 )
			{
				pObj->nlen = 3;
				pObj->value = (unsigned char *)HSMalloc(3);
				memcpy(pObj->value,gDefaultBearerCapability,3);
			}
			else
			{
				pObj->nlen = pLen;
				pObj->value = (unsigned char *)HSMalloc(pLen);
				memcpy(pObj->value,pValue,pLen);
			}
			return HS_OK;
		case e_Q931ElementType_CallingPartyNumber:
		case e_Q931ElementType_CalledPartyNumber:
			if( pValue == HS_NULL || pLen == 0 )
			{
				pObj->nlen = 1;
				pObj->value = (unsigned char *)HSMalloc(1);
				pObj->value[0] = gDefaultPartyNumber;
			}
			else
			{
				pObj->nlen = pLen;
				pObj->value = (unsigned char *)HSMalloc(pLen);
				memcpy(pObj->value,pValue,pLen);
			}
			return HS_OK;
		case e_Q931ElementType_ProgressIndicator:
			if( pValue == HS_NULL || pLen == 0 )
			{
				pObj->nlen = 2;
				pObj->value = (unsigned char *)HSMalloc(2);
				memcpy(pObj->value,gDefaultProgressIndicator,2);
			}
			else
			{
				pObj->nlen = pLen;
				pObj->value = (unsigned char *)HSMalloc(pLen);
				memcpy(pObj->value,pValue,pLen);
			}
			return HS_OK;
		case e_Q931ElementType_Signal:
			if( pValue == HS_NULL || pLen == 0 )
			{
				pObj->nlen = 1;
				pObj->value = (unsigned char *)HSMalloc(1);
				pObj->value[0] = gDefaultSignal;
			}
			else
			{
				pObj->nlen = pLen;
				pObj->value = (unsigned char *)HSMalloc(pLen);
				memcpy(pObj->value,pValue,pLen);
			}
			return HS_OK;
		case e_Q931ElementType_UserToUser:
			if( pValue != HS_NULL && pLen )
			{
				pObj->ilen = pLen;
				pObj->value = (unsigned char*)HSMalloc(pLen);
				memcpy(pObj->value,pValue,pLen);
			}
			return HS_OK;

		case e_Q931ElementType_Cause:
		case e_Q931ElementType_ExtendedFacility:
		case e_Q931ElementType_CallIdentity:
		case e_Q931ElementType_CallState:	
		case e_Q931ElementType_ChannelIdentification:	
		case e_Q931ElementType_Facility:
		case e_Q931ElementType_NotificationIndicator:
		case e_Q931ElementType_DateTime:
		case e_Q931ElementType_KeypadFacility:
		case e_Q931ElementType_ConnectedNumber:
		case e_Q931ElementType_ConnectedSubAddress:
		case e_Q931ElementType_CallingPartySubaddress:
		case e_Q931ElementType_CalledPartySubaddress:	
		case e_Q931ElementType_Display:
			break;
		default:
			return HS_ERR_ASN_OUTOF_RANGE;
	}

	if( pValue != HS_NULL && pLen )
	{
		pObj->nlen = pLen;
		pObj->value = (unsigned char*)HSMalloc(pLen);
		memcpy(pObj->value,pValue,pLen);
	}

	return HS_OK;
}


/* Q931Element_SetValueInVisible */
HS_RESULT Q931Element_SetValueInVisible( void *pObject, Q931ElementType pType, char *pValue )
{
	if( pType == e_Q931ElementType_CallingPartyNumber || pType == e_Q931ElementType_CalledPartyNumber )
	{
		unsigned char tValue[HS_MAX_DIGIT];

		if( pValue == HS_NULL ) return HS_ERR_NULL_PARAM;
		tValue[0] = gDefaultPartyNumber;
		memcpy(tValue+1,pValue,strlen(pValue));
		return Q931Element_SetValue( pObject, pType, tValue, strlen(pValue) + 1 );
	}
	
	return Q931Element_SetValue( pObject, pType, pValue, strlen(pValue) );
}


/* Q931Element_SetCauseValue */
HS_RESULT Q931Element_SetCauseValue( void *pObject, Q931CauseType pCause )
{
	unsigned char tValue[2];

	if( pCause > 127 ) return HS_ERR_ASN_OUTOF_RANGE;

	/*Location:User, ITU-T Standard*/
	tValue[0] = 0x80;
	tValue[1] = 0x80+((unsigned char)pCause);

	return Q931Element_SetValue( pObject, e_Q931ElementType_Cause, tValue, 2 );
}


/* Q931Element_GetCauseValue */
Q931CauseType Q931Element_GetCauseValue( void *pObject )
{
	Q931Element *pObj = (Q931Element *)pObject;
	
	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on Q931Element_GetCauseValue(NULL) -> null parameter" );
#endif
		return e_Q931CauseType_NormalCallClearing;
	}

	if( pObj->value == HS_NULL || pObj->nlen != 2 ) return e_Q931CauseType_NormalCallClearing;
	return (Q931CauseType)( 0x7f & pObj->value[1] );
}

/* Q931Element_Print */
#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT Q931Element_Print( void *pObject )
{
	unsigned short i, len;
	Q931Element *pObj = (Q931Element *)pObject;
	
	if( pObj == HS_NULL )
	{
		HSPrint( "\n Error on Q931Element_Print(NULL) -> null parameter" );
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->discriminator == e_Q931ElementType_SendingComplete )
	{
		HSPrint( "\n  SendingComplete{\n  }" );
		return HS_OK;
	}

	if( pObj->discriminator == e_Q931ElementType_UserToUser )
	{
		len = pObj->ilen;
		HSPrint( "\n  UserToUser {" );
	}
	else
	{
		len = (unsigned short)(pObj->nlen);
		HSPrint( "\n  %s {", GetQ931ElementTypeName(pObj->discriminator) );
	}

	if( pObj->value != HS_NULL )
	{
		for( i=0; i<len; i++ )
		{
			if( i%8 == 0 ) HSPrint( "  " );
			if( i%16 == 0 ) HSPrint( "\n    " );
			HSPrint( " %02X", pObj->value[i] );
		}
	}

	HSPrint( "\n  }" );
	return HS_OK;
}
#endif



/**************************************************************************/
/* Q931Message memebers
*/
/* new_Q931Message */
HS_RESULT new_Q931Message( void *pObject )
{
	Q931Message *pObj = (Q931Message *)pObject;
	
	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on new_Q931Message(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}

	pObj->wholeDiscriminator = 0x08;	/*Q.931*/
	pObj->origin = TRUE;
	pObj->crv = 0;
	pObj->type = e_Q931MessageType_NationalEscape;

	pObj->units.unit = HS_NULL;
	pObj->units.next = HS_NULL;
	pObj->size = 0;
	pObj->uuie = HS_NULL;

	return HS_OK;
}


/* delete_Q931Message */
HS_RESULT delete_Q931Message( void *pObject )
{
	HS_RESULT tRet;

	Q931Message *pObj = (Q931Message *)pObject;
	
	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on delete_Q931Message(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}

	if( (tRet=Q931Message_Clear(pObj)) != HS_OK )
		return tRet;

	return HS_OK;
}


/* Q931Message_Clear */
HS_RESULT Q931Message_Clear( void *pObject )
{
	HS_UINT i;
	UnitOf *rounder = HS_NULL;
	UnitOf *deleter = HS_NULL;
	Q931Message *pObj = (Q931Message *)pObject;
	
	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on Q931Message_Clear(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->size == 0 )
	{
		if( pObj->units.unit != HS_NULL || pObj->units.next != HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error Q931Message_Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_ERR_ASN_CONFLICT;
		}
		else
			return HS_OK;
	}

	rounder = &(pObj->units);

	for( i=0; i<pObj->size; i++ )
	{
		if( rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error Q931Message_Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_ERR_ASN_CONFLICT;
		}
		if( rounder->unit == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error Q931Message_Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_ERR_ASN_CONFLICT;
		}

		delete_Q931Element( rounder->unit );
		HSFree(rounder->unit);

		deleter = rounder;
		rounder = (UnitOf *)rounder->next;

		if( i ) HSFree( deleter );
	}

	if( rounder != HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error Q931Message_Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
		return HS_ERR_ASN_CONFLICT;
	}

	pObj->units.unit = HS_NULL;
	pObj->units.next = HS_NULL;
	pObj->size = 0;

	if( pObj->uuie != HS_NULL )
	{
		delete_ASNH225H323_UserInformation( pObj->uuie );
		HSFree( pObj->uuie );
		pObj->uuie = HS_NULL;
	}

	return HS_OK;
}


/* Q931Message_AddElement */
HS_RESULT Q931Message_AddElement( void *pObject, Q931Element *pElement )
{
	HS_UINT i;
	UnitOf *rounder = HS_NULL;
	Q931Message *pObj = (Q931Message *)pObject;
	
	if( pObj == HS_NULL || pElement == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on Q931Message_AddElement(%x,%x) -> null parameter", pObj, pElement );
#endif
		return HS_ERR_NULL_PARAM;
	}

	rounder = &(pObj->units);
	if( pObj->size == 0 )
	{
		if( rounder->next != HS_NULL || rounder->next != HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on Q931Message_AddElement(pObj,pElement) -> confilict size 1" );
#endif
			return HS_ERR_ASN_CONFLICT;
		}
		else
		{
			rounder->unit = pElement;
			pObj->size = 1;
			return HS_OK;
		}
	}

	for( i=0; i<(pObj->size-1); i++ )
	{
		if( rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on Q931Message_AddElement(pObj,pElement) -> confilict size 2" );
#endif
			return HS_ERR_ASN_CONFLICT;
		}
		rounder = rounder->next;
	}

	if( rounder != HS_NULL )
	{
		if( rounder->unit != HS_NULL && rounder->next == HS_NULL )
		{
			rounder->next = (UnitOf*)HSMalloc( sizeof(UnitOf) );
			rounder = (UnitOf*)(rounder->next);
			rounder->unit = pElement;
			rounder->next = HS_NULL;
			pObj->size++;

			return HS_OK;
		}
	}

#ifdef HS_DEBUG_ASN
	HSPrint( "\n Error on Q931Message_AddElement(pObj,pElement) -> confilict size 3" );
#endif
	return HS_ERR_ASN_CONFLICT;
}


/* Q931Message_AddElementEx */
HS_RESULT Q931Message_AddElementEx( Q931Message *pMsg, Q931ElementType pType, unsigned char *pValue, unsigned pLen )
{
	HS_RESULT tRet;
	Q931Element *element = HS_NULL;

	if( pMsg == HS_NULL ) return HS_ERR_NULL_PARAM;

	element = (Q931Element*)HSMalloc( sizeof(Q931Element) );
	if( element == HS_NULL ) return HS_ERR_MALLOC;

	new_Q931Element(element);
	if( (tRet=Q931Element_SetValue(element, pType, pValue, pLen)) != HS_OK )
	{
		delete_Q931Element(element);
		HSFree(element);
		return tRet;
	}

	if( (tRet=Q931Message_AddElement(pMsg,element)) != HS_OK )
	{
		delete_Q931Element(element);
		HSFree(element);
		return tRet;
	}

	return HS_OK;
}


/* Q931Message_AddElementInVisible */
HS_RESULT Q931Message_AddElementInVisible( Q931Message *pMsg, Q931ElementType pType, char *pValue )
{
	HS_RESULT tRet;
	Q931Element *element = HS_NULL;

	if( pMsg == HS_NULL ) return HS_ERR_NULL_PARAM;

	element = (Q931Element*)HSMalloc( sizeof(Q931Element) );
	if( element == HS_NULL ) return HS_ERR_MALLOC;

	new_Q931Element(element);
	if( (tRet=Q931Element_SetValueInVisible(element, pType, pValue)) != HS_OK )
	{
		delete_Q931Element(element);
		HSFree(element);
		return tRet;
	}

	if( (tRet=Q931Message_AddElement(pMsg,element)) != HS_OK )
	{
		delete_Q931Element(element);
		HSFree(element);
		return tRet;
	}

	return HS_OK;
}


/* Q931Message_AddElementCauseValue */
HS_RESULT Q931Message_AddElementCauseValue( Q931Message *pMsg, Q931CauseType pCause )
{
	HS_RESULT tRet;
	Q931Element *element = HS_NULL;

	if( pMsg == HS_NULL ) return HS_ERR_NULL_PARAM;

	element = (Q931Element*)HSMalloc( sizeof(Q931Element) );
	if( element == HS_NULL ) return HS_ERR_MALLOC;

	new_Q931Element(element);
	if( (tRet=Q931Element_SetCauseValue(element, pCause)) != HS_OK )
	{
		delete_Q931Element(element);
		HSFree(element);
		return tRet;
	}

	if( (tRet=Q931Message_AddElement(pMsg,element)) != HS_OK )
	{
		delete_Q931Element(element);
		HSFree(element);
		return tRet;
	}

	return HS_OK;
}


/* Q931Message_GetElement */
Q931Element *Q931Message_GetElement( void *pObject, Q931ElementType pType )
{
	HS_UINT i;
	UnitOf *rounder = HS_NULL;
	Q931Element *element;
	Q931Message *pObj = (Q931Message *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on Q931Message_GetElement(NULL) -> null parameter" );
#endif
		return HS_NULL;
	}

	if( pObj->size == 0 ) return HS_NULL;

	rounder = &(pObj->units);
	for( i=0; i<pObj->size; i++ )
	{
		if( rounder == HS_NULL ) return HS_NULL;
		if( rounder->unit == HS_NULL ) return HS_NULL;
		element = (Q931Element *)(rounder->unit);

		if( element->discriminator == pType ) return element;
		rounder = rounder->next;
	}

	return HS_NULL;
}


/* Q931Message_DeleteElement */
HS_RESULT Q931Message_DeleteElement( void *pObject, Q931ElementType pType )
{
	HS_UINT i;
	UnitOf *prever = HS_NULL;
	UnitOf *rounder = HS_NULL;
	Q931Element *element;
	Q931Message *pObj = (Q931Message *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on Q931Message_DeleteElement(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->size == 0 ) return HS_ERR;

	rounder = &(pObj->units);
	for( i=0; i<pObj->size; i++ )
	{
		if( rounder == HS_NULL ) return HS_ERR_ASN_CONFLICT;
		if( rounder->unit == HS_NULL ) return HS_ERR_ASN_CONFLICT;
		element = (Q931Element *)(rounder->unit);

		if( element->discriminator == pType )
		{
			delete_Q931Element( rounder->unit );
			HSFree( rounder->unit );

			if( prever == HS_NULL )
			{
				rounder->unit = HS_NULL;
				pObj->size = 0;
			}
			else
			{
				prever->next = rounder->next;
				HSFree( rounder );
				pObj->size--;
			}

			return HS_OK;
		}

		prever = rounder;
		rounder = rounder->next;
	}

	return HS_ERR;
}


/* Q931Message_Decode */
HS_RESULT Q931Message_Decode( void *pObject, AsnStream *pStrm )
{
	Q931ElementType tType;
	unsigned char nLength;
	unsigned short iLength;
	unsigned char tValue[2];

	HS_RESULT tRet;
	Q931Element *element = HS_NULL;
	Q931Message *pObj = (Q931Message *)pObject;
	
	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on Q931Message_Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif
		return HS_ERR_NULL_PARAM;
	}

	if( (tRet=Q931Message_Clear(pObj)) != HS_OK ) return tRet;
	AsnStream_Aligning( pStrm );

	if( (tRet=AsnStream_BytesDecoding(pStrm,&(pObj->wholeDiscriminator),1)) != HS_OK ) return tRet;
	if( (tRet=AsnStream_BytesDecoding(pStrm,tValue,1)) != HS_OK ) return tRet;
	if( tValue[0] != 2 ) return HS_ERR_ASN_LENGTH;
	if( (tRet=AsnStream_BytesDecoding(pStrm,tValue,2)) != HS_OK ) return tRet;

	if( tValue[0] & 0x80 )	pObj->origin = FALSE;
	else					pObj->origin = TRUE;

	tValue[0] = tValue[0] & 0x7f;
	Uchar2Ushort(&(pObj->crv),tValue);

	if( (tRet=AsnStream_BytesDecoding(pStrm,tValue,1)) != HS_OK ) return tRet;
	pObj->type = (Q931MessageType)tValue[0];

	while(1)
	{
		if( pStrm->byteOffset == pStrm->size ) break;

		if( AsnStream_BytesDecoding(pStrm,tValue,1) != HS_OK ) break;
		tType = (Q931ElementType)tValue[0];

		if( tType < e_Q931ElementType_BearerCapability || tType > e_Q931ElementType_SendingComplete ) break;

		if(tType == e_Q931ElementType_UserToUser)
		{
			if( AsnStream_BytesDecoding(pStrm,tValue,2) != HS_OK ) break;
			Uchar2Ushort( &(iLength), tValue );

			if( CheckBytesOverflow(pStrm,iLength) != HS_OK ) break;
			else
			{
				AsnStream tStrm;
				new_AsnStream( &tStrm, iLength, TRUE );
				
				element = (Q931Element*)HSMalloc( sizeof(Q931Element) );
				new_Q931Element( element );

				/*uuie-protocol descriminator x.208,x.209 coded*/
				if( AsnStream_BytesDecoding(pStrm,tValue,1) != HS_OK ) break;
				iLength--;
				
				Q931Element_SetValue( element, tType, pStrm->datas+pStrm->byteOffset, iLength );
				pStrm->byteOffset += iLength;

				pObj->uuie = (ASNH225H323_UserInformation*)HSMalloc( sizeof(ASNH225H323_UserInformation) );
				new_ASNH225H323_UserInformation( pObj->uuie );
				ASNH225H323_UserInformation_MakeMold( pObj->uuie );

				memcpy(tStrm.datas,element->value,iLength);
				ASNH225H323_UserInformation_Decode( pObj->uuie, &tStrm );
			}
		}
		else if( tType == e_Q931ElementType_SendingComplete )
		{
			element = (Q931Element*)HSMalloc( sizeof(Q931Element) );
			new_Q931Element( element );
			Q931Element_SetValue( element, tType, HS_NULL, 0 );
		}
		else
		{
			if( AsnStream_BytesDecoding(pStrm,tValue,1) != HS_OK ) break;
			nLength = tValue[0];
			if( CheckBytesOverflow(pStrm,nLength) != HS_OK ) break;

			element = (Q931Element*)HSMalloc( sizeof(Q931Element) );
			new_Q931Element( element );
			Q931Element_SetValue( element, tType, pStrm->datas+pStrm->byteOffset, nLength );
			pStrm->byteOffset += nLength;
		}

		Q931Message_AddElement( pObj, element );
	}

	return HS_OK;
}


/* Q931Message_Encode */
HS_RESULT Q931Message_Encode( void *pObject, AsnStream *pStrm )
{
	HS_UINT i;
	unsigned char tValue[2];
	unsigned short iLength;

	HS_RESULT tRet;
	AsnStream uuieStrm;
	UnitOf *rounder = HS_NULL;
	Q931Element *element = HS_NULL;
	Q931Message *pObj = (Q931Message *)pObject;

	/*parameter check*/
	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on Q931Message_Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->uuie == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on Q931Message_Encode(pObj,pStrm) -> no uuie", pObj, pStrm );
#endif
		return HS_ERR_NULL_PARAM;
	}

	/*uuie encoding*/
	new_AsnStream(&uuieStrm, 1024, TRUE);
	tRet=ASNH225H323_UserInformation_Encode(pObj->uuie, &uuieStrm);
	if( tRet != HS_OK )
	{
		delete_AsnStream(&uuieStrm);
		return tRet;
	}

	AsnStream_Aligning(&uuieStrm);
	tRet = Q931Message_AddElementEx(pObj, e_Q931ElementType_UserToUser, uuieStrm.datas, uuieStrm.byteOffset);
	if( tRet != HS_OK )
	{
		delete_AsnStream(&uuieStrm);
		return tRet;
	}
	delete_AsnStream(&uuieStrm);

	/*q931 header encoding*/
	AsnStream_Aligning( pStrm );
	if( (tRet=AsnStream_BytesEncoding(pStrm,&(pObj->wholeDiscriminator),1)) != HS_OK ) return tRet;

	tValue[0] = 2;
	if( (tRet=AsnStream_BytesEncoding(pStrm,tValue,1)) != HS_OK ) return tRet;
	
	Ushort2Uchar(tValue,&(pObj->crv));
	if( pObj->origin == FALSE ) tValue[0] = tValue[0] | 0x80;
	if( (tRet=AsnStream_BytesEncoding(pStrm,tValue,2)) != HS_OK ) return tRet;

	tValue[0] = (unsigned char)(pObj->type);
	if( (tRet=AsnStream_BytesEncoding(pStrm,tValue,1)) != HS_OK ) return tRet;

	/*q931 elements encoding*/
	rounder = &(pObj->units);
	for( i=0; i<pObj->size; i++ )
	{
		if( rounder == HS_NULL ) return HS_ERR_ASN_CONFLICT;
		if( rounder->unit == HS_NULL ) return HS_ERR_ASN_CONFLICT;

		element = (Q931Element*)(rounder->unit);
		if( element->discriminator == e_Q931ElementType_SendingComplete )
		{
			tValue[0] = (unsigned char)(element->discriminator);
			if( (tRet=AsnStream_BytesEncoding(pStrm,tValue,1)) != HS_OK ) return tRet;
		}
		else if( element->discriminator == e_Q931ElementType_UserToUser )
		{
			if( element->value == HS_NULL && element->ilen != 0 ) return HS_ERR_ASN_LENGTH;

			tValue[0] = (unsigned char)(element->discriminator);
			if( (tRet=AsnStream_BytesEncoding(pStrm,tValue,1)) != HS_OK ) return tRet;

			iLength = element->ilen + 1;
			Ushort2Uchar(tValue,&(iLength));
			if( (tRet=AsnStream_BytesEncoding(pStrm,tValue,2)) != HS_OK ) return tRet;

			tValue[0] = 0x05;/*uuie-protocol descriminator x.208,x.209 coded*/
			if( (tRet=AsnStream_BytesEncoding(pStrm,tValue,1)) != HS_OK ) return tRet;

			if( element->ilen != 0 )
			{
				if( (tRet=CheckBytesOverflow(pStrm,element->ilen)) != HS_OK ) return tRet;
				memcpy(pStrm->datas+pStrm->byteOffset, element->value, element->ilen);
				pStrm->byteOffset += element->ilen;
			}
		}
		else
		{
			if( element->value == HS_NULL && element->nlen != 0 ) return HS_ERR_ASN_LENGTH;

			tValue[0] = (unsigned char)(element->discriminator);
			if( (tRet=AsnStream_BytesEncoding(pStrm,tValue,1)) != HS_OK ) return tRet;
			
			tValue[0] = element->nlen;
			if( (tRet=AsnStream_BytesEncoding(pStrm,tValue,1)) != HS_OK ) return tRet;

			if( element->nlen != 0 )
			{
				if( (tRet=CheckBytesOverflow(pStrm,element->nlen)) != HS_OK ) return tRet;
				memcpy(pStrm->datas+pStrm->byteOffset, element->value, element->nlen);
				pStrm->byteOffset += element->nlen;
			}
		}

		rounder = rounder->next;
	}

	if( rounder != HS_NULL ) return HS_ERR_ASN_CONFLICT;
	return HS_OK;
}


/* Q931Message_Print */
#ifdef HS_DEBUG_ASN_PRINT
HS_RESULT Q931Message_Print( void *pObject, HS_UINT pDepth, char *pTypeName )
{
	HS_UINT i;
	UnitOf *rounder = HS_NULL;
	Q931Message *pObj = (Q931Message *)pObject;
	
	if( pObj == HS_NULL  )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on Q931Message_Print(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}

	HSPrint( "\n%s {", pTypeName==HS_NULL? "Q931Message":pTypeName );
	
	HSPrint( "\n  Protocol Discriminator: %02X", pObj->wholeDiscriminator );
	HSPrint( "\n  %s", pObj->origin? "Source":"Dest" );
	HSPrint( "\n  CRV: %u", pObj->crv );
	HSPrint( "\n  MessageType: %d\n", pObj->type );

	rounder = &(pObj->units);
	for( i=0; i<pObj->size; i++ )
	{
		if( rounder->unit == HS_NULL )
		{
			HSPrint( "\n Error on Q931Message_Print(pObj,pDepth,%s) -> conflict size", pTypeName==HS_NULL? "Q931Message":pTypeName );
			return HS_ERR_ASN_CONFLICT;
		}

		Q931Element_Print( rounder->unit );
		rounder = rounder->next;
	}

	if( pObj->uuie != HS_NULL )	ASNH225H323_UserInformation_Print( pObj->uuie, 1, "H323_UserInformation" );
	
	HSPrint( "\n}" );
	return HS_OK;
}
#endif
