/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<H323DemonHandler.c>	2005-03-27,10:40

*/

#include "H323DemonHandler.h"



/************************************************************************************/
/* Event Handlers
*/

	/****************************************************************************************/
   /*
  User Command Messages : Event Handler {
*/
#ifdef HS_LICENCE_LOCK_H323
HS_UINT gLicenceLockMaxCall = 100;
#endif
HS_RESULT H323DemonEventMakeCall(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	HS_USHORT tSN;
	HS_RESULT tRet = HS_OK;
	TcpConnect *tTcpCon = HS_NULL;
	ICall *tCall = (ICall*)pParam;

	HS_UCHAR tDestIp[4];
	HS_USHORT tDestPort;

	if( tCall == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pStack == HS_NULL || pEndpoint == HS_NULL || pCalls == HS_NULL || pWaits == HS_NULL )
	{
		delete_ICall(tCall);
		HSFree(tCall);
		return HS_ERR_NULL_PARAM;
	}

#ifdef HS_LICENCE_LOCK_H323
	if( gLicenceLockMaxCall != 0 ) gLicenceLockMaxCall--;
	else
	{
		if( pEndpoint->CallbackCallRemoved != HS_NULL )
			pEndpoint->CallbackCallRemoved(tCall->handle, e_CallCloseReasonMax, 0x100 );

		delete_ICall(tCall);
		HSFree(tCall);
		return HS_ERR_H323;
	}
#endif

	ICall_Make(tCall, pEndpoint);

	if( tCall->sourceAliases.size == 0 )
		ASNH225AliasAddresses_Copy( &(tCall->sourceAliases), &(pEndpoint->aliases) );

	if( IGatekeeper_IsRegisted(&(pEndpoint->gatekeeper)) == HS_YES && tCall->routeState != e_RouteState_Direct )
	{
		RasMsg *arq = MakeARQ(pEndpoint, tCall, &tSN);

		tCall->routeState = e_RouteState_Route;

		if( arq == HS_NULL )
		{
			if( pEndpoint->CallbackCallRemoved != HS_NULL )
				pEndpoint->CallbackCallRemoved(tCall->handle, e_CallCloseReason_Admission, 0x100 );

			delete_ICall(tCall);
			HSFree(tCall);
			return HS_ERR_H323_MAKE_MESSAGE;
		}

		tRet = AttemptRasMessage(pStack, pEndpoint, pWaits, tCall->handle, tSN, arq);
		if( tRet != HS_OK )
		{
			delete_RasMsg(arq);
			HSFree(arq);

			if( pEndpoint->CallbackCallRemoved != HS_NULL )
				pEndpoint->CallbackCallRemoved(tCall->handle, e_CallCloseReason_Admission, 0x100 );

			delete_ICall(tCall);
			HSFree(tCall);
			return tRet;
		}

		delete_RasMsg(arq);
		HSFree(arq);

		NoLockList_AttachData(pCalls, tCall);
		return tRet;
	}

	/*point to point call*/
	tCall->routeState = e_RouteState_Direct;

	tRet = ASNH225TransportAddress2Tsap( tDestIp, &tDestPort, &(tCall->dcsaTsap) );
	if( tRet != HS_OK )
	{
		if( pEndpoint->CallbackCallRemoved != HS_NULL )
			pEndpoint->CallbackCallRemoved(tCall->handle, e_CallCloseReason_UnknownDestination, 0x100 );

		delete_ICall(tCall);
		HSFree(tCall);
		return HS_ERR_H323_NOTFOUND;
	}

	tTcpCon = newm_TcpConnectEx(tDestIp,tDestPort,pStack->h323DemonQ,tCall->handle);
	if( tTcpCon == HS_NULL )
	{
		if( pEndpoint->CallbackCallRemoved != HS_NULL )
			pEndpoint->CallbackCallRemoved(tCall->handle, e_CallCloseReason_Undefined, 0x100 );

		delete_ICall(tCall);
		HSFree(tCall);
		return HS_ERR_H323;
	}

	_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_MAKE_SOCKET_SET, 0, (LPARAM)(tCall->handle) );
	_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_TRY_CONNECT, 0, (LPARAM)tTcpCon);

	return NoLockList_AttachData(pCalls, tCall);
}


HS_RESULT H323DemonEventChangeGk(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits)
{
	HS_RESULT tRet = HS_OK;
	Registrar *qMsg = (Registrar*)pParam;

	if( qMsg == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pEndpoint == HS_NULL || pWaits==HS_NULL )
	{
		delete_Registrar(qMsg);
		HSFree(qMsg);
		return HS_ERR_NULL_PARAM;
	}

	/* unregistration the old gatekeeper.
	*/
	if( pEndpoint->gatekeeper.registrar != HS_NULL )
	{
		if( pEndpoint->gatekeeper.registrar->sn != HS_INVALID_RAS_SEQUENCE_NUMBER )
		{
			/* mean.. 'there is #RasWaitSet# and RAS timeout timer in trying registration.'
			   user want to re-registration or un-registration. so,
			   remove #RasWaitSet# and kill RAS timeout timer.
			*/
			NoLockList_DeleteRasWaitSetBySequenceNumber(pWaits,pEndpoint->gatekeeper.registrar->sn);
			HSKillTimer( HS_TID_RAS_TRANSACTION, pEndpoint->gatekeeper.registrar->sn );
		}

		if( pEndpoint->gatekeeper.registrar->registed == TRUE )
			TryUnregistration( pStack, pEndpoint, pWaits );

		delete_Registrar( pEndpoint->gatekeeper.registrar );
		HSFree(pEndpoint->gatekeeper.registrar);
		pEndpoint->gatekeeper.registrar = HS_NULL;
	}

	/* user want to re-registration or un-registration. so,
	   kill TTL timer(in registed state) or attempt timer(in unregisted state).
	*/
	HSKillTimer( HS_TID_REGISTER, 0 );

	if( qMsg->target[0] != '\0' )
	{
		pEndpoint->gatekeeper.registrar = qMsg;
		if( (tRet=TryRegistration(pStack, pEndpoint, pWaits, FALSE)) != HS_OK )
		{
			delete_Registrar( pEndpoint->gatekeeper.registrar );
			HSFree( pEndpoint->gatekeeper.registrar );
			pEndpoint->gatekeeper.registrar = HS_NULL;

			HSSetTimer(
				pEndpoint->gatekeeper.attemptTime*1000,
				pStack->h323DemonQ, HS_TID_REGISTER, 0, e_TimerType_Once
			);

			return tRet;
		}
	}
	else
	{
		HSSetTimer(
			pEndpoint->gatekeeper.attemptTime*1000,
			pStack->h323DemonQ, HS_TID_REGISTER, 0, e_TimerType_Once
		);

		delete_Registrar(qMsg);
		HSFree(qMsg);
	}

	return HS_OK;
}


HS_RESULT H323DemonEventAcceptCall(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;
	HS_CALL_HANDLE tHandle = (HS_CALL_HANDLE)pParam;

	HS_RESULT tRet;
	Q931Message *tQ931Msg = HS_NULL;

	HS_UCHAR tH245Ip[4];
	HS_USHORT tH245Port;
	TcpConnect *tTryListen = NULL;

	BOOL tFacility245Listen = HS_NO;
	BOOL tFacilityFastStart = HS_NO;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL )
		return HS_ERR_NULL_PARAM;

	if( (tChain=NoLockList_FindCallByHandle(pCalls, tHandle)) == HS_NULL )
		return HS_ERR_H323_NOTFOUND;
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
		return HS_ERR_H323_CONFLICT;
	if( tCall->q931State != e_Q931State_ReadyConnect )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! CONFLICT(ACCEPT_CALL)" );
#endif
		return HS_ERR_H323;
	}

	/*h245 listen*/
	if( tCall->isTunneling==HS_NO &&
		(tCall->h245ListenPoint == e_H245ListenPoint_Facility ||
		tCall->h245ListenPoint == e_H245ListenPoint_Connect)
	)
	{
		tRet = ASNH225TransportAddress2Tsap(tH245Ip, &tH245Port, &(pEndpoint->csaTsap));
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_H323
				HSPrint( "\n $ H245 Listen Fail on AcceptCall." );
#endif
				return tRet;
		}
		else
		{
			tTryListen = newm_TcpConnectEx(tH245Ip,(HS_USHORT)(tCall->h245ListenPort),0,tCall->handle);
			if( tTryListen == HS_NULL )
			{
#ifdef HS_DEBUG_H323
				HSPrint( "\n $ H245 Listen Fail on AcceptCall." );
#endif
				return HS_ERR_H323_MAKE_QMESSAGE;
			}
			else
			{
				_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_TRY_LISTEN, 0, (LPARAM)tTryListen);
				Sleep(10);	/*time to try listen*/
			}
		}
	}

	/* sending facility, if faststart response point is a facility or h245 listen point is a facility
	*/
	tFacility245Listen = (tCall->isFastStart==HS_YES && tCall->fastStartPoint == e_FastStartResponsePoint_Facility);
	tFacilityFastStart = (tCall->isTunneling==HS_NO && tCall->h245ListenPoint == e_H245ListenPoint_Facility);

	if( tFacility245Listen || tFacilityFastStart )
	{
		ASNH225FacilityReasonChoice tFacilityReason;

		if( tFacility245Listen ) tFacilityReason = e_ASNH225FacilityReasonChoice_startH245;
		else					 tFacilityReason = e_ASNH225FacilityReasonChoice_undefinedReason;

		if( (tQ931Msg=MakeQ931Facility(pStack,pEndpoint,pWaits,tCall,tFacilityReason)) == HS_NULL )
		{
			if( tFacility245Listen ) tCall->h245ListenPoint = e_H245ListenPoint_Connect;
			if( tFacilityFastStart ) tCall->fastStartPoint = e_FastStartResponsePoint_Connect;
		}
		else
		{
			if( AttemptQ931Message(pStack, pEndpoint, pWaits, tQ931Msg, tCall) != HS_OK )
			{
				if( tFacility245Listen ) tCall->h245ListenPoint = e_H245ListenPoint_Connect;
				if( tFacilityFastStart ) tCall->fastStartPoint = e_FastStartResponsePoint_Connect;
			}
			else
			{
				if( tQ931Msg->uuie != HS_NULL &&
					ASNH225Facility_UUIE_IsIncludeOptionField(
						tQ931Msg->uuie->m_h323_uu_pdu.m_h323_message_body.alter,
						e_ASNH225Facility_UUIEOptionMap_fastStart
					) && tCall->fsActive == HS_NO
				) OpenMediaWithFastStart(pStack, pEndpoint, tCall);
			}

			delete_Q931Message(tQ931Msg);
			HSFree(tQ931Msg);
			tQ931Msg = HS_NULL;
		}
	}

	/*sending connect*/
	if( (tQ931Msg=MakeQ931Connect(pStack,pEndpoint,pWaits,tCall)) == HS_NULL )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! MakeQ931Connect Error on AcceptCall, Call Closing.." );
#endif
		return CallClosing(pStack, pEndpoint, pCalls, pWaits, tCall,
			e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
			e_Q931CauseType_NormalCallClearing
		);
	}
	if( (tRet=AttemptQ931Message(pStack, pEndpoint, pWaits, tQ931Msg, tCall)) != HS_OK )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! AttemptQ931Connect Error on AcceptCall, Call Closing.. -> tRet(%u)", tRet );
#endif

		delete_Q931Message(tQ931Msg);
		HSFree(tQ931Msg);

		return CallClosing(pStack, pEndpoint, pCalls, pWaits, tCall,
			e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
			e_Q931CauseType_NormalCallClearing
		);
	}
	if( tQ931Msg->uuie != HS_NULL &&
		ASNH225Connect_UUIE_IsIncludeOptionField(
			tQ931Msg->uuie->m_h323_uu_pdu.m_h323_message_body.alter,
			e_ASNH225Connect_UUIEOptionMap_fastStart
		) && tCall->fsActive == HS_NO
	) OpenMediaWithFastStart(pStack, pEndpoint, tCall);

	delete_Q931Message(tQ931Msg);
	HSFree(tQ931Msg);

	tCall->q931State = e_Q931State_Connect;
	/* start IRR message timer, if irrFrequency exist
		NOTE: this function must called in #e_Q931State_Connect# state
	*/
	StartIRRTimer(pStack, tCall);

	return HS_OK;
}


HS_RESULT H323DemonEventRemoveCall(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;
	QMRemoveCall *qMsg = (QMRemoveCall*)pParam;
	Q931CauseType tCause;

	if( qMsg == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL )
	{
		delete_QMRemoveCall(qMsg);
		return HS_ERR_NULL_PARAM;
	}
	if( (tChain=NoLockList_FindCallByHandle(pCalls,qMsg->handle)) == HS_NULL )
	{
		delete_QMRemoveCall(qMsg);
		return HS_ERR_H323_NOTFOUND;
	}
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
	{
		delete_QMRemoveCall(qMsg);
		return HS_ERR_H323_CONFLICT;
	}

	switch(tCall->q931State)
	{
#if 0
		case e_Q931State_Idle:
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! CONFLICT(REMOVE_CALL)" );
#endif
			delete_QMRemoveCall(qMsg);
			return HS_ERR_H323;
#endif
		case e_Q931State_ReadySetup:
			_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_REMOVE_SOCKET_SET, 0, (LPARAM)(tCall->handle));
			if( pEndpoint->CallbackCallRemoved != HS_NULL )
				pEndpoint->CallbackCallRemoved(tCall->handle,qMsg->reason,e_Q931CauseType_CallRejected);
			delete_QMRemoveCall(qMsg);
			return NoLockList_DeleteChain(pCalls,tChain);

		case e_Q931State_Idle:
		case e_Q931State_Setup:
		case e_Q931State_CallProceed:
		case e_Q931State_Alert:
		case e_Q931State_ReadyConnect:
		case e_Q931State_Connect:
			delete_QMRemoveCall(qMsg);
			tCause = e_Q931CauseType_NormalCallClearing;
			switch(qMsg->reason)
			{
				case e_CallCloseReason_Busy:			tCause=e_Q931CauseType_UserBusy;	break;
				case e_CallCloseReason_NoAnswer:		tCause=e_Q931CauseType_NoAnswer;	break;
				case e_CallCloseReason_InvalidNumber:	tCause=e_Q931CauseType_UnallocatedNumber;	break;
			}
			return CallClosing(
				pStack, pEndpoint, pCalls, pWaits, tCall, qMsg->reason,
				e_ASNH225DisengageReasonChoice_normalDrop, tCause
			);
	}/*switch(Q931State)*/

#ifdef HS_DEBUG_H323
	HSPrint( "\n $ ! OutOfState(REMOVE_CALL), Call Closing.." );
#endif
	CallClosing(
		pStack, pEndpoint, pCalls, pWaits, tCall, qMsg->reason,
		e_ASNH225DisengageReasonChoice_normalDrop,
		e_Q931CauseType_NormalCallClearing
	);
	delete_QMRemoveCall(qMsg);
	return HS_ERR_H323_NOUNIT;
}


HS_RESULT H323DemonEventChangeBandwidth	(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;
	QMBandwidth *qMsg = (QMBandwidth*)pParam;

	HS_USHORT tSN;
	HS_RESULT tRet = HS_OK;
	RasMsg *brq = HS_NULL;

	if( qMsg == HS_NULL ) return HS_ERR_NULL_PARAM;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL )
	{
		delete_QMBandwidth(qMsg);
		return HS_ERR_NULL_PARAM;
	}
	if( IGatekeeper_IsRegisted(&(pEndpoint->gatekeeper))==HS_NO )
	{
		delete_QMBandwidth(qMsg);
		return HS_OK;
	}
	if( (tChain=NoLockList_FindCallByHandle(pCalls,qMsg->handle)) == HS_NULL )
	{
		delete_QMBandwidth(qMsg);
		return HS_ERR_H323_NOTFOUND;
	}
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
	{
		NoLockList_DeleteChain(pCalls,tChain);
		delete_QMBandwidth(qMsg);
		return HS_ERR_H323_CONFLICT;
	}
	if( tCall->q931State != e_Q931State_Connect )
	{
		delete_QMBandwidth(qMsg);
		return HS_OK;
	}

	/* sending brq
	*/
	if( (brq=MakeBRQ(pEndpoint,tCall,qMsg->bandwidth,&tSN)) == HS_NULL )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! MakeBRQ Error on ConnectHandler(), Skip." );
#endif
		delete_QMBandwidth(qMsg);
		return HS_ERR_H323_MAKE_MESSAGE;
	}

	tRet = AttemptRasMessage(pStack, pEndpoint, pWaits, tCall->handle, tSN, brq);

	delete_QMBandwidth(qMsg);
	delete_RasMsg(brq);
	HSFree(brq);
	return tRet;
}


HS_RESULT H323DemonEventH245UserInputIndication(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	HS_RESULT tRet = HS_OK;
	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;
	ControlMsg *tH245Msg = HS_NULL;
	QMH245UserInputIndication *qMsg = (QMH245UserInputIndication*)pParam;

	/* parameter check
	*/
	if( qMsg == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL )
	{
		delete_QMH245UserInputIndication(qMsg);
		return HS_ERR_NULL_PARAM;
	}

	/* finding call
	*/
	if( (tChain=NoLockList_FindCallByHandle(pCalls,qMsg->handle)) == HS_NULL )
	{
		delete_QMH245UserInputIndication(qMsg);
		return HS_ERR_H323_NOTFOUND;
	}
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
	{
		NoLockList_DeleteChain(pCalls,tChain);
		delete_QMH245UserInputIndication(qMsg);
		return HS_ERR_H323_CONFLICT;
	}

	tH245Msg = qMsg->msg;
	qMsg->msg = HS_NULL;
	delete_QMH245UserInputIndication(qMsg);

	if( tH245Msg == HS_NULL ) return HS_ERR_H323_NOUNIT;

	tRet = NoLockList_AttachData( &(tCall->h245Charge), tH245Msg );
	if( tRet != HS_OK )
	{
		delete_ControlMsg(tH245Msg);
		HSFree(tH245Msg);
		return tRet;
	}

	return DischargeH245Message(pStack, pEndpoint, pWaits, tCall);
}


HS_RESULT H323DemonEventChangeAliases(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	HS_UINT i;
	HS_RESULT tRet = HS_OK;
	ChainUnit *rounder = HS_NULL;
	NoLockList *tList = (NoLockList*)pParam;

	if( tList==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;

	if( IGatekeeper_IsRegisted(&(pEndpoint->gatekeeper)) == HS_YES ) return HS_ERR_H323_CONFLICT;
	if( pCalls->size > 0 ) return HS_ERR_H323_CONFLICT;

	NoLockList_Clear( &(pEndpoint->aliases) );

	rounder = tList->units;
	for( i=0; i<tList->size; i++ )
	{
		if( rounder==HS_NULL ) return HS_ERR_H323_CONFLICT;
		if( rounder->data==HS_NULL ) return HS_ERR_H323_NOUNIT;

		HS_TRY( NoLockList_AttachData(&(pEndpoint->aliases), rounder->data) )

		rounder->data = HS_NULL;
		rounder = rounder->next;
	}

	return HS_OK;
}
/*
 User Command Messages : Event Handler }
  */
	/****************************************************************************************/


HS_RESULT H323DemonEventTimer(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	HS_RESULT tRet = HS_OK;
	HSTimerEvent *tTimerEvent = (HSTimerEvent*)pParam;

	if( tTimerEvent == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL )
	{
		deletem_HSTimerEvent(tTimerEvent);
		return HS_ERR_NULL_PARAM;
	}

#ifdef HS_DEBUG_H323DEMON
	HSPrint(" (%u/%u)", tTimerEvent->mTid, tTimerEvent->mSub);
#endif

	switch( tTimerEvent->mTid )
	{
		/* RAS transaction timeout
		*/
		case HS_TID_RAS_TRANSACTION:
			tRet = RasTransactionTimerHandler(pStack, pEndpoint, pCalls, pWaits, tTimerEvent);
			break;
		/* register timeout or register TTL expired
		*/
		case HS_TID_REGISTER:
			tRet = RegisterTimerHandler(pStack, pEndpoint, pWaits);
			break;
		/* irrFrequency timeout
		*/
		case HS_TID_IRR:
			tRet = IRRTimerHandler(pStack, pEndpoint, pCalls, pWaits, tTimerEvent);
			break;
		/* unknown timer id
		*/
		default:
			HSKillTimer(tTimerEvent->mTid, tTimerEvent->mSub);
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	deletem_HSTimerEvent(tTimerEvent);
	return tRet;
}


HS_RESULT H323DemonEventQ931Connected(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	HS_RESULT tRet;
	Q931Message *tQ931 = HS_NULL;

	HS_UCHAR tH245Ip[4];
	HS_USHORT tH245Port;
	TcpConnect *tTryListen = NULL;

	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;
	TcpConnected *tConnected = (TcpConnected*)pParam;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL || tConnected==HS_NULL ) return HS_ERR_NULL_PARAM;

	/*get call from list*/
	if( (tChain=NoLockList_FindCallByHandle(pCalls,tConnected->mHandle)) == HS_NULL )
	{
		deletem_TcpConnected(tConnected);
		return HS_ERR_H323_NOTFOUND;
	}
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
	{
		deletem_TcpConnected(tConnected);
		return HS_ERR_H323_CONFLICT;
	}

	/* socket check
	*/
	if( tConnected->mSocket == HS_INVALID_SOCKET )
	{
		if( ReleaseAdmission(pStack, pEndpoint, tCall, pWaits, e_ASNH225DisengageReasonChoice_undefinedReason) != HS_OK )
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ DRQ Fail, Skip." );
#endif

		if( pEndpoint->CallbackCallRemoved != HS_NULL )
			pEndpoint->CallbackCallRemoved(tCall->handle,e_CallCloseReason_Connection,0x100);

		deletem_TcpConnected(tConnected);
		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_REMOVE_SOCKET_SET, 0, (LPARAM)(tCall->handle) );
		return NoLockList_DeleteChain(pCalls, tChain);
	}

	/*h245 listen : early h245 mode*/
	if( tCall->isTunneling==HS_NO && tCall->h245ListenPoint==e_H245ListenPoint_Setup )
	{
		tRet = ASNH225TransportAddress2Tsap(tH245Ip, &tH245Port, &(pEndpoint->csaTsap));
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_H323
				HSPrint( "\n $ H245 Listen Fail on Setup, Do Tunneling." );
#endif
				tCall->isTunneling = HS_YES;
				tRet = HS_ERR_H323;
		}
		else
		{
			tTryListen = newm_TcpConnectEx(tH245Ip,(HS_USHORT)(tCall->h245ListenPort),0,tCall->handle);
			if( tTryListen==NULL )
			{
#ifdef HS_DEBUG_H323
				HSPrint( "\n $ H245 Listen Fail on Setup, Do Tunneling." );
#endif
				tCall->isTunneling = HS_YES;
				tRet = HS_ERR_H323_MAKE_QMESSAGE;
			}
			else
			{
				_HSThreadSendMessage(pStack->enetDemonQ,HS_QMESSAGE_TRY_LISTEN,0,(LPARAM)tTryListen);
				Sleep(10);	/*time to try listen*/
			}
		}
	}

	/*ready to tunneled h245 messages*/
	if( tCall->isTunneling == HS_YES )
	{
		NoLockList_AttachData( &(tCall->h245Charge), MakeTerminalCapabilitySet(pEndpoint, tCall) );
		tCall->h245State.msdValue = (HS_UINT)(rand()%HS_MASTER_SLAVE_MAX_VALUE);
		NoLockList_AttachData( &(tCall->h245Charge), MakeMasterSlaveDetermination(pEndpoint, tCall->h245State.msdValue) );
	}

	/*make message*/
	tQ931 = MakeQ931Setup(pStack, pEndpoint, pWaits, tCall);
	if( tQ931 == HS_NULL )
	{
		if( ReleaseAdmission(pStack, pEndpoint, tCall, pWaits, e_ASNH225DisengageReasonChoice_undefinedReason) != HS_OK )
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ DRQ Fail, Skip." );
#endif

		if( pEndpoint->CallbackCallRemoved != HS_NULL )
			pEndpoint->CallbackCallRemoved(tCall->handle,e_CallCloseReason_Undefined,0x100);

		deletem_TcpConnected(tConnected);
		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_REMOVE_SOCKET_SET, 0, (LPARAM)(tCall->handle) );
		return NoLockList_DeleteChain(pCalls, tChain);
	}

	/*send message*/
	if( (tRet=AttemptQ931Message(pStack, pEndpoint, pWaits, tQ931, tCall)) != HS_OK )
	{
		if( ReleaseAdmission(pStack, pEndpoint, tCall, pWaits, e_ASNH225DisengageReasonChoice_undefinedReason) != HS_OK )
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ DRQ Fail, Skip." );
#endif

		if( pEndpoint->CallbackCallRemoved != HS_NULL )
			pEndpoint->CallbackCallRemoved(tCall->handle,e_CallCloseReason_Undefined,0x100);

		delete_Q931Message(tQ931);
		HSFree(tQ931);
		deletem_TcpConnected(tConnected);
		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_REMOVE_SOCKET_SET, 0, (LPARAM)(tCall->handle) );
		NoLockList_DeleteChain(pCalls, tChain);
		return tRet;
	}

	if( tRet == HS_OK ) tCall->q931State = e_Q931State_Setup;

	delete_Q931Message(tQ931);
	HSFree(tQ931);
	deletem_TcpConnected(tConnected);
	return tRet;
}


HS_RESULT H323DemonEventH245Connected(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;
	TcpConnected *qMsg = (TcpConnected*)pParam;

	if( qMsg==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL )
		return HS_ERR_NULL_PARAM;
	if( (tChain=NoLockList_FindCallByHandle(pCalls,qMsg->mHandle)) == HS_NULL )
	{
		deletem_TcpConnected(qMsg);
		return HS_ERR_H323_NOTFOUND;
	}
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
	{
		deletem_TcpConnected(qMsg);
		return HS_ERR_H323_CONFLICT;
	}

	if( tCall->isTunneling == HS_NO )
	{
		if( qMsg->mSocket == HS_INVALID_SOCKET)
		{
#ifdef HS_DEBUG_H323DEMON
			HSPrint( "\n Error : Fail to make h245 socket." );
#endif
			deletem_TcpConnected(qMsg);
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, tCall,
				e_CallCloseReason_H245Session, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}

		tCall->h245State.session = HS_YES;

		if( tCall->h245State.sendTCS == HS_NO )
			NoLockList_AttachData( &(tCall->h245Charge), MakeTerminalCapabilitySet(pEndpoint, tCall) );

		if( tCall->h245State.sendMSD == HS_NO )
		{
			tCall->h245State.msdValue = (HS_UINT)(rand()%HS_MASTER_SLAVE_MAX_VALUE);
			NoLockList_AttachData( &(tCall->h245Charge), MakeMasterSlaveDetermination(pEndpoint, tCall->h245State.msdValue) );
		}

		DischargeH245Message(pStack, pEndpoint, pWaits, tCall);
	}
	else
	{
#ifdef HS_DEBUG_H323DEMON
		HSPrint( "\n Error : h245 socket made in tunneling mode." );
#endif
		closesocket(qMsg->mSocket);
	}

	deletem_TcpConnected(qMsg);
	return HS_OK;
}


HS_RESULT H323DemonEventReceiveRasMessage(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	HS_RESULT tRet = HS_OK;
	AsnStream tStrm;
	RasMsg *eMsg = HS_NULL;
	QMRasMessage *qMsg = (QMRasMessage*)pParam;

	if( qMsg==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( qMsg->raw==HS_NULL || qMsg->len==0 ) return HS_ERR_NULL_PARAM;

	/* alloc RAS message format
	*/
	if( (eMsg=(RasMsg*)HSMalloc(sizeof(RasMsg))) == HS_NULL )
	{
		delete_QMRasMessage(qMsg);
		return HS_ERR_H323_MALLOC;
	}
	new_RasMsg(eMsg);
	new_AsnStream(&tStrm, qMsg->len, TRUE);
	memcpy(tStrm.datas,qMsg->raw,qMsg->len);

	/* ethernet rawdata callback function check
	*/
	if( pEndpoint->CallbackReceiveRasRawData != HS_NULL )
	{
		if( pEndpoint->CallbackReceiveRasRawData(pStack,HS_INVALID_CALL_HANDLE,&tStrm) == FALSE )
		{
			delete_RasMsg(eMsg);
			HSFree(eMsg);
			delete_QMRasMessage(qMsg);
			delete_AsnStream(&tStrm);
			return HS_OK;
		}
	}

	/* decode message
	*/
	if( (tRet=RasMsg_Decode(eMsg,&tStrm)) != HS_OK )
	{
		if( pEndpoint->CallbackAsnDecodingError != HS_NULL )
			pEndpoint->CallbackAsnDecodingError(pStack,HS_INVALID_CALL_HANDLE,&tStrm);

		delete_RasMsg(eMsg);
		HSFree(eMsg);
		delete_QMRasMessage(qMsg);
		delete_AsnStream(&tStrm);
		return tRet;
	}

	/* RAS message callback function check
	*/
	if( pEndpoint->CallbackReceiveRasMessage != HS_NULL )
	{
		if( pEndpoint->CallbackReceiveRasMessage(pStack,HS_INVALID_CALL_HANDLE,eMsg) == FALSE )
		{
			delete_RasMsg(eMsg);
			HSFree(eMsg);
			delete_QMRasMessage(qMsg);
			delete_AsnStream(&tStrm);
			return HS_OK;
		}
	}

	/* RAS message handling
	*/
	switch( eMsg->choice )
	{
		/* RAS response messages from GK
		*/
		case e_ASNH225RasMessageChoice_gatekeeperConfirm:
			tRet=GCFHandler( eMsg->alter, pStack, pEndpoint, pWaits );
			break;
		case e_ASNH225RasMessageChoice_gatekeeperReject:
			tRet=GRJHandler( eMsg->alter, pStack, pEndpoint, pWaits );
			break;
		case e_ASNH225RasMessageChoice_registrationConfirm:
			tRet=RCFHandler( eMsg->alter, pStack, pEndpoint, pWaits );
			break;
		case e_ASNH225RasMessageChoice_registrationReject:
			tRet=RRJHandler( eMsg->alter, pStack, pEndpoint, pWaits );
			break;
		case e_ASNH225RasMessageChoice_unregistrationConfirm:
			tRet=UCFHandler( eMsg->alter, pEndpoint, pWaits );
			break;
		case e_ASNH225RasMessageChoice_unregistrationReject:
			tRet=URJHandler( eMsg->alter, pEndpoint, pWaits );
			break;
		case e_ASNH225RasMessageChoice_admissionConfirm:
			tRet=ACFHandler( eMsg->alter, pStack, pEndpoint, pCalls, pWaits );
			break;
		case e_ASNH225RasMessageChoice_admissionReject:
			tRet=ARJHandler( eMsg->alter, pStack, pEndpoint, pCalls, pWaits );
			break;
		case e_ASNH225RasMessageChoice_bandwidthConfirm:
			tRet=BCFHandler( eMsg->alter, pEndpoint, pCalls, pWaits );
			break;
		case e_ASNH225RasMessageChoice_bandwidthReject:
			tRet=BRJHandler( eMsg->alter, pEndpoint, pWaits );
			break;
		case e_ASNH225RasMessageChoice_disengageConfirm:
			tRet=DCFHandler( eMsg->alter, pEndpoint, pWaits );
			break;
		case e_ASNH225RasMessageChoice_disengageReject:
			tRet=DRJHandler( eMsg->alter, pEndpoint, pWaits );
			break;
		case e_ASNH225RasMessageChoice_infoRequestAck:
		case e_ASNH225RasMessageChoice_infoRequestNak:
			tRet=HS_OK;
			break;
		/* RAS request messages from GK
		*/
		case e_ASNH225RasMessageChoice_unregistrationRequest:
			tRet=URQHandler( eMsg->alter, pStack, pEndpoint, pWaits );
			break;
		case e_ASNH225RasMessageChoice_bandwidthRequest:
			tRet=BRQHandler( eMsg->alter, pStack, pEndpoint, pCalls, pWaits );
			break;
		case e_ASNH225RasMessageChoice_disengageRequest:
			tRet=DRQHandler( eMsg->alter, pStack, pEndpoint, pCalls, pWaits );
			break;
		case e_ASNH225RasMessageChoice_infoRequest:
			tRet=IRQHandler( eMsg->alter, pStack, pEndpoint, pCalls, pWaits );
			break;
		/* not receivable messages on endoint
		*/
		case e_ASNH225RasMessageChoice_gatekeeperRequest:
		case e_ASNH225RasMessageChoice_registrationRequest:
		case e_ASNH225RasMessageChoice_admissionRequest:
		case e_ASNH225RasMessageChoice_locationRequest:
		case e_ASNH225RasMessageChoice_infoRequestResponse:
		case e_ASNH225RasMessageChoice_resourcesAvailableIndicate:
		case e_ASNH225RasMessageChoice_serviceControlIndication:
		case e_ASNH225RasMessageChoice_serviceControlResponse:
			tRet=UnknownRASHandler( eMsg, pStack, pEndpoint, pWaits );
			break;
		/* stack not supported messages, yet.
		*/
		case e_ASNH225RasMessageChoice_locationConfirm:
		case e_ASNH225RasMessageChoice_locationReject:
		case e_ASNH225RasMessageChoice_nonStandardMessage:
		case e_ASNH225RasMessageChoice_unknownMessageResponse:
		case e_ASNH225RasMessageChoice_requestInProgress:
		case e_ASNH225RasMessageChoice_resourcesAvailableConfirm:
			tRet = HS_ERR_NOT_SUPPORT_YET;
			break;
		default:
			tRet = HS_ERR_H323_CONFLICT;
			break;
	}

	/*closing*/
	delete_RasMsg(eMsg);
	HSFree(eMsg);
	delete_QMRasMessage(qMsg);
	delete_AsnStream(&tStrm);
	return tRet;
}


HS_RESULT H323DemonEventReceiveQ931Message(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	HS_UINT i;
	HS_RESULT tRet = HS_OK;
	AsnStream tStrm;
	Q931Message *eMsg = HS_NULL;
	QMTcpMessage *qMsg = (QMTcpMessage*)pParam;

	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;

	if( qMsg==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( qMsg->raw==HS_NULL || qMsg->len==0 ) return HS_ERR_NULL_PARAM;

	/* finding call
	*/
	if( (tChain=NoLockList_FindCallByHandle(pCalls, qMsg->handle)) == HS_NULL )
	{
#ifdef HS_DEBUG_H323DEMON
		HSPrint( "\n $ Receive Q931 Message of Invalid CallChain, delete SocketSet" );
#endif
		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_REMOVE_SOCKET_SET, 0, (LPARAM)(qMsg->handle));

		delete_QMTcpMessage(qMsg);
		return HS_ERR_H323_NOTFOUND;
	}
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
	{
#ifdef HS_DEBUG_H323DEMON
		HSPrint( "\n $ Receive Q931 Message of Invalid Call, delete Chain/SocketSet" );
#endif
		NoLockList_DeleteChain(pCalls, tChain);
		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_REMOVE_SOCKET_SET, 0, (LPARAM)(qMsg->handle));

		delete_QMTcpMessage(qMsg);
		return HS_ERR_H323_CONFLICT;
	}
	/* decode message
	*/
	if( (eMsg=(Q931Message*)HSMalloc(sizeof(Q931Message))) == HS_NULL )
	{
		delete_QMTcpMessage(qMsg);
		return HS_ERR_H323_MALLOC;
	}

	new_Q931Message(eMsg);
	new_AsnStream(&tStrm, qMsg->len, TRUE);
	memcpy(tStrm.datas,qMsg->raw,qMsg->len);
	/* rawdata callback function check
	*/
	if( pEndpoint->CallbackReceiveQ931RawData != HS_NULL )
	{
		if( pEndpoint->CallbackReceiveQ931RawData(pStack,tCall->handle,&tStrm) == FALSE )
		{
			delete_AsnStream(&tStrm);
			delete_Q931Message(eMsg);
			HSFree(eMsg);
			delete_QMTcpMessage(qMsg);
			return HS_OK;
		}
	}

	/* decode message
	*/
	if( (tRet=Q931Message_Decode(eMsg,&tStrm)) != HS_OK )
	{
		if( pEndpoint->CallbackAsnDecodingError != HS_NULL )
			pEndpoint->CallbackAsnDecodingError(pStack,tCall->handle,&tStrm);

		delete_Q931Message(eMsg);
		HSFree(eMsg);
		delete_QMTcpMessage(qMsg);
		delete_AsnStream(&tStrm);
		return tRet;
	}
	if( eMsg->uuie == HS_NULL )
	{
#ifdef HS_DEBUG_H323DEMON
		HSPrint( "\n $ Q931 Message have no UUIE." );
#endif
		delete_Q931Message(eMsg);
		HSFree(eMsg);
		delete_QMTcpMessage(qMsg);
		delete_AsnStream(&tStrm);
		return HS_ERR_H323_NOUNIT;
	}

	/* uuiesRequested IRR message send
	*/
	if( IsIRRwithPDU(eMsg->uuie->m_h323_uu_pdu.m_h323_message_body.choice, &(tCall->uuiesRequested)) )
	{
		HS_USHORT tSN;
		RasMsg *tIrr = MakeIRRwithPDU(pEndpoint, tCall, &(eMsg->uuie->m_h323_uu_pdu), HS_NO, &tSN);

		if( tIrr != HS_NULL )
		{
			AttemptRasMessage(pStack, pEndpoint, pWaits, tCall->handle, HS_INVALID_RAS_SEQUENCE_NUMBER, tIrr);
			delete_RasMsg(tIrr);
			HSFree(tIrr);
		}
	}

	/*message callback function check*/
	if( pEndpoint->CallbackReceiveQ931Message != HS_NULL )
	{
		if( pEndpoint->CallbackReceiveQ931Message(pStack,tCall->handle,eMsg) == FALSE )
		{
			delete_Q931Message(eMsg);
			HSFree(eMsg);
			delete_QMTcpMessage(qMsg);
			delete_AsnStream(&tStrm);
			return HS_OK;
		}
	}

	/*q931 message handling*/
	switch( eMsg->uuie->m_h323_uu_pdu.m_h323_message_body.choice )
	{
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_setup:
			tRet=SetupHandler(eMsg, pStack, pEndpoint, pCalls, pWaits, tCall);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_callProceeding:
			tRet=CallProceedHandler(eMsg, pStack, pEndpoint, pCalls, pWaits, tCall);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_connect:
			tRet=ConnectHandler(eMsg, pStack, pEndpoint, pCalls, pWaits, tCall);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_alerting:
			tRet=AlertHandler(eMsg, pStack, pEndpoint, pCalls, pWaits, tCall);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_information:
			tRet=InformationHandler(eMsg);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_releaseComplete:
			tRet=ReleaseCompleteHandler(eMsg, pStack, pEndpoint, pCalls, pWaits, tCall);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_facility:
			tRet=FacilityHandler(eMsg, pStack, pEndpoint, pCalls, pWaits, tCall);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_progress:
			tRet=ProgressHandler(eMsg);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_empty:
			tRet=EmptyHandler(eMsg);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_status:
			tRet=StatusHandler(eMsg);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_statusInquiry:
			tRet=StatusInquiryHandler(eMsg);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_setupAcknowledge:
			tRet=SetupAcknowledgeHandler(eMsg);
			break;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_notify:
			tRet=NotifyHandler(eMsg);
			break;
		default:
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	/*H.245 tunneling procedure*/
	if( tRet == HS_OK )
	{
		if( ASNH225H323_UU_PDU_IsIncludeOptionField(&(eMsg->uuie->m_h323_uu_pdu), e_ASNH225H323_UU_PDUOptionMap_h245Tunneling) &&
			eMsg->uuie->m_h323_uu_pdu.m_h245Tunneling.value &&
			eMsg->uuie->m_h323_uu_pdu.m_h323_message_body.choice != e_ASNH225H323_UU_PDU_h323_message_bodyChoice_releaseComplete
		)
		{
			tCall->isTunneling = HS_YES;

			if( ASNH225H323_UU_PDU_IsIncludeOptionField(&(eMsg->uuie->m_h323_uu_pdu), e_ASNH225H323_UU_PDUOptionMap_h245Control) )
			{
				HS_UCHAR *tRaw = HS_NULL;
				QMTcpMessage *qTunnelMsg = HS_NULL;
				AsnOctetString *tOctetString = HS_NULL;

				for( i=0; i<eMsg->uuie->m_h323_uu_pdu.m_h245Control.size; i++ )
				{
					tOctetString = AsnSequenceOf_GetUnit( &(eMsg->uuie->m_h323_uu_pdu.m_h245Control), i );
					if( tOctetString == HS_NULL ) continue;
					if( tOctetString->value==HS_NULL || tOctetString->length==0 ) continue;
					if( (tRaw=(HS_UCHAR*)HSMalloc(tOctetString->length)) == HS_NULL ) continue;
					memcpy(tRaw,tOctetString->value,tOctetString->length);
					if( (qTunnelMsg=new_QMTcpMessage(tRaw,tOctetString->length,tCall->handle)) == HS_NULL )
					{
						HSFree(tRaw);	tRaw = HS_NULL;
						continue;
					}

					HSPrint( "\n[+Tunnel] H323->H245_MESSAGE" );
					tRet=H323DemonEventReceiveH245Message((long)qTunnelMsg,pStack,pEndpoint,pCalls,pWaits);
					HSPrint( "\n[-Tunnel] H323:%u", tRet );
				}
			}
		}
	}

	/*closing*/
	delete_Q931Message(eMsg);
	HSFree(eMsg);
	delete_QMTcpMessage(qMsg);
	delete_AsnStream(&tStrm);
	return tRet;
}


HS_RESULT H323DemonEventReceiveH245Message(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	HS_RESULT tRet = HS_OK;
	AsnStream tStrm;
	ControlMsg *eMsg = HS_NULL;
	QMTcpMessage *qMsg = (QMTcpMessage*)pParam;

	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;

	if( qMsg==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL )
		return HS_ERR_NULL_PARAM;
	if( qMsg->raw==HS_NULL || qMsg->len==0 ) return HS_ERR_NULL_PARAM;

	/*get call*/
	if( (tChain=NoLockList_FindCallByHandle(pCalls, qMsg->handle)) == HS_NULL )
	{
#ifdef HS_DEBUG_H323DEMON
		HSPrint( "\n $ Receive H245 Message of Invalid CallChain, delete SocketSet" );
#endif
		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_REMOVE_SOCKET_SET, 0, (LPARAM)(qMsg->handle));

		delete_QMTcpMessage(qMsg);
		return HS_ERR_H323_NOTFOUND;
	}
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
	{
#ifdef HS_DEBUG_H323DEMON
		HSPrint( "\n $ Receive H245 Message of Invalid Call, delete Chain/SocketSet" );
#endif
		NoLockList_DeleteChain(pCalls, tChain);
		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_REMOVE_SOCKET_SET, 0, (LPARAM)(qMsg->handle));

		delete_QMTcpMessage(qMsg);
		return HS_ERR_H323_CONFLICT;
	}

	/*decode message*/
	if( (eMsg=(ControlMsg*)HSMalloc(sizeof(ControlMsg))) == HS_NULL )
	{
		delete_QMTcpMessage(qMsg);
		return HS_ERR_H323_MALLOC;
	}

	new_ControlMsg(eMsg);
	new_AsnStream(&tStrm, qMsg->len, TRUE);
	memcpy(tStrm.datas,qMsg->raw,qMsg->len);
	/*rawdata callback function check*/
	if( pEndpoint->CallbackReceiveH245RawData != HS_NULL )
	{
		if( pEndpoint->CallbackReceiveH245RawData(pStack,tCall->handle,&tStrm) == FALSE )
		{
			delete_AsnStream(&tStrm);
			delete_ControlMsg(eMsg);
			HSFree(eMsg);
			delete_QMTcpMessage(qMsg);
			return tRet;
		}
	}

	/*decode message*/
	if( (tRet=ControlMsg_Decode(eMsg,&tStrm)) != HS_OK )
	{
		if( pEndpoint->CallbackAsnDecodingError != HS_NULL )
			pEndpoint->CallbackAsnDecodingError(pStack,tCall->handle,&tStrm);

		delete_ControlMsg(eMsg);
		HSFree(eMsg);
		delete_QMTcpMessage(qMsg);
		delete_AsnStream(&tStrm);
		return tRet;
	}

	/*message callback function check*/
	if( pEndpoint->CallbackReceiveH245Message != HS_NULL )
	{
		if( pEndpoint->CallbackReceiveH245Message(pStack,tCall->handle,eMsg) == FALSE )
		{
			delete_ControlMsg(eMsg);
			HSFree(eMsg);
			delete_QMTcpMessage(qMsg);
			delete_AsnStream(&tStrm);
			return HS_OK;
		}
	}

	switch( eMsg->choice )
	{
		case e_ASNH245MultimediaSystemControlMessageChoice_request:
			tRet=H245RequestHandler(eMsg->alter, pStack, pEndpoint, pCalls, pWaits, tCall);
			break;
		case e_ASNH245MultimediaSystemControlMessageChoice_response:
			tRet=H245ResponseHandler(eMsg->alter, pStack, pEndpoint, pCalls, pWaits, tCall);
			break;
		case e_ASNH245MultimediaSystemControlMessageChoice_command:
			tRet=H245CommandHandler(eMsg->alter, pStack, pEndpoint, pCalls, pWaits, tCall);
			break;
		case e_ASNH245MultimediaSystemControlMessageChoice_indication:
			tRet=H245IndicationHandler(eMsg->alter, pStack, pEndpoint, pCalls, pWaits, tCall);
			break;
		default:
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	/*closing*/
	if( tRet == HS_OK_SAVE_IN_TCS )
		tCall->remoteTCS = eMsg;
	else if( tRet == HS_OK_SAVE_IN_OLC )
		NoLockList_AttachOpenLogicalChannelSet( &(tCall->receivedOlcSetList), eMsg );
	else
	{
		delete_ControlMsg(eMsg);
		HSFree(eMsg);
	}

	delete_QMTcpMessage(qMsg);
	delete_AsnStream(&tStrm);
	return DischargeH245Message(pStack, pEndpoint, pWaits, tCall);
}


HS_RESULT H323DemonEventQ931Accepted(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls)
{
	ICall *tCall = HS_NULL;
	QMAccepted *qMsg = (QMAccepted*)pParam;

	if( qMsg==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pEndpoint==HS_NULL || pCalls==HS_NULL )
	{
		delete_QMAccepted(qMsg);
		return HS_ERR_NULL_PARAM;
	}
	if( (tCall=(ICall*)HSMalloc(sizeof(ICall))) == HS_NULL )
	{
		delete_QMAccepted(qMsg);
		return HS_ERR_H323_MALLOC;
	}

	new_ICall(tCall, HS_YES);
	tCall->handle = qMsg->handle;

	ICall_Make(tCall, pEndpoint);
	tCall->q931State = e_Q931State_ReadySetup;

	NoLockList_AttachData(pCalls, tCall);

	if( pEndpoint->CallbackCallIncoming != HS_NULL )
	{
		pEndpoint->CallbackCallIncoming(pStack,tCall);
	}

	delete_QMAccepted(qMsg);
	return HS_OK;
}


HS_RESULT H323DemonEventH245Accepted(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;
	QMAccepted *qMsg = (QMAccepted*)pParam;

	if( qMsg==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL )
		return HS_ERR_NULL_PARAM;
	if( (tChain=NoLockList_FindCallByHandle(pCalls,qMsg->handle)) == HS_NULL )
	{
		delete_QMAccepted(qMsg);
		return HS_ERR_H323_NOTFOUND;
	}
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
	{
		delete_QMAccepted(qMsg);
		return HS_ERR_H323_CONFLICT;
	}

	if( tCall->isTunneling == HS_NO )
	{
		tCall->h245State.session = HS_YES;

		if( tCall->h245State.sendTCS == HS_NO )
			NoLockList_AttachData( &(tCall->h245Charge), MakeTerminalCapabilitySet(pEndpoint, tCall) );

		if( tCall->h245State.sendMSD == HS_NO )
		{
			tCall->h245State.msdValue = (HS_UINT)(rand()%HS_MASTER_SLAVE_MAX_VALUE);
			NoLockList_AttachData( &(tCall->h245Charge), MakeMasterSlaveDetermination(pEndpoint, tCall->h245State.msdValue) );
		}

		DischargeH245Message(pStack, pEndpoint, pWaits, tCall);
	}

	delete_QMAccepted(qMsg);
	return HS_OK;
}


HS_RESULT H323DemonEventQ931Close(long pParam, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;
	HS_CALL_HANDLE tHandle = pParam;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( (tChain=NoLockList_FindCallByHandle(pCalls,tHandle)) == HS_NULL ) return HS_ERR_H323_NOTFOUND;
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL ) return HS_ERR_H323_CONFLICT;

	if( pEndpoint->CallbackException != HS_NULL )
		pEndpoint->CallbackException(e_exception_q931closed,HS_OK);

	return CallClosing(pStack, pEndpoint, pCalls, pWaits, tCall,
		e_CallCloseReason_Connection, e_ASNH225DisengageReasonChoice_undefinedReason,
		e_Q931CauseType_Congestion );
}


HS_RESULT H323DemonEventH245Close(long pParam, NoLockList *pCalls)
{
	ChainUnit *tChain = HS_NULL;
	HS_CALL_HANDLE tHandle = pParam;

	if( pCalls == HS_NULL ) return HS_ERR_NULL_PARAM;

	tChain = NoLockList_FindCallByHandle(pCalls, tHandle);
	if( tChain != HS_NULL )
	{
		ICall *tCall = (ICall*)(tChain->data);
		if( tCall != HS_NULL ) tCall->h245State.session = HS_NO;
	}

	return HS_OK;
}

