/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<RtpRcver.c>	2006-01-11,23:47

*/

#ifdef _WIN32_WCE
#include "codec/Typedef.h"
#include "codec/va_g729ab_intf.h"
#include "codec/OS_depend_func.h"
#else
#include <process.h>
#include "codec/va_g729.h"
#endif

#include "codec/g711.h"
#include "codec/gsm/gsm.h"
#include "codec/iLBC/iLBC_decode.h"

#include "RtpRcver.h"





typedef struct iLBC_Dec_Inst_t_*	iLBCDec;





/*
 *
 * RtpRcver member functions
 *
 */
HS_RESULT new_RtpRcver(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	pObj->mIsActive = FALSE;

	pObj->mSocket = INVALID_SOCKET;
	pObj->mLocalPort = HS_INVALID_TSAP_PORT;
	pObj->mSenderSocket = INVALID_SOCKET;

	pObj->mMyQ = HS_INVALID_QID;
	pObj->mSpeakerQ = HS_INVALID_QID;

	pObj->mType = e_RtpPayloadTypeMax;
	pObj->mFpp = 1;
	pObj->mSleep = 5;
	pObj->mSeqNum = 124;
	pObj->mTimeStamp = 1120;
	pObj->mSsrc = 0;

	pObj->mHold = FALSE;

#ifdef _WIN32_WCE
	pObj->m729Handle;
#endif
	pObj->mGsm = NULL;
	pObj->mIlbc = NULL;
	return HS_OK;
}


RtpRcver *newm_RtpRcver()
{
	RtpRcver *tResult = NULL;

	if( (tResult=(RtpRcver*)HSMalloc(sizeof(RtpRcver)))==NULL )
		return NULL;
	if( new_RtpRcver(tResult) != HS_OK )
	{
		HSFree(tResult);
		return NULL;
	}
	return tResult;
}


HS_RESULT delete_RtpRcver(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	if( pObj->mSocket != INVALID_SOCKET )
	{
		closesocket(pObj->mSocket);
		pObj->mSocket = INVALID_SOCKET;
	}

	return HS_OK;
}


HS_RESULT deletem_RtpRcver(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	delete_RtpRcver(pObj);
	HSFree(pObj);
	return HS_OK;
}


BOOL RtpRcver_IsActive(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;

	if( pObj==NULL ) return FALSE;
	return pObj->mIsActive;
}


HS_RESULT RtpRcver_OpenCodec(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mType)
	{
		case e_RtpPayloadType_Gsm:
			if( (pObj->mGsm=gsm_create())==NULL )
				return HS_ERR;
			break;
		case e_RtpPayloadType_G729:
#ifdef _WIN32_WCE
			if( (pObj->m729Handle=D_IF_g729ab_init())==NULL )
				return HS_ERR;
#else
			va_g729a_init_decoder();
#endif
			break;
		case e_RtpPayloadType_iLbc:
			if( (pObj->mIlbc=(void*)HSMalloc((unsigned)sizeof(struct iLBC_Dec_Inst_t_)))==NULL )
				return HS_ERR;
			initDecode((iLBCDec)(pObj->mIlbc),30/*ms*/,1);
			break;
		default:
			break;
	}

	return HS_OK;
}


HS_RESULT RtpRcver_CloseCodec(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mType)
	{
		case e_RtpPayloadType_Gsm:
			if( pObj->mGsm != NULL )
			{
				gsm_destroy((gsm)(pObj->mGsm));
				pObj->mGsm = NULL;
			}
			break;
		case e_RtpPayloadType_G729:
#ifdef _WIN32_WCE
			if( pObj->m729Handle != NULL )
			{
				D_IF_g729ab_exit(pObj->m729Handle);
				pObj->m729Handle = NULL;
			}
#endif
			break;
		case e_RtpPayloadType_iLbc:
			if( pObj->mIlbc != NULL )
			{
				HSFree(pObj->mIlbc);
				pObj->mIlbc = NULL;
			}
			break;
		default:
			break;
	}

	return HS_OK;
}


HS_UINT RtpRcver_GetFrameSize(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;

	if( pObj==NULL ) return 0;

	switch(pObj->mType)
	{
		case e_RtpPayloadType_Pcmu:
		case e_RtpPayloadType_Pcma:
			return HS_G711_PKT_SIZE;
		case e_RtpPayloadType_Gsm:
			return HS_GSM_PKT_SIZE;
		case e_RtpPayloadType_G729:
			return HS_G729_PKT_SIZE;
		case e_RtpPayloadType_iLbc:
			return HS_ILBC_PKT_SIZE;
		default:
			break;
	}
	return 0;
}


HS_RESULT RtpRcver_CheckRtpHeader(void *pObject,HS_UCHAR *pHeader)
{
	HS_USHORT tSeqNum = 0;
	HS_UINT tTimeStamp = 0;
	HS_UINT tSsrc = 0;
	RtpRcver *pObj = (RtpRcver*)pObject;

	if( pObj==NULL || pHeader==NULL )
		return HS_ERR_NULL_PARAM;
	if( pObj->mHold==TRUE )
		return HS_ERR;

#ifdef HS_BIG_ENDIAN
	memcpy((void*)(&tSeqNum),(void*)(pHeader+2),2);
	memcpy((void*)(&tTimeStamp),(void*)(pHeader+4),4);
	memcpy((void*)(&tSsrc),(void*)(pHeader+8),4);
#else
	((HS_UCHAR*)(&tSeqNum))[1] = pHeader[2];
	((HS_UCHAR*)(&tSeqNum))[0] = pHeader[3];
	((HS_UCHAR*)(&tTimeStamp))[3] = pHeader[4];
	((HS_UCHAR*)(&tTimeStamp))[2] = pHeader[5];
	((HS_UCHAR*)(&tTimeStamp))[1] = pHeader[6];
	((HS_UCHAR*)(&tTimeStamp))[0] = pHeader[7];
	((HS_UCHAR*)(&tSsrc))[3] = pHeader[8];
	((HS_UCHAR*)(&tSsrc))[2] = pHeader[9];
	((HS_UCHAR*)(&tSsrc))[1] = pHeader[10];
	((HS_UCHAR*)(&tSsrc))[0] = pHeader[11];
#endif

	if( pHeader[0] != 0x80 ) return HS_ERR;
	if( (pHeader[1] & 0x7F) != (HS_UCHAR)(pObj->mType) ) return HS_ERR;
	if( tSeqNum!=0 && tSeqNum <= pObj->mSeqNum ) return HS_ERR;
	if( tTimeStamp > 1000 && tTimeStamp <= pObj->mTimeStamp ) return HS_ERR;

	pObj->mSeqNum = tSeqNum;
	pObj->mTimeStamp = tTimeStamp;
	pObj->mSsrc = tSsrc;
	return HS_OK;
}


HS_UCHAR *RtpRcver_Decode(void *pObject,HS_UINT *pOut,HS_UCHAR *pIn,HS_UINT pSize)
{
	HS_UINT i, tSize = 0;
	HS_SHORT *tShort = NULL;
	HS_UCHAR *tResult = NULL;
	RtpRcver *pObj = (RtpRcver*)pObject;
#if _WIN32_WCE
	HS_UCHAR tBuffer[HS_G729_PKT_SIZE+6];
	tBuffer[0] = 0x02;
#endif

	*pOut = 0;
	if( pObj==NULL || pOut==NULL || pIn==NULL || pSize==0 ) return NULL;
	switch(pObj->mType)
	{
		case e_RtpPayloadType_Pcmu:
		case e_RtpPayloadType_Pcma:
			tSize = pObj->mFpp * HS_G711_PCM_SIZE;
			break;
		case e_RtpPayloadType_Gsm:
			tSize = pObj->mFpp * HS_GSM_PCM_SIZE;
			break;
		case e_RtpPayloadType_G729:
			tSize = pObj->mFpp * HS_G729_PCM_SIZE;
			break;
		case e_RtpPayloadType_iLbc:
			tSize = pObj->mFpp * HS_ILBC_PCM_SIZE;
			break;
		default:
			return NULL;
	}

	if( (tResult=(HS_UCHAR*)HSMalloc(tSize))==NULL ) return NULL;
	memset(tResult,0,tSize);
	switch(pObj->mType)
	{
		case e_RtpPayloadType_Pcmu:
		case e_RtpPayloadType_Pcma:
			if( G711Decode(pIn,pSize,tResult,&((int)tSize),pObj->mType) != 0 )
			{
				HSFree(tResult);
				return NULL;
			}
			*pOut = tSize;
			return tResult;
		case e_RtpPayloadType_Gsm:
			if( pObj->mGsm==NULL )
			{
				HSFree(tResult);
				return NULL;
			}
			for(i=0; i<pObj->mFpp; i++)
				gsm_decode((gsm)(pObj->mGsm),pIn+i*HS_GSM_PKT_SIZE,(short*)(tResult+i*HS_GSM_PCM_SIZE));
			*pOut = i*HS_GSM_PCM_SIZE;
			break;
		case e_RtpPayloadType_G729:
#ifdef _WIN32_WCE
			if( pObj->m729Handle==NULL )
			{
				HSFree(tResult);
				return NULL;
			}
#endif
			for(i=0; i<pObj->mFpp; i++)
			{
#ifdef _WIN32_WCE
				memcpy(tBuffer+1,pIn+i*HS_G729_PKT_SIZE,HS_G729_PKT_SIZE);
				if( D_IF_g729ab_decode(pObj->m729Handle,tBuffer,(short*)(tResult+i*HS_G729_PCM_SIZE),0) != 0) continue;
#else
				va_g729a_decoder(pIn+i*HS_G729_PKT_SIZE,(short*)(tResult+i*HS_G729_PCM_SIZE),0);
#endif
			}
			*pOut = i*HS_G729_PCM_SIZE;
			break;
		case e_RtpPayloadType_iLbc:
		{
			int j;
			float tFloat;
			float tBlock[BLOCKL_MAX];
			iLBCDec tDec = (iLBCDec)(pObj->mIlbc);

			tShort = (short*)tResult;
			if( tDec==NULL )
			{
				HSFree(tResult);
				return NULL;
			}

			iLBC_decode(tBlock,pIn,tDec,1);

			for(j=0; j<tDec->blockl; j++)
			{
				tFloat = tBlock[j];
				if( tFloat < MIN_SAMPLE )
					tFloat = MIN_SAMPLE;
				else if( tFloat > MAX_SAMPLE )
					tFloat = MAX_SAMPLE;

				tShort[j] = (short)tFloat;
			}
			*pOut = tDec->blockl * 2;
			break;
		}
		default:
			HSFree(tResult);
			return NULL;
	}

	return tResult;
}





/*
 *
 * RtpRcver thread functions
 *
 */
void RtpRcver_Demon(void *pObject)
{
	MSG tQmsg;
	BOOL tIsFirstPkt = TRUE;
	HS_UINT tFrameSize = 0;
	HS_UCHAR *tRaw = NULL;
	RtpRcver *pObj = (RtpRcver*)pObject;

	int tLen;
	HS_UCHAR tBuffer[1024];
	fd_set	tFdSet;
	int		tFdMax;
	struct timeval tTv;

	tTv.tv_sec=1;
	tTv.tv_usec=0;

	if( pObj==NULL ) return;
	if( RtpRcver_OpenCodec(pObj) != HS_OK ) return;
	if( (tFrameSize=RtpRcver_GetFrameSize(pObj))==0 ) return;

	pObj->mIsActive = TRUE;
	while(pObj->mIsActive)
	{
		/* check queue message
		*/
		if( PeekMessage(&tQmsg,NULL,HS_QM,HS_QM_RTP_MAX,PM_REMOVE) )
		{
            switch(tQmsg.message)
			{
				case HS_QM_STOP:
					pObj->mIsActive = FALSE;
					break;
				case HS_QM_CHECK:
					HSPrint("\nthread checking> RtpRcver-live");
					break;
				case HS_QM_RTP_SEND_SOCKET:
					pObj->mSenderSocket = (SOCKET)(tQmsg.lParam);
					break;
				case HS_QM_HOLD:
					pObj->mHold = TRUE;
					break;
				case HS_QM_RESUME:
					pObj->mHold = FALSE;
					break;
			}
		}
		if( pObj->mIsActive==FALSE ) continue;

		/* check recevie packet
		*/
		FD_ZERO(&tFdSet);
		FD_SET(pObj->mSocket,&tFdSet);
		tFdMax = pObj->mSocket+1;

		if( pObj->mSenderSocket != INVALID_SOCKET )
		{
			FD_SET(pObj->mSenderSocket,&tFdSet);
			if( tFdMax < (int)(pObj->mSenderSocket) )
				tFdMax = pObj->mSenderSocket +1;
		}

		if( (tLen=select(tFdMax+1,&tFdSet,NULL,NULL,&tTv)) < 1 ) continue;

		tLen = 0;
		if( FD_ISSET(pObj->mSocket,&tFdSet) != FALSE )
			tLen = recv(pObj->mSocket,tBuffer,1024,0);
		if( pObj->mSenderSocket != INVALID_SOCKET )
		{
			if( FD_ISSET(pObj->mSenderSocket,&tFdSet) != FALSE )
				tLen = recv(pObj->mSenderSocket,tBuffer,1024,0);
		}

		if( tLen < HS_RTP_HEADER_SIZE ) continue;
		if( (tLen-HS_RTP_HEADER_SIZE)%tFrameSize != 0 ) continue;
		if( RtpRcver_CheckRtpHeader(pObj,tBuffer) != HS_OK ) continue;

		if( tIsFirstPkt==TRUE )
		{
			pObj->mFpp = (tLen-HS_RTP_HEADER_SIZE)/tFrameSize;
			tIsFirstPkt = FALSE;
		}

		if( (tRaw=RtpRcver_Decode(pObj,(HS_UINT*)(&tLen),tBuffer+HS_RTP_HEADER_SIZE,tFrameSize*(pObj->mFpp)))==NULL ) continue;
		PostThreadMessage(pObj->mSpeakerQ,HS_QM_RTP,(WPARAM)tLen,(LPARAM)tRaw);
		tRaw = NULL;
	}

	RtpRcver_CloseCodec(pObj);
	return;
}


unsigned __stdcall RtpRcver_Thread(void* pArg)
{
	RtpRcver *pObj = (RtpRcver*)pArg;

	if( pObj==NULL )
	{
#ifdef _WIN32_WCE
		ExitThread(0);
#else
		_endthreadex(0);
#endif
		return 0;
	}
	if( pObj->mSocket==INVALID_SOCKET )
	{
#ifdef _WIN32_WCE
		ExitThread(0);
#else
		_endthreadex(0);
#endif
		return 0;
	}
	if( pObj->mSpeakerQ==HS_INVALID_QID )
	{
#ifdef _WIN32_WCE
		ExitThread(0);
#else
		_endthreadex(0);
#endif
		return 0;
	}

	/* thread
	*/
	RtpRcver_Demon(pObj);

	deletem_RtpRcver(pObj);
	HSPrint("\nRtpRcver-End");
	
#ifdef _WIN32_WCE
	ExitThread(0);
#else
	_endthreadex(0);
#endif
	return 0;
}


HS_QID RtpRcver_Start(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;
	if( pObj==NULL ) return HS_INVALID_QID;

#ifdef _WIN32_WCE
	CreateThread(NULL,0,&RtpRcver_Thread,pObj,0,&(pObj->mMyQ));
#else
	_beginthreadex(NULL,0,&RtpRcver_Thread,pObj,0,&(pObj->mMyQ));
#endif
	return pObj->mMyQ;
}


HS_RESULT RtpRcver_Stop(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_STOP,(WPARAM)0,(LPARAM)NULL);
	return HS_OK;
}


HS_RESULT RtpRcver_Hold(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_HOLD,(WPARAM)0,(LPARAM)NULL);
	return HS_OK;
}


HS_RESULT RtpRcver_Resume(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_RESUME,(WPARAM)0,(LPARAM)NULL);
	return HS_OK;
}


HS_RESULT RtpRcver_Check(void *pObject)
{
	RtpRcver *pObj = (RtpRcver*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_CHECK,(WPARAM)0,(LPARAM)NULL);
	return HS_OK;
}


HS_RESULT RtpRcver_SendSocket(void *pObject,SOCKET pSocket)
{
	RtpRcver *pObj = (RtpRcver*)pObject;
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->mMyQ == HS_INVALID_QID ) return HS_ERR;

	PostThreadMessage(pObj->mMyQ,HS_QM_RTP_SEND_SOCKET,(WPARAM)0,(LPARAM)pSocket);
	return HS_OK;
}


