/*

	<rc2.c>		2005-12-05,22:26

*/

#include <malloc.h>
#include <string.h>

#include "rc2.h"





#define RC2Xor(out,in1,in2)																\
{																						\
	((unsigned int*)(out))[0] = ((unsigned int*)(in1))[0] ^ ((unsigned int*)(in2))[0];	\
	((unsigned int*)(out))[1] = ((unsigned int*)(in1))[1] ^ ((unsigned int*)(in2))[1];	\
}





int new_Rc2Context(void *pObject)
{
	Rc2Context *pObj = (Rc2Context*)pObject;

	if( pObj==NULL ) return 2/*null param*/;

	pObj->mKey = NULL;
	pObj->mResult = NULL;
	pObj->mResultLen = 0;

	return 0/*ok*/;
}


Rc2Context *newm_Rc2Context()
{
	int tRet = 0;
	Rc2Context *tResult = NULL;

	if( (tResult=(Rc2Context*)HSMalloc(sizeof(Rc2Context)))==NULL ) return NULL;
	if( (tRet=new_Rc2Context(tResult)) != 0 )
	{
		HSFree(tResult);
		return NULL;
	}

	return tResult;
}


int delete_Rc2Context(void *pObject)
{
	Rc2Context *pObj = (Rc2Context*)pObject;

	if( pObj==NULL ) return 2/*null param*/;

	if( pObj->mKey != NULL )
	{
		HSFree(pObj->mKey);
		pObj->mKey = NULL;
	}
	if( pObj->mResult != NULL )
	{
		HSFree(pObj->mResult);
		pObj->mResult = NULL;
	}
	pObj->mResultLen = 0;

	return 0/*ok*/;
}


int deletem_Rc2Context(void *pObject)
{
	int tRet = 0;

	if( (tRet=delete_Rc2Context(pObject)) != 0 ) return tRet;

	HSFree(pObject);
	return 0;
}


/* implemented by RFC 2268 section 2. Key expension
*/
int Rc2Context_SetKey(void *pObject,unsigned char *pKey,unsigned int pByteLen,unsigned int pEffective)
{
	unsigned char *PITABLE;
	unsigned char _L[RC2_KEY_SIZE];
	unsigned int i, _T, _T1, _T8, _TM;
	unsigned char *tKeyPtr = _L;
	Rc2Context *pObj = (Rc2Context*)pObject;

	if( pObj==NULL || pKey==NULL || pByteLen==0 || pEffective==0 ) return 2/*null param*/;
	if( pObj->mKey != NULL )
	{
		HSFree(pObj->mKey);
		pObj->mKey = NULL;
	}
	if( (pObj->mKey=(RC2_KEY*)HSMalloc(sizeof(RC2_KEY)))==NULL ) return 3/*err HSMalloc*/;


	_T = pByteLen;
	_T1 = pEffective;
	_T8 = (_T1+7)/8;
	_TM = 0xFF % rc2_pow(2,(8+_T1-(8*_T8)));
	PITABLE = sBox;
	memcpy(_L,pKey,pByteLen);

	for(i=_T; i<RC2_KEY_SIZE; i++)
	{
		_L[i] = PITABLE[ (_L[i-1]+_L[i-_T]) % 256 ];
	}
	_L[128-_T8] = PITABLE[ _L[128-_T8] & _TM ];
	for(i=127-_T8; i<128; i--)
	{
		_L[i] = PITABLE[ _L[i+1] ^ _L[i+_T8] ];
	}

	for(i=0; i<RC2_KEY_SIZE_WORDS; i++)
	{
		pObj->mKey->key[i] = mgetLWord(tKeyPtr);
	}

	return 0/*ok*/;
}


static int Rc2EncryptEcb(unsigned char *pOutput, unsigned char *pInput, RC2_KEY *pKey)
{
	unsigned int word0, word1, word2, word3;
	unsigned int *tKey = NULL;
	unsigned char *bufPtr = NULL;

	if( pOutput==NULL || pInput==NULL || pKey==NULL ) return 2/*null param*/;
	tKey = pKey->key;
	bufPtr = pInput;

	word0 = mgetLWord( bufPtr );
	word1 = mgetLWord( bufPtr );
	word2 = mgetLWord( bufPtr );
	word3 = mgetLWord( bufPtr );

	encRound( word0, word1, word2, word3, tKey, 0 );
	encRound( word0, word1, word2, word3, tKey, 1 );
	encRound( word0, word1, word2, word3, tKey, 2 );
	encRound( word0, word1, word2, word3, tKey, 3 );
	encRound( word0, word1, word2, word3, tKey, 4 );

	addSboxes(word0, word1, word2, word3, tKey );
	encRound( word0, word1, word2, word3, tKey, 5 );
	encRound( word0, word1, word2, word3, tKey, 6 );
	encRound( word0, word1, word2, word3, tKey, 7 );
	encRound( word0, word1, word2, word3, tKey, 8 );
	encRound( word0, word1, word2, word3, tKey, 9 );
	encRound( word0, word1, word2, word3, tKey, 10 );

	addSboxes(word0, word1, word2, word3, tKey );
	encRound( word0, word1, word2, word3, tKey, 11 );
	encRound( word0, word1, word2, word3, tKey, 12 );
	encRound( word0, word1, word2, word3, tKey, 13 );
	encRound( word0, word1, word2, word3, tKey, 14 );
	encRound( word0, word1, word2, word3, tKey, 15 );

	mputLWord( pOutput, word0 );
	mputLWord( pOutput, word1 );
	mputLWord( pOutput, word2 );
	mputLWord( pOutput, word3 );

	return 0;
}


static int Rc2DecryptEcb(unsigned char *pOutput, unsigned char *pInput, RC2_KEY *pKey)
{
	unsigned int word0, word1, word2, word3;
	unsigned int *tKey = NULL;
	unsigned char *bufPtr = NULL;

	if( pOutput==NULL || pInput==NULL || pKey==NULL ) return 2/*null param*/;
	tKey = pKey->key;
	bufPtr = pInput;

	word0 = mgetLWord( bufPtr );
	word1 = mgetLWord( bufPtr );
	word2 = mgetLWord( bufPtr );
	word3 = mgetLWord( bufPtr );

	decRound( word0, word1, word2, word3, tKey, 15 );
	decRound( word0, word1, word2, word3, tKey, 14 );
	decRound( word0, word1, word2, word3, tKey, 13 );
	decRound( word0, word1, word2, word3, tKey, 12 );
	decRound( word0, word1, word2, word3, tKey, 11 );

	subSboxes(word0, word1, word2, word3, tKey );
	decRound( word0, word1, word2, word3, tKey, 10 );
	decRound( word0, word1, word2, word3, tKey, 9 );
	decRound( word0, word1, word2, word3, tKey, 8 );
	decRound( word0, word1, word2, word3, tKey, 7 );
	decRound( word0, word1, word2, word3, tKey, 6 );
	decRound( word0, word1, word2, word3, tKey, 5 );

	subSboxes(word0, word1, word2, word3, tKey );
	decRound( word0, word1, word2, word3, tKey, 4 );
	decRound( word0, word1, word2, word3, tKey, 3 );
	decRound( word0, word1, word2, word3, tKey, 2 );
	decRound( word0, word1, word2, word3, tKey, 1 );
	decRound( word0, word1, word2, word3, tKey, 0 );

	mputLWord( pOutput, word0 );
	mputLWord( pOutput, word1 );
	mputLWord( pOutput, word2 );
	mputLWord( pOutput, word3 );

	return 0;
}


int Rc2Context_EncryptEcb(void *pObject, unsigned char *pData, unsigned int pLen)
{
	unsigned int i, tBlockSize, tRemainLen;
	Rc2Context *pObj = (Rc2Context*)pObject;

	if( pObj==NULL || pData==NULL || pLen==0 ) return 2/*null param*/;
	if( pObj->mResult != NULL )
	{
		HSFree(pObj->mResult);
		pObj->mResult = NULL;
	}
	if( (pObj->mResult=(unsigned char*)HSMalloc(pLen))==NULL ) return 3/*err HSMalloc*/;
	pObj->mResultLen = pLen;

	tBlockSize = (unsigned int)(pLen/8);
	tRemainLen = (unsigned int)(pLen%8);

	for(i=0; i<tBlockSize; i++)
		Rc2EncryptEcb(pObj->mResult+i*8,pData+i*8,pObj->mKey);

	/* do not encrypt remain bytes.
	   remain bytes will be treated by API user.
	*/
	if( tRemainLen != 0 )
		memcpy(pObj->mResult+i*8,pData+i*8,tRemainLen);

	return 0;
}


int Rc2Context_DecryptEcb(void *pObject, unsigned char *pData, unsigned int pLen)
{
	unsigned int i, tBlockSize, tRemainLen;
	Rc2Context *pObj = (Rc2Context*)pObject;

	if( pObj==NULL || pData==NULL || pLen==0 ) return 2/*null param*/;
	if( pObj->mResult != NULL )
	{
		HSFree(pObj->mResult);
		pObj->mResult = NULL;
	}
	if( (pObj->mResult=(unsigned char*)HSMalloc(pLen))==NULL ) return 3/*err HSMalloc*/;
	pObj->mResultLen = pLen;

	tBlockSize = (unsigned int)(pLen/8);
	tRemainLen = (unsigned int)(pLen%8);

	for(i=0; i<tBlockSize; i++)
		Rc2DecryptEcb(pObj->mResult+i*8,pData+i*8,pObj->mKey);

	/* do not decrypt remain bytes.
	   remain bytes will be treated by API user.
	*/
	if( tRemainLen != 0 )
		memcpy(pObj->mResult+i*8,pData+i*8,tRemainLen);

	return 0;
}


int Rc2Context_EncryptCbc(void *pObject, unsigned char *pIv, unsigned char *pData, unsigned int pLen)
{
	unsigned char tXorer[8];
	unsigned int i, tBlockSize, tRemainLen;
	Rc2Context *pObj = (Rc2Context*)pObject;
	
	if( pObj==NULL || pIv==NULL || pData==NULL || pLen==0 ) return 2/*null param*/;
	if( pObj->mResult != NULL )
	{
		HSFree(pObj->mResult);
		pObj->mResult = NULL;
	}
	if( (pObj->mResult=(unsigned char*)HSMalloc(pLen))==NULL ) return 3/*err HSMalloc*/;
	pObj->mResultLen = pLen;

	tBlockSize = (unsigned int)(pLen/8);
	tRemainLen = (unsigned int)(pLen%8);

	memcpy(tXorer,pIv,8);
	for(i=0; i<tBlockSize; i++)
	{
		RC2Xor(pObj->mResult+i*8,pData+i*8,tXorer);
		Rc2EncryptEcb(pObj->mResult+i*8,pObj->mResult+i*8,pObj->mKey);
		memcpy(tXorer,pObj->mResult+i*8,8);
	}

	/* do not encrypt remain bytes.
	   remain bytes will be treated by API user.
	*/
	if( tRemainLen != 0 )
		memcpy(pObj->mResult+i*8,pData+i*8,tRemainLen);

	return 0;
}


int Rc2Context_DecryptCbc(void *pObject, unsigned char *pIv, unsigned char *pData, unsigned int pLen)
{
	unsigned char tXorer[8];
	unsigned int i, tBlockSize, tRemainLen;
	Rc2Context *pObj = (Rc2Context*)pObject;

	if( pObj==NULL || pIv==NULL || pData==NULL || pLen==0 ) return 2/*null param*/;
	if( pObj->mResult != NULL )
	{
		HSFree(pObj->mResult);
		pObj->mResult = NULL;
	}
	if( (pObj->mResult=(unsigned char*)HSMalloc(pLen))==NULL ) return 3/*err HSMalloc*/;
	pObj->mResultLen = pLen;

	tBlockSize = (unsigned int)(pLen/8);
	tRemainLen = (unsigned int)(pLen%8);

	memcpy(tXorer,pIv,8);
	for(i=0; i<tBlockSize; i++)
	{
		Rc2DecryptEcb(pObj->mResult+i*8,pData+i*8,pObj->mKey);
		RC2Xor(pObj->mResult+i*8,pObj->mResult+i*8,tXorer);
		memcpy(tXorer,pData+i*8,8);
	}

	/* do not encrypt remain bytes.
	   remain bytes will be treated by API user.
	*/
	if( tRemainLen != 0 )
		memcpy(pObj->mResult+i*8,pData+i*8,tRemainLen);

	return 0;
}


int Rc2Context_EncryptOfb(void *pObject, unsigned char *pIv, unsigned char *pData, unsigned int pLen)
{
	unsigned char tXorer[8];
	unsigned int i, j, tBlockSize, tRemainLen;
	Rc2Context *pObj = (Rc2Context*)pObject;

	if( pObj==NULL || pIv==NULL || pData==NULL || pLen==0 ) return 2/*null param*/;
	if( pObj->mResult != NULL )
	{
		HSFree(pObj->mResult);
		pObj->mResult = NULL;
	}
	if( (pObj->mResult=(unsigned char*)HSMalloc(pLen))==NULL ) return 3/*err HSMalloc*/;
	pObj->mResultLen = pLen;

	tBlockSize = (unsigned int)(pLen/8);
	tRemainLen = (unsigned int)(pLen%8);

	memcpy(tXorer,pIv,8);
	for(i=0; i<tBlockSize; i++)
	{
		Rc2EncryptEcb(tXorer,tXorer,pObj->mKey);
		RC2Xor(pObj->mResult+i*8,pData+i*8,tXorer);
	}

	if( tRemainLen != 0 )
	{
		Rc2EncryptEcb(tXorer,tXorer,pObj->mKey);
		for(j=0; j<tRemainLen; j++)
			pObj->mResult[i*8+j] = pData[i*8+j] ^ tXorer[j];
	}

	return 0;
}


int Rc2Context_DecryptOfb(void *pObject, unsigned char *pIv, unsigned char *pData, unsigned int pLen)
{
	return Rc2Context_EncryptOfb(pObject,pIv,pData,pLen);
}


int Rc2Context_EncryptCfb(void *pObject, unsigned char *pIv, unsigned char *pData, unsigned int pLen)
{
	unsigned char tXorer[8];
	unsigned int i, j, tBlockSize, tRemainLen;
	Rc2Context *pObj = (Rc2Context*)pObject;
	
	if( pObj==NULL || pIv==NULL || pData==NULL || pLen==0 ) return 2/*null param*/;
	if( pObj->mResult != NULL )
	{
		HSFree(pObj->mResult);
		pObj->mResult = NULL;
	}
	if( (pObj->mResult=(unsigned char*)HSMalloc(pLen))==NULL ) return 3/*err HSMalloc*/;
	pObj->mResultLen = pLen;

	tBlockSize = (unsigned int)(pLen/8);
	tRemainLen = (unsigned int)(pLen%8);

	memcpy(tXorer,pIv,8);
	for(i=0; i<tBlockSize; i++)
	{
		Rc2EncryptEcb(tXorer,tXorer,pObj->mKey);
		RC2Xor(pObj->mResult+i*8,pData+i*8,tXorer);
		memcpy(tXorer,pObj->mResult+i*8,8);
	}

	if( tRemainLen != 0 )
	{
		Rc2EncryptEcb(tXorer,tXorer,pObj->mKey);
		for(j=0; j<tRemainLen; j++)
			pObj->mResult[i*8+j] = pData[i*8+j] ^ tXorer[j];
	}

	return 0;
}


int Rc2Context_DecryptCfb(void *pObject, unsigned char *pIv, unsigned char *pData, unsigned int pLen)
{
	unsigned char tXorer[8];
	unsigned int i, j, tBlockSize, tRemainLen;
	Rc2Context *pObj = (Rc2Context*)pObject;
	
	if( pObj==NULL || pIv==NULL || pData==NULL || pLen==0 ) return 2/*null param*/;
	if( pObj->mResult != NULL )
	{
		HSFree(pObj->mResult);
		pObj->mResult = NULL;
	}
	if( (pObj->mResult=(unsigned char*)HSMalloc(pLen))==NULL ) return 3/*err HSMalloc*/;
	pObj->mResultLen = pLen;

	tBlockSize = (unsigned int)(pLen/8);
	tRemainLen = (unsigned int)(pLen%8);

	memcpy(tXorer,pIv,8);
	for(i=0; i<tBlockSize; i++)
	{
		Rc2EncryptEcb(tXorer,tXorer,pObj->mKey);
		RC2Xor(pObj->mResult+i*8,pData+i*8,tXorer);
		memcpy(tXorer,pData+i*8,8);
	}

	if( tRemainLen != 0 )
	{
		Rc2EncryptEcb(tXorer,tXorer,pObj->mKey);
		for(j=0; j<tRemainLen; j++)
			pObj->mResult[i*8+j] = pData[i*8+j] ^ tXorer[j];
	}

	return 0;
}


unsigned char *Rc2Context_GetResult(void *pObject, unsigned int *pLen)
{
	unsigned char *tResult = NULL;
	Rc2Context *pObj = (Rc2Context*)pObject;

	if( pObj==NULL ) return NULL;

	tResult = pObj->mResult;
	pObj->mResult = NULL;
	*pLen = pObj->mResultLen;
	pObj->mResultLen = 0;
	return tResult;
}
