#include "stdafx.h"
#include "granny/grntype.h"
#include "granny/grannyfile.h"
#include "granny/GrannyTextures.h"
#include "Config.h"
#include "Debug.h"
#include <iostream>
#include <cassert>

using namespace std;

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

// Utility Function
void calculateBoneRests( Bone *bone );

//static void toLower(char &c) { c = ::tolower(c); }


//  Just a pointer to a font style..
//  Fonts supported by GLUT are: GLUT_BITMAP_8_BY_13, 
//  GLUT_BITMAP_9_BY_15, GLUT_BITMAP_TIMES_ROMAN_10, 
//  GLUT_BITMAP_TIMES_ROMAN_24, GLUT_BITMAP_HELVETICA_10,
//  GLUT_BITMAP_HELVETICA_12, and GLUT_BITMAP_HELVETICA_18.
GLvoid *font_style = GLUT_BITMAP_9_BY_15;

static void printw (float x, float y, float z, char* format, ...)
{
    va_list arg_list;
    char str[256];
	int i;
    
    va_start(arg_list, format);
    vsprintf(str, format, arg_list);
    va_end(arg_list);
    
    glRasterPos3f (x, y, z);

    for (i = 0; str[i] != '\0'; i++)
        glutBitmapCharacter(font_style, str[i]);
}

extern int polycount;

cGrannyFile::cGrannyFile()
{
	m_texture = 0;
	m_initialized = false;
	m_texfilename = "";
	left_hand_bone = -1;
	right_hand_bone = -1;
	master_bone = -1;
}

cGrannyFile::~cGrannyFile()
{
	// Destroy the Texture we're binding
	//delete texture;
	if (pGrannyTextureLoader)
		pGrannyTextureLoader->FreeTexture(m_texfilename);
	m_texture = NULL;
}

bool isMasterName(const char *str)
{
	char *s = _strlwr(_strdup(str));
	bool ismaster = false;

	if (strstr(s, "master") || strstr(s, "mesh"))	
		ismaster = true;

	free(s);

	return ismaster;
}

// assing bone name, find master bone 
//  - bone.id -> bone.ojbptr -> bone.obj -> textid -> textchunk -> bone.name
void cGrannyFile::initBone()
{
	if (!m_initialized) return;

	std::vector<Bone *> &bones = getBones().bones;
	std::vector<dword> &boneObjects = getTies().boneObjects;
	std::vector<dword> &boneObjPtrs = getTies().boneObjPtrs;

	dword lodNm = findString("__ObjectName");

	int temp_lhand_bone = -1, temp_rhand_bone = -1;
	master_bone = left_hand_bone = right_hand_bone = -1;

	for (int i = 0; i < bones.size(); i++) {
		Bone* bone = bones[i];
		// bonetie.boneObjPtr: connect bone.id to boneObj
		dword objptr = boneObjPtrs[bone->id];
		// bonetie.boneObjects: connect bone.id to textchunk(bone name)
		dword obj = boneObjects[objptr-1];
		// objects.getValue(obj-1,key); objects[obj]->getValue(key);
		dword textid = getValue(obj,lodNm);
		// assign bone name
		bone->name = findID(textid);

		const char *s = bone->name.c_str();
		if (isMasterName(s) && master_bone == -1)
			master_bone = bone->id;
		else if (_stricmp(s, "CP_Grasp_Rhand") == 0)
			right_hand_bone  = bone->id;
		else if (_stricmp(s, "CP_Grasp_Lhand") == 0)
			left_hand_bone  = bone->id;
		else if (_stricmp(s, "Bip01 L Hand") == 0)
			temp_lhand_bone = bone->id;
		else if (_stricmp(s, "Bip01 R Hand") == 0)
			temp_rhand_bone = bone->id;
	}

#if 1
	// td anim, some aos anim(id 73) has not cp_grasp_ bone, we use hane bone.
	if (!nConfig::aos || left_hand_bone  == -1) 
		left_hand_bone  = temp_lhand_bone;
	if (!nConfig::aos || right_hand_bone == -1) 
		right_hand_bone = temp_rhand_bone;
#endif
}

int cGrannyFile::getBoneID(const char *name)
{
	if (!m_initialized) return -1;

	std::vector<Bone *> &bones = getBones().bones;

	if (isMasterName(name)) 
		return master_bone;

	for (int i = 0; i < bones.size(); i++) {
		Bone* bone = bones[i];
		if (_stricmp(bone->name.c_str(), name) == 0)
			return bone->id;
	}

	return -1;
}

void cGrannyFile::print()
{
	if (!m_initialized) return;

	//-------------------------------
	// print granny file info
	//------------------------------

	std::vector<Bone *> &bones = getBones().bones;
	std::list<Mesh> &meshes = getMeshes().meshes;
	std::vector<BoneTie *> &boneties = getTies().boneties;
	std::vector<dword> &boneObjects = getTies().boneObjects;
	std::vector<dword> &boneObjPtrs = getTies().boneObjPtrs;
	TextChunk &tchunk = getTextChunk();
	std::vector<Object*> &objects = getObjects().objects;
	
	Animations &anims = getAnimations();
	std::vector<BoneAnim> &boneAnims = anims.bones;

	pDebug.Log(LEVEL_INFO3, "granny file begin--------------------------------");
	pDebug.Log(LEVEL_INFO3, "granny file loading...: %s", m_filename.c_str());
	pDebug.Log(LEVEL_INFO3, "texture file: %s", getTextureName().c_str());
	pDebug.Log(LEVEL_INFO3, "mesh count: %d", meshes.size());
	pDebug.Log(LEVEL_INFO3, "bone count: %d", bones.size());
	pDebug.Log(LEVEL_INFO3, "bone ties count: %d",  boneties.size());
	pDebug.Log(LEVEL_INFO3, "bone object count: %d", boneObjects.size());
	pDebug.Log(LEVEL_INFO3, "bone object ptr count: %d", boneObjPtrs.size());
	pDebug.Log(LEVEL_INFO3, "bone anim count: %d length: %f", anims.size(), anims.length());


	if (nConfig::logMesh) {
		pDebug.Log(LEVEL_INFO2,"Mesh      --------------------------------");
		int i = 0;
		for (list<Mesh>::iterator it = meshes.begin(); it != meshes.end(); it++, i++) {
			Mesh& m = *it;
			pDebug.Log(LEVEL_INFO2,"mesh[%d] points:%d, normals: %d, polygons:%d, weights:%d, texturemap:%d", 
				i, m.points.size(), m.normals.size(), m.polygons.size(), m.weights.size(), m.textureMap.size());
			pDebug.Log(LEVEL_INFO2,"        bound: x(%+.3f~%+.3f) y(%+.3f~%+.3f) z(%+.3f~%+.3f)",
				m.minX, m.maxX, m.minY, m.maxY, m.minZ, m.maxZ);
		}
	}

	if (nConfig::logTextChunk) {
		pDebug.Log(LEVEL_INFO2,"textChunk    --------------------------------");
		for (int i=0; i < tchunk.size(); i++)
			pDebug.Log(LEVEL_INFO2, "text[%2d] \"%s\"", i, tchunk.findID(i).c_str());
	}

	dword lodNm = findString("__ObjectName");

	if (nConfig::logObject) {
		pDebug.Log(LEVEL_INFO2,"textKey     --------------------------------");

		pDebug.Log(LEVEL_INFO2, "__Standard    key: %2d", findString("__Standard"));
		pDebug.Log(LEVEL_INFO2, "__FileName    key: %2d", findString("__FileName"));
		pDebug.Log(LEVEL_INFO2, "__ObjectName  key: %2d", findString("__ObjectName"));
		pDebug.Log(LEVEL_INFO2, "__Description key: %2d", findString("__Description"));
		pDebug.Log(LEVEL_INFO2, "__Root        key: %2d", findString("__Root"));

		pDebug.Log(LEVEL_INFO2,"object     --------------------------------");

		pDebug.Log(LEVEL_INFO2, "object[id, key] value");
		for (int i = 0; i < objects.size(); i++) {
			Object *o = objects[i];
			for (int k = 0; k < o->numKeys; k++)
				pDebug.Log(LEVEL_INFO2, "object[%2d,%2d] %2d", i, o->keys[k], o->values[k]);
		}
	}

#if 0
	// not used
	pDebug.Log(LEVEL_INFO2, "bontiesBone--------------------------------");
	for (int i = 0; i < boneties.size(); i++) {
		BoneTie* tie= boneties[i];
		pDebug.Log(LEVEL_INFO2, "boneties[%2d] bone:%2d", i, tie->bone);
	}
#endif

	if (nConfig::logObjPtr) {
		pDebug.Log(LEVEL_INFO2, "boneObjPtr --------------------------------");
		for (int i = 0 ; i < boneObjPtrs.size(); i++) {
			int ptr = boneObjPtrs[i]-1;
			//int obj = boneObjects[ptr]-1;
			//int txt = getValue(obj,lodNm);
			pDebug.Log(LEVEL_INFO2, "boneobjptr[%2d] boneoject[%2d]", 
				i, ptr);
		}
	}

	if (nConfig::logBoneObj) {
		pDebug.Log(LEVEL_INFO2, "boneObject --------------------------------");
		for (int i = 0 ; i < boneObjects.size(); i++) {
			pDebug.Log(LEVEL_INFO2, "boneobject[%2d] object[%2d,%2d]", i, boneObjects[i]-1, lodNm);
		}
	}

	if (nConfig::logBoneAnim) {
		pDebug.Log(LEVEL_INFO2, "boneAnim   --------------------------------");
		for (int i = 0; i < boneAnims.size(); i++) {
			dword id = boneAnims[i].id;
			dword obj = boneObjects[id-1];
			dword textid = getValue(obj,lodNm);
			pDebug.Log(LEVEL_INFO2,
				"BoneAnim[%2d] : ObjPtr:%2d T: %2d Q: %2d U:%2d Len:%.3f \"%s\"", 
				i, id-1, 
				boneAnims[i].translates.size(),   // boneAnims[i].translateTimeline.size(), 
				boneAnims[i].quaternions.size(),  // boneAnims[i].quaternionTimeline.size(), 
				boneAnims[i].numUnknowns, boneAnims[i].length,
				findID(textid).c_str());
		}
	}

	if (nConfig::logBone) {
		pDebug.Log(LEVEL_INFO2, "Bone       --------------------------------");
		for (int i = 0; i < bones.size(); i++) {
			Bone* bone = bones[i];
			Point& p = bone->translate, q=bone->quaternion;

			// bonetie.boneObjPtr: bone.id  boneObj  
			dword objptr = boneObjPtrs[bone->id];
			// bonetie.boneObjects: bone.id  textchunk(bone name)  
			dword obj = boneObjects[objptr-1];
			// objects.getValue(obj-1,key); objects[obj]->getValue(key);
			dword textid = getValue(obj,lodNm);

			pDebug.Log(LEVEL_INFO2,"Bone[%2d] ID:%2d P:%2d ObjPtr:%2d Obj:%2d "
				"T(%+.3f, %+.3f, %+.3f) Q(%+.3f, %+.3f, %+.3f, %+.3f) \"%s\"", 
				i, bone->id, bone->parent, objptr-1, obj-1,
				p.points[0],p.points[1],p.points[2],
				q.points[0],q.points[1],q.points[2], q.points[3],
				//bone->name.c_str());
				findID(textid).c_str());
		}
	}
	pDebug.Log(LEVEL_INFO3, "granny file end  --------------------------------");
}

void cGrannyFile::load( std::string filename, std::string basepath )
{
	std::fstream * file = new fstream( filename.c_str(), ios::in|ios::binary );	

	if( !file->is_open() ) {
		pDebug.Log("Error: File not found: %s", filename.c_str());
		return;
	}

	m_stream = new cGrannyStream(file, filename);
	file->close();
	delete file;
	
	glPushMatrix();
	glLoadIdentity();

	m_stream->seekg( 0x40 );	// skip header (Could be FileType magic)

	dword chunk=m_stream->readDword();
	switch (chunk)
	{
	case 0xCA5E0000:
		mainChunk();
		break;
	default:
		{
			pDebug.Log(LEVEL_ERROR, "Unknown main Chunk: 0x%x", chunk);
			//assert(!"Unknown main Chunk:");
			return;
		}
	}

	if( getTextureName() != "" )
		loadTexture( basepath.c_str() );

	std::vector<Bone *> &bones = getBones().bones;
	if( bones.size() > 0 )
	{
		calculateBoneRests( bones[0] );
	}


	delete m_stream;
	m_stream = NULL;
	m_initialized = true;
	m_pathname = filename;
	m_filename = ::getFileName(filename);

	glPopMatrix();

	initBone();

	print();
}

bool cGrannyFile::export(std::string filename)
{
	if (!m_initialized) return false;

	pDebug.Log("exporting file... : %s.grn -> %s.smd", 
		::getFileName(m_filename).c_str(), ::getFileName(filename, '\\').c_str());

	FILE* fp = fopen(filename.c_str(), "wt");
	if (!fp) return false;

	std::vector<Bone *> bones = getBones().bones;
	BoneTies &boneTies = getTies();
	std::list<Mesh> &meshes = getMeshes().meshes;

	fprintf(fp, "version 1\n");
	fprintf(fp, "nodes\n");
	for (int i = 0; i < bones.size(); i++) {
		Bone* bone = bones[i];
		fprintf(fp, "%2d \"%s\" %2d\n", bone->id, bone->name.c_str(), i==0?-1:bone->parent);
	}
	fprintf(fp, "end\n");
	fprintf(fp, "skeleton\n");
	fprintf(fp, "time 0\n");

	for (int i = 0; i < bones.size(); i++) {
		Bone* bone = bones[i];
		Point p = QuaternionToEuler(bone->quaternion);
		float *t = bone->translate.points, *q= p.points;

		fprintf(fp, "%2d %.6f %.6f %.6f %.6f %.6f %.6f\n", 
			bone->id,  t[0]*40, t[1]*40, t[2]*40, q[0], q[1], q[2]);
	}

	fprintf(fp, "end\n");
	
	std::string textureName = getTextureName();
	std::list<Mesh>::iterator imesh = meshes.begin();
	Textures &textures = getTextures();
	if (imesh != meshes.end()) {
		fprintf(fp, "triangles\n");
		dword poly = 0;
		std::vector<gPolygon>::iterator polygon = imesh->polygons.begin();
		for (; polygon != imesh->polygons.end(); polygon++, poly++) 
		{
			fprintf(fp, "%s\n", textureName.c_str());
			for (int i = 0; i < 3; i++) {
				int node = polygon->nodes[i];
				dword bone = 1;

				if (imesh->weights.size() > 0) {
					assert(imesh->weights.size() > node);
					BoneWeight &weight =  imesh->weights[node];
					int maxBoneIdx = 0;
					float maxBoneWeight = 0.0f;
					for (int w = 0; w < weight.numWeights; w++) {
						if (maxBoneWeight < weight.weights[w]) maxBoneIdx = w;
					}
					bone = boneTies.boneties[weight.bones[maxBoneIdx]]->bone;
				}

				float *p = imesh->points[node].points;
				float *n = imesh->normals[node].points;
				float *t = imesh->textureMap[textures.polygons[poly].nodes[i+1]].points;

				fprintf(fp, "%2d %.6f %.6f %.6f %.6f %.6f %.6f %.6f %.6f\n",
					bone, p[0]*40, p[1]*40, p[2]*40, n[0], n[1], n[2], t[0], -t[1]);
			}
		}

		fprintf(fp, "end\n");
	}

	fclose(fp); fp = NULL;

	return true;
}

void cGrannyFile::addTime( float t )
{
}

string cGrannyFile::getTextureName()
{
	return m_object.getTextureName();
}

Meshes &cGrannyFile::getMeshes()
{
	return m_object.getMeshes();
}

Bones &cGrannyFile::getBones()
{
	return m_object.getBones();
}

BoneTies &cGrannyFile::getTies()
{
	return m_object.getTies();
}

Textures &cGrannyFile::getTextures()
{
	return m_object.getTextures();
}

BoneAnim &cGrannyFile::getBoneAnim( dword id )
{
	return m_object.getBoneAnim( id );
}

Animations &cGrannyFile::getAnimations()
{
	return m_object.getAnimations();
}

TextChunk &cGrannyFile::getTextChunk()
{
	return m_object.getTextChunk();
}

dword cGrannyFile::findString(string str)
{
	return m_object.findString(str);
}

dword cGrannyFile::getValue(dword obj,dword key)
{
	return m_object.getValue(obj,key);
}
dword cGrannyFile::findValue(dword key,dword value)
{
	return m_object.findValue(key,value);
}

std::string cGrannyFile::findID(dword id)
{
	return m_object.findID(id);
}

void cGrannyFile::mainChunk()
{
	dword children = m_stream->readDword();

	for( int i = 0; i < 6; ++i )
		m_stream->readDword(); // CRC?

	for( dword child = 0; child < children; ++child )
	{
		// Chunk Header
		dword chunk = m_stream->readDword();

		switch( chunk )
		{
			// Final Chunk (End-of-File?)
		case 0xCA5E0101:
			pDebug.Log(LEVEL_INFO1, "[loading final chunk...]");
			m_final.load( m_stream );
			break;

			// Copyright Chunk
		case 0xCA5E0102:
			pDebug.Log(LEVEL_INFO1, "[loading copyright chunk...]");
			m_copyright.load( m_stream );
			break;

			// Object Chunk
		case 0xCA5E0103:
			pDebug.Log(LEVEL_INFO1, "[loading object chunk...]");
			m_object.load( m_stream );
			break;
		default:
			{
				pDebug.Log(LEVEL_ERROR,"Unknown GRN Chunk: 0x%x", chunk);
				//assert(!"Unknown GRN Chunk");
				return;
			}
		}
	}
}

bool cGrannyFile::loadTexture( const char *basepath )
{
	std::string filename;

	if (m_texture && pGrannyTextureLoader)
		pGrannyTextureLoader->FreeTexture(m_texfilename);

	m_texture = NULL;
	m_texfilename = getTextureName();

	if (pGrannyTextureLoader) 
	{
		//pDebug.Log("texture load: %s\n", m_texfilename.c_str());
		m_texture = pGrannyTextureLoader->LoadTexture(m_texfilename);
	}

	return (m_texture);
}

void calculateBoneRests( Bone *bone )
{
	GrnMatrix matrix;
	matrix.setTransform(bone->quaternion, bone->translate);
	glMultMatrixf(matrix.matrix);

	glGetFloatv(GL_MODELVIEW_MATRIX,matrix.matrix);

	bone->matrix=matrix;
	bone->matrix.invert();
	bone->curMatrix=matrix;
	bone->curMatrix*=bone->matrix;

	bone->curQuaternion = bone->quaternion;
	bone->curTranslate = bone->translate;

	vector<Bone *>::iterator ibone;
	for (ibone=bone->children.begin();
		ibone!=bone->children.end();ibone++)
	{
		glPushMatrix();
		calculateBoneRests(*ibone);
		glPopMatrix();
	}
}


cDeformedArray * cGrannyFile::createDeformed (cGrannyFile * animation, float time, list<Mesh>::iterator imesh)
{
	cDeformedArray * deformed = NULL;

	Meshes &meshes = getMeshes();
	Bones &bones = getBones();
	Textures &textures = getTextures();
	BoneTies &boneTies = getTies();


	glPushMatrix();
	glLoadIdentity();
	if( animation )
		animation->getSkeleton( bones.bones[0], time ); 
	else
		calculateBoneRests(bones.bones[0]);
	glPopMatrix();

	vector<BoneWeight>::iterator iwt;
	dword pnt = 0;
	if (imesh->weights.size() > 0) {
		deformed = new cDeformedArray (imesh->weights.size());
		float * data = deformed->data();

		for( iwt = imesh->weights.begin(); iwt != imesh->weights.end(); ++iwt, ++pnt )
		{
			data[0] = 0;  
			data[1] = 0;  
			data[2] = 0;
			for(unsigned int wt = 0; wt < iwt->numWeights; ++wt )
			{
				Point post;
				float weight=iwt->weights[wt];
				dword bone=boneTies.boneties[iwt->bones[wt]]->bone;
				post=bones.bones[bone]->curMatrix*imesh->points[pnt];
				data[0]+=(post.points[0]*weight);
				data[1]+=(post.points[1]*weight);
				data[2]+=(post.points[2]*weight); 
			}

			data += 3;
		}
	} else if (imesh->points.size() > 0) {
		deformed = new cDeformedArray (imesh->points.size());
		float * data = deformed->data();

		for (unsigned int i = 0; i < imesh->points.size(); i++) {
			Point post;
			float weight=1.0f;
			post=bones.bones[1]->curMatrix*imesh->points[i];
			//post=imesh->points[i];
			*data = post.points[0]*weight; data++;
			*data = post.points[1]*weight; data++;
			*data = post.points[2]*weight; data++;
		}
	}

	if (animation) {
		deformed->matrix_left_hand = animation->matrix_left_hand;
		deformed->matrix_right_hand = animation->matrix_right_hand;
	}

	return deformed;
}

void cGrannyAnimation::drawSkeleton(Bone* bone, float curTime)
{
	float X,Y,Z,x,y,z,w;
	dword rid=0,mid=0;
	GrnMatrix matrix;

	//if (!m_animBones)
	//	return;


	if( (m_animBones.size() <= bone->id || m_animBones[ bone->id ] == (dword)-1))
	{
		//pDebug.Log("can't found id %d anim bone", bone->id);
		return;
	}
	else
	{
		BoneAnim &boneAnim = getBoneAnim( m_animBones[ bone->id ] );
		vector<float>::iterator movi, roti;

		//pDebug.Log("getSkeleton bone id : %d, boneAnim id : %d", bone->id, boneAnim.id);

		// Problem here, i suppose that some animations are looped back/forward
		if( curTime > boneAnim.length )
		{
			curTime = 0.0f;
		}

		for( roti = boneAnim.quaternionTimeline.begin();
			roti != boneAnim.quaternionTimeline.end() && (*roti) < curTime;
			roti++, rid++ );

		for( movi = boneAnim.translateTimeline.begin();
			movi != boneAnim.translateTimeline.end() && (*movi) < curTime;
			movi++, mid++ );

		Point t=boneAnim.translates[ mid ],q=boneAnim.quaternions[ rid ];

		X = t.points[ 0 ];
		Y = t.points[ 1 ];
		Z = t.points[ 2 ];

		// Interpolate when between Keyframes
		if( curTime != (*movi) )
		{
			float F1 = *movi;
			movi++;
			if (movi!=boneAnim.translateTimeline.end())
			{
				float F2 = *movi;
				float x2 = boneAnim.translates[mid+1].points[0];
				float y2 = boneAnim.translates[mid+1].points[1];
				float z2 = boneAnim.translates[mid+1].points[2];
				float tm = ( curTime - F1 ) / ( F2 - F1 );
				t.points[0] += tm * ( x2 - X );
				t.points[1] += tm * ( y2 - Y );
				t.points[2] += tm * ( z2 - Z );
			}
		}

		matrix.setTransform(q, t);

		Point p1(0,0,0),p2(0,0,0);
		p2=matrix*p1;

		// bone line
		glColor3f(1,1,0);
		glBegin(GL_LINES);
		glVertex3fv(p1.points);
		glVertex3fv(p2.points);
		glEnd();

		glMultMatrixf( matrix.matrix );

		// bone point
		glPointSize(5);
		glColor3f(1,0,0);
		glBegin(GL_POINTS);
		glVertex3f(0,0,0);
		glEnd();
		glPointSize(1);

		// bone name
		if (nConfig::printBoneName) {
			glColor3f(0.7f,0.7f,0.7f);
			printw(0,0,0,"(%d) %s", bone->id, bone->name.c_str());
		}

	}

	for(vector<Bone *>::iterator ibone = bone->children.begin(); ibone != bone->children.end(); ++ibone )
	{
		glPushMatrix();
		drawSkeleton( *ibone, curTime);
		glPopMatrix();
	}
}

void cGrannyFile::drawSkeleton(Bone* bone, float curTime)
{
	if (!bone) return;

	glPushMatrix();

	GrnMatrix m;
	m.setTransform(bone->quaternion, bone->translate);
	
	Point p1(0,0,0),p2;
	p2 = m*p1;

	// bone line
	glColor3f(1,1,0);
	glBegin(GL_LINES);
	glVertex3fv(p1.points);
	glVertex3fv(p2.points);
	glEnd();

	glMultMatrixf(m.matrix);

	// bone point
	glPointSize(5);
	glColor3f(1,0,0);
	glBegin(GL_POINTS);
	glVertex3f(0,0,0);
	glEnd();
	glPointSize(1);

	// bone name
	if (nConfig::printBoneName) {
		glColor3f(0.7f,0.7f,0.7f);
		printw(0,0,0,"(%d) %s", bone->id, bone->name.c_str());
	}
	
	vector<Bone *>::iterator ibone;
	for( ibone = bone->children.begin(); ibone != bone->children.end(); ++ibone )
	{
		drawSkeleton(*ibone, 0);	
	}
	glPopMatrix();
}

int cGrannyFile::getFrame (cGrannyFile * animation, float & curTime)
{
	if (animation) {
		if (curTime >= animation->length())
			curTime = 0.0f;
		if (animation->length() > 0.0f)
			return (int) (curTime / animation->length() * 25.0f);
	}
	return 0;
}

Point cGrannyFile::getModelOrg(float *w, float *h, float *d)
{
	Meshes &meshes = getMeshes();
	list<Mesh>::iterator imesh = meshes.meshes.begin();

	if (imesh == meshes.meshes.end()) return Point();

	if (w) *w = abs(imesh->maxX-imesh->minX);
	if (h) *h = abs(imesh->maxY-imesh->minY);
	if (d) *d = abs(imesh->maxZ-imesh->minZ);

	return Point(-(imesh->maxX+imesh->minX)/2, -(imesh->maxZ+imesh->minZ)/2, -(imesh->maxZ+imesh->minZ)/2);
}

void cGrannyFile::drawBoundBox(const Mesh &mesh)
{
	// bounding box
	Point p[24];

	p[0].points[0] = mesh.minX; p[0].points[1] = mesh.maxY; p[0].points[2] = mesh.minZ; 
	p[1].points[0] = mesh.maxX; p[1].points[1] = mesh.maxY; p[1].points[2] = mesh.minZ; 
	p[2] = p[1];
	p[3].points[0] = mesh.maxX; p[3].points[1] = mesh.minY; p[3].points[2] = mesh.minZ;
	p[4] = p[3];
	p[5].points[0] = mesh.minX; p[5].points[1] = mesh.minY; p[5].points[2] = mesh.minZ; 
	p[6] = p[5];
	p[7] = p[0];

	p[8].points[0]  = mesh.minX; p[8].points[1]  = mesh.maxY; p[8].points[2]  = mesh.maxZ; 
	p[9].points[0]  = mesh.maxX; p[9].points[1]  = mesh.maxY; p[9].points[2]  = mesh.maxZ; 
	p[10] = p[9];
	p[11].points[0] = mesh.maxX; p[11].points[1] = mesh.minY; p[11].points[2] = mesh.maxZ;
	p[12] = p[11];
	p[13].points[0] = mesh.minX; p[13].points[1] = mesh.minY; p[13].points[2] = mesh.maxZ; 
	p[14] = p[13];
	p[15] = p[8];

	p[16] = p[0];
	p[17] = p[8];
	p[18] = p[1];
	p[19] = p[9];
	p[20] = p[3];
	p[21] = p[11];
	p[22] = p[5];
	p[23] = p[13];

	glColor3f(0,0,1.0f);
	glBegin(GL_LINES);
	for (int i = 0; i < 24; i++)
		glVertex3fv(p[i].points);
	glEnd();
	glColor3f(1.0f,1.0f,1.0f);
}

void cGrannyFile::Render(cGrannyFile *animation, float & curTime)
{
	if (!initialized())
		return;

	cDeformedArray * deformed = NULL;

	int frame = getFrame(animation, curTime);

	float time = 0.0f;

	if (animation) {
		time = ((float) frame) / 25.0f * animation->length();
		deformed = animation->getDeformed(frame);
	}

	Meshes &meshes = getMeshes();
	Bones &bones = getBones();
	Textures &textures = getTextures();
	BoneTies &boneTies = getTies();
	list<Mesh>::iterator imesh;

	//if (meshes.meshes.size() != 1) {
	//	//pDebug.Log(LEVEL_ERROR, "Warning: More than one Granny Mesh is not supported!\n");
	//	return;
	//}

	imesh = meshes.meshes.begin();

	//glTranslatef(-(imesh->maxX+imesh->minX)/2, -(imesh->maxZ+imesh->minZ)/2, -(imesh->maxZ+imesh->minZ)/2);

	if (nConfig::drawBoundBox && imesh != meshes.meshes.end())
		drawBoundBox(*imesh);

	if (!deformed) {
		deformed = createDeformed(animation, time, imesh);
		if (animation)
			animation->addDeformed(deformed, frame);
	} else {
		if (animation) {
			animation->matrix_left_hand  = deformed->matrix_left_hand;
			animation->matrix_right_hand = deformed->matrix_right_hand;
		}
	}

	if (nConfig::drawBone)
	{
		glColor3f(1.0f,1.0f,0);

		if (animation) {
			animation->drawSkeleton(bones.bones[0],time);
			//animation->drawSkeleton(animation->getBones().bones[0],curTime);
		} else {
			drawSkeleton(getBones().bones[0],0);
		}
		
		glColor3f(1.0f,1.0f,1.0f);
	}

	// we would need to recalculate normals here.. screw that dude.
	// I doubt OSI does (although since I have yet to actually run UO:TD,
	// I don't know for sure).

	if (deformed) 
	{
		float * data = deformed->data();
		dword poly = 0;
		vector< gPolygon >::iterator polygon;
		
		glBindTexture( GL_TEXTURE_2D, getTexture() );
		
		glBegin( GL_TRIANGLES );

		float *vertex = NULL;
		
		for ( polygon = imesh->polygons.begin(); polygon != imesh->polygons.end(); polygon++, poly++ )
		{
			polycount ++;
			for ( int i = 0; i < 3; ++i )
			{
				if (imesh->normals.size() > 0 && 
					imesh->normals.size() > polygon->nodes[i])
				{
					//assert(imesh->normals.size() > polygon->nodes[i]); // crash id:834
					Point &p = imesh->normals[ polygon->nodes[ i ] ];
					glNormal3fv( p.points);
				}

				// Do we have texture-map information?
				if (imesh->textureMap.size() > 0 &&
					textures.polygons.size() > poly &&
					imesh->textureMap.size() > textures.polygons[ poly ].nodes[ i + 1 ])
				{
					//assert(textures.polygons[ poly ].nodes.size() > i + 1);	
					Point &p = imesh->textureMap[ textures.polygons[ poly ].nodes[ i + 1 ] ];
					glTexCoord2fv(p.points);
				
				}

				vertex = data + (polygon->nodes[i] * 3);
				glVertex3fv(vertex);
			}
		}

		glEnd();

		glBindTexture( GL_TEXTURE_2D, 0 );

		if (!animation)
			delete deformed;
	}
}

void cGrannyFile::getSkeleton( Bone *bone, float & curTime )
{
}

cGrannyAnimation::cGrannyAnimation()
{
//	cGrannyFile::cGrannyFile();
//	m_animBones = NULL;
	m_length = 0.0f;
	m_assignModel = NULL;
}

cGrannyAnimation::~cGrannyAnimation()
{
	std::map <int, cDeformedArray *>::iterator iter;
	for (iter = m_cache.begin(); iter != m_cache.end(); iter++)
		delete iter->second;
	m_cache.clear();

	//delete [] m_animBones;
}


// connect animation.bone to model.bone by bone.name
void cGrannyAnimation::Assign (cGrannyFile * model)
{
	if (!initialized() || !model || !model->initialized())
		return;

	m_assignModel = model;

	left_hand_bone  = model->left_hand_bone;
	right_hand_bone = model->right_hand_bone;

	BoneTies &boneTies = getTies();
	BoneTies &boneLodTies = model->getTies();
	Animations &anims = getAnimations();
	vector<BoneAnim> &animBones = anims.bones;
	//vector<Bone*> bones = getBones().bones;

	dword anmNm = findString("__ObjectName");
	dword lodNm = model->findString("__ObjectName");

	m_animBones.assign(model->getBones().bones.size(), (dword)-1);

	// for each animation bone
	for (int i = 0; i < animBones.size(); i++)
	{
		dword boneId = 0;
		dword animId = animBones[i].id; // objptr
		dword id = boneTies.boneObjects[ animId - 1 ]; // object
		dword textId = getValue( id, anmNm );
		std::string boneStr = findID( textId );
		//std::string boneStr = bones[animId-1]->name;

		boneId = model->getBoneID(boneStr.c_str());

		// model  bone.id   BoneAnim  
		if (boneId != (dword)-1)
			m_animBones[ boneId ] = i;

		pDebug.Log(LEVEL_INFO2, "Assign BoneAnim[%2d] to Model Bone :%2d, \"%s\"", i, boneId, boneStr.c_str());
	}

	for (int i = 0; i < m_animBones.size(); i++)
		pDebug.Log(LEVEL_INFO2, "Model Bone ID %2d : BoneAnim[%2d]", i, m_animBones[i]);

	m_length = m_object.getAnimLength();
}

void cGrannyAnimation::getSkeleton( Bone *bone, float & curTime )
{
	float X,Y,Z,x,y,z,w;
	dword rid=0,mid=0;
	GrnMatrix matrix;

	//if (!m_animBones || !bone)
	if (m_animBones.empty() || !bone)
		return;

#if 0
	if ((left_hand_bone  != -1) && ((int) bone->id == left_hand_bone)) {
		glPushMatrix();
		glRotatef(-135.0f, 1.0f, 0.0f, 0.0f);
		glRotatef(-45.0f, 0.0f, 1.0f, 0.0f);
		//glRotatef(45.0f, 0.0f, 0.0f, 1.0f);
		glGetFloatv(GL_MODELVIEW_MATRIX,matrix_left_hand.matrix);
		glPopMatrix();
	}

	if ((right_hand_bone != -1) && ((int) bone->id == right_hand_bone)) {
		glPushMatrix();
		glRotatef(90.0f, 1.0f, 0.0f, 0.0f);
		glRotatef(135.0f, 0.0f, 1.0f, 0.0f);
		//glRotatef(45.0f, 0.0f, 0.0f, 1.0f);
		glGetFloatv(GL_MODELVIEW_MATRIX,matrix_right_hand.matrix);
		glPopMatrix();
	}
#endif

	if( (m_animBones[ bone->id ] == (dword)-1))
	{
		//pDebug.Log("can't found id %d anim bone", bone->id);
		return;
	}
	else
	{
		BoneAnim &boneAnim = getBoneAnim( m_animBones[ bone->id ] );
		vector<float>::iterator movi, roti;

		//pDebug.Log("getSkeleton bone id : %d, boneAnim id : %d", bone->id, boneAnim.id);

		// Problem here, i suppose that some animations are looped back/forward
		if( curTime > boneAnim.length )
		{
			curTime = 0.0f;
		}

		for( roti = boneAnim.quaternionTimeline.begin();
			roti != boneAnim.quaternionTimeline.end() && (*roti) < curTime;
			roti++, rid++ );

		for( movi = boneAnim.translateTimeline.begin();
			movi != boneAnim.translateTimeline.end() && (*movi) < curTime;
			movi++, mid++ );

		Point t=boneAnim.translates[ mid ],q=boneAnim.quaternions[ rid ];

		X = t.points[ 0 ];
		Y = t.points[ 1 ];
		Z = t.points[ 2 ];

		// Interpolate when between Keyframes
		if( curTime != (*movi) )
		{
			float F1 = *movi;
			movi++;
			if (movi!=boneAnim.translateTimeline.end())
			{
				float F2 = *movi;
				float x2 = boneAnim.translates[mid+1].points[0];
				float y2 = boneAnim.translates[mid+1].points[1];
				float z2 = boneAnim.translates[mid+1].points[2];
				float tm = ( curTime - F1 ) / ( F2 - F1 );
				t.points[0] += tm * ( x2 - X );
				t.points[1] += tm * ( y2 - Y );
				t.points[2] += tm * ( z2 - Z );
			}
		}

		matrix.setTransform(q, t);

		bone->curQuaternion = q;
		bone->curTranslate = t;

		glMultMatrixf( matrix.matrix );
		glGetFloatv( GL_MODELVIEW_MATRIX, matrix.matrix );
		bone->curMatrix = matrix;
		bone->curMatrix *= bone->matrix;

		if ((left_hand_bone  != -1) && ((int) bone->id == left_hand_bone))
			matrix_left_hand = matrix;

		if ((right_hand_bone != -1) && ((int) bone->id == right_hand_bone))
			matrix_right_hand = matrix;
	}

	vector<Bone *>::iterator ibone;
	for( ibone = bone->children.begin(); ibone != bone->children.end(); ++ibone )
	{
		glPushMatrix();
		getSkeleton( *ibone, curTime);
		glPopMatrix();
	}
}

cDeformedArray * cGrannyAnimation::getDeformed (int index)
{
	std::map <int, cDeformedArray *>::iterator iter = m_cache.find(index);
	if (iter != m_cache.end()) 
		return iter->second;
	return NULL;
}

void cGrannyAnimation::addDeformed (cDeformedArray * deformed, int index)
{
	std::map <int, cDeformedArray *>::iterator iter = m_cache.find(index);
	if (iter != m_cache.end()) {
		delete iter->second;
		m_cache.erase(index);
		//pDebug.Log("cGrannyAnimation::addDeformed - id:%d mist!", index);
	}

	m_cache.insert(make_pair(index, deformed));

}

cDeformedArray::cDeformedArray(int size)
{
	assert(size > 0);
	m_data = new float[size * 3];
}


cDeformedArray::~cDeformedArray()
{
	delete [] m_data;
}


float * cDeformedArray::data()
{
	return m_data;
}

bool cGrannyAnimation::export(std::string filename)
{
	if (!m_initialized || !m_assignModel) return false;

	pDebug.Log("exporting file... : %s.grn -> %s.smd", 
		::getFileName(m_filename).c_str(), ::getFileName(filename, '\\').c_str());

	FILE* fp = fopen(filename.c_str(), "wt");
	if (!fp) return false;

	std::vector<Bone *> bones = getBones().bones;
	//Bones &assignBones = assignModel->getBones();

	fprintf(fp,"version 1\n");
	fprintf(fp,"nodes\n");

	for (int i = 0; i < bones.size(); i++) {
		Bone* bone = bones[i];
		fprintf(fp,"%2d \"%s\" %2d\n", bone->id, bone->name.c_str(), i==0?-1:bone->parent);
	}
	fprintf(fp,"end\n");
	fprintf(fp,"skeleton\n");

	std::vector<dword> tmp(m_animBones);
	m_animBones.clear();
	for (int i = 0; i < bones.size(); i++)
		m_animBones.push_back(bones[i]->id);

	for (int time = 0; time * 0.1 <= length(); time++) {
		fprintf(fp,"time %d\n", time);

		float curTime = time * 0.1;

		glPushMatrix();
		glLoadIdentity();
		calculateBoneRests(bones[0]);
		getSkeleton(bones[0], curTime);
		glPopMatrix();

		for (int i = 0; i < bones.size(); i++) {
			Bone* bone = bones[i];
			
			/*GrnMatrix matrix;
			matrix.setQuaternion(bone->curQuaternion);			
			Point p = matrix.getRotationRadians();*/

			//Point p = DegToRad(QuaternionToEuler(bone->curQuaternion));
			Point p = QuaternionToEuler(bone->curQuaternion);

			float *t = bone->curTranslate.points, *q= p.points;

			fprintf(fp,"%2d %.6f %.6f %.6f %.6f %.6f %.6f\n", 
				bone->id,  t[0]*40, t[1]*40, t[2]*40, q[0], q[1], q[2]);
		}
	}

	m_animBones.swap(tmp);

	fprintf(fp,"end\n");
	fclose(fp); fp = NULL;


	return true;
}

