using System;
using System.Drawing;
using System.Net.Sockets;
using System.Reflection;
using VNC.RFBProtocolHandling;
using System.Collections;
using System.Threading;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using DrawingSupport;
using VNC.RFBDrawing.UpdateDecoders;
using VNC.RFBDrawing.PixelDecoders;
using VNC.Config;

#region

// author: Dominic Ullmann, dominic_ullmann@swissonline.ch
// Version: 1.02
        
// VNC-Client for .NET
// Copyright (C) 2002 Dominic Ullmann

// author: Noh Seoung Hyun, gerranwizard@hotmail.com
// Version: 1.0

// VNC-Client for .NET Compact Framework
// Copyright (C) 2003 Noh Seoung Hyun

// This program is free software; 
// you can redistribute is and/or modify it under the terms of 
// the GNU General Public License as published by the Free Software Foundation;
// either version 2 of the License, or (at your option) any later version.

#endregion

namespace VNC.RFBDrawing 
{
	/// <summary>
	/// The RFBSurface class contains the data of the remote frame buffer and handles updates of the
	/// remote frame buffer
	/// </summary>
	public class RFBSurface 
	{ 
		/// <summary> reference to protocol-Handler </summary>
		private RFBProtocolHandler protocolHandler;		

		private int bufferWidth, bufferHeight;
		/// <summary> the width of the remote framebuffer </summary>
		public int BufferWidth 
		{
			get 
			{
				if (!connected) 
				{ 
					throw new Exception("not connected"); 
				}
			 
				return bufferWidth; 
			}
		}
		
		/// <summary> the height of the remote framebuffer </summary>
		public int BufferHeight 
		{
			get 
			{ 
				if (!connected) 
				{ 
					throw new Exception("not connected"); 
				}
				  
				return bufferHeight; 
			}
		}
		private byte depth;
		/// <summary> the depth selected for usage at client side. The depth is choosable by the client during connection handshake </summary>
		public byte Depth 
		{
			get 
			{ 
				if (!connected) 
				{ 
					throw new Exception("not connected"); 
				}
				  
				return depth; 
			}
		}

		/// <summary> the vies connected to this surface </summary>
		private ArrayList views = new ArrayList();
		//private System.Drawing.Imaging.PixelFormat format;
		/// <summary> decodes pixelvalues from stream </summary>
		private PixelDecoder pixDecod; 
		/// <summary> the drawSupport object: facility for drawing </summary>
		private DrawSupport drawSup;
		
		/// <summary> the decoders for decoding updates, O(1) access in normal case </summary>
		private Hashtable decoders = new Hashtable();
		/// <summary> the decoders sorted by the priority the client wishes to use them </summary>
		private ArrayList decoderPriorities = new ArrayList();
		
		private string serverName;
		/// <summary> the name of the server received during connection establishment </summary>
		public string ServerName 
		{
			get 
			{	
				if (!connected) 
				{ 
					throw new Exception("not connected"); 
				}
					
				return serverName; 
			}	
		}
		
		/// <summary> the configuration information </summary>
		private VNCConfiguration config;
		
		// server, port: the information needed for connection to the VNC-Server
		private String server;
		public String Server
		{
			get { return server; }
		}

		private int port;
		public int Port
		{
			get { return port; }
		}

		private bool connected;
		/// <summary> is this surface has an established connection to a VNC-Server </summary>
		public bool Connected 
		{
			get { return connected; }
		}
					
		/// <summary> constructor for an RFBSurface </summary>
		/// <param name="server">the server to connect to</param>
		/// <param name="port">the port to connect to</param>
		/// <param name="config">the configuration information read from the config-file</param>						
		public RFBSurface(String server, int port, VNCConfiguration config) 
		{
			this.server = server;
			this.port = port;
			this.config = config;
			// create the decoders, after the handshake is complete, the decoders are initalized (before starting listening to regular messages)
			createDecoders();			
		}		

		// ****************************************************************************************************************
		// ************************************ methods for connecting and disconnecting the surface 
		// ****************************************************************************************************************

		/// <summary> Register views to notify
		/// Views displays the content of the RFBSurface
		/// </summary>
		internal void connectSurfaceToView(RFBView view) 
		{
			Monitor.Enter(this);
			if (views.Contains(view)) { Monitor.Exit(this); throw new Exception("surface already connected to view"); }

			if (views.Count == 0) { // first view connected
				views.Add(view);				
				// connecting surface
				establishConnection(server, port);
				
				// creating draw-Support:
				drawSup = new DrawSupport(bufferWidth, bufferHeight, depth);	
			
				view.setRFBSize(bufferWidth, bufferHeight);

				// provide draw-Support Object to get access to drawing facilities
				view.setDrawSupport(drawSup);
				// initalize the decoders
				connectDecoders();

				// start listening for regular server-messages	
				protocolHandler.startMessageProcessing(); // now ready to receive noninitalization-messages from RFBServer!				
				// send update request to server, to get content of the RFB
				getFullUpdate();
			} 
			else 
			{ 
				// not the first view: enqueue only
				views.Add(view);
				view.setRFBSize(bufferWidth, bufferHeight);
				// provide draw-Support Object to get access to drawing facilities
				try 
				{
					view.setDrawSupport(drawSup);				
				} catch (Exception e) {
					// Exception: removing the view from the view-List, leaving the monitor! and rethrow exception
					disconnectView(view);
					Monitor.Exit(this);
					throw e;
				}
			}
			
			Monitor.Exit(this);			
		}

		/// <summary> disconnect a view from the surface </summary>
		internal void disconnectView(RFBView view) 
		{
			Monitor.Enter(this); /* synchronize with adding/notification */
			views.Remove(view);
			if (views.Count == 0) {
				// last view disconnected: closing connection
				closeConnection();
			}
			Monitor.Exit(this);
		}
		
		/// <summary> is called, when last view gets disconnected </summary>
		public void closeConnection() 
		{
			protocolHandler.closeConnection();
			drawSup.Dispose();
		}

		/// <summary> connects this RFBSurface to the RFBServer,
		/// afterwards this surface stores pixeldata and informs the server of
		/// events
		/// </summary>
		private void establishConnection(string server, int port)
		{
			// creating Connection-Handler for Connection to the RFB-Server
			protocolHandler = new RFBProtocolHandler(server, port, this);
			ServerData data = protocolHandler.handshake();

			bufferWidth = data.fbWidth;
			bufferHeight = data.fbHeight;
			serverName = data.serverName;
			depth = data.pixForm.depth;

			// creating buffer and decoder
			// chooses the format I like and sending setFormat to Server
			// decoder: strategy-pattern for short decoding routines
			switch (depth) 
			{
				case 16:
					pixDecod = new Pixel16bitDecoder();
					Console.WriteLine("16 bit modus selected");
					break;
				case 24:
					pixDecod = new Pixel24bitDecoder();
					Console.WriteLine("24 bit modus selected");
					break;
				case 32: 
					pixDecod = new Pixel32bitDecoder();
					Console.WriteLine("32 bit modus selected");
					break;
			}	
		
			// sending setPixelFormat-Message:
			// the format is defined by the choosen decoder
			// I will modify this region.
	//		protocolHandler.sendSetPixelFormat(pixDecod.getFormatDescription());
			// sends supported encodings
			protocolHandler.sendSetEncodings(decoderPriorities);

			// listening is started, when surface is connected to a View
			connected = true; // here the connection handshake is completed, connection to the server is ok
			Console.WriteLine("framebuffer size: " + bufferWidth + " " + bufferHeight);				
		}
		
		/// <summary> create the configured Decoders. </summary>
		/// <remarks> for extending the RFB-Protocol by an own decoder, add it to the decoder section
		/// in the VNCClient.Config.xml file.
		/// Use the fully qualified name of the decoder in the config-file.
		/// If the decoder is in a separate dll, append the name of the dll to the name after a comma,
		/// eg. VNC.RFBDrawing.UpdateDecoders.MyOwnDecoder,decoder for MyOwnDecoder in decoder.dll
		/// A decoder which should be usable here, must provide a no-argument constructor.
		/// </remarks>
		private void createDecoders() 
		{	
			// use the decoders from the config
			int nrOfDecodersWorking = 0;
			foreach (String dec in config.Decoders) 
			{
				// try to instantiate the decoder
				try 
				{
					Type decoderType = Type.GetType(dec, true);
					Decoder decoder = (Decoder) decoderType.Assembly.CreateInstance(decoderType.FullName);
					registerDecoder(decoder);
					nrOfDecodersWorking++;
					Console.WriteLine("created decoder " + dec + ": " + decoder);	
				} 
				catch (Exception e) 
				{
					Console.WriteLine("WARNING: error instantiating decoder " + dec + ": " + e);
				}
			}

			if (nrOfDecodersWorking == 0) { throw new Exception("no decoders could be instantiated!"); }
		}
		
		/// <summary> registers a Decoder in the order the client wishes to use it, highest priority decoder
		/// must be registered first
		/// </summary>
		private void registerDecoder(Decoder decoder) 
		{
			// add in list, sorted by priority the client wishes to use decoder
			decoderPriorities.Add(decoder);
		}


		/// <summary> connect the decoders to the DrawSupport instance; after completion of this method, the decoders are usable </summary>
		private void connectDecoders() 
		{
			foreach (Decoder decoder in decoderPriorities) 
			{
				decoder.initalize(this, protocolHandler.Stream, pixDecod, drawSup);
				// add for accessing decoder
				decoders[decoder.getEncodingNr()] = decoder;
			}
		}
		
		// *******************************************************************************
		// ****************** methods used during regular rfb-protocol message exchange
		// *******************************************************************************
		
		/// <summary> gets a full update from the server </summary>
		public void getFullUpdate() 
		{
			// send update request to server, to get content of the RFB
			protocolHandler.sendFBNonIncrementalUpdateRequest(0, 0, (int)bufferWidth, (int)bufferHeight);
		}
				
		/// <summary> notify interested views, changement discribed by x,y,width,height </summary>
		private void notifyView(int x, int y, int width, int height) 
		{
			Monitor.Enter(this); /* synchronize with adding/removing of views */
   			IEnumerator enumerator = views.GetEnumerator();
   			while (enumerator.MoveNext()) {
	   			((RFBView)enumerator.Current).notifyUpdate(x,y,width,height);
	   		}
	   		Monitor.Exit(this);
		}
		
		/// <summary>
		/// decodes a received update with the encoding encoding
		/// </summary>
		public void decodeUpdate(uint encoding, int xpos, int ypos, int width, int height) 
		{
			// get the decoder for this update
			Decoder dec = (Decoder)decoders[encoding];
			if (dec == null) 
			{
				Console.WriteLine("encoding not supported: " + encoding + "; check the VNCClient.config.xml if a decoder should be present!");	
				throw new Exception("encoding not Supported: " + encoding);
			}
			// update is decoded and drawn by the Decoder
			dec.decode(xpos, ypos, width, height);
		}

		/// <summary>
		/// this method decides what to do on an update receiving
		/// </summary>
		public void gotRFBUpdate()
		{
			// sending network-events as early as possible to reduce delay
			
			// sending a new update request to stay informed of changes of the remote frame buffer
			// incremental updated are ok, because the buffer contains a probably old, but valid content
			protocolHandler.sendFBIncrementalUpdateRequest(0, 0, (int)bufferWidth, (int)bufferHeight);
		}
				
		/// <summary>
		/// this method decides how to proceed after a successful update of the framebuffer
		/// </summary>
		public void updateDone(int minX, int minY, int maxX, int maxY) 
		{
			// draw the update to the screen!
			notifyView(minX, minY, maxX-minX, maxY-minY);
		}

		// --------------
		// Event-Handling
		// --------------
		/// <summary> handles a pointer event, used by the connected views to inform the surface </summary>
		public void handlePointerEvent(byte buttonMask, int x, int y) 
		{
			// for slow connections: it would be possible to drop some events, but this slows down faster
			// connections
			protocolHandler.sendPointerEvent(buttonMask, x, y);
		}
		
		/// <summary> handles a key event, used by the connected views to inform the surface </summary>
		public void handleKeyEvent(uint keySym, bool pressed) 
		{
			protocolHandler.sendKeyEvent(keySym, pressed);
		}		
	}
}